//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 10209 $
// $Date: 2013-09-23 10:03:49 +0200 (Mon, 23 Sep 2013) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Dsss Auto Correlation module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/Correlator/verilog/rtl/DsssAutoCorr.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module DsssAutoCorr #(parameter DATAINWIDTH  = 13, // Input data from TXRX FE and delay lines
                      parameter DATAOUTWIDTH = 20  // Accumulated AutoCorr output
                     )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  AGCClk,
            //Reset resynchronized to AGCClk
            input    wire                                  nAGCRst,

            //Control Signals
            //Enable for Dsss Auto Correlation block, from AGC FSM
            input    wire                                  DsssACBlockEn,
            //Add zeros instead of samples from delay block for first 0.8,1.6,2.4 or 3.2 us
            input    wire                                  AddZeroValue,
            //Selects Accumulation of Auto correlation values for 1 or 2 us
            input    wire                                  DsssACModeSel,
            //Enable for input data
            input    wire                                  DataInEn,
            //Data Input
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DQ,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn40DI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn40DQ,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn60DI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn60DQ,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Output Accumulated sum for AutoCorrelation
            output wire signed     [DATAOUTWIDTH-1:0]      AccSumI,
            output wire signed     [DATAOUTWIDTH-1:0]      AccSumQ,
            //Enable for output data
            output wire                                    DsssACDataEn
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DATAINWIDTH:0]     Mult1DataOutI;
wire    signed     [DATAINWIDTH:0]     Mult1DataOutQ;
wire    signed     [DATAINWIDTH:0]     Mult2DataOutI;
wire    signed     [DATAINWIDTH:0]     Mult2DataOutQ;
wire    signed     [DATAINWIDTH:0]     MuxMult2OutI;
wire    signed     [DATAINWIDTH:0]     MuxMult2OutQ;
wire                                   NullDataIn20DI;
wire                                   NullDataIn20DQ;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [DATAINWIDTH-1:0]   MultIp1I;
reg     signed     [DATAINWIDTH-1:0]   MultIp1Q;
reg                                    MultIp2I;
reg                                    MultIp2Q;
reg                                    NullDataMultIp2I;
reg                                    NullDataMultIp2Q;
reg     signed     [DATAINWIDTH+6:0]   AccSumDataI;
reg     signed     [DATAINWIDTH+6:0]   AccSumDataQ;
reg                                    DataInEn1D;
reg                                    DataInEn2D;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Data is registered with delayed version of DataInEn
  always @ (posedge AGCClk or negedge nAGCRst) begin:EnDelayBlk
    if (nAGCRst == 1'b0) begin
      DataInEn1D <= 1'b0;
      DataInEn2D <= 1'b0;
    end
    else begin
      if (DsssACBlockEn == 1'b0) begin
        DataInEn1D <= 1'b0;
        DataInEn2D <= 1'b0;
      end
      else begin
          DataInEn1D <= DataInEn;
          DataInEn2D <= DataInEn1D; 
      end
    end
  end //EnDelayBlk

//Multiplexer for input MultIp1 of second multiplier
  always @ (*) begin: MuxMultIp1
    case (DsssACModeSel)
      1'b0: begin
         MultIp1I = DataIn20DI;
         MultIp1Q = DataIn20DQ;
      end
      default: begin
         MultIp1I = DataIn40DI;
         MultIp1Q = DataIn40DQ;
      end
    endcase

  end //MuxMultIp1

//Input data is at 20Msps. Registering is done with delayed Enable signals
//Multiplexer for input MultIp2 of second multiplier
  always @ (*) begin: MuxMultIp2
    case (DsssACModeSel)
      1'b0: begin
         MultIp2I         = DataIn40DI[DATAINWIDTH-1];
         MultIp2Q         = DataIn40DQ[DATAINWIDTH-1];
         NullDataMultIp2I = (DataIn40DI == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;
         NullDataMultIp2Q = (DataIn40DQ == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;
      end
      default: begin
         MultIp2I         = DataIn60DI[DATAINWIDTH-1];
         MultIp2Q         = DataIn60DQ[DATAINWIDTH-1];
         NullDataMultIp2I = (DataIn60DI == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;
         NullDataMultIp2Q = (DataIn60DQ == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;
      end
    endcase

  end //MuxMultIp2

//Mult1
  ACQuantMult # (.DATAINWIDTH(DATAINWIDTH))
    U_ACQuantMult1(
                .DataInI(DataInI),
                .DataInQ(DataInQ),
                .QuantDataInI(DataIn20DI[DATAINWIDTH-1]),
                .QuantDataInQ(DataIn20DQ[DATAINWIDTH-1]),
                .NullDataInI(NullDataIn20DI),
                .NullDataInQ(NullDataIn20DQ),
                .DataOutI(Mult1DataOutI),
                .DataOutQ(Mult1DataOutQ)
                );

  assign NullDataIn20DI = (DataIn20DI == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;
  assign NullDataIn20DQ = (DataIn20DQ == $signed({{DATAINWIDTH}{1'b0}})) ? 1'b1 : 1'b0;

//Mult2
  ACQuantMult # (.DATAINWIDTH(DATAINWIDTH))
    U_ACQuantMult2(
                .DataInI(MultIp1I),
                .DataInQ(MultIp1Q),
                .QuantDataInI(MultIp2I),
                .QuantDataInQ(MultIp2Q),
                .NullDataInI(NullDataMultIp2I),
                .NullDataInQ(NullDataMultIp2Q),
                .DataOutI(Mult2DataOutI),
                .DataOutQ(Mult2DataOutQ)
                );

//Muxing the Output of Mult2 with zero when AddZeroValue is set to 1
//This is to avoid invalid inputs from the delay line initially
  assign MuxMult2OutI = (AddZeroValue) ? $signed({{DATAINWIDTH+1}{1'b0}}): Mult2DataOutI;
  assign MuxMult2OutQ = (AddZeroValue) ? $signed({{DATAINWIDTH+1}{1'b0}}): Mult2DataOutQ;

//Delay. Registering the data path
  always @ (posedge AGCClk or negedge nAGCRst) begin:Delay1Blk
    if (nAGCRst == 1'b0) begin
      AccSumDataI <= $signed({{(DATAINWIDTH+7)}{1'b0}});
      AccSumDataQ <= $signed({{(DATAINWIDTH+7)}{1'b0}});
    end
    else begin
      if (DsssACBlockEn == 1'b0) begin
        AccSumDataI <= $signed({{(DATAINWIDTH+7)}{1'b0}});
        AccSumDataQ <= $signed({{(DATAINWIDTH+7)}{1'b0}});
      end
      else if (DataInEn1D == 1'b1) begin
        AccSumDataI <= $signed({{6{Mult1DataOutI[DATAINWIDTH]}},Mult1DataOutI[DATAINWIDTH:0]}
                       -{{6{MuxMult2OutI[DATAINWIDTH]}},MuxMult2OutI[DATAINWIDTH:0]})
                       +AccSumDataI;
        AccSumDataQ <= $signed({{6{Mult1DataOutQ[DATAINWIDTH]}},Mult1DataOutQ[DATAINWIDTH:0]}
                       -{{6{MuxMult2OutQ[DATAINWIDTH]}},MuxMult2OutQ[DATAINWIDTH:0]})
                       +AccSumDataQ;
      end
    end
  end //Delay1Blk

//Output Assignment
  assign AccSumI = AccSumDataI;
  assign AccSumQ = AccSumDataQ;

  assign DsssACDataEn = DataInEn2D;

endmodule // DsssAutoCorr

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
