//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 10209 $
// $Date: 2013-09-23 10:03:49 +0200 (Mon, 23 Sep 2013) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : OFDM Auto Correlation top module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/Correlator/verilog/rtl/OfdmAutoCorrTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module OfdmAutoCorrTop #(parameter NRX = 4,   // Number of RX chains
                         parameter DATAINWIDTH  = 13,  // Input data from TXRX FE and delay lines
                         parameter DATAOUTWIDTH = 22   // Accumulated AutoCorr output
                        )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  AGCClk,
            //Reset resynchronized to AGCClk
            input    wire                                  nAGCRst,

            //Control Signals
            //Enable for Ofdm Auto Correlation block, from AGC FSM
            input    wire                                  OfdmACBlockEn,
            //Force first 0.8,1.6,2.4 or 3.2us of outputs as zero, from AGC FSM
            input    wire                                  SetZeroOutput,
            //Add zeros instead of samples from delay block for first 0.8,1.6,2.4 or 3.2 us
            input    wire                                  AddZeroValue,
            //Selects Accumulation of Auto correlation values for 0.8,1.6,2.4 or 3.2 us
            input    wire              [1:0]               OfdmACModeSel,
            //Enable for input data
            input    wire                                  DataInEn,

            //Data Input RxChain1
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DQ1,

            //Data Input RxChain2
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DQ2,

            //Data Input RxChain3
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DI3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DQ3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DI3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DQ3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DI3,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DQ3,

            //Data Input RxChain4
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DI4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn32DQ4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DI4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn48DQ4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DI4,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn64DQ4,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Output Accumulated sum for AutoCorrelation
            output   wire          [DATAOUTWIDTH-1:0]      OfdmACAccSum,
            //Enable for output data
            output   wire                                  OfdmACDataEn
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam DIV3 = 10'd341;          // Division by 3 = *341>>>10

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [DATAINWIDTH+8:0]   ModApproxOut;
wire    signed     [DATAINWIDTH+8:0]   SumAccDataI;
wire    signed     [DATAINWIDTH+8:0]   SumAccDataQ;
wire               [DATAINWIDTH+8:0]   MuxModApprox;
wire               [DATAINWIDTH+18:0]  MuxModApproxDiv3;

wire               [3:0]               DataOutEn;
wire    signed     [DATAINWIDTH+6:0]   AccSumI   [3:0];
wire    signed     [DATAINWIDTH+6:0]   AccSumQ   [3:0];

wire    signed     [DATAINWIDTH-1:0]   DataInI[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataInQ[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn16DI[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn16DQ[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn32DI[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn32DQ[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn48DI[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn48DQ[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn64DI[3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn64DQ[3:0];

//Genvar
genvar                                 j,i;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [DATAINWIDTH+8:0]   RegSumI;
reg     signed     [DATAINWIDTH+8:0]   RegSumQ;
reg                [DATAINWIDTH+8:0]   RegMuxModApprox;
reg                [4:0]               DelDataOutEn;
reg                [4:0]               DelSetZeroOutput;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
//Assign Inputs to Array variable
  assign DataInI[0]    = DataInI1;
  assign DataInQ[0]    = DataInQ1;
  assign DataIn16DI[0] = DataIn16DI1;
  assign DataIn16DQ[0] = DataIn16DQ1;
  assign DataIn32DI[0] = DataIn32DI1;
  assign DataIn32DQ[0] = DataIn32DQ1;
  assign DataIn48DI[0] = DataIn48DI1;
  assign DataIn48DQ[0] = DataIn48DQ1;
  assign DataIn64DI[0] = DataIn64DI1;
  assign DataIn64DQ[0] = DataIn64DQ1;

  assign DataInI[1]    = DataInI2;
  assign DataInQ[1]    = DataInQ2;
  assign DataIn16DI[1] = DataIn16DI2;
  assign DataIn16DQ[1] = DataIn16DQ2;
  assign DataIn32DI[1] = DataIn32DI2;
  assign DataIn32DQ[1] = DataIn32DQ2;
  assign DataIn48DI[1] = DataIn48DI2;
  assign DataIn48DQ[1] = DataIn48DQ2;
  assign DataIn64DI[1] = DataIn64DI2;
  assign DataIn64DQ[1] = DataIn64DQ2;

  assign DataInI[2]    = DataInI3;
  assign DataInQ[2]    = DataInQ3;
  assign DataIn16DI[2] = DataIn16DI3;
  assign DataIn16DQ[2] = DataIn16DQ3;
  assign DataIn32DI[2] = DataIn32DI3;
  assign DataIn32DQ[2] = DataIn32DQ3;
  assign DataIn48DI[2] = DataIn48DI3;
  assign DataIn48DQ[2] = DataIn48DQ3;
  assign DataIn64DI[2] = DataIn64DI3;
  assign DataIn64DQ[2] = DataIn64DQ3;

  assign DataInI[3]    = DataInI4;
  assign DataInQ[3]    = DataInQ4;
  assign DataIn16DI[3] = DataIn16DI4;
  assign DataIn16DQ[3] = DataIn16DQ4;
  assign DataIn32DI[3] = DataIn32DI4;
  assign DataIn32DQ[3] = DataIn32DQ4;
  assign DataIn48DI[3] = DataIn48DI4;
  assign DataIn48DQ[3] = DataIn48DQ4;
  assign DataIn64DI[3] = DataIn64DI4;
  assign DataIn64DQ[3] = DataIn64DQ4;

//Data is registered with delayed version of DataInEn
  always @ (posedge AGCClk or negedge nAGCRst) begin:EnDelayBlk
    if (nAGCRst == 1'b0) begin
      DelDataOutEn     <= 5'd0;
      DelSetZeroOutput <= 5'd0;
    end
    else begin
      if (OfdmACBlockEn == 1'b0) begin
        DelDataOutEn     <= 5'd0;
        DelSetZeroOutput <= 5'd0;
      end
      else begin
        DelDataOutEn[0] <= DataOutEn[0];
        DelDataOutEn[4:1] <= DelDataOutEn[3:0];
        DelSetZeroOutput[0] <= SetZeroOutput;
        DelSetZeroOutput[4:1] <= DelSetZeroOutput[3:0];
      end
    end
  end //EnDelayBlk

generate
  for (j=0; j<NRX; j=j+1) begin: GenOfdmAutoCorr
    OfdmAutoCorr # (.DATAINWIDTH(DATAINWIDTH),
                    .DATAOUTWIDTH(DATAINWIDTH+7))
      U_OfdmAutoCorr(
                   //Inputs
                   .AGCClk(AGCClk),
                   .nAGCRst(nAGCRst),
                   .OfdmACBlockEn(OfdmACBlockEn),
                   .AddZeroValue(AddZeroValue),
                   .OfdmACModeSel(OfdmACModeSel),
                   .DataInEn(DataInEn),
                   .DataInI(DataInI[j]),
                   .DataInQ(DataInQ[j]),
                   .DataIn16DI(DataIn16DI[j]),
                   .DataIn16DQ(DataIn16DQ[j]),
                   .DataIn32DI(DataIn32DI[j]),
                   .DataIn32DQ(DataIn32DQ[j]),
                   .DataIn48DI(DataIn48DI[j]),
                   .DataIn48DQ(DataIn48DQ[j]),
                   .DataIn64DI(DataIn64DI[j]),
                   .DataIn64DQ(DataIn64DQ[j]),

                   //Outputs
                   .OfdmACDataEn(DataOutEn[j]),
                   .AccSumI(AccSumI[j]),
                   .AccSumQ(AccSumQ[j])
                   );
  end // GenOfdmAutoCorr
endgenerate

//Adding the outputs from all the RX chains
generate
  for(i=NRX;i<4;i=i+1) begin: GenAdder

     assign AccSumI[i]   = $signed({{(DATAINWIDTH+7)}{1'b0}});
     assign AccSumQ[i]   = $signed({{(DATAINWIDTH+7)}{1'b0}});
     assign DataOutEn[i] = 1'b0;

  end //GenAdder
endgenerate

  assign SumAccDataI =  $signed({{2{AccSumI[0][DATAINWIDTH+6]}},AccSumI[0][DATAINWIDTH+6:0]} +
                        {{2{AccSumI[1][DATAINWIDTH+6]}},AccSumI[1][DATAINWIDTH+6:0]} +
                        {{2{AccSumI[2][DATAINWIDTH+6]}},AccSumI[2][DATAINWIDTH+6:0]} +
                        {{2{AccSumI[3][DATAINWIDTH+6]}},AccSumI[3][DATAINWIDTH+6:0]});

  assign SumAccDataQ =  $signed({{2{AccSumQ[0][DATAINWIDTH+6]}},AccSumQ[0][DATAINWIDTH+6:0]} +
                        {{2{AccSumQ[1][DATAINWIDTH+6]}},AccSumQ[1][DATAINWIDTH+6:0]} +
                        {{2{AccSumQ[2][DATAINWIDTH+6]}},AccSumQ[2][DATAINWIDTH+6:0]} +
                        {{2{AccSumQ[3][DATAINWIDTH+6]}},AccSumQ[3][DATAINWIDTH+6:0]});

//Registering Sum Data
  always @ (posedge AGCClk or negedge nAGCRst) begin: RegSumData
    if (nAGCRst == 1'b0) begin
      RegSumI <= $signed({{(DATAINWIDTH+9)}{1'b0}});
      RegSumQ <= $signed({{(DATAINWIDTH+9)}{1'b0}});
    end
    else begin
      if (OfdmACBlockEn == 1'b0) begin
        RegSumI <= $signed({{(DATAINWIDTH+9)}{1'b0}});
        RegSumQ <= $signed({{(DATAINWIDTH+9)}{1'b0}});
      end
      else begin
        RegSumI <= SumAccDataI;
        RegSumQ <= SumAccDataQ;
      end
    end
  end // RegSumData

//Mod Approx
  ModApprox # (
              .SUM_WIDTH(DATAINWIDTH+9),
              .CORRVAL_WIDTH(DATAINWIDTH+9)
              )
              U_ModApprox (
                //Inputs
                .PhyClk(AGCClk),
                .nPhyRst(nAGCRst),
                .ISum(RegSumI),
                .QSum(RegSumQ),
                .ComputeOn(OfdmACBlockEn),

                //Output
                .CorrVal(ModApproxOut)
              );

//Muxing the output
  assign MuxModApprox = (DelSetZeroOutput[4])? {{(DATAINWIDTH+9)}{1'b0}}:
                          ModApproxOut;

// Division by 3 for sliding window length set to 2.4us
  assign MuxModApproxDiv3 = MuxModApprox * {{DATAINWIDTH-1{1'b0}},DIV3};

//Registering Mux Data. 2 clock delay is added to make latency 7 clocks for
//all correlation modules
  always @ (posedge AGCClk or negedge nAGCRst) begin: RegOutData
    if (nAGCRst == 1'b0) begin
      RegMuxModApprox <= {{(DATAINWIDTH+9)}{1'b0}} ;
    end
    else begin
      if (OfdmACBlockEn == 1'b0) begin
        RegMuxModApprox <= {{(DATAINWIDTH+9)}{1'b0}};
      end
      else if (DelDataOutEn[3] == 1'b1)begin
        if (OfdmACModeSel == 2'h0)
          RegMuxModApprox <= MuxModApprox;
        else if (OfdmACModeSel == 2'h1)
          RegMuxModApprox <= {1'b0, MuxModApprox[DATAINWIDTH+8:1]};
        else
          RegMuxModApprox <= MuxModApproxDiv3[DATAINWIDTH+18:10];
      end
    end
  end // RegOutData

//Output Assignment
  assign OfdmACAccSum = RegMuxModApprox;
  assign OfdmACDataEn = DelDataOutEn[4];

endmodule // OfdmAutoCorrTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
