//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 25573 $
// $Date: 2016-03-31 17:54:36 +0200 (Thu, 31 Mar 2016) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : DSSS Cross correlator
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/RIUCORE/AGC/DSSSContCorr/verilog/rtl/DSSSContCrossCorr.v $
//
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module DSSSContCrossCorr
(
  // Clock and reset
  input  wire        reset_n,       // Global reset.
  input  wire        clk,           // Clock (80 Mhz)

  // Control
  input  wire        cb_en,
  input  wire [2:0]  cb_config,

  // Input data
  input  wire [14:0] data_i_in,
  input  wire [14:0] data_q_in,
  input  wire        data_valid,

  // Signal quality
  output reg  [21:0] cb_bc
  );


//////////////////////////////////////////////////////////////////////////////
// Local Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// Width of the accumulator
localparam ACCU_SIZE_G = 22;

// Kc coefs
localparam [17:0] MULT_COEF2_CT  = 18'd2;
localparam [17:0] MULT_COEF4_CT  = 18'd4;
localparam [17:0] MULT_COEF6_CT  = 18'd6;
localparam [17:0] MULT_COEF8_CT  = 18'd8;
localparam [17:0] MULT_COEF10_CT = 18'd10;
localparam [17:0] MULT_COEF12_CT = 18'd12;

// 32-Kc coefs
localparam [19:0] MULT_COEF20_CT = 20'd20;
localparam [19:0] MULT_COEF22_CT = 20'd22;
localparam [19:0] MULT_COEF24_CT = 20'd24;
localparam [19:0] MULT_COEF26_CT = 20'd26;
localparam [19:0] MULT_COEF28_CT = 20'd28;
localparam [19:0] MULT_COEF30_CT = 20'd30;

// This value is used to count up to 79. (80 MHz -> 1MHz)
localparam [6:0] COUNT_DSSS_CT  = 7'b1001111;

// This value is used as the symbol cyclic synchronization signal.
localparam [6:0] SYNC_SYMBOL_CT = 7'b0000000;

//////////////////////////////////////////////////////////////////////////////
// Internal Regs declarations
//////////////////////////////////////////////////////////////////////////////

  reg        data_valid_ff1;
  reg        data_valid_ff2;
  reg        data_valid_ff3;

  // Accumulator bank of registers.
  reg [19:0] accu_bank [ACCU_SIZE_G-1:0];

  // Signals for the three max values in the register bank.
  reg [19:0] max_value1;
  reg [19:0] max_value2;
  reg [19:0] max_value3;
  // Sum of the three max values.
  reg [21:0] max_sum_next;

  reg [24:0] accu_add_mult;

  reg [21:0] mult_coef;

  // Signals for the symbol counter (counts 1 MHz in DSSS).
  reg [6:0]  symbol_count;
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  wire [17:0] mult_mux;
  wire [19:0] mult32_coeff;

  wire [29:0] data_mult_i;
  wire [29:0] data_mult_q;
  wire [30:0] squared_sum;
  wire [21:0] squared_sum_cut_9lsb;
  wire [17:0] squared_sum_sat;
  wire [20:0] loop_in;
  wire [21:0] accu_add_int;
  wire [19:0] accu_add_sat;
  wire [19:0] accu_add_cut;
  
  integer     i,j,k;
  
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // Delay line
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      data_valid_ff1 <= 1'b0;
      data_valid_ff2 <= 1'b0;
      data_valid_ff3 <= 1'b0;
    end
    else begin
      data_valid_ff1 <= data_valid;
      data_valid_ff2 <= data_valid_ff1;
      data_valid_ff3 <= data_valid_ff2;
    end
  end

  // Accumulator bank input value.
  assign data_mult_i = $signed({data_i_in[14],data_i_in}) * $signed({data_i_in[14],data_i_in});
  assign data_mult_q = $signed({data_q_in[14],data_q_in}) * $signed({data_q_in[14],data_q_in});
  assign squared_sum = {1'b0,data_mult_i} + {1'b0,data_mult_q};

  // Cut 9 lsb
  assign squared_sum_cut_9lsb = squared_sum[30:9];

  // Sat to 18-bits
  SatUnsigned # (
    .INPUT_WIDTH(22),
    .OUTPUT_WIDTH(18)
    )
    u_SatSum (
      .InputData(squared_sum_cut_9lsb),
      .SatData(squared_sum_sat)
      );

  // Mult coeff from config mode
  assign mult_mux = (cb_config == 3'd5) ? MULT_COEF12_CT :
                    (cb_config == 3'd4) ? MULT_COEF10_CT :
                    (cb_config == 3'd3) ? MULT_COEF8_CT  :
                    (cb_config == 3'd2) ? MULT_COEF6_CT  :
                    (cb_config == 3'd1) ? MULT_COEF4_CT  :
                                          MULT_COEF2_CT;

  // Mult by coef
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      mult_coef <= 22'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff1)
          mult_coef <= squared_sum_sat * mult_mux;
      end
      else
        mult_coef <= 22'b0;
    end
  end

  // Cut 1 lsb
  assign loop_in = mult_coef[21:1];
  
  // Accumulator
  assign accu_add_int = {1'b0,loop_in} + {2'b0,accu_add_cut};
  
  // Accumulator with saturation
  SatUnsigned # (
    .INPUT_WIDTH(22),
    .OUTPUT_WIDTH(20)
    )
    u_SatAccu (
      .InputData(accu_add_int),
      .SatData(accu_add_sat)
      );

  // Register bank update process.
  // This process accumulates the absolute values in a shift register bank.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      // Reset accumulator bank.
      for (i=0; i<ACCU_SIZE_G; i=i+1) begin
        accu_bank[i] <= 20'b0;
      end
    end
    else begin
      if (cb_en) begin
        if (data_valid_ff2) begin
          // Left shift of accumulator register bank.
          for (k=1; k<ACCU_SIZE_G; k=k+1) begin
            accu_bank[k] <= accu_bank[k-1];
          end
          // Store new input value.
          accu_bank[0]   <= accu_add_sat;
        end
      end
      else begin
        // Reset accumulator bank.
        for (j=0; j<ACCU_SIZE_G; j=j+1) begin
          accu_bank[j] <= 20'b0;
        end
      end
    end
  end

  assign mult32_coeff = (cb_config == 3'd5) ? MULT_COEF20_CT :
                        (cb_config == 3'd4) ? MULT_COEF22_CT :
                        (cb_config == 3'd3) ? MULT_COEF24_CT :
                        (cb_config == 3'd2) ? MULT_COEF26_CT :
                        (cb_config == 3'd1) ? MULT_COEF28_CT :
                                              MULT_COEF30_CT;

  // Mult into loop.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      accu_add_mult <= 25'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff2)
          accu_add_mult <= accu_bank[21] * mult32_coeff;
      end
      else
        accu_add_mult <= 25'b0;
    end
  end

  assign accu_add_cut = accu_add_mult[24:5];

  // Peak detection process.
  // This process detects the three maximum accumulated values over a symbol
  // period. The synchronization is done on SYNC_SYMBOL_CT, as for the
  // index.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      // Reset max values.
      max_value1 <= 20'b0;
      max_value2 <= 20'b0;
      max_value3 <= 20'b0;
    end
    else begin
      // Reset max values every Symbol period. At the same time, max_value1 is
      // updated with the current accu bank input (first max value of the next
      // period).
      if (cb_en) begin
        if (symbol_count == SYNC_SYMBOL_CT) begin
          max_value1  <= accu_add_sat; 
          max_value2  <= 20'b0;
          max_value3  <= 20'b0;
        end
        else if (data_valid_ff3) begin // 22 MHz.
        
          // The data input to the register bank is compared to the three max
          // values. If it is greater than one of those, it is inserted in the
          // corresponding register and the others max values or reordered.
          // 
          //  new value range| < max3 | [max3, max2[ | [max2, max1[ | >= max1
          //  -------------------------------------------------------------// 
          //     next_max1   |  max1  |     max1     |     max1     |   new 
          //     next_max2   |  max2  |     max2     |     new      |   max1
          //     next_max3   |  max3  |     new      |     max2     |   max2
                
          // If the data takes the same value twice, keep the last index (use >=).
          if (accu_add_sat >= max_value3) begin
            if (accu_add_sat >= max_value1) begin         // Max of maxima detected.
              max_value1 <= accu_add_sat;      // Load new max values.
              max_value2 <= max_value1;
              max_value3 <= max_value2;
            end
            else if (accu_add_sat >= max_value2) begin
              max_value2 <= accu_add_sat;      // Load new max value.
              max_value3 <= max_value2;
            end
            else
              max_value3 <= accu_add_sat;      // Load new max value.
          end
        end
      end
      else begin
        // Reset max values.
        max_value1 <= 20'b0;
        max_value2 <= 20'b0;
        max_value3 <= 20'b0;
      end

    end
  end

  // Counter process.
  // This counter is used to obtain from the 80 Mhz clock:
  //   a 1 Mhz sampling in DSSS mode (counts up to 79)
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      symbol_count <= {7{1'b1}};
    else begin
      if (cb_en) begin
        // Symbol counter operation.
        if (symbol_count == COUNT_DSSS_CT)
          symbol_count <= 7'b0;
        else  
          symbol_count <= symbol_count + 7'd1;
      end
      else // Reset the counter.
      symbol_count <= {7{1'b1}};
    end
  end

  // Adder for the three max values.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      max_sum_next <= 22'b0;
    else begin
      if (cb_en) begin
        if (symbol_count == SYNC_SYMBOL_CT)
          max_sum_next <= {2'b0,max_value1} + {2'b0,max_value2} + {2'b0,max_value3};
      end
      else
        max_sum_next <= 22'b0;
    end
  end
  
  // Generation of the Signal Quality.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      cb_bc <= 22'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff3)
          cb_bc <= max_sum_next;  // Store signal quality.
      end
      else
        cb_bc <= 22'b0;
    end
  end
 
endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////

`default_nettype wire
