//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 25524 $
// $Date: 2016-03-30 19:11:02 +0200 (Wed, 30 Mar 2016) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Reference level path of DSSS continuous correlator
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/RIUCORE/AGC/DSSSContCorr/verilog/rtl/DSSSRefLevel.v $
//
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module DSSSRefLevel
(
  // Clock and reset
  input  wire        reset_n,       // Global reset.
  input  wire        clk,           // Clock (80 Mhz)

  // Control
  input  wire        cb_en,
  input  wire [2:0]  cb_config,

  // Input data
  input  wire [9:0]  data_i_i,
  input  wire [9:0]  data_q_i,
  input  wire        data_valid,

  // Reference level
  output wire [21:0] cb_rl,
  output wire        cb_rl_valid
  );

//////////////////////////////////////////////////////////////////////////////
// Internal Regs declarations
//////////////////////////////////////////////////////////////////////////////

  reg        data_valid_ff1;
  reg        data_valid_ff2;
  reg        data_valid_ff3;
  
  reg [4:0]  counter_1MHz;
  reg [3:0]  counter_k_mult_const;
  reg [3:0]  counter_g_mult_const;

  reg        floor_ceil_choice;
  
  reg [24:0] in_loop;
  reg [21:0] in_accu_sat_ff;

  reg [31:0] accu_mult_32;
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  wire [19:0] data_mult_i;
  wire [19:0] data_mult_q;
  wire [19:0] squared_sum;

  wire [6:0]  const_k;
  
  wire [17:0] squared_module_cut;

  wire [28:0] in_accu;
  wire [22:0] in_accu_floor;
  wire [22:0] in_accu_ceil;
  wire [22:0] in_accu_6lsb;
  wire [21:0] in_accu_sat;

  wire [5:0]  const_g;
  wire [31:0] accu_shift;
  wire [27:0] accu_mult_g;
  wire [27:0] accumulator;
  
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // Delay line
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      data_valid_ff1 <= 1'b0;
      data_valid_ff2 <= 1'b0;
      data_valid_ff3 <= 1'b0;
    end
    else begin
      data_valid_ff1 <= data_valid;
      data_valid_ff2 <= data_valid_ff1;
      data_valid_ff3 <= data_valid_ff2;
    end
  end

  // Creation of squared_sum
  // i2 + q2 @ 22MS/s
  assign data_mult_i = $signed({data_i_i[9],data_i_i})*$signed({data_i_i[9],data_i_i});
  assign data_mult_q = $signed({data_q_i[9],data_q_i})*$signed({data_q_i[9],data_q_i});

  // ||x||2
  assign squared_sum = data_mult_i + data_mult_q;

  // cut 2LSB
  assign squared_module_cut = squared_sum[19:2];

  // Counter to deliver mult_k one after another at 1 MHz
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      counter_1MHz         <= 5'b0;
      counter_k_mult_const <= 4'b0;
    end
    else begin
      if (cb_en) begin
        if (data_valid_ff1) begin
          if (counter_1MHz == 5'd21) begin
            if (counter_k_mult_const < (4'd9 - {1'b0,cb_config}))
              counter_k_mult_const <= counter_k_mult_const + 4'd1;
            counter_1MHz <= 5'b0;
          end
          else
            counter_1MHz <= counter_1MHz + 5'd1;
        end
      end
      else begin
        counter_1MHz         <= 5'b0;
        counter_k_mult_const <= 4'b0;
      end
    end
  end

  // LUT for K.
  MultK u_MultK (
      // Clocks & Reset
      .clk              (clk),
      .reset_n          (reset_n),
      // Controls
      .counter_k        (counter_k_mult_const),
      // Test control signal
      .k_value          (const_k)
      );

  // const_k * squared_module_cut
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      in_loop <= 25'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff1)
          in_loop <= squared_module_cut * {11'b0,const_k};
      end
      else
        in_loop <= 25'b0;
    end
  end

  // Accumulator
  assign in_accu = {4'b0,in_loop} + {1'b0,accumulator};
  
  // Floor/ceil 6lsb alternatively
  assign in_accu_floor = in_accu[28:6];
  assign in_accu_ceil  = (in_accu[5:0]==6'b0) ? in_accu_floor : in_accu_floor + 23'd1;

  // floor_ceil_choice select at 22MHz alternatively the in_accu_floor or the
  // in_accu_ceil signal.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      floor_ceil_choice <= 1'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff3)
          floor_ceil_choice <= ~ floor_ceil_choice;
      end
      else
        floor_ceil_choice <= 1'b0;
    end
  end

  assign in_accu_6lsb = (floor_ceil_choice) ? in_accu_floor : in_accu_ceil;

  // Saturation to 22-bit
  SatUnsigned # (
    .INPUT_WIDTH(23),
    .OUTPUT_WIDTH(22)
    )
    u_SatAccu (
      .InputData(in_accu_6lsb),
      .SatData(in_accu_sat)
      );

  // Delay line
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      in_accu_sat_ff <= 22'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff2)
          in_accu_sat_ff <= in_accu_sat;
      end
      else 
        in_accu_sat_ff <= 22'b0;
    end
  end

  // Ouput assignment
  assign cb_rl       = in_accu_sat_ff;
  assign cb_rl_valid = data_valid_ff3;

  // Counter to deliver g : 1 cycle (at 22MHz) after counter_k_mult_const
  // counter_g_mult_const is at 1MHz.
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      counter_g_mult_const <= 4'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff3)
          counter_g_mult_const <= counter_k_mult_const;
      end
      else
        counter_g_mult_const <= 4'b0;
    end
  end

  // LUT for G.
  MultG u_MultG (
      // Clocks & Reset
      .clk              (clk),
      .reset_n          (reset_n),
      // Controls
      .counter_g        (counter_g_mult_const),
      // Test control signal
      .g_value          (const_g)
      );

  // Goal is to do (1024-const_g)*in_accu_sat
  // First clock cycle do 1024*in_accu_sat (=shift 10) and const_g*in_accu_sat
  assign accu_shift  = {in_accu_sat_ff, 10'b0};
  assign accu_mult_g = in_accu_sat_ff * {16'b0,const_g};
  
  // Second clock cycle do substraction
  always @ (posedge clk or negedge reset_n)
  begin
    if (reset_n == 1'b0)
      accu_mult_32 <= 32'b0;
    else begin
      if (cb_en) begin
        if (data_valid_ff3)
          accu_mult_32 <= accu_shift - {4'b0,accu_mult_g};
      end
      else
        accu_mult_32 <= 32'b0;
    end
  end
  
  // Round by 4
  USgnRound # (
    .INPUT_WIDTH(32),
    .OUTPUT_WIDTH(28)
    )
    u_AccuRound (
      .InputData(accu_mult_32),
      .RoundData(accumulator)
      );

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////

`default_nettype wire
