////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 19268 $
// $Date: 2015-05-06 17:38:19 +0200 (Wed, 06 May 2015) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : ADC Power Estimation Block for AGC
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/PowerEst/verilog/rtl/ADCPowEst.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

`default_nettype none

module ADCPowEst #(parameter ADC_WIDTH       = 12, //Input Data Width
                   parameter BW_PARAM        = 80, //Bandwidth (20/40/80)
                   parameter ADCPOWSAT_WIDTH = 14  //Width of Power Estimation for Sat Output
                  )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active Low Reset
            input    wire                                  ADCPowClk, //ADC Power Clock (max of 40/80/160MHz support)
                          
            //Control Signals
            input    wire                                  AGCEn,   //Block Enable
            input    wire                                  Force20, //Force20 Mode
            input    wire                                  Force40, //Force40 Mode
            input    wire                                  DelLineClr, //Clear Delay Lines

            //Registers
            input    wire              [1:0]               ADCSWLSatConfig, //Delay Line O/p Sel
            input    wire              [1:0]               ADCSWLConfig, //Delay Line Output Sel

            //Data - from ADC
            input    wire   signed     [ADC_WIDTH-1:0]     RxADCRe, //Real Component
            input    wire   signed     [ADC_WIDTH-1:0]     RxADCIm, //Imaginary Component

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signals
            output   reg                                   ADCPowLinValidforFSM, //Power Valid
            output   reg                                   ADCSatPowLinValidforFSM, //Power Valid

            //Power Estimate
            output   reg       [ADCPOWSAT_WIDTH-1:0]       ADCSatPowLin, //Pow Est for Sat
            output   reg       [(2*ADC_WIDTH)+5:0]         ADCPowLin //Pow Est for Gain Set
            );

//////////////////////////////////////////////////////////////////////////////
//  Local parameter Declarations
//////////////////////////////////////////////////////////////////////////////
localparam DELAY_LINE_WIDTH = (BW_PARAM/20*32)-1;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire        [ADC_WIDTH-1:0]         RxDataIntAbs;
wire                                RxDataIntValid2D;
wire                                RxDataIntValid4D;
wire        [ADC_WIDTH-6:0]         RxDataInt1Rnd;
wire        [ADC_WIDTH-6:0]         RxDataIntAbsRnd;
wire        [ADCPOWSAT_WIDTH-1:0]   ADCSatPowLinRnd;
wire        [ADCPOWSAT_WIDTH-2:0]   ADCSatPowLinRnd1;
wire        [ADCPOWSAT_WIDTH-3:0]   ADCSatPowLinRnd2;
wire                                ComputeOn;
wire        [(2*(ADC_WIDTH))+3:0]   ADCPowLinIntRnd2;
wire        [(2*(ADC_WIDTH))+4:0]   ADCPowLinIntRnd1;
wire        [3:0]                   TermCnt1;
wire signed [(2*ADC_WIDTH):0]       RxDataPowDiff;
wire signed [(2*(ADC_WIDTH-5)):0]   RxDataSatPowDiff;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                SampleToggle;
reg signed [ADC_WIDTH-1:0]         RxDataIntRe;
reg signed [ADC_WIDTH-1:0]         RxDataIntIm;
reg                                RxDataIntValid;
reg        [ADC_WIDTH-1:0]         RxDelayLine[DELAY_LINE_WIDTH:0];
reg        [ADC_WIDTH-1:0]         RxDataInt1;
reg        [ADC_WIDTH-1:0]         RxDataInt2;
reg        [ADC_WIDTH-6:0]         RxDataInt3;
reg        [ADC_WIDTH-6:0]         RxDataInt4;
reg        [(2*(ADC_WIDTH-5))-1:0] RxDataPow0;
reg        [(2*(ADC_WIDTH-5))-1:0] RxDataPow1;
reg        [(2*ADC_WIDTH)-1:0]     RxDataPow2;
reg        [(2*ADC_WIDTH)-1:0]     RxDataPow3;
reg        [(2*(ADC_WIDTH-5))+5:0] ADCSatPowLinInt;
reg        [(2*(ADC_WIDTH))+5:0]   ADCPowLinInt;
reg        [7:0]                   DelayCnt0;
reg        [7:0]                   TermCnt0;
reg        [3:0]                   DelayCnt1;
reg        [3:0]                   RxDataIntValidDel;

integer                            i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module estimates the power of the ADC samples. There are 2 outputs, one
//which is used for detecting saturation and the other which is used for gain
//setting.

//Down sample incoming data by 2
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: SampleToggle_Blk
   if (nAGCRst == 1'b0)
      SampleToggle <= 1'b0;
   else if (AGCEn == 1'b0)
      SampleToggle <= 1'b0;
   else 
      SampleToggle <= ~SampleToggle;
end //SampleToggle_Blk

always @ (posedge ADCPowClk or negedge nAGCRst)
begin: DownSample_Blk
   if (nAGCRst == 1'b0) begin
      RxDataIntRe    <= $signed({{ADC_WIDTH}{1'b0}});
      RxDataIntIm    <= $signed({{ADC_WIDTH}{1'b0}});
      RxDataIntValid <= 1'b0;
   end
   else if (AGCEn == 1'b0) begin
      RxDataIntRe    <= $signed({{ADC_WIDTH}{1'b0}});
      RxDataIntIm    <= $signed({{ADC_WIDTH}{1'b0}});
      RxDataIntValid <= 1'b0;
   end
   else if (SampleToggle) begin
      RxDataIntRe    <= RxADCRe;
      RxDataIntIm    <= RxADCIm;
      RxDataIntValid <= 1'b1;
   end
   else begin
      RxDataIntValid <= 1'b0;
   end
end //DownSample_Blk

//Instantiate ModApprox Block
//This block is reused from TBE, and hence the port names are not
//representative in the AGC context.
ModApprox #(
            .SUM_WIDTH(ADC_WIDTH),    //Input Width
            .CORRVAL_WIDTH(ADC_WIDTH) //Output Width
           )
            U_MdApprox1(
                        //Inputs
                        .PhyClk(ADCPowClk),
                        .nPhyRst(nAGCRst),
                        .ComputeOn(ComputeOn),
                        .ISum(RxDataIntRe),
                        .QSum(RxDataIntIm),

                        //Output
                        .CorrVal(RxDataIntAbs)
                       );

assign ComputeOn = 1'b1;

//Delay RxDataIntValid for 2 clocks to qualify output of ModApprox block
//which has a 2 clock latency.
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: RxDataIntValidDel_Blk
   if (nAGCRst == 1'b0)
      RxDataIntValidDel <= 4'b0;
   else if (AGCEn == 1'b0)
      RxDataIntValidDel <= 4'b0;
   else 
      RxDataIntValidDel <= {RxDataIntValidDel[2:0], RxDataIntValid};
end //RxDataIntValidDel_Blk

assign RxDataIntValid2D = RxDataIntValidDel[1];

//Delay Line for Data
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: DelayLine_Blk
   if (nAGCRst == 1'b0) begin
      for (i = 0; i <= DELAY_LINE_WIDTH; i = i+1)
         RxDelayLine[i] <= {{ADC_WIDTH}{1'b0}};
   end
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
      for (i = 0; i <= DELAY_LINE_WIDTH; i = i+1)
         RxDelayLine[i] <= {{ADC_WIDTH}{1'b0}};
   end
   else if (RxDataIntValid2D == 1'b1) begin
      RxDelayLine[0] <= RxDataIntAbs;
      for (i = 1; i <= DELAY_LINE_WIDTH; i = i+1)
         RxDelayLine[i] <= RxDelayLine[i-1];
   end
end //DelayLine_Blk
      
//Power for Saturation Detection
//Select one of the delayed outputs based on Bandwidth, Force20,
//Force40 and ADCSWLSatConfig
generate if (BW_PARAM == 20) begin : DELAYLINE20BLK
   always @ (*)
   begin: RxDataInt1_20_Blk
      //ADC = 20 MS/s
      case (ADCSWLSatConfig)
         2'b00:   RxDataInt1 = RxDelayLine[7];
         2'b01:   RxDataInt1 = RxDelayLine[15];
         default: RxDataInt1 = RxDelayLine[31];
      endcase
   end //RxDataInt1_20_Blk
end //BW_PARAM = 20
else if (BW_PARAM == 40) begin : DELAYLINE40BLK //BW_PARAM = 40
   always @ (*)
   begin: RxDataInt1_40_Blk
      if (Force20 == 1'b0) begin
         //ADC = 40 MS/s
         case (ADCSWLSatConfig)
            2'b00:   RxDataInt1 = RxDelayLine[15];
            2'b01:   RxDataInt1 = RxDelayLine[31];
            default: RxDataInt1 = RxDelayLine[63];
         endcase
      end
      else begin
         //ADC = 20 MS/s
         case (ADCSWLSatConfig)
            2'b00:   RxDataInt1 = RxDelayLine[7];
            2'b01:   RxDataInt1 = RxDelayLine[15];
            default: RxDataInt1 = RxDelayLine[31];
         endcase
      end
   end //RxDataInt1_40_Blk

end //BW_PARAM = 40
else begin : DELAYLINE80BLK //BW_PARAM = 80
   always @ (*)
   begin: RxDataInt1_Blk
      if (Force40 == 1'b0 && Force20 == 1'b0) begin
         //ADC = 80 MS/s
         case (ADCSWLSatConfig)
            2'b00:   RxDataInt1 = RxDelayLine[31];
            2'b01:   RxDataInt1 = RxDelayLine[63];
            default: RxDataInt1 = RxDelayLine[127];
         endcase
      end
      else if (Force40 == 1'b1 && Force20 == 1'b0) begin
         //ADC = 40 MS/s
         case (ADCSWLSatConfig)
            2'b00:   RxDataInt1 = RxDelayLine[15];
            2'b01:   RxDataInt1 = RxDelayLine[31];
            default: RxDataInt1 = RxDelayLine[63];
         endcase
      end
      else begin
         //ADC = 20 MS/s
         case (ADCSWLSatConfig)
            2'b00:   RxDataInt1 = RxDelayLine[7];
            2'b01:   RxDataInt1 = RxDelayLine[15];
            default: RxDataInt1 = RxDelayLine[31];
         endcase
      end
   end //RxDataInt1_Blk

end //BW_PARAM = 80
endgenerate


//Rounding for RxDataInt1 and RxDataIntAbs
USgnRound #(
            .INPUT_WIDTH(ADC_WIDTH),
            .OUTPUT_WIDTH(ADC_WIDTH-5)
           )
            U_ROUNDSATPOW0(
                        .InputData(RxDataInt1),
                        .RoundData(RxDataInt1Rnd)
                       );

USgnRound #(
            .INPUT_WIDTH(ADC_WIDTH),
            .OUTPUT_WIDTH(ADC_WIDTH-5)
           )
            U_ROUNDSATPOW1(
                        .InputData(RxDataIntAbs),
                        .RoundData(RxDataIntAbsRnd)
                       );

//Register the output of the rounding blocks
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: RndReg_Blk
   if (nAGCRst == 1'b0) begin
      RxDataInt3 <= {{(ADC_WIDTH-5)}{1'b0}};
      RxDataInt4 <= {{(ADC_WIDTH-5)}{1'b0}};
   end
   else begin
      RxDataInt3 <= RxDataIntAbsRnd;
      RxDataInt4 <= RxDataInt1Rnd;
   end
end //RndReg_Blk

//Find the square of RxDataInt3 and RxDataInt4
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: Mult0_Blk
   if (nAGCRst == 1'b0) begin
      RxDataPow0 <= {{(2*(ADC_WIDTH-5))}{1'b0}};
      RxDataPow1 <= {{(2*(ADC_WIDTH-5))}{1'b0}};
   end
   else begin
      RxDataPow0 <= RxDataInt3 * RxDataInt3;
      RxDataPow1 <= RxDataInt4 * RxDataInt4;
   end
end //Mult0_Blk

//Delay RxDataIntValid by 4 clocks to qualify output of Multiplier
assign RxDataIntValid4D = RxDataIntValidDel[3];

//Power variation
assign RxDataSatPowDiff = {1'b0,RxDataPow0} - {1'b0,RxDataPow1};

//Accumulator
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: SatAcc_Blk
   if (nAGCRst == 1'b0)
      ADCSatPowLinInt <= {{((2*(ADC_WIDTH-5))+6)}{1'b0}};
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1)
      ADCSatPowLinInt <= {{((2*(ADC_WIDTH-5))+6)}{1'b0}};
   else if (RxDataIntValid4D == 1'b1)
   begin
      //Accumulator must be always positive, so check diff before addition
      if (($signed({1'b0,ADCSatPowLinInt}) >= $signed({{6{RxDataSatPowDiff[(2*(ADC_WIDTH-5))-1]}},RxDataSatPowDiff})) || !RxDataSatPowDiff[(2*(ADC_WIDTH-5))])
        ADCSatPowLinInt <= ADCSatPowLinInt + {{5{RxDataSatPowDiff[(2*(ADC_WIDTH-5))-1]}},RxDataSatPowDiff};
      else
        ADCSatPowLinInt <= {{((2*(ADC_WIDTH-5))+6)}{1'b0}};
   end
end //SatAcc_Blk


//The output width of ADCSatPowLin is set to ADCPOWSAT_WIDTH.
//Rounding has to be done by 6 bits.

//Rounding by 6 bits
USgnRound #(
            .INPUT_WIDTH(ADCPOWSAT_WIDTH+6),
            .OUTPUT_WIDTH(ADCPOWSAT_WIDTH)
           )
            U_ROUNDSATPOW2(
                        .InputData(ADCSatPowLinInt),
                        .RoundData(ADCSatPowLinRnd)
                       );

//Rounding by 7 bits
generate if (BW_PARAM == 80 || BW_PARAM == 40) begin : ROUNDSAT8040
USgnRound #(
            .INPUT_WIDTH(ADCPOWSAT_WIDTH+6),
            .OUTPUT_WIDTH(ADCPOWSAT_WIDTH-1)
           )
            U_ROUNDSATPOW3(
                        .InputData(ADCSatPowLinInt),
                        .RoundData(ADCSatPowLinRnd1)
                       );
end
else begin : NOROUNDSAT8040
   assign ADCSatPowLinRnd1 = {{ADCPOWSAT_WIDTH-1}{1'b0}};
end
endgenerate

//Rounding by 8 bits
generate if (BW_PARAM == 80) begin : ROUNDSAT80
   USgnRound #(
               .INPUT_WIDTH(ADCPOWSAT_WIDTH+6),
               .OUTPUT_WIDTH(ADCPOWSAT_WIDTH-2)
              )
               U_ROUNDSATPOW4(
                           .InputData(ADCSatPowLinInt),
                           .RoundData(ADCSatPowLinRnd2)
                          );
end
else begin : NOROUNDSAT80
   assign ADCSatPowLinRnd2 = {{ADCPOWSAT_WIDTH-2}{1'b0}};
end
endgenerate

//Choose the required output
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: SatOutput_Blk
   if (nAGCRst == 1'b0)
      ADCSatPowLin <= {{ADCPOWSAT_WIDTH}{1'b0}};
   else if (Force40 == 1'b0 && Force20 == 1'b0)
      ADCSatPowLin <= {2'b00,ADCSatPowLinRnd2};
   else if (Force40 == 1'b1 && Force20 == 1'b0)
      ADCSatPowLin <= {1'b0,ADCSatPowLinRnd1};
   else
      ADCSatPowLin <= ADCSatPowLinRnd;
end //SatOutput_Blk

//Power for Gain Setting
//Select one of the delayed outputs based on Force20 and Force40,
//and ADCSWLConfig
generate if (BW_PARAM == 20) begin : DELAYLINE_2_20BLK
   always @ (posedge ADCPowClk or negedge nAGCRst)
   begin: RxDataInt2_20_Blk
      if (nAGCRst == 1'b0)
         RxDataInt2 <= {{ADC_WIDTH}{1'b0}};
      else begin
         //ADC = 20 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[7];
            2'b01:   RxDataInt2 <= RxDelayLine[15];
            default: RxDataInt2 <= RxDelayLine[31];
         endcase
      end
   end //RxDataInt2_Blk
end //BW_PARAM = 20
else if (BW_PARAM == 40) begin : DELAYLINE_2_40BLK //BW_PARAM = 40
   always @ (posedge ADCPowClk or negedge nAGCRst)
   begin: RxDataInt2_40_Blk
      if (nAGCRst == 1'b0)
         RxDataInt2 <= {{ADC_WIDTH}{1'b0}};
      else if (Force20 == 1'b0) begin
         //ADC = 40 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[15];
            2'b01:   RxDataInt2 <= RxDelayLine[31];
            default: RxDataInt2 <= RxDelayLine[63];
         endcase
      end
      else begin
         //ADC = 20 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[7];
            2'b01:   RxDataInt2 <= RxDelayLine[15];
            default: RxDataInt2 <= RxDelayLine[31];
         endcase
      end
   end //RxDataInt2_Blk
end //BW_PARAM = 40
else begin : DELAYLINE_2_80BLK //BW_PARAM = 80
   always @ (posedge ADCPowClk or negedge nAGCRst)
   begin: RxDataInt2_Blk
      if (nAGCRst == 1'b0)
         RxDataInt2 <= {{ADC_WIDTH}{1'b0}};
      else if (Force40 == 1'b0 && Force20 == 1'b0) begin
         //ADC = 80 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[31];
            2'b01:   RxDataInt2 <= RxDelayLine[63];
            default: RxDataInt2 <= RxDelayLine[127];
         endcase
      end
      else if (Force40 == 1'b1 && Force20 == 1'b0) begin
         //ADC = 40 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[15];
            2'b01:   RxDataInt2 <= RxDelayLine[31];
            default: RxDataInt2 <= RxDelayLine[63];
         endcase
      end
      else begin
         //ADC = 20 MS/s
         case (ADCSWLConfig)
            2'b00:   RxDataInt2 <= RxDelayLine[7];
            2'b01:   RxDataInt2 <= RxDelayLine[15];
            default: RxDataInt2 <= RxDelayLine[31];
         endcase
      end
   end //RxDataInt2_Blk

end //BW_PARAM = 80
endgenerate

//Find the square of RxDataInt2 and RxDataIntAbs
//Since RxDataInt2 has one clock latency for the muxing, the input
//data (RxDataIntAbs) also has to be delayed by a clock
//This is equivalent to using RxDelayLine[0]
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: Mult1_Blk
   if (nAGCRst == 1'b0) begin
      RxDataPow2 <= {{(2*ADC_WIDTH)}{1'b0}};
      RxDataPow3 <= {{(2*ADC_WIDTH)}{1'b0}};
   end
   else begin
      RxDataPow2 <= RxDelayLine[0] * RxDelayLine[0];
      RxDataPow3 <= RxDataInt2 * RxDataInt2;
   end
end //Mult1_Blk

//Power variation
assign RxDataPowDiff = {1'b0,RxDataPow2} - {1'b0,RxDataPow3};

//Accumulator
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: Acc_Blk
   if (nAGCRst == 1'b0)
      ADCPowLinInt <= {{((2*ADC_WIDTH)+6)}{1'b0}};
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1)
      ADCPowLinInt <= {{((2*(ADC_WIDTH))+6)}{1'b0}};
   else if (RxDataIntValid4D == 1'b1)
   begin
      //Accumulator must be always positive, so check diff before addition
      if (($signed({1'b0,ADCPowLinInt}) >= $signed({{6{RxDataPowDiff[2*ADC_WIDTH]}},RxDataPowDiff})) || !RxDataPowDiff[2*ADC_WIDTH])
        ADCPowLinInt <= ADCPowLinInt + {{5{RxDataPowDiff[2*ADC_WIDTH]}},RxDataPowDiff};
      else
        ADCPowLinInt <= {{((2*ADC_WIDTH)+6)}{1'b0}};
   end
end //Acc_Blk

//Rounding by 2 bits
generate if (BW_PARAM == 80) begin : ROUND80
   USgnRound #(
               .INPUT_WIDTH((2*ADC_WIDTH)+6),
               .OUTPUT_WIDTH((2*ADC_WIDTH)+4)
              )
               U_ROUNDPOW0(
                           .InputData(ADCPowLinInt),
                           .RoundData(ADCPowLinIntRnd2)
                          );
end
else begin : NOROUND80
   assign ADCPowLinIntRnd2 = {{(2*ADC_WIDTH)+4}{1'b0}};
end
endgenerate

//Rounding by 1 bits
generate if (BW_PARAM == 80 || BW_PARAM == 40) begin : ROUND8040
USgnRound #(
            .INPUT_WIDTH((2*ADC_WIDTH)+6),
            .OUTPUT_WIDTH((2*ADC_WIDTH)+5)
           )
            U_ROUNDPOW1(
                        .InputData(ADCPowLinInt),
                        .RoundData(ADCPowLinIntRnd1)
                       );
end
else begin : NOROUND8040
   assign ADCPowLinIntRnd1 = {{(2*ADC_WIDTH)+5}{1'b0}};
end
endgenerate

//Output rounded
always @ (posedge ADCPowClk or negedge nAGCRst)
begin: LinOut_Blk
   if (nAGCRst == 1'b0)
      ADCPowLin <= {{((2*(ADC_WIDTH))+6)}{1'b0}};
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1)
      ADCPowLin <= {{((2*(ADC_WIDTH))+6)}{1'b0}};
   else if (Force40 == 1'b0 && Force20 == 1'b0)
      ADCPowLin <= {2'b00,ADCPowLinIntRnd2}; //ADC = 80 MS/s
   else if (Force40 == 1'b1 && Force20 == 1'b0)
      ADCPowLin <= {1'b0,ADCPowLinIntRnd1};  //ADC = 40 MS/s
   else
      ADCPowLin <= ADCPowLinInt;          //ADC = 20 MS/s
end //LinOut_Blk

//Generate ADCPowLinValidforFSM
//This signal indicates to the FSM that the output power can be used for
//its decision making. This should take care of the delay line length as well
//as the hardware latency.
always @ (*)
begin: TermDelaySel0_Blk
   case (ADCSWLConfig)
      2'b00  : TermCnt0 = (Force20) ? 8'd15 : (Force40) ? 8'd31  : 8'd63;  // 0.4 us
      2'b01  : TermCnt0 = (Force20) ? 8'd31 : (Force40) ? 8'd63  : 8'd127; // 0.8 us
      default: TermCnt0 = (Force20) ? 8'd63 : (Force40) ? 8'd127 : 8'd255; // 1.6 us
   endcase
end //TermDelaySel0_Blk

always @ (posedge ADCPowClk or negedge nAGCRst)
begin: Valid0_Blk
   if (nAGCRst == 1'b0) begin
      ADCPowLinValidforFSM <= 1'b0;
      DelayCnt0            <= 8'd0;
   end
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
      ADCPowLinValidforFSM <= 1'b0;
      DelayCnt0            <= 8'd0;
   end
   else if (DelayCnt0 < TermCnt0) begin
      ADCPowLinValidforFSM <= 1'b0;
      DelayCnt0            <= DelayCnt0 + 8'b1;
   end
   else begin
      ADCPowLinValidforFSM <= 1'b1;
   end
end //Valid0_Blk

//Generate ADCSatPowLinValidforFSM after 0.1us
assign TermCnt1 = (Force20) ? 4'd3 : (Force40) ? 4'd7 : 4'd15;

always @ (posedge ADCPowClk or negedge nAGCRst)
begin: Valid1_Blk
   if (nAGCRst == 1'b0) begin
      ADCSatPowLinValidforFSM <= 1'b0;
      DelayCnt1               <= 4'd0;
   end
   else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
      ADCSatPowLinValidforFSM <= 1'b0;
      DelayCnt1               <= 4'd0;
   end
   else if (DelayCnt1 < TermCnt1) begin
      ADCSatPowLinValidforFSM <= 1'b0;
      DelayCnt1               <= DelayCnt1 + 4'b1;
   end
   else begin
      ADCSatPowLinValidforFSM <= 1'b1;
   end
end //Valid1_Blk

endmodule //ADCPowEst

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
