////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Power Estimation Top Level for AGC
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/PowerEst/verilog/rtl/AGCPowEstTop.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

`default_nettype none

module AGCPowEstTop #(parameter ADC_WIDTH       = 12, //ADC Data Width
                      parameter INBD_WIDTH      = 13, //Front End Output Width
                      parameter BW_PARAM        = 80, //Bandwidth (20/40/80)
                      parameter FSB_ITER        =  6, //Iterations for finding FSB
                      parameter INBDPOW_WIDTH   = 24, //Data width of In Band Power
                      parameter ADCPOWSAT_WIDTH = 14, //Data width of ADC Power for Sat detect
                      parameter ADCPOW_WIDTH    = 30, //Data width of ADC Power
                      parameter DBVPOW_WIDTH    = 12, //Data width of dBV power
                      parameter DBMPOW_WIDTH    = 13, //Data width of dBm power
                      parameter NRX             =  4  //Number of Rx Chains
                     )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active LOW Reset
            input    wire                                  AGCClk,  //AGC Clock 80MHz
            input    wire                                  ADCPowClk,  //ADC Power Clock (max of 40/80/160MHz support)

            //Control Signals
            input    wire                                  Force20, //Force20 Mode
            input    wire                                  Force40, //Force40 Mode
            input    wire                                  DelLineClr, //Clear Delay Lines
            input    wire                                  RxAGC20PriDataValid,  //Qualifies Data 20 Pri
            input    wire                                  RxAGC20SecDataValid, //Qualifies Data 20 Sec
            input    wire                                  RxAGC40SecDataValid, //Qualifies Data 40 Sec
            input    wire   signed     [7:0]               RxGaindB0, //Gain of RF in dB (Chain 0)
            input    wire   signed     [7:0]               RxGaindB1, //Gain of RF in dB (Chain 1)
            input    wire   signed     [7:0]               RxGaindB2, //Gain of RF in dB (Chain 2)
            input    wire   signed     [7:0]               RxGaindB3, //Gain of RF in dB (Chain 3)
            input    wire                                  ADCPowEstEn, //Enable for ADCPowEst
            input    wire                                  InBdSecPowEstEn, //Enable InBand 20 and 40 secondary PowEst
            input    wire                                  InBd20PowEstEn, //Enable InBand 20 primary PowEst
            input    wire              [1:0]               AntSelCCA20s, //Chain selection for 20 secondary
            input    wire              [1:0]               AntSelCCA40s, //Chain selection for 40 secondary

            //Registers
            input    wire              [1:0]               ADCSWLSatConfig, //Delay Line O/p Sel
            input    wire              [1:0]               ADCSWLConfig, //Delay Line Output Sel
            input    wire              [2:0]               InBd20SWLConfig, //Delay Line Output Sel 20 primary
            input    wire              [2:0]               InBd20SecSWLConfig, //Delay Line Output Sel 20 secondary
            input    wire              [2:0]               InBd40SecSWLConfig, //Delay Line Output Sel 40 secondary
            input    wire              [3:0]               NBitADC, //Effective ADC Width
            input    wire   signed     [7:0]               VPeakADCqdBV, //ADC Peak Voltage in dB
            input    wire              [3:0]               AntLossdB0, //Antenna Loss in dB
            input    wire              [3:0]               AntLossdB1, //Antenna Loss in dB
            input    wire              [3:0]               AntLossdB2, //Antenna Loss in dB
            input    wire              [3:0]               AntLossdB3, //Antenna Loss in dB
            input    wire              [1:0]               RegInBdRnd, //Rounding for InBand Power
            input    wire   signed     [DBVPOW_WIDTH-5:0]  RegADCPowBiasqdBV, //Bias for ADC Power in dBV
            input    wire   signed     [DBMPOW_WIDTH-5:0]  RegADCPowBiasqdBm, //Bias for ADC Power in dBm
            input    wire   signed     [DBMPOW_WIDTH-5:0]  RegInBd40SecPowBiasqdBm, //Bias for InBd40 secondary in dBm
            input    wire   signed     [DBMPOW_WIDTH-5:0]  RegInBd20PowBiasqdBm, //Bias for InBd20 primary in dBm
            input    wire   signed     [DBMPOW_WIDTH-5:0]  RegInBd20SecPowBiasqdBm,//Bias for InBd20 secondary in dBm
            input    wire   signed     [DBVPOW_WIDTH-3:0]  InBddBVThr,  //InBand dBVrms Threshold
            input    wire   signed     [DBMPOW_WIDTH-3:0]  InBddBmThr, //InBand dBm Threshold
            input    wire   signed     [DBVPOW_WIDTH-3:0]  ADCdBVThr,  //ADC dBVrms Threshold
            input    wire   signed     [DBMPOW_WIDTH-3:0]  ADCdBmThr, //ADC dBm Threshold
            input    wire   signed     [7:0]               FE20Gain, //Gain of FE Filter in 20MHz
            input    wire   signed     [7:0]               FE40Gain, //Gain of FE Filter in 40MHz
            
            //Data
            //From ADC
            input    wire   signed     [ADC_WIDTH-1:0]     ADCReRx0, //Real Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCImRx0, //Imaginary Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCReRx1, //Real Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCImRx1, //Imaginary Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCReRx2, //Real Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCImRx2, //Imaginary Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCReRx3, //Real Component
            input    wire   signed     [ADC_WIDTH-1:0]     ADCImRx3, //Imaginary Component

            //From FE Primary 20MHz
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriReRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriImRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriReRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriImRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriReRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriImRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriReRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20PriImRx3, //Imaginary Component

            //From FE Secondary 20MHz
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecReRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecImRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecReRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecImRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecReRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecImRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecReRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC20SecImRx3, //Imaginary Component

            //From FE Secondary 40MHz
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecReRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecImRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecReRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecImRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecReRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecImRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecReRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    RxAGC40SecImRx3, //Imaginary Component

            //Inputs from Shared Delay Line
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe8DRx0,  //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm8DRx0,  //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe16DRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm16DRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe32DRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm32DRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe48DRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm48DRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe64DRx0, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm64DRx0, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe8DRx1,  //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm8DRx1,  //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe16DRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm16DRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe32DRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm32DRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe48DRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm48DRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe64DRx1, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm64DRx1, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe8DRx2,  //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm8DRx2,  //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe16DRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm16DRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe32DRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm32DRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe48DRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm48DRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe64DRx2, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm64DRx2, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe8DRx3,  //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm8DRx3,  //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe16DRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm16DRx3, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe32DRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm32DRx3, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe48DRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm48DRx3, //Imaginary Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCRe64DRx3, //Real Component
            input    wire   signed     [INBD_WIDTH-1:0]    AGCIm64DRx3, //Imaginary Component

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signals - Power Valid
            output   reg                                   ADCPowdBVValid,    //ADC Power in dBV Valid
            output   wire                                  ADCSatPowdBmValid, //ADC Sat Power in dBm Valid
            output   reg                                   ADCSatPowValid,    //ADC Sat Power Valid
            output   wire                                  InBd20PowValid,    //InBd 20 Pri Power Valid
            output   wire                                  InBd20PowCCAValid, //InBd 20 Pri Power Valid for CCA
            output   wire                                  InBd20SecPowValid, //InBd 20 Sec Power Valid
            output   wire                                  InBd40SecPowValid, //InBd 40 Sec Power Valid

            //ADC Power Estimate in dBVrms
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx0, //Power in Rx Chain 0
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx1, //Power in Rx Chain 1
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx2, //Power in Rx Chain 2
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx3, //Power in Rx Chain 3

            //ADC Power Estimate in dBm
            output   wire   signed     [DBMPOW_WIDTH-1:0]  ADCSatPowdBmRx0, //Power in Rx Chain 0
            output   wire   signed     [DBMPOW_WIDTH-1:0]  ADCSatPowdBmRx1, //Power in Rx Chain 1
            output   wire   signed     [DBMPOW_WIDTH-1:0]  ADCSatPowdBmRx2, //Power in Rx Chain 2
            output   wire   signed     [DBMPOW_WIDTH-1:0]  ADCSatPowdBmRx3, //Power in Rx Chain 3

            //ADC Power Estimate in dBVrms - for saturation detection
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx0, //Power in Rx Chain 0
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx1, //Power in Rx Chain 1
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx2, //Power in Rx Chain 2
            output   wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx3, //Power in Rx Chain 3

            //In Band Power in dBm for 20 MHz Primary channel
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx0, //Power in Rx Chain 0
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx1, //Power in Rx Chain 1
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx2, //Power in Rx Chain 2
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx3, //Power in Rx Chain 3

            //In Band Power in dBm for 40 MHz channel
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd40SecPowdBm, //Power in 40 secondary

            //In Band Power in dBm for 20 MHz Secondary channel
            output   wire   signed     [DBMPOW_WIDTH-1:0]  InBd20SecPowdBm  //Power in 20 secondary
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [DBVPOW_WIDTH-1:0] CONST_ZERO_DBVPOW_WIDTH = {{DBVPOW_WIDTH}{1'b0}};
localparam signed [DBMPOW_WIDTH-1:0] CONST_ZERO_DBMPOW_WIDTH = {{DBMPOW_WIDTH}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVIntRx0;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVIntRx1;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVIntRx2;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVIntRx3;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVIntRx0;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVIntRx1;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVIntRx2;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVIntRx3;
wire             [ADCPOW_WIDTH-1:0]    ADCPowLinRx0;
wire             [ADCPOW_WIDTH-1:0]    ADCPowLinRx1;
wire             [ADCPOW_WIDTH-1:0]    ADCPowLinRx2;
wire             [ADCPOW_WIDTH-1:0]    ADCPowLinRx3;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd40SecPowLin;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd40SecPowdBVInt;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVIntRx0;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVIntRx1;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVIntRx2;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVIntRx3;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffRx0;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffRx1;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffRx2;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffRx3;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20SecPowdBVEff;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd40SecPowdBVEff;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd20PowLinRx0;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd20PowLinRx1;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd20PowLinRx2;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd20PowLinRx3;
wire    signed   [INBDPOW_WIDTH-1:0]   InBd20SecPowLin;
wire    signed   [DBVPOW_WIDTH-1:0]    InBd20SecPowdBVInt;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCqdBVThr;
wire    signed   [DBMPOW_WIDTH-1:0]    ADCqdBmThr;
wire    signed   [DBVPOW_WIDTH-1:0]    InBdqdBVThr;
wire    signed   [DBMPOW_WIDTH-1:0]    InBdqdBmThr;
wire    signed   [DBMPOW_WIDTH-1:0]    dBmTemp0;
wire    signed   [DBMPOW_WIDTH-1:0]    dBmTemp1;
wire    signed   [DBMPOW_WIDTH-1:0]    dBmTemp2;
wire    signed   [DBMPOW_WIDTH-1:0]    dBmTemp3;
wire             [5:0]                 AntLossqdB0;
wire             [5:0]                 AntLossqdB1;
wire             [5:0]                 AntLossqdB2;
wire             [5:0]                 AntLossqdB3;
wire    signed   [9:0]                 RxGainqdB0;
wire    signed   [9:0]                 RxGainqdB1;
wire    signed   [9:0]                 RxGainqdB2;
wire    signed   [9:0]                 RxGainqdB3;
wire                                   ADCSatPowLinValidforFSMRx0;
wire                                   ADCSatPowLinValidforFSMRx1;
wire                                   ADCSatPowLinValidforFSMRx2;
wire                                   ADCSatPowLinValidforFSMRx3;
wire                                   ADCPowLinValidforFSMRx0;
wire                                   ADCPowLinValidforFSMRx1;
wire                                   ADCPowLinValidforFSMRx2;
wire                                   ADCPowLinValidforFSMRx3;
wire                                   ADCSatPowdBVValidforFSMRx0;
wire                                   ADCSatPowdBVValidforFSMRx1;
wire                                   ADCSatPowdBVValidforFSMRx2;
wire                                   ADCSatPowdBVValidforFSMRx3;
wire                                   ADCPowdBVValidforFSMRx0;
wire                                   ADCPowdBVValidforFSMRx1;
wire                                   ADCPowdBVValidforFSMRx2;
wire                                   ADCPowdBVValidforFSMRx3;
wire                                   InBd40SecPowLinValidforFSM;
wire                                   InBd40SecPowdBVValidforFSM;
wire                                   InBd40SecPowdBVValidforFSMMux;
wire                                   InBd40SecPowValidTemp;
wire                                   InBd20PowLinValidforFSMRx0;
wire                                   InBd20PowLinValidforCCARx0;
wire                                   InBd20PowLinValidforCCARx0Mux;
wire                                   InBd20PowLinValidforFSMRx1;
wire                                   InBd20PowLinValidforFSMRx2;
wire                                   InBd20PowLinValidforFSMRx3;
wire                                   InBd20PowdBVValidforFSMRx0;
wire                                   InBd20PowdBVValidforFSMRx0Mux;
wire                                   InBd20SecPowLinValidforFSM;
wire                                   InBd20SecPowdBVValidforFSM;
wire                                   InBd20SecPowdBVValidforFSMMux;
wire                                   InBd20SecPowValidTemp;
wire             [ADCPOWSAT_WIDTH-1:0] ADCSatPowLinRx0Int;
wire             [ADCPOWSAT_WIDTH-1:0] ADCSatPowLinRx1Int;
wire             [ADCPOWSAT_WIDTH-1:0] ADCSatPowLinRx2Int;
wire             [ADCPOWSAT_WIDTH-1:0] ADCSatPowLinRx3Int;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVTempRx0;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVTempRx1;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVTempRx2;
wire    signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVTempRx3;
wire                                   AddZeroInBd20CntGt0;
wire                                   AddZeroInBd20;
wire                                   AddZeroInBdSec20CntGt0;
wire                                   AddZeroInBdSec20;
wire                                   AddZeroInBdSec40CntGt0;
wire                                   AddZeroInBdSec40;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx0Int;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx1Int;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx2Int;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx3Int;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx0Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx1Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx2Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20PowdBmRx3Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20SecPowdBmTemp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd20SecPowdBmInt;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd40SecPowdBmTemp;
reg     signed   [DBMPOW_WIDTH-1:0]    InBd40SecPowdBmInt;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffSatRx0;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffSatRx1;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffSatRx2;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd20PowdBVEffSatRx3;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd20SecPowdBVEffSat;
reg     signed   [DBVPOW_WIDTH-1:0]    InBd40SecPowdBVEffSat;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVRx0Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVRx1Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVRx2Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCPowdBVRx3Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx0Int;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx1Int;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx2Int;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx3Int;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx0Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx1Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx2Temp;
reg     signed   [DBMPOW_WIDTH-1:0]    ADCSatPowdBmRx3Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVRx0Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVRx1Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVRx2Temp;
reg     signed   [DBVPOW_WIDTH-1:0]    ADCSatPowdBVRx3Temp;
reg     signed   [7:0]                 RxGaindB0D;
reg     signed   [7:0]                 RxGaindB1D;
reg     signed   [7:0]                 RxGaindB2D;
reg     signed   [7:0]                 RxGaindB3D;
reg     signed   [DBMPOW_WIDTH-1:0]    dBmTemp20s;
reg     signed   [DBMPOW_WIDTH-1:0]    dBmTemp40s;
reg     signed   [9:0]                 LSBLessADCPow;
reg     signed   [9:0]                 LSBLessADCPowSat;
reg     signed   [9:0]                 LSBLessInBd20Pow;
reg     signed   [9:0]                 LSBLessInBd20SecPow;
reg     signed   [9:0]                 LSBLessInBd40SecPow;
reg              [6:0]                 AddZeroInBd20Cnt;
reg              [6:0]                 AddZeroInBdSec20Cnt;
reg              [6:0]                 AddZeroInBdSec40Cnt;
reg     signed   [INBD_WIDTH-1:0]      RxAGC20SecRe;
reg     signed   [INBD_WIDTH-1:0]      RxAGC20SecIm;
reg     signed   [INBD_WIDTH-1:0]      RxAGC40SecRe;
reg     signed   [INBD_WIDTH-1:0]      RxAGC40SecIm;
reg                                    RxAGC20PriDataValid1D;
reg                                    RxAGC20PriDataValid2D;
reg                                    RxAGC20PriDataValid3D;
reg                                    InBd20PowEstEnD;
reg                                    InBdSecPowEstEnD;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module instantiates the various power estimation blocks required by the
//AGC scheme

//Set LSBLess for Lin2dBVrms conversion for each power estimation according
//window size duration: ADCSWLConfig
always @ (*)
   begin: LSBLessADCPow_Blk
     case (ADCSWLConfig)
       2'b00: LSBLessADCPow   = 10'sd48;
       2'b01: LSBLessADCPow   = 10'sd64;
       2'b10: LSBLessADCPow   = 10'sd80;
       default: LSBLessADCPow = 10'sd48;
     endcase
   end //LSBLessADCPow_Blk

always @ (*)
   begin: LSBLessADCPowSat_Blk
     case (ADCSWLSatConfig)
       2'b00: LSBLessADCPowSat   = -10'sd208;
       2'b01: LSBLessADCPowSat   = -10'sd192;
       2'b10: LSBLessADCPowSat   = -10'sd176;
       default: LSBLessADCPowSat = -10'sd208;
     endcase
   end //LSBLessADCPowSat_Blk

always @ (*)
   begin: LSBLessInBd20Pow_Blk
     case (InBd20SWLConfig)
       3'b000: LSBLessInBd20Pow  = -10'sd80;
       3'b001: LSBLessInBd20Pow  = -10'sd64;
       3'b010: LSBLessInBd20Pow  = -10'sd48;
       3'b011: LSBLessInBd20Pow  = -10'sd39;
       3'b100: LSBLessInBd20Pow  = -10'sd32;
       default: LSBLessInBd20Pow = -10'sd80;
     endcase
   end //LSBLessInBd20Pow_Blk

always @ (*)
   begin: LSBLessInBd20SecPow_Blk
     case (InBd20SecSWLConfig)
       3'b000: LSBLessInBd20SecPow  = -10'sd80;
       3'b001: LSBLessInBd20SecPow  = -10'sd64;
       3'b010: LSBLessInBd20SecPow  = -10'sd48;
       3'b011: LSBLessInBd20SecPow  = -10'sd39;
       3'b100: LSBLessInBd20SecPow  = -10'sd32;
       default: LSBLessInBd20SecPow = -10'sd80;
     endcase
   end //LSBLessInBd20SecPow_Blk

always @ (*)
   begin: LSBLessInBd40SecPow_Blk
     case (InBd40SecSWLConfig)
       3'b000: LSBLessInBd40SecPow  = -10'sd68;
       3'b001: LSBLessInBd40SecPow  = -10'sd52;
       3'b010: LSBLessInBd40SecPow  = -10'sd36;
       3'b011: LSBLessInBd40SecPow  = -10'sd27;
       3'b100: LSBLessInBd40SecPow  = -10'sd20;
       default: LSBLessInBd40SecPow = -10'sd68;
     endcase
   end //LSBLessInBd40SecPow_Blk

//ADC Power Estimation
//For Rx Chain 0
ADCPowEst # (
             .ADC_WIDTH(ADC_WIDTH),
             .BW_PARAM(BW_PARAM),
             .ADCPOWSAT_WIDTH(ADCPOWSAT_WIDTH)
            ) ADCPowEst0 (
                          //Inputs
                          .ADCPowClk(ADCPowClk),
                          .nAGCRst(nAGCRst),
                          .RxADCRe(ADCReRx0),
                          .RxADCIm(ADCImRx0),
                          .AGCEn(ADCPowEstEn),
                          .Force20(Force20),
                          .Force40(Force40),
                          .DelLineClr(DelLineClr),
                          .ADCSWLSatConfig(ADCSWLSatConfig),
                          .ADCSWLConfig(ADCSWLConfig),

                          //Outputs
                          .ADCSatPowLin(ADCSatPowLinRx0Int),
                          .ADCSatPowLinValidforFSM(ADCSatPowLinValidforFSMRx0),
                          .ADCPowLin(ADCPowLinRx0),
                          .ADCPowLinValidforFSM(ADCPowLinValidforFSMRx0)
                         );

//Convert ADCSatPowLinRx0 and ADCPowLinRx0 to dBVrms
Lin2dBVrms # (
              .INPUT_WIDTH(ADCPOW_WIDTH),
              .OUTPUT_WIDTH(DBVPOW_WIDTH),
              .FSB_ITER(FSB_ITER)
             ) Lin2dBADCPow0 (
                         //Inputs
                         .AGCClk(AGCClk),
                         .nAGCRst(nAGCRst),
                         .LinValIn(ADCPowLinRx0),
                         .LSBLess(LSBLessADCPow),
                         .NBitADC(NBitADC),
                         .VPeakADCqdBV(VPeakADCqdBV),
                         .LinDataValidforFSM(ADCPowLinValidforFSMRx0),

                         //Output
                         .qdBVOut(ADCPowdBVIntRx0),
                         .qdBVValidforFSM(ADCPowdBVValidforFSMRx0)
                        );

Lin2dBVrms # (
              .INPUT_WIDTH(ADCPOWSAT_WIDTH),
              .OUTPUT_WIDTH(DBVPOW_WIDTH),
              .FSB_ITER(FSB_ITER)
             ) Lin2dBADCSatPow0 (
                         //Inputs
                         .AGCClk(AGCClk),
                         .nAGCRst(nAGCRst),
                         .LinValIn(ADCSatPowLinRx0Int),
                         .LSBLess(LSBLessADCPowSat),
                         .NBitADC(NBitADC),
                         .VPeakADCqdBV(VPeakADCqdBV),
                         .LinDataValidforFSM(ADCSatPowLinValidforFSMRx0),

                         //Output
                         .qdBVOut(ADCSatPowdBVIntRx0),
                         .qdBVValidforFSM(ADCSatPowdBVValidforFSMRx0)
                        );

//Convert ADCPowdBVInt and ADCSatPowdBVInt to ADCPowdBV and ADCSatPowdBV

//Add a bias to the ADC Power
assign ADCPowdBVTempRx0 = $signed(ADCPowdBVIntRx0 + {{4{RegADCPowBiasqdBV[DBVPOW_WIDTH-5]}},RegADCPowBiasqdBV});

//Saturate the dBV Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCPowOut0_Blk
      if (nAGCRst == 1'b0)
         ADCPowdBVRx0Temp <= CONST_ZERO_DBVPOW_WIDTH;
      else if (ADCPowdBVTempRx0 < ADCqdBVThr)
         ADCPowdBVRx0Temp <= ADCqdBVThr;
      else
         ADCPowdBVRx0Temp <= ADCPowdBVTempRx0;
   end //ADCPowOut0_Blk

assign ADCPowdBVRx0 = ADCPowdBVRx0Temp;

//Saturate the dBV Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCSatPowOut0_Blk
      if (nAGCRst == 1'b0)
         ADCSatPowdBVRx0Temp <= CONST_ZERO_DBVPOW_WIDTH;
      else if (ADCSatPowdBVIntRx0 < ADCqdBVThr)
         ADCSatPowdBVRx0Temp <= ADCqdBVThr;
      else
         ADCSatPowdBVRx0Temp <= ADCSatPowdBVIntRx0;
   end //ADCSatPowOut0_Blk

assign ADCSatPowdBVRx0 = ADCSatPowdBVRx0Temp;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCPowdBVValid_Blk
      if (nAGCRst == 1'b0)
         ADCPowdBVValid <= 1'b0;
      else
         ADCPowdBVValid <= ADCPowdBVValidforFSMRx0;
   end //ADCPowdBVValid_Blk

always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCSatPowValid_Blk
      if (nAGCRst == 1'b0)
         ADCSatPowValid <= 1'b0;
      else
         ADCSatPowValid <= ADCSatPowdBVValidforFSMRx0;
   end //ADCSatPowValid_Blk

//Convert dBVrms to dBm
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCSatdBmInt0_Blk
      if (nAGCRst == 1'b0)
         ADCSatPowdBmRx0Int <= CONST_ZERO_DBMPOW_WIDTH;
      else
         ADCSatPowdBmRx0Int <= $signed({ADCSatPowdBVIntRx0[DBVPOW_WIDTH-1],ADCSatPowdBVIntRx0} +
                               dBmTemp0 + {{4{RegADCPowBiasqdBm[DBMPOW_WIDTH-5]}},RegADCPowBiasqdBm});
   end //ADCdBmInt0_Blk

//Saturate the dBm Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCSatPowdBm0_Blk
      if (nAGCRst == 1'b0)
         ADCSatPowdBmRx0Temp <= CONST_ZERO_DBMPOW_WIDTH;
      else if (ADCSatPowdBmRx0Int < ADCqdBmThr)
         ADCSatPowdBmRx0Temp <= ADCqdBmThr;
      else
         ADCSatPowdBmRx0Temp <= ADCSatPowdBmRx0Int;
   end //ADCSatPowdBm0_Blk

assign ADCSatPowdBmRx0 = ADCSatPowdBmRx0Temp;

//Generate ADCSatPowdBmValid
DelayLine # (
             .DATAWIDTH(1), //Width of ADCSatPowdBVValidforFSMRx0
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLine0 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(~ADCPowEstEn),
                            .DataIn(ADCSatPowdBVValidforFSMRx0),
                            .DataOut(ADCSatPowdBmValid)
                           );

//For Rx Chain 1
generate
   if (NRX > 1) begin : U_ADCPowEst_gen1
      ADCPowEst # (
                   .ADC_WIDTH(ADC_WIDTH),
                   .BW_PARAM(BW_PARAM),
                   .ADCPOWSAT_WIDTH(ADCPOWSAT_WIDTH)
                  ) ADCPowEst1 (
                                //Inputs
                                .ADCPowClk(ADCPowClk),
                                .nAGCRst(nAGCRst),
                                .RxADCRe(ADCReRx1),
                                .RxADCIm(ADCImRx1),
                                .AGCEn(ADCPowEstEn),
                                .Force20(Force20),
                                .Force40(Force40),
                                .DelLineClr(DelLineClr),
                                .ADCSWLSatConfig(ADCSWLSatConfig),
                                .ADCSWLConfig(ADCSWLConfig),

                                //Outputs
                                .ADCSatPowLin(ADCSatPowLinRx1Int),
                                .ADCSatPowLinValidforFSM(ADCSatPowLinValidforFSMRx1),
                                .ADCPowLin(ADCPowLinRx1),
                                .ADCPowLinValidforFSM(ADCPowLinValidforFSMRx1)
                               );

      //Convert ADCSatPowLinRx1 and ADCPowLinRx1 to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCPow1 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCPowLinRx1),
                               .LSBLess(LSBLessADCPow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCPowLinValidforFSMRx1),

                               //Output
                               .qdBVOut(ADCPowdBVIntRx1),
                               .qdBVValidforFSM(ADCPowdBVValidforFSMRx1)
                              );

      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOWSAT_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCSatPow1 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCSatPowLinRx1Int),
                               .LSBLess(LSBLessADCPowSat),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCSatPowLinValidforFSMRx1),

                               //Output
                               .qdBVOut(ADCSatPowdBVIntRx1),
                               .qdBVValidforFSM(ADCSatPowdBVValidforFSMRx1)
                              );

      //Add a bias to the ADC Power
      assign ADCPowdBVTempRx1 = $signed(ADCPowdBVIntRx1 + {{4{RegADCPowBiasqdBV[DBVPOW_WIDTH-5]}},RegADCPowBiasqdBV});

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCPowOut1_Blk
            if (nAGCRst == 1'b0)
               ADCPowdBVRx1Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCPowdBVTempRx1 < ADCqdBVThr)
               ADCPowdBVRx1Temp <= ADCqdBVThr;
            else
               ADCPowdBVRx1Temp <= ADCPowdBVTempRx1;
         end //ADCPowOut1_Blk

      assign ADCPowdBVRx1 = ADCPowdBVRx1Temp;

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowOut1_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBVRx1Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCSatPowdBVIntRx1 < ADCqdBVThr)
               ADCSatPowdBVRx1Temp <= ADCqdBVThr;
            else
               ADCSatPowdBVRx1Temp <= ADCSatPowdBVIntRx1;
         end //ADCSatPowOut1_Blk

      assign ADCSatPowdBVRx1 = ADCSatPowdBVRx1Temp;

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatdBmInt1_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx1Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               ADCSatPowdBmRx1Int <= $signed({ADCSatPowdBVIntRx1[DBVPOW_WIDTH-1],ADCSatPowdBVIntRx1} +
                                     dBmTemp1 + {{4{RegADCPowBiasqdBm[DBMPOW_WIDTH-5]}},RegADCPowBiasqdBm});
         end //ADCdBmInt1_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowdBm1_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx1Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (ADCSatPowdBmRx1Int < ADCqdBmThr)
               ADCSatPowdBmRx1Temp <= ADCqdBmThr;
            else
               ADCSatPowdBmRx1Temp <= ADCSatPowdBmRx1Int;
         end //ADCSatPowdBm1_Blk

      assign ADCSatPowdBmRx1 = ADCSatPowdBmRx1Temp;

  end //NRX > 1
  else begin
      assign ADCSatPowdBVRx1 = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCPowdBVRx1    = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCSatPowdBmRx1 = CONST_ZERO_DBMPOW_WIDTH;
  end
endgenerate

//For Rx Chain 2
generate
   if (NRX > 2) begin : U_ADCPowEst_gen2
      ADCPowEst # (
                   .ADC_WIDTH(ADC_WIDTH),
                   .BW_PARAM(BW_PARAM),
                   .ADCPOWSAT_WIDTH(ADCPOWSAT_WIDTH)
                  ) ADCPowEst2 (
                                //Inputs
                                .ADCPowClk(ADCPowClk),
                                .nAGCRst(nAGCRst),
                                .RxADCRe(ADCReRx2),
                                .RxADCIm(ADCImRx2),
                                .AGCEn(ADCPowEstEn),
                                .Force20(Force20),
                                .Force40(Force40),
                                .DelLineClr(DelLineClr),
                                .ADCSWLSatConfig(ADCSWLSatConfig),
                                .ADCSWLConfig(ADCSWLConfig),

                                //Outputs
                                .ADCSatPowLin(ADCSatPowLinRx2Int),
                                .ADCSatPowLinValidforFSM(ADCSatPowLinValidforFSMRx2),
                                .ADCPowLin(ADCPowLinRx2),
                                .ADCPowLinValidforFSM(ADCPowLinValidforFSMRx2)
                               );

      //Convert ADCSatPowLinRx2 and ADCPowLinRx2 to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCPow2 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCPowLinRx2),
                               .LSBLess(LSBLessADCPow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCPowLinValidforFSMRx2),

                               //Output
                               .qdBVOut(ADCPowdBVIntRx2),
                               .qdBVValidforFSM(ADCPowdBVValidforFSMRx2)
                              );

      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOWSAT_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCSatPow2 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCSatPowLinRx2Int),
                               .LSBLess(LSBLessADCPowSat),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCSatPowLinValidforFSMRx2),

                               //Output
                               .qdBVOut(ADCSatPowdBVIntRx2),
                               .qdBVValidforFSM(ADCSatPowdBVValidforFSMRx2)
                              );

      //Add a bias to the ADC Power
      assign ADCPowdBVTempRx2 = $signed(ADCPowdBVIntRx2 + {{4{RegADCPowBiasqdBV[DBVPOW_WIDTH-5]}},RegADCPowBiasqdBV});

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCPowOut2_Blk
            if (nAGCRst == 1'b0)
               ADCPowdBVRx2Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCPowdBVTempRx2 < ADCqdBVThr)
               ADCPowdBVRx2Temp <= ADCqdBVThr;
            else
               ADCPowdBVRx2Temp <= ADCPowdBVTempRx2;
         end //ADCPowOut2_Blk

      assign ADCPowdBVRx2 = ADCPowdBVRx2Temp;

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowOut2_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBVRx2Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCSatPowdBVIntRx2 < ADCqdBVThr)
               ADCSatPowdBVRx2Temp <= ADCqdBVThr;
            else
               ADCSatPowdBVRx2Temp <= ADCSatPowdBVIntRx2;
         end //ADCSatPowOut2_Blk

      assign ADCSatPowdBVRx2 = ADCSatPowdBVRx2Temp;

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatdBmInt2_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx2Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               ADCSatPowdBmRx2Int <= $signed({ADCSatPowdBVIntRx2[DBVPOW_WIDTH-1],ADCSatPowdBVIntRx2} +
                                     dBmTemp2 + {{4{RegADCPowBiasqdBm[DBMPOW_WIDTH-5]}},RegADCPowBiasqdBm});
         end //ADCdBmInt2_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowdBm2_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx2Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (ADCSatPowdBmRx2Int < ADCqdBmThr)
               ADCSatPowdBmRx2Temp <= ADCqdBmThr;
            else
               ADCSatPowdBmRx2Temp <= ADCSatPowdBmRx2Int;
         end //ADCSatPowdBm0_Blk

      assign ADCSatPowdBmRx2 = ADCSatPowdBmRx2Temp;

   end //NRX > 2
   else begin
      assign ADCSatPowdBVRx2 = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCPowdBVRx2    = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCSatPowdBmRx2 = CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate

//For Rx Chain 3
generate
   if (NRX > 3) begin : U_ADCPowEst_gen3
      ADCPowEst # (
                   .ADC_WIDTH(ADC_WIDTH),
                   .BW_PARAM(BW_PARAM),
                   .ADCPOWSAT_WIDTH(ADCPOWSAT_WIDTH)
                  ) ADCPowEst3 (
                                //Inputs
                                .ADCPowClk(ADCPowClk),
                                .nAGCRst(nAGCRst),
                                .RxADCRe(ADCReRx3),
                                .RxADCIm(ADCImRx3),
                                .AGCEn(ADCPowEstEn),
                                .Force20(Force20),
                                .Force40(Force40),
                                .DelLineClr(DelLineClr),
                                .ADCSWLSatConfig(ADCSWLSatConfig),
                                .ADCSWLConfig(ADCSWLConfig),

                                //Outputs
                                .ADCSatPowLin(ADCSatPowLinRx3Int),
                                .ADCSatPowLinValidforFSM(ADCSatPowLinValidforFSMRx3),
                                .ADCPowLin(ADCPowLinRx3),
                                .ADCPowLinValidforFSM(ADCPowLinValidforFSMRx3)
                               );

      //Convert ADCSatPowLinRx3 and ADCPowLinRx3 to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCPow3 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCPowLinRx3),
                               .LSBLess(LSBLessADCPow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCPowLinValidforFSMRx3),

                               //Output
                               .qdBVOut(ADCPowdBVIntRx3),
                               .qdBVValidforFSM(ADCPowdBVValidforFSMRx3)
                              );

      Lin2dBVrms # (
                    .INPUT_WIDTH(ADCPOWSAT_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBADCSatPow3 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(ADCSatPowLinRx3Int),
                               .LSBLess(LSBLessADCPowSat),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(ADCSatPowLinValidforFSMRx3),

                               //Output
                               .qdBVOut(ADCSatPowdBVIntRx3),
                               .qdBVValidforFSM(ADCSatPowdBVValidforFSMRx3)
                              );

      //Add a bias to the ADC Power
      assign ADCPowdBVTempRx3 = $signed(ADCPowdBVIntRx3 + {{4{RegADCPowBiasqdBV[DBVPOW_WIDTH-5]}},RegADCPowBiasqdBV});

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCPowOut3_Blk
            if (nAGCRst == 1'b0)
               ADCPowdBVRx3Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCPowdBVTempRx3 < ADCqdBVThr)
               ADCPowdBVRx3Temp <= ADCqdBVThr;
            else
               ADCPowdBVRx3Temp <= ADCPowdBVTempRx3;
         end //ADCPowOut3_Blk

      assign ADCPowdBVRx3 = ADCPowdBVRx3Temp;

      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowOut3_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBVRx3Temp <= CONST_ZERO_DBVPOW_WIDTH;
            else if (ADCSatPowdBVIntRx3 < ADCqdBVThr)
               ADCSatPowdBVRx3Temp <= ADCqdBVThr;
            else
               ADCSatPowdBVRx3Temp <= ADCSatPowdBVIntRx3;
         end //ADCSatPowOut3_Blk

      assign ADCSatPowdBVRx3 = ADCSatPowdBVRx3Temp;

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatdBmInt3_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx3Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               ADCSatPowdBmRx3Int <= $signed({ADCSatPowdBVIntRx3[DBVPOW_WIDTH-1],ADCSatPowdBVIntRx3} +
                                     dBmTemp3 + {{4{RegADCPowBiasqdBm[DBMPOW_WIDTH-5]}},RegADCPowBiasqdBm});
         end //ADCSatdBmInt3_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: ADCSatPowdBm3_Blk
            if (nAGCRst == 1'b0)
               ADCSatPowdBmRx3Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (ADCSatPowdBmRx3Int < ADCqdBmThr)
               ADCSatPowdBmRx3Temp <= ADCqdBmThr;
            else
               ADCSatPowdBmRx3Temp <= ADCSatPowdBmRx3Int;
         end //ADCSatPowdBm3_Blk

      assign ADCSatPowdBmRx3 = ADCSatPowdBmRx3Temp;

   end //NRX > 3
   else begin
      assign ADCSatPowdBVRx3 = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCPowdBVRx3    = CONST_ZERO_DBVPOW_WIDTH;
      assign ADCSatPowdBmRx3 = CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate

//Capture Rx gain on rising edge of InBd20PowEstEn
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RxGainCapture_Blk
      if (nAGCRst == 1'b0) begin
         RxGaindB0D <= 8'b0;
         RxGaindB1D <= 8'b0;
         RxGaindB2D <= 8'b0;
         RxGaindB3D <= 8'b0;
      end
      else if (InBd20PowEstEn == 1'b1 && InBd20PowEstEnD == 1'b0) begin
         RxGaindB0D <= RxGaindB0;
         RxGaindB1D <= RxGaindB1;
         RxGaindB2D <= RxGaindB2;
         RxGaindB3D <= RxGaindB3;
      end
   end //RxGainCapture_Blk

//Convert dBVInt to dBVrms
assign RxGainqdB0  = $signed({RxGaindB0D,2'b0});
assign RxGainqdB1  = $signed({RxGaindB1D,2'b0});
assign RxGainqdB2  = $signed({RxGaindB2D,2'b0});
assign RxGainqdB3  = $signed({RxGaindB3D,2'b0});

assign AntLossqdB0 = $signed({AntLossdB0,2'b0});
assign AntLossqdB1 = $signed({AntLossdB1,2'b0});
assign AntLossqdB2 = $signed({AntLossdB2,2'b0});
assign AntLossqdB3 = $signed({AntLossdB3,2'b0});

assign dBmTemp0    = $signed({7'h0,AntLossqdB0} - {{3{RxGainqdB0[9]}},RxGainqdB0} + 13'd52);
assign dBmTemp1    = $signed({7'h0,AntLossqdB1} - {{3{RxGainqdB1[9]}},RxGainqdB1} + 13'd52);
assign dBmTemp2    = $signed({7'h0,AntLossqdB2} - {{3{RxGainqdB2[9]}},RxGainqdB2} + 13'd52);
assign dBmTemp3    = $signed({7'h0,AntLossqdB3} - {{3{RxGainqdB3[9]}},RxGainqdB3} + 13'd52);

//Convert Thresholds in qdB
assign InBdqdBVThr = $signed({InBddBVThr, 2'b00});
assign InBdqdBmThr = $signed({InBddBmThr, 2'b00});
assign ADCqdBVThr  = $signed({ADCdBVThr, 2'b00});
assign ADCqdBmThr  = $signed({ADCdBmThr, 2'b00});

//In Band Power Estimation using Shared Delay Line for 20 MHz packet
//For Rx Chain 0
InBdPowEstShDel # (
                   .INPUT_WIDTH(INBD_WIDTH),
                   .OUTPUT_WIDTH(INBDPOW_WIDTH)
                  ) 
                    U_InBd20PriPow0 (
                                 //Inputs
                                 .AGCClk(AGCClk),
                                 .nAGCRst(nAGCRst),
                                 .RxAGCRe(RxAGC20PriReRx0),
                                 .RxAGCIm(RxAGC20PriImRx0),
                                 .RxDataValid(RxAGC20PriDataValid),
                                 .AGCEn(InBd20PowEstEn),
                                 .AddZeroValue(AddZeroInBd20),
                                 .DelLineClr(DelLineClr),
                                 .InBdSWLConfig(InBd20SWLConfig),
                                 .RegInBdRnd(RegInBdRnd),
                                 .RxAGCRe8D(AGCRe8DRx0),
                                 .RxAGCIm8D(AGCIm8DRx0),
                                 .RxAGCRe16D(AGCRe16DRx0),
                                 .RxAGCIm16D(AGCIm16DRx0),
                                 .RxAGCRe32D(AGCRe32DRx0),
                                 .RxAGCIm32D(AGCIm32DRx0),
                                 .RxAGCRe48D(AGCRe48DRx0),
                                 .RxAGCIm48D(AGCIm48DRx0),
                                 .RxAGCRe64D(AGCRe64DRx0),
                                 .RxAGCIm64D(AGCIm64DRx0),

                                 //Outputs
                                 .InBdPowLin(InBd20PowLinRx0),
                                 .InBdPowLinValidforFSM(InBd20PowLinValidforFSMRx0),
                                 .InBdPowLinValidforCCA(InBd20PowLinValidforCCARx0)
                                 );

//Convert linear power value to dBVrms
Lin2dBVrms # (
              .INPUT_WIDTH(INBDPOW_WIDTH),
              .OUTPUT_WIDTH(DBVPOW_WIDTH),
              .FSB_ITER(FSB_ITER)
             ) Lin2dBInbd20Pow0 (
                         //Inputs
                         .AGCClk(AGCClk),
                         .nAGCRst(nAGCRst),
                         .LinValIn(InBd20PowLinRx0),
                         .LSBLess(LSBLessInBd20Pow),
                         .NBitADC(NBitADC),
                         .VPeakADCqdBV(VPeakADCqdBV),
                         .LinDataValidforFSM(InBd20PowLinValidforFSMRx0),

                         //Output
                         .qdBVOut(InBd20PowdBVIntRx0),
                         .qdBVValidforFSM(InBd20PowdBVValidforFSMRx0)
                        );

//Convert dBVInt to dBVrms
assign InBd20PowdBVEffRx0 = InBd20PowdBVIntRx0 - $signed({{DBVPOW_WIDTH-8{FE20Gain[7]}},FE20Gain});

//Saturate the dBVrms Value
always @ (*)
   begin: InBd20PowdBV0_Blk
      if (InBd20PowdBVEffRx0 < InBdqdBVThr)
         InBd20PowdBVEffSatRx0 = InBdqdBVThr;
      else
         InBd20PowdBVEffSatRx0 = InBd20PowdBVEffRx0;
   end //InBd20PowdBV0_Blk

//Convert dBVrms to dBm
always @ (posedge AGCClk or negedge nAGCRst)
   begin: InBd20dBmInt0_Blk
      if (nAGCRst == 1'b0)
         InBd20PowdBmRx0Int <= CONST_ZERO_DBMPOW_WIDTH;
      else
         InBd20PowdBmRx0Int <= $signed({InBd20PowdBVEffSatRx0[DBVPOW_WIDTH-1],InBd20PowdBVEffSatRx0} +
                               dBmTemp0 + {{4{RegInBd20PowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd20PowBiasqdBm});
   end //InBd20dBmInt0_Blk

//Saturate the dBm Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: InBd20PowdBm0_Blk
      if (nAGCRst == 1'b0)
         InBd20PowdBmRx0Temp <= CONST_ZERO_DBMPOW_WIDTH;
      else if (InBd20PowdBmRx0Int < InBdqdBmThr)
         InBd20PowdBmRx0Temp <= InBdqdBmThr;
      else
         InBd20PowdBmRx0Temp <= InBd20PowdBmRx0Int;
   end //InBd20PowdBm0_Blk

assign InBd20PowdBmRx0 = InBd20PowdBmRx0Temp;

// Disable data valid when InBdSecPowEstEn=0
assign InBd20PowdBVValidforFSMRx0Mux = InBd20PowdBVValidforFSMRx0 & InBd20PowEstEn;

//Generate InBd20PowValid
DelayLine # (
             .DATAWIDTH(1), //Width of InBd20PowdBVValidforFSMRx0
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLine1 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(~InBd20PowEstEn),
                            .DataIn(InBd20PowdBVValidforFSMRx0Mux),
                            .DataOut(InBd20PowValid)
                           );

// Disable data valid when InBd20PowEstEn=0
assign InBd20PowLinValidforCCARx0Mux = InBd20PowLinValidforCCARx0 & InBd20PowEstEn;

//Generate InBd20PowCCAValid
DelayLine # (
             .DATAWIDTH(1), //Width of InBd20PowLinValidforCCARx0
             .LATENCY(8)    //8 clocks latency
             )
              U_DelayLine2 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(~InBd20PowEstEn),
                            .DataIn(InBd20PowLinValidforCCARx0Mux),
                            .DataOut(InBd20PowCCAValid)
                           );

//For Rx Chain 1
generate
   if (NRX > 1) begin : U_InBd20PowPriEstShDel_gen1
      InBdPowEstShDel # (
                         .INPUT_WIDTH(INBD_WIDTH),
                         .OUTPUT_WIDTH(INBDPOW_WIDTH)
                        ) 
                          U_InBd20PriPow1 (
                                       //Inputs
                                       .AGCClk(AGCClk),
                                       .nAGCRst(nAGCRst),
                                       .RxAGCRe(RxAGC20PriReRx1),
                                       .RxAGCIm(RxAGC20PriImRx1),
                                       .RxDataValid(RxAGC20PriDataValid),
                                       .AGCEn(InBd20PowEstEn),
                                       .AddZeroValue(AddZeroInBd20),
                                       .DelLineClr(DelLineClr),
                                       .InBdSWLConfig(InBd20SWLConfig),
                                       .RegInBdRnd(RegInBdRnd),
                                       .RxAGCRe8D(AGCRe8DRx1),
                                       .RxAGCIm8D(AGCIm8DRx1),
                                       .RxAGCRe16D(AGCRe16DRx1),
                                       .RxAGCIm16D(AGCIm16DRx1),
                                       .RxAGCRe32D(AGCRe32DRx1),
                                       .RxAGCIm32D(AGCIm32DRx1),
                                       .RxAGCRe48D(AGCRe48DRx1),
                                       .RxAGCIm48D(AGCIm48DRx1),
                                       .RxAGCRe64D(AGCRe64DRx1),
                                       .RxAGCIm64D(AGCIm64DRx1),

                                       //Outputs
                                       .InBdPowLin(InBd20PowLinRx1),
                                       .InBdPowLinValidforFSM(InBd20PowLinValidforFSMRx1),
                                       .InBdPowLinValidforCCA()
                                       );

      //Convert linear power value to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(INBDPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBInbd20Pow1 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(InBd20PowLinRx1),
                               .LSBLess(LSBLessInBd20Pow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(InBd20PowLinValidforFSMRx1),

                               //Output
                               .qdBVOut(InBd20PowdBVIntRx1),
                               .qdBVValidforFSM()
                              );

      //Convert dBVInt to dBVrms
      assign InBd20PowdBVEffRx1 = InBd20PowdBVIntRx1 - $signed({{DBVPOW_WIDTH-8{FE20Gain[7]}},FE20Gain});

      //Saturate the dBVrms Value
      always @ (*)
         begin: InBd20PowdBV1_Blk
            if (InBd20PowdBVEffRx1 < InBdqdBVThr)
               InBd20PowdBVEffSatRx1 = InBdqdBVThr;
            else
               InBd20PowdBVEffSatRx1 = InBd20PowdBVEffRx1;
         end //InBd20PowdBV1_Blk

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20dBmInt1_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx1Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               InBd20PowdBmRx1Int <= $signed({InBd20PowdBVEffSatRx1[DBVPOW_WIDTH-1],InBd20PowdBVEffSatRx1} +
                                     dBmTemp1 + {{4{RegInBd20PowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd20PowBiasqdBm});
         end //InBd20dBmInt1_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20PowdBm1_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx1Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (InBd20PowdBmRx1Int < InBdqdBmThr)
               InBd20PowdBmRx1Temp <= InBdqdBmThr;
            else
               InBd20PowdBmRx1Temp <= InBd20PowdBmRx1Int;
         end //InBd20PowdBm1_Blk

      assign InBd20PowdBmRx1 = InBd20PowdBmRx1Temp;

   end //NRX > 1
   else begin
      assign InBd20PowdBmRx1 = CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate

//For Rx Chain 2
generate
   if (NRX > 2) begin : U_InBd20PowPriEstShDel_gen2
      InBdPowEstShDel # (
                         .INPUT_WIDTH(INBD_WIDTH),
                         .OUTPUT_WIDTH(INBDPOW_WIDTH)
                        ) 
                          U_InBd20PriPow2 (
                                       //Inputs
                                       .AGCClk(AGCClk),
                                       .nAGCRst(nAGCRst),
                                       .RxAGCRe(RxAGC20PriReRx2),
                                       .RxAGCIm(RxAGC20PriImRx2),
                                       .RxDataValid(RxAGC20PriDataValid),
                                       .AGCEn(InBd20PowEstEn),
                                       .AddZeroValue(AddZeroInBd20),
                                       .DelLineClr(DelLineClr),
                                       .InBdSWLConfig(InBd20SWLConfig),
                                       .RegInBdRnd(RegInBdRnd),
                                       .RxAGCRe8D(AGCRe8DRx2),
                                       .RxAGCIm8D(AGCIm8DRx2),
                                       .RxAGCRe16D(AGCRe16DRx2),
                                       .RxAGCIm16D(AGCIm16DRx2),
                                       .RxAGCRe32D(AGCRe32DRx2),
                                       .RxAGCIm32D(AGCIm32DRx2),
                                       .RxAGCRe48D(AGCRe48DRx2),
                                       .RxAGCIm48D(AGCIm48DRx2),
                                       .RxAGCRe64D(AGCRe64DRx2),
                                       .RxAGCIm64D(AGCIm64DRx2),

                                       //Outputs
                                       .InBdPowLin(InBd20PowLinRx2),
                                       .InBdPowLinValidforFSM(InBd20PowLinValidforFSMRx2),
                                       .InBdPowLinValidforCCA()
                                       );

      //Convert linear power value to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(INBDPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBInbd20Pow2 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(InBd20PowLinRx2),
                               .LSBLess(LSBLessInBd20Pow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(InBd20PowLinValidforFSMRx2),

                               //Output
                               .qdBVOut(InBd20PowdBVIntRx2),
                               .qdBVValidforFSM()
                              );

      //Convert dBVInt to dBVrms
      assign InBd20PowdBVEffRx2 = InBd20PowdBVIntRx2 - $signed({{DBVPOW_WIDTH-8{FE20Gain[7]}},FE20Gain});

      //Saturate the dBVrms Value
      always @ (*)
         begin: InBd20PowdBV2_Blk
            if (InBd20PowdBVEffRx2 < InBdqdBVThr)
               InBd20PowdBVEffSatRx2 = InBdqdBVThr;
            else
               InBd20PowdBVEffSatRx2 = InBd20PowdBVEffRx2;
         end //InBd20PowdBV2_Blk

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20dBmInt2_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx2Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               InBd20PowdBmRx2Int <= $signed({InBd20PowdBVEffSatRx2[DBVPOW_WIDTH-1],InBd20PowdBVEffSatRx2} +
                                     dBmTemp2 + {{4{RegInBd20PowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd20PowBiasqdBm});
         end //InBd20dBmInt2_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20PowdBm2_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx2Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (InBd20PowdBmRx2Int < InBdqdBmThr)
               InBd20PowdBmRx2Temp <= InBdqdBmThr;
            else
               InBd20PowdBmRx2Temp <= InBd20PowdBmRx2Int;
         end //InBd20PowdBm2_Blk

      assign InBd20PowdBmRx2 = InBd20PowdBmRx2Temp;

   end //NRX > 2
   else begin
      assign InBd20PowdBmRx2 = CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate

//For Rx Chain 3
generate
   if (NRX > 3) begin : U_InBd20PowPriEstShDel_gen3
      InBdPowEstShDel # (
                         .INPUT_WIDTH(INBD_WIDTH),
                         .OUTPUT_WIDTH(INBDPOW_WIDTH)
                        ) 
                          U_InBd20PriPow3 (
                                       //Inputs
                                       .AGCClk(AGCClk),
                                       .nAGCRst(nAGCRst),
                                       .RxAGCRe(RxAGC20PriReRx3),
                                       .RxAGCIm(RxAGC20PriImRx3),
                                       .RxDataValid(RxAGC20PriDataValid),
                                       .AGCEn(InBd20PowEstEn),
                                       .AddZeroValue(AddZeroInBd20),
                                       .DelLineClr(DelLineClr),
                                       .InBdSWLConfig(InBd20SWLConfig),
                                       .RegInBdRnd(RegInBdRnd),
                                       .RxAGCRe8D(AGCRe8DRx3),
                                       .RxAGCIm8D(AGCIm8DRx3),
                                       .RxAGCRe16D(AGCRe16DRx3),
                                       .RxAGCIm16D(AGCIm16DRx3),
                                       .RxAGCRe32D(AGCRe32DRx3),
                                       .RxAGCIm32D(AGCIm32DRx3),
                                       .RxAGCRe48D(AGCRe48DRx3),
                                       .RxAGCIm48D(AGCIm48DRx3),
                                       .RxAGCRe64D(AGCRe64DRx3),
                                       .RxAGCIm64D(AGCIm64DRx3),

                                       //Outputs
                                       .InBdPowLin(InBd20PowLinRx3),
                                       .InBdPowLinValidforFSM(InBd20PowLinValidforFSMRx3),
                                       .InBdPowLinValidforCCA()
                                       );

      //Convert linear power value to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(INBDPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBInbd20Pow3 (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(InBd20PowLinRx3),
                               .LSBLess(LSBLessInBd20Pow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(InBd20PowLinValidforFSMRx3),

                               //Output
                               .qdBVOut(InBd20PowdBVIntRx3),
                               .qdBVValidforFSM()
                              );

      //Convert dBVInt to dBVrms
      assign InBd20PowdBVEffRx3 = InBd20PowdBVIntRx3 - $signed({{DBVPOW_WIDTH-8{FE20Gain[7]}},FE20Gain});

      //Saturate the dBVrms Value
      always @ (*)
         begin: InBd20PowdBV3_Blk
            if (InBd20PowdBVEffRx3 < InBdqdBVThr)
               InBd20PowdBVEffSatRx3 = InBdqdBVThr;
            else
               InBd20PowdBVEffSatRx3 = InBd20PowdBVEffRx3;
         end //InBd20PowdBV3_Blk

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20dBmInt3_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx3Int <= CONST_ZERO_DBMPOW_WIDTH;
            else
               InBd20PowdBmRx3Int <= $signed({InBd20PowdBVEffSatRx3[DBVPOW_WIDTH-1],InBd20PowdBVEffSatRx3} +
                                     dBmTemp3 + {{4{RegInBd20PowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd20PowBiasqdBm});
         end //InBd20dBmInt3_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20PowdBm3_Blk
            if (nAGCRst == 1'b0)
               InBd20PowdBmRx3Temp <= CONST_ZERO_DBMPOW_WIDTH;
            else if (InBd20PowdBmRx3Int < InBdqdBmThr)
               InBd20PowdBmRx3Temp <= InBdqdBmThr;
            else
               InBd20PowdBmRx3Temp <= InBd20PowdBmRx3Int;
         end //InBd20PowdBm3_Blk

      assign InBd20PowdBmRx3 = InBd20PowdBmRx3Temp;

   end // NRX > 3
   else begin
      assign InBd20PowdBmRx3 = CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate


//In Band Power in Secondary 20 MHz Channel.
//Required only when BW_PARAM = 40 or 80.
//According to AntSelCCA20s
generate
   if (BW_PARAM == 40 || BW_PARAM == 80) begin : U_InBd20SecPowEst_gen

      // Select Rx path according to AntSelCCA20s from AGC FSM
      always @ (*)
      begin: RxAGC20SecSel_Blk
            case (AntSelCCA20s)
               2'b00: begin
                        RxAGC20SecRe = RxAGC20SecReRx0;
                        RxAGC20SecIm = RxAGC20SecImRx0;
                      end
               2'b01: begin
                        RxAGC20SecRe = RxAGC20SecReRx1;
                        RxAGC20SecIm = RxAGC20SecImRx1;
                      end
               2'b10: begin
                        RxAGC20SecRe = RxAGC20SecReRx2;
                        RxAGC20SecIm = RxAGC20SecImRx2;
                      end
               default: begin
                        RxAGC20SecRe = RxAGC20SecReRx3;
                        RxAGC20SecIm = RxAGC20SecImRx3;
                      end
            endcase
      end //RxAGC20SecSel_Blk

      InBdPowEst #(
                      //Parameters
                    .INPUT_WIDTH(INBD_WIDTH),
                    .OUTPUT_WIDTH(INBDPOW_WIDTH),
                    .BW_PARAM(20) // Force 20MHz bandwith
                     ) U_InBd20SecPowEst (
                                        //Inputs
                                        .AGCClk(AGCClk),
                                        .nAGCRst(nAGCRst),
                                        .AGCEn(InBdSecPowEstEn),
                                        .RxAGCRe(RxAGC20SecRe),
                                        .RxAGCIm(RxAGC20SecIm),
                                        .RxDataValid(RxAGC20SecDataValid),
                                        .AddZeroValue(AddZeroInBdSec20),
                                        .DelLineClr(DelLineClr),
                                        .InBdSWLConfig(InBd20SecSWLConfig),
                                        .RegInBdRnd(RegInBdRnd),

                                        //Outputs
                                        .InBdPowLin(InBd20SecPowLin),
                                        .InBdPowLinValidforFSM(InBd20SecPowLinValidforFSM)
                                       );
      
      //Convert linear power value to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(INBDPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBInbd20SecPow (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(InBd20SecPowLin),
                               .LSBLess(LSBLessInBd20SecPow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(InBd20SecPowLinValidforFSM),

                               //Output
                               .qdBVOut(InBd20SecPowdBVInt),
                               .qdBVValidforFSM(InBd20SecPowdBVValidforFSM)
                              );

      //Convert dBVInt to dBVrms
      assign InBd20SecPowdBVEff = InBd20SecPowdBVInt - $signed({{DBVPOW_WIDTH-8{FE20Gain[7]}},FE20Gain});
      
      //Saturate the dBVrms Value
      always @ (*)
         begin: InBd20SecPowdBV_Blk
            if (InBd20SecPowdBVEff < InBdqdBVThr)
               InBd20SecPowdBVEffSat = InBdqdBVThr;
            else
               InBd20SecPowdBVEffSat = InBd20SecPowdBVEff;
         end //InBd20SecPowdBV_Blk

      // Select Rx gain according to AntSelCCA20s from AGC FSM
      always @ (*)
      begin: dBmTemp20sSel_Blk
            case (AntSelCCA20s)
               2'b00:   dBmTemp20s = dBmTemp0;
               2'b01:   dBmTemp20s = dBmTemp1;
               2'b10:   dBmTemp20s = dBmTemp2;
               default: dBmTemp20s = dBmTemp3;
            endcase
      end //dBmTemp20sSel_Blk
      
      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd20SecPowdBmTemp_Blk
            if (nAGCRst == 1'b0)
               InBd20SecPowdBmTemp <= CONST_ZERO_DBMPOW_WIDTH;
            else
               InBd20SecPowdBmTemp <= $signed({InBd20SecPowdBVEffSat[DBVPOW_WIDTH-1],InBd20SecPowdBVEffSat} +
                                      dBmTemp20s + {{4{RegInBd20SecPowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd20SecPowBiasqdBm});

         end //InBd20SecPowdBmTemp0_Blk

      //Saturate the dBm Value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: SecPowdBm_Blk
            if (nAGCRst == 1'b0)
               InBd20SecPowdBmInt <= CONST_ZERO_DBMPOW_WIDTH;
            else if (InBd20SecPowdBmTemp < InBdqdBmThr)
               InBd20SecPowdBmInt <= InBdqdBmThr;
            else
               InBd20SecPowdBmInt <= InBd20SecPowdBmTemp;

         end //SecPowdBm_Blk

      // Disable data valid when InBdSecPowEstEn=0
      assign InBd20SecPowdBVValidforFSMMux = InBd20SecPowdBVValidforFSM & InBdSecPowEstEn;

      //Generate InBd20SecPowValid
      DelayLine # (
                   .DATAWIDTH(1), //Width of InBd20SecPowdBVValidforFSMMux
                   .LATENCY(2)    //2 clocks latency
                   )
                    U_DelayLine3 (
                                  .PhyClk(AGCClk),
                                  .nPhyRst(nAGCRst),
                                  .Bypass(~InBdSecPowEstEn),
                                  .DataIn(InBd20SecPowdBVValidforFSMMux),
                                  .DataOut(InBd20SecPowValidTemp)
                                 );
   end //BW_PARAM == 40 || BW_PARAM == 80
   else begin
      assign InBd20SecPowValidTemp = 1'b0;
   end
endgenerate

assign InBd20SecPowValid = InBd20SecPowValidTemp;

generate
   if (BW_PARAM == 20) begin : U_InBd20SecPowdBm_nogen
      assign InBd20SecPowdBm = CONST_ZERO_DBMPOW_WIDTH;
   end //BW_PARAM = 20 
   else begin : U_InBd20SecPowdBm_gen
      assign InBd20SecPowdBm = InBd20SecPowdBmInt;
   end
endgenerate


//In Band Power Estimation using Self Delay Line for 80 MHz packets
generate
   if (BW_PARAM == 80) begin : U_InBd40SecPowEst_gen
      
      // Select Rx path according to AntSelCCA40s from AGC FSM
      always @ (*)
      begin: RxAGC40SecSel_Blk
            case (AntSelCCA40s)
               2'b00: begin
                        RxAGC40SecRe = RxAGC40SecReRx0;
                        RxAGC40SecIm = RxAGC40SecImRx0;
                      end
               2'b01: begin
                        RxAGC40SecRe = RxAGC40SecReRx1;
                        RxAGC40SecIm = RxAGC40SecImRx1;
                      end
               2'b10: begin
                        RxAGC40SecRe = RxAGC40SecReRx2;
                        RxAGC40SecIm = RxAGC40SecImRx2;
                      end
               default: begin
                        RxAGC40SecRe = RxAGC40SecReRx3;
                        RxAGC40SecIm = RxAGC40SecImRx3;
                      end
            endcase
      end //RxAGC40SecSel_Blk
      
      InBdPowEst # (
                    .INPUT_WIDTH(INBD_WIDTH),
                    .OUTPUT_WIDTH(INBDPOW_WIDTH),
                    .BW_PARAM(40) // Force 40MHz bandwith
                   ) 
                     U_InBd40SecPow (
                                  //Inputs
                                  .AGCClk(AGCClk),
                                  .nAGCRst(nAGCRst),
                                  .AGCEn(InBdSecPowEstEn),
                                  .RxAGCRe(RxAGC40SecRe),
                                  .RxAGCIm(RxAGC40SecIm),
                                  .RxDataValid(RxAGC40SecDataValid),
                                  .AddZeroValue(AddZeroInBdSec40),
                                  .DelLineClr(DelLineClr),
                                  .InBdSWLConfig(InBd40SecSWLConfig),
                                  .RegInBdRnd(RegInBdRnd),

                                  //Outputs
                                  .InBdPowLin(InBd40SecPowLin),
                                  .InBdPowLinValidforFSM(InBd40SecPowLinValidforFSM)
                                  );

      //Convert linear power value to dBVrms
      Lin2dBVrms # (
                    .INPUT_WIDTH(INBDPOW_WIDTH),
                    .OUTPUT_WIDTH(DBVPOW_WIDTH),
                    .FSB_ITER(FSB_ITER)
                   ) Lin2dBInbd40SecPow (
                               //Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .LinValIn(InBd40SecPowLin),
                               .LSBLess(LSBLessInBd40SecPow),
                               .NBitADC(NBitADC),
                               .VPeakADCqdBV(VPeakADCqdBV),
                               .LinDataValidforFSM(InBd40SecPowLinValidforFSM),

                               //Output
                               .qdBVOut(InBd40SecPowdBVInt),
                               .qdBVValidforFSM(InBd40SecPowdBVValidforFSM)
                              );

      //Convert dBVInt to dBVrms
      assign InBd40SecPowdBVEff = InBd40SecPowdBVInt - $signed({{DBVPOW_WIDTH-8{FE40Gain[7]}},FE40Gain});

      //Saturate the dBVrms Value
      always @ (*)
         begin: InBd40SecPowdBV_Blk
            if (InBd40SecPowdBVEff < InBdqdBVThr)
               InBd40SecPowdBVEffSat = InBdqdBVThr;
            else
               InBd40SecPowdBVEffSat = InBd40SecPowdBVEff;
         end //InBd40SecPowdBV_Blk

      // Select Rx gain according to AntSelCCA40s from AGC FSM
      always @ (*)
      begin: dBmTemp40sSel_Blk
            case (AntSelCCA40s)
               2'b00:   dBmTemp40s = dBmTemp0;
               2'b01:   dBmTemp40s = dBmTemp1;
               2'b10:   dBmTemp40s = dBmTemp2;
               default: dBmTemp40s = dBmTemp3;
            endcase
      end //dBmTemp40sSel_Blk

      //Convert dBVrms to dBm
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd40SecPowdBm_Blk
            if (nAGCRst == 1'b0)
               InBd40SecPowdBmTemp <= CONST_ZERO_DBMPOW_WIDTH;
            else
               InBd40SecPowdBmTemp <= $signed({InBd40SecPowdBVEffSat[DBVPOW_WIDTH-1],InBd40SecPowdBVEffSat} +
                                      dBmTemp40s + {{4{RegInBd40SecPowBiasqdBm[DBMPOW_WIDTH-5]}},RegInBd40SecPowBiasqdBm});
         end //InBd40SecPowdBm_Blk

      //Saturate the dBm value
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: InBd40SecdBm_Blk
            if (nAGCRst == 1'b0)
               InBd40SecPowdBmInt <= CONST_ZERO_DBMPOW_WIDTH;
            else if (InBd40SecPowdBmTemp < InBdqdBmThr)
               InBd40SecPowdBmInt <= InBdqdBmThr;
            else
               InBd40SecPowdBmInt <= InBd40SecPowdBmTemp;
         end //InBd40SecdBm_Blk
      
      // Disable data valid when InBdSecPowEstEn=0
      assign InBd40SecPowdBVValidforFSMMux = InBd40SecPowdBVValidforFSM & InBdSecPowEstEn;
      
      //Generate InBd40SecPowValid
      DelayLine # (
                   .DATAWIDTH(1), //Width of InBd40SecPowdBVValidforFSMMux
                   .LATENCY(2)    //2 clocks latency
                   )
                    U_DelayLine4 (
                                  .PhyClk(AGCClk),
                                  .nPhyRst(nAGCRst),
                                  .Bypass(~InBdSecPowEstEn),
                                  .DataIn(InBd40SecPowdBVValidforFSMMux),
                                  .DataOut(InBd40SecPowValidTemp)
                                 );

   end //BW_PARAM = 80 
   else begin
      assign InBd40SecPowValidTemp = 1'b0;
   end
endgenerate

assign InBd40SecPowValid = InBd40SecPowValidTemp;

generate
   if (BW_PARAM == 80) begin : U_InBd40SecPowdBm_gen
      assign InBd40SecPowdBm = InBd40SecPowdBmInt;
   end //BW_PARAM = 80 
   else begin
      assign InBd40SecPowdBm =CONST_ZERO_DBMPOW_WIDTH;
   end
endgenerate

//Generate AddZeroInBd20
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddZeroInBd20Cnt_Blk
      if (nAGCRst == 1'b0)
         AddZeroInBd20Cnt <= 7'b0;
      else if (InBd20PowEstEn == 1'b0)
         AddZeroInBd20Cnt <= 7'b0;
      else if (InBd20PowEstEn == 1'b1 && InBd20PowEstEnD == 1'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBd20Cnt <= 7'b1;
      else if (AddZeroInBd20Cnt == 7'd7 && InBd20SWLConfig == 3'h0 && RxAGC20PriDataValid3D == 1'b1)  // 0.4us
         AddZeroInBd20Cnt <= 7'b0;
      else if (AddZeroInBd20Cnt == 7'd15 && InBd20SWLConfig == 3'h1 && RxAGC20PriDataValid3D == 1'b1) // 0.8us
         AddZeroInBd20Cnt <= 7'b0;
      else if (AddZeroInBd20Cnt == 7'd31 && InBd20SWLConfig == 3'h2 && RxAGC20PriDataValid3D == 1'b1) // 1.6us
         AddZeroInBd20Cnt <= 7'b0;
      else if (AddZeroInBd20Cnt == 7'd47 && InBd20SWLConfig == 3'h3 && RxAGC20PriDataValid3D == 1'b1) // 2.4us
         AddZeroInBd20Cnt <= 7'b0;
      else if (AddZeroInBd20Cnt == 7'd63 && InBd20SWLConfig == 3'h4 && RxAGC20PriDataValid3D == 1'b1) // 3.2us
         AddZeroInBd20Cnt <= 7'b0;
      else if (AddZeroInBd20Cnt != 7'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBd20Cnt <= AddZeroInBd20Cnt + 7'b1;
   end //AddZeroInBd20Cnt_Blk

assign AddZeroInBd20CntGt0 = (AddZeroInBd20Cnt != 7'b0) ? 1'b1 : 1'b0;
assign AddZeroInBd20       = (InBd20PowEstEn & ~InBd20PowEstEnD) | AddZeroInBd20CntGt0;

//Generate AddZeroInBdSec20
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddZeroInBdSec20Cnt_Blk
      if (nAGCRst == 1'b0)
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (InBdSecPowEstEn == 1'b0)
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (InBdSecPowEstEn == 1'b1 && InBdSecPowEstEnD == 1'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBdSec20Cnt <= 7'b1;
      else if (AddZeroInBdSec20Cnt == 7'd7 && InBd20SecSWLConfig == 3'h0 && RxAGC20PriDataValid3D == 1'b1)  // 0.4us
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (AddZeroInBdSec20Cnt == 7'd15 && InBd20SecSWLConfig == 3'h1 && RxAGC20PriDataValid3D == 1'b1) // 0.8us
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (AddZeroInBdSec20Cnt == 7'd31 && InBd20SecSWLConfig == 3'h2 && RxAGC20PriDataValid3D == 1'b1) // 1.6us
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (AddZeroInBdSec20Cnt == 7'd47 && InBd20SecSWLConfig == 3'h3 && RxAGC20PriDataValid3D == 1'b1) // 2.4us
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (AddZeroInBdSec20Cnt == 7'd63 && InBd20SecSWLConfig == 3'h4 && RxAGC20PriDataValid3D == 1'b1) // 3.2us
         AddZeroInBdSec20Cnt <= 7'b0;
      else if (AddZeroInBdSec20Cnt != 7'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBdSec20Cnt <= AddZeroInBdSec20Cnt + 7'b1;
   end //AddZeroInBdSec20Cnt_Blk

assign AddZeroInBdSec20CntGt0 = (AddZeroInBdSec20Cnt != 7'b0) ? 1'b1 : 1'b0;
assign AddZeroInBdSec20       = (InBdSecPowEstEn & ~InBdSecPowEstEnD) | AddZeroInBdSec20CntGt0;

//Generate AddZeroInBdSec40
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddZeroInBdSec40Cnt_Blk
      if (nAGCRst == 1'b0)
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (InBdSecPowEstEn == 1'b0)
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (InBdSecPowEstEn == 1'b1 && InBdSecPowEstEnD == 1'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBdSec40Cnt <= 7'b1;
      else if (AddZeroInBdSec40Cnt == 7'd7 && InBd40SecSWLConfig == 3'h0 && RxAGC20PriDataValid3D == 1'b1)  // 0.4us
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (AddZeroInBdSec40Cnt == 7'd15 && InBd40SecSWLConfig == 3'h1 && RxAGC20PriDataValid3D == 1'b1) // 0.8us
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (AddZeroInBdSec40Cnt == 7'd31 && InBd40SecSWLConfig == 3'h2 && RxAGC20PriDataValid3D == 1'b1) // 1.6us
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (AddZeroInBdSec40Cnt == 7'd47 && InBd40SecSWLConfig == 3'h3 && RxAGC20PriDataValid3D == 1'b1) // 2.4us
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (AddZeroInBdSec40Cnt == 7'd63 && InBd40SecSWLConfig == 3'h4 && RxAGC20PriDataValid3D == 1'b1) // 3.2us
         AddZeroInBdSec40Cnt <= 7'b0;
      else if (AddZeroInBdSec40Cnt != 7'b0 && RxAGC20PriDataValid3D == 1'b1)
         AddZeroInBdSec40Cnt <= AddZeroInBdSec40Cnt + 7'b1;
   end //AddZeroInBdSec40Cnt_Blk

assign AddZeroInBdSec40CntGt0 = (AddZeroInBdSec40Cnt != 7'b0) ? 1'b1 : 1'b0;
assign AddZeroInBdSec40       = (InBdSecPowEstEn & ~InBdSecPowEstEnD) | AddZeroInBdSec40CntGt0;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: InBdPowEstEnD_Blk
      if (nAGCRst == 1'b0) begin
         InBd20PowEstEnD       <= 1'b0;
         InBdSecPowEstEnD      <= 1'b0;
         RxAGC20PriDataValid1D <= 1'b0;
         RxAGC20PriDataValid2D <= 1'b0;
         RxAGC20PriDataValid3D <= 1'b0;
      end
      else begin
         if (!InBd20PowEstEn)
           InBd20PowEstEnD  <= InBd20PowEstEn;
         else if (RxAGC20PriDataValid3D)
           InBd20PowEstEnD  <= InBd20PowEstEn;
         if (!InBdSecPowEstEn)
           InBdSecPowEstEnD <= InBdSecPowEstEn;
         else if (RxAGC20PriDataValid3D)
           InBdSecPowEstEnD <= InBdSecPowEstEn;
         RxAGC20PriDataValid1D <= RxAGC20PriDataValid;
         RxAGC20PriDataValid2D <= RxAGC20PriDataValid1D;
         RxAGC20PriDataValid3D <= RxAGC20PriDataValid2D;
      end
   end //InBdPowEstEnD_Blk


endmodule //AGCPowEstTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
