////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 19268 $
// $Date: 2015-05-06 17:38:19 +0200 (Wed, 06 May 2015) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : In Band Power Est Block for AGC with Self Delay Line
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/PowerEst/verilog/rtl/InBdPowEst.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

module InBdPowEst #(parameter INPUT_WIDTH  = 13, //Input Data Width
                    parameter OUTPUT_WIDTH = 24, //Output Data Width
                    parameter BW_PARAM     = 40  //Bandwidth (20/40)
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active Low Reset
            input    wire                                  AGCClk,  //AGC Clock

            //Control Signals
            input    wire                                  AGCEn,   //Block Enable
            input    wire                                  AddZeroValue, //Add zeros instead of samples from delay block for first 0.4,0.8,1.6,2.4 or 3.2 us
            input    wire                                  RxDataValid, //Qualifies input data
            input    wire                                  DelLineClr, //Clear Delay Line

            //Registers
            input    wire[2:0]                             InBdSWLConfig, //Delay Line Output Sel
            input    wire[1:0]                             RegInBdRnd, //Rounding of Abs value

            //Data - from Front End
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe, //Real Component
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm, //Imaginary Component

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signal
            output    reg                                  InBdPowLinValidforFSM, //Power Valid
            //Power Estimate
            output    reg         [OUTPUT_WIDTH-1:0]       InBdPowLin //In Band Power
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam                          CONSTZERO              = 32'd0;     // Zero
localparam signed [INPUT_WIDTH-1:0] CONST_ZERO_INPUT_WIDTH = {{INPUT_WIDTH}{1'b0}};
localparam                          DELAY_LINE_WIDTH       = (BW_PARAM/20*64)-1;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire        [INPUT_WIDTH-1:0]         RxDataIntAbs;
wire        [INPUT_WIDTH-4:0]         RxDataIntAbsRnd3;
wire        [INPUT_WIDTH-3:0]         RxDataIntAbsRnd2;
wire        [INPUT_WIDTH-2:0]         RxDataIntAbsRnd1;
wire                                  RxDataIntValid3D;
wire                                  RxDataIntValid5D;
wire        [OUTPUT_WIDTH-1:0]        InBdPowLinSat;
wire        [(2*(INPUT_WIDTH-3))+4:0] InBdPowLinRnd;
wire        [INPUT_WIDTH-4:0]         RxDataIntAbsSat;
wire        [(2*(INPUT_WIDTH-3))-1:0] MuxRxDataPow1;
wire signed [(2*(INPUT_WIDTH-3)):0]   RxDataPowDiff;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg signed [INPUT_WIDTH-1:0]         RxDataIntRe;
reg signed [INPUT_WIDTH-1:0]         RxDataIntIm;
reg        [INPUT_WIDTH-4:0]         RxDataInt1;
reg                                  RxDataIntValid;
reg        [INPUT_WIDTH-4:0]         RxDelayLine[DELAY_LINE_WIDTH:0];
reg        [(2*(INPUT_WIDTH-3))-1:0] RxDataPow0;
reg        [(2*(INPUT_WIDTH-3))-1:0] RxDataPow1;
reg        [(2*(INPUT_WIDTH-3))+6:0] InBdPowLinInt;
reg        [8:0]                     TermCnt0;
reg        [8:0]                     DelayCnt0;
reg        [INPUT_WIDTH-1:0]         RxDataIntAbsRnd;
reg                                  AddZeroValueDel;

integer i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module calculates the In Band Power Estimate using a delay line. This
//line is not shared with other blocks, hence the name "Self Delay Line".

//Register the Inputs when RxDataValid is high
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DataInput_Blk
      if (nAGCRst == 1'b0) begin
         RxDataIntRe    <= CONST_ZERO_INPUT_WIDTH;
         RxDataIntIm    <= CONST_ZERO_INPUT_WIDTH;
         RxDataIntValid <= 1'b0;
      end
      else if (AGCEn == 1'b0) begin
         RxDataIntRe    <= CONST_ZERO_INPUT_WIDTH;
         RxDataIntIm    <= CONST_ZERO_INPUT_WIDTH;
         RxDataIntValid <= 1'b0;
      end
      else if (RxDataValid == 1'b1) begin
         RxDataIntRe    <= RxAGCRe;
         RxDataIntIm    <= RxAGCIm;
         RxDataIntValid <= 1'b1;
      end
      else begin
         RxDataIntValid <= 1'b0;
      end
   end //DataInput_Blk

//Find the absolute value of the input using Mod Approx module
//This block is reused from TBE, and hence the port names are not
//representative in the AGC context.
ModApprox #(
            .SUM_WIDTH(INPUT_WIDTH),    //Input Width
            .CORRVAL_WIDTH(INPUT_WIDTH) //Output Width
           )
            U_MdApprox0(
                        //Inputs
                        .PhyClk(AGCClk),
                        .nPhyRst(nAGCRst),
                        .ComputeOn(AGCEn),
                        .ISum(RxDataIntRe),
                        .QSum(RxDataIntIm),

                        //Output
                        .CorrVal(RxDataIntAbs)
                       );

//Round the absolute value by 1/2/3 bits
//Choose the output depending on RegInBdRnd and then saturate it.
//Round by 1 bit
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-1)
           )
            U_ROUNDPOW1(
                        .InputData(RxDataIntAbs),
                        .RoundData(RxDataIntAbsRnd1)
                       );

//Round by 2 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-2)
           )
            U_ROUNDPOW2(
                        .InputData(RxDataIntAbs),
                        .RoundData(RxDataIntAbsRnd2)
                       );

//Round by 3 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-3)
           )
            U_ROUNDPOW3(
                        .InputData(RxDataIntAbs),
                        .RoundData(RxDataIntAbsRnd3)
                       );

//Choose the output of rounding depending on RegInBdRnd
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RndSel_Blk
      if (nAGCRst == 1'b0)
         RxDataIntAbsRnd <= {{(INPUT_WIDTH)}{1'b0}};
      else if (RegInBdRnd == 2'b00)
         RxDataIntAbsRnd <= RxDataIntAbs;     //No Rounding
      else if (RegInBdRnd == 2'b01)
         RxDataIntAbsRnd <= {1'b0,RxDataIntAbsRnd1}; //1 bit Rounding
      else if (RegInBdRnd == 2'b10)
         RxDataIntAbsRnd <= {2'b0,RxDataIntAbsRnd2}; //2 bits Rounding
      else
         RxDataIntAbsRnd <= {3'b0,RxDataIntAbsRnd3}; //3 bits Rounding
   end //RndSel_Blk

//Saturate to (INPUT_WIDTH-3)
SatUnsigned # (
               .INPUT_WIDTH(INPUT_WIDTH),
               .OUTPUT_WIDTH(INPUT_WIDTH-3)
              ) U_SATUSG4 (
                           .InputData(RxDataIntAbsRnd),
                           .SatData(RxDataIntAbsSat)
                          );

//Delay RxDataIntValid by 3 clocks to qualify output of Saturation Block
DelayLine # (
             .DATAWIDTH(1), //Width of RxDataIntValid
             .LATENCY(3)    //3 clocks latency
             )
              U_DelayLine0 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(~AGCEn),
                            .DataIn(RxDataIntValid),
                            .DataOut(RxDataIntValid3D)
                           );

//Feed the output of round to a delay line. The delay line should be fed only
//when RxDataValidInt3D is high.
//Delay Line for Data
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayLine_Blk
      if (nAGCRst == 1'b0) begin
         for (i = 0; i <= DELAY_LINE_WIDTH; i = i+1)
            RxDelayLine[i] <= {{(INPUT_WIDTH-3)}{1'b0}};
      end
      else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
         for (i = 0; i <= DELAY_LINE_WIDTH; i = i+1)
            RxDelayLine[i] <= {{(INPUT_WIDTH-3)}{1'b0}};
      end
      else if (RxDataIntValid3D == 1'b1) begin
         RxDelayLine[0] <= RxDataIntAbsSat;
         for (i = 1; i <= DELAY_LINE_WIDTH; i = i+1)
            RxDelayLine[i] <= RxDelayLine[i-1];
      end
   end //DelayLine_Blk

//Select one of the delayed outputs based on BW_PARAM
//and InBdSWLConfig
generate
   if (BW_PARAM == 20) begin : SEL20BLK
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: RxDataInt1_20_Blk
            if (nAGCRst == 1'b0) begin
               RxDataInt1 <= {{(INPUT_WIDTH-3)}{1'b0}};
            end
            else begin
               case (InBdSWLConfig)
                  3'b000:  RxDataInt1 <= RxDelayLine[7];             // 0.4us 
                  3'b001:  RxDataInt1 <= RxDelayLine[15];            // 0.8us 
                  3'b010:  RxDataInt1 <= RxDelayLine[31];            // 1.6us 
                  3'b011:  RxDataInt1 <= RxDelayLine[47];            // 2.4us 
                  default: RxDataInt1 <= RxDelayLine[63];            // 3.2us 
               endcase
            end
         end //RxDataInt1_Blk
   end //BW_PARAM = 20
   else begin : SEL40BLK //BW_PARAM = 40
      always @ (posedge AGCClk or negedge nAGCRst)
         begin: RxDataInt1_Blk
            if (nAGCRst == 1'b0) begin
               RxDataInt1 <= {{(INPUT_WIDTH-3)}{1'b0}};
            end
            else begin
               case (InBdSWLConfig)
                  3'b000:  RxDataInt1 <= RxDelayLine[15];           // 0.4us         
                  3'b001:  RxDataInt1 <= RxDelayLine[31];           // 0.8us         
                  3'b010:  RxDataInt1 <= RxDelayLine[63];           // 1.6us         
                  3'b011:  RxDataInt1 <= RxDelayLine[95];           // 2.4us         
                  default: RxDataInt1 <= RxDelayLine[127];          // 3.2us         
               endcase
            end
         end //RxDataInt1_Blk
   end //BW_PARAM = 40
endgenerate

//Find the square of RxDataInt1 and RxDataIntAbsSat
//Since RxDataInt1 has one clock latency for the muxing, the input
//data (RxDataIntAbsSat) also has to be delayed by a clock
//This is equivalent to using RxDelayLine[0]
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Mult1_Blk
      if (nAGCRst == 1'b0) begin
         RxDataPow0 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
         RxDataPow1 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
      end
      else begin
         RxDataPow0 <= RxDelayLine[0] * RxDelayLine[0];
         RxDataPow1 <= RxDataInt1 * RxDataInt1;
      end
   end //Mult1_Blk

//Muxing the Output of RxDataPow1 with zero when AddZeroValue is set to 1
//This is to avoid invalid inputs from the delay line initially
assign MuxRxDataPow1 = (AddZeroValue || AddZeroValueDel) ? CONSTZERO[(2*(INPUT_WIDTH-3))-1:0] : RxDataPow1;

//Delay AddZeroValue for 5 clocks to qualify MuxRxDataPow1
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddZeroValueDel_Blk
      if (nAGCRst == 1'b0)
         AddZeroValueDel <= 1'b0;
      else if (AGCEn == 1'b0)
         AddZeroValueDel <= 1'b0;
      else if (RxDataIntValid5D == 1'b1)
         AddZeroValueDel <= AddZeroValue;
   end //AddZeroValueDel_Blk

//Delay RxDataIntValid3D by 2 clocks to qualify output of Multiplier
DelayLine # (
             .DATAWIDTH(1), //Width of RxDataIntValid3D
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLine1 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(~AGCEn),
                            .DataIn(RxDataIntValid3D),
                            .DataOut(RxDataIntValid5D)
                           );
//Power variation
assign RxDataPowDiff = {1'b0,RxDataPow0} - {1'b0,MuxRxDataPow1};

//Accumulator
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Acc_Blk
      if (nAGCRst == 1'b0)
         InBdPowLinInt <= {{((2*(INPUT_WIDTH-3)+7))}{1'b0}};
      else if (AGCEn == 1'b0 || DelLineClr == 1'b1)
         InBdPowLinInt <= {{((2*(INPUT_WIDTH-3)+7))}{1'b0}};
      else if (RxDataIntValid5D == 1'b1)
      begin
        //Accumulator must be always positive, so check diff before addition
        if (($signed({1'b0,InBdPowLinInt}) >= $signed({{7{RxDataPowDiff[2*(INPUT_WIDTH-3)]}},RxDataPowDiff})) || !RxDataPowDiff[(2*(INPUT_WIDTH-3))])
          InBdPowLinInt <= InBdPowLinInt + {{6{RxDataPowDiff[2*(INPUT_WIDTH-3)]}},RxDataPowDiff};
        else
          InBdPowLinInt <= {{((2*(INPUT_WIDTH-3)+7))}{1'b0}};
      end
   end //Acc_Blk

//Round accumulated value by 2 bits
USgnRound #(
            .INPUT_WIDTH((2*(INPUT_WIDTH-3))+7),
            .OUTPUT_WIDTH((2*(INPUT_WIDTH-3))+5)
           )
            U_ROUNDPOW4(
                        .InputData(InBdPowLinInt),
                        .RoundData(InBdPowLinRnd)
                       );

//Saturate the rounded value to OUTPUT_WIDTH
SatUnsigned # (
               .INPUT_WIDTH((2*(INPUT_WIDTH-3))+5),
               .OUTPUT_WIDTH(OUTPUT_WIDTH)
              ) U_SATPOW0 (
                           .InputData(InBdPowLinRnd),
                           .SatData(InBdPowLinSat)
                          );

//Register the final output
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Output_Blk
      if (nAGCRst == 1'b0)
         InBdPowLin <= {{OUTPUT_WIDTH}{1'b0}};
      else
         InBdPowLin <= InBdPowLinSat;
   end //Output_Blk

//Generate InBdPowLinValidforFSM
//This signal indicates to the FSM that the output power can be used for
//its decision making. This should take care of the delay line length as well
//as the hardware latency.
always @ (*)
   begin: TermDelaySel0_Blk
      case (InBdSWLConfig)
         3'b000:  TermCnt0 = 9'd30;  // 0.4us 
         3'b001:  TermCnt0 = 9'd62;  // 0.8us
         3'b010:  TermCnt0 = 9'd126; // 1.6us
         3'b011:  TermCnt0 = 9'd190; // 2.4us
         default: TermCnt0 = 9'd254; // 3.2us
      endcase
   end //TermDelaySel0_Blk

always @ (posedge AGCClk or negedge nAGCRst)
   begin: Valid0_Blk
      if (nAGCRst == 1'b0) begin
         InBdPowLinValidforFSM <= 1'b0;
         DelayCnt0 <= 9'd0;
      end
      else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
         InBdPowLinValidforFSM <= 1'b0;
         DelayCnt0 <= 9'd0;
      end
      else if (DelayCnt0 < TermCnt0) begin
         InBdPowLinValidforFSM <= 1'b0;
         DelayCnt0 <= DelayCnt0 + 9'b1;
      end
      else begin
         InBdPowLinValidforFSM <= 1'b1;
      end
   end //Valid0_Blk

endmodule //InBdPowEst

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
