////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 19268 $
// $Date: 2015-05-06 17:38:19 +0200 (Wed, 06 May 2015) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : In Band Power Est Block for AGC with Shared Del Line
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/PowerEst/verilog/rtl/InBdPowEstShDel.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

module InBdPowEstShDel #(parameter INPUT_WIDTH  = 13, //Input Data Width
                         parameter OUTPUT_WIDTH = 24  //Output Data Width
                        )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active Low Reset
            input    wire                                  AGCClk,  //AGC Clock

            //Control Signals
            input    wire                                  AGCEn,   //Block Enable
            input    wire                                  AddZeroValue, //Add zeros instead of samples from delay block for first 0.4,0.8,1.6,2.4 or 3.2 us
            input    wire                                  RxDataValid, //Qualifies input data
            input    wire                                  DelLineClr, //Clear Delay Line

            //Registers
            input    wire              [2:0]               InBdSWLConfig, //Delay Line Output Sel
            input    wire              [1:0]               RegInBdRnd, //Rounding of Abs value

            //Data - from Front End
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe, //Real Component
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm, //Imaginary Component

            //Data - from AGCDelLine
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe8D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm8D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe16D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm16D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe32D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm32D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe48D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm48D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCRe64D,
            input    wire   signed     [INPUT_WIDTH-1:0]   RxAGCIm64D,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            ////Control Signal
            output   reg                                   InBdPowLinValidforFSM, //Power Valid
            output   reg                                   InBdPowLinValidforCCA, //Power Valid
            //Power Estimate
            output   reg       [OUTPUT_WIDTH-1:0]          InBdPowLin //In Band Power
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam                           CONSTZERO     = 32'd0;
localparam        [8:0]              CCA_MIN_DELAY = 9'd100;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [INPUT_WIDTH-1:0]   RxDataInAbs;
wire               [INPUT_WIDTH-2:0]   RxDataInAbsRnd1;
wire               [INPUT_WIDTH-3:0]   RxDataInAbsRnd2;
wire               [INPUT_WIDTH-4:0]   RxDataInAbsRnd3;
wire               [INPUT_WIDTH-1:0]   RxDataDelAbs;
wire               [INPUT_WIDTH-2:0]   RxDataDelAbsRnd1;
wire               [INPUT_WIDTH-3:0]   RxDataDelAbsRnd2;
wire               [INPUT_WIDTH-4:0]   RxDataDelAbsRnd3;
wire                                   RxDataValid5D;
wire               [OUTPUT_WIDTH-1:0]  InBdPowLinRnd;
wire               [INPUT_WIDTH-4:0]   RxDataInAbsSat;
wire               [INPUT_WIDTH-4:0]   RxDataDelAbsSat;
wire               [(2*(INPUT_WIDTH-3))-1:0] MuxRxDataPow1;
wire    signed     [(2*(INPUT_WIDTH-3)):0]   RxDataPowDiff;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [INPUT_WIDTH-1:0]   RxDataDelRe;
reg     signed     [INPUT_WIDTH-1:0]   RxDataDelIm;
reg                [INPUT_WIDTH-4:0]   RxDataInt0;
reg                [INPUT_WIDTH-4:0]   RxDataInt1;
reg                [(2*(INPUT_WIDTH-3))-1:0] RxDataPow0;
reg                [(2*(INPUT_WIDTH-3))-1:0] RxDataPow1;
reg                [(2*(INPUT_WIDTH-3))+5:0] InBdPowLinInt;
reg                [8:0]               TermCnt0;
reg                [8:0]               DelayCnt0;
reg                [INPUT_WIDTH-1:0]   RxDataInAbsRnd;
reg                [INPUT_WIDTH-1:0]   RxDataDelAbsRnd;
reg                [4:0]               RxDataValidDel;
reg                                    AddZeroValueDel;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module calculates the In Band Power Estimate using a delay line which
//is shared with correlators. The delay line is present outside the block.
//This module calculates power for a 20 MHz case only

//Find the absolute value of the current input using Mod Approx module.
//This block is reused from TBE, and hence the port names are not
//representative in the AGC context.
ModApprox #(
            .SUM_WIDTH(INPUT_WIDTH),    //Input Width
            .CORRVAL_WIDTH(INPUT_WIDTH) //Output Width
           )
            U_MdApprox0(
                        //Inputs
                        .PhyClk(AGCClk),
                        .nPhyRst(nAGCRst),
                        .ComputeOn(AGCEn),
                        .ISum(RxAGCRe),
                        .QSum(RxAGCIm),

                        //Output
                        .CorrVal(RxDataInAbs)
                       );

//Round the absolute value by 1/2/3 bits
//Choose the output depending on RegInBdRnd and then saturate it.
//Round by 1 bit
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-1)
           )
            U_ROUNDPOW1(
                        .InputData(RxDataInAbs),
                        .RoundData(RxDataInAbsRnd1)
                       );

//Round by 2 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-2)
           )
            U_ROUNDPOW2(
                        .InputData(RxDataInAbs),
                        .RoundData(RxDataInAbsRnd2)
                       );

//Round by 3 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-3)
           )
            U_ROUNDPOW3(
                        .InputData(RxDataInAbs),
                        .RoundData(RxDataInAbsRnd3)
                       );

//Choose the output of rounding depending on RegInBdRnd
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RndSel0_Blk
      if (nAGCRst == 1'b0)
         RxDataInAbsRnd <= {{INPUT_WIDTH}{1'b0}};
      else if (RxDataValidDel[1] == 1'b1) begin
        if (RegInBdRnd == 2'b00)
           RxDataInAbsRnd <= RxDataInAbs;              //No Rounding
        else if (RegInBdRnd == 2'b01)
           RxDataInAbsRnd <= {1'b0,RxDataInAbsRnd1};   //1 bit Rounding
        else if (RegInBdRnd == 2'b10)
           RxDataInAbsRnd <= {2'b00,RxDataInAbsRnd2};  //2 bits Rounding
        else
           RxDataInAbsRnd <= {3'b000,RxDataInAbsRnd3}; //3 bits Rounding
      end
   end //RndSel0_Blk

//Saturate to (INPUT_WIDTH-3)
SatUnsigned # (
               .INPUT_WIDTH(INPUT_WIDTH),
               .OUTPUT_WIDTH(INPUT_WIDTH-3)
              ) U_SATUSG5 (
                           .InputData(RxDataInAbsRnd),
                           .SatData(RxDataInAbsSat)
                          );

//Select one of the delay values (coming as inputs) based on the register
//value InBdSWLConfig
always @ (*)
   begin: RxDataDel_Blk
      case (InBdSWLConfig)
         3'b000: begin
            RxDataDelRe = RxAGCRe8D;
            RxDataDelIm = RxAGCIm8D;
         end

         3'b001: begin
            RxDataDelRe = RxAGCRe16D;
            RxDataDelIm = RxAGCIm16D;
         end

         3'b010: begin
            RxDataDelRe = RxAGCRe32D;
            RxDataDelIm = RxAGCIm32D;
         end

         3'b011: begin
            RxDataDelRe = RxAGCRe48D;
            RxDataDelIm = RxAGCIm48D;
         end

         default: begin
            RxDataDelRe = RxAGCRe64D;
            RxDataDelIm = RxAGCIm64D;
         end
      endcase //InBdSWLConfig
   end //RxDataDel_Blk

//Find the absolute value of the delayed version
ModApprox #(
            .SUM_WIDTH(INPUT_WIDTH),    //Input Width
            .CORRVAL_WIDTH(INPUT_WIDTH) //Output Width
           )
            U_MdApprox1(
                        //Inputs
                        .PhyClk(AGCClk),
                        .nPhyRst(nAGCRst),
                        .ComputeOn(AGCEn),
                        .ISum(RxDataDelRe),
                        .QSum(RxDataDelIm),

                        //Output
                        .CorrVal(RxDataDelAbs)
                       );

//Round the absolute value by 1/2/3 bits
//Choose the output depending on RegInBdRnd and then saturate it.
//Round by 1 bit
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-1)
           )
            U_ROUNDPOW4(
                        .InputData(RxDataDelAbs),
                        .RoundData(RxDataDelAbsRnd1)
                       );

//Round by 2 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-2)
           )
            U_ROUNDPOW5(
                        .InputData(RxDataDelAbs),
                        .RoundData(RxDataDelAbsRnd2)
                       );

//Round by 3 bits
USgnRound #(
            .INPUT_WIDTH(INPUT_WIDTH),
            .OUTPUT_WIDTH(INPUT_WIDTH-3)
           )
            U_ROUNDPOW6(
                        .InputData(RxDataDelAbs),
                        .RoundData(RxDataDelAbsRnd3)
                       );

//Choose the output of rounding depending on RegInBdRnd
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RndSel1_Blk
      if (nAGCRst == 1'b0)
         RxDataDelAbsRnd <= {{INPUT_WIDTH}{1'b0}};
      else if (RxDataValidDel[1] == 1'b1) begin
         if (RegInBdRnd == 2'b00)
           RxDataDelAbsRnd <= RxDataDelAbs;              //No Rounding
        else if (RegInBdRnd == 2'b01)
           RxDataDelAbsRnd <= {1'b0,RxDataDelAbsRnd1};   //1 bit Rounding
        else if (RegInBdRnd == 2'b10)
           RxDataDelAbsRnd <= {2'b00,RxDataDelAbsRnd2};  //2 bits Rounding
        else
           RxDataDelAbsRnd <= {3'b000,RxDataDelAbsRnd3}; //3 bits Rounding
      end
   end //RndSel1_Blk

//Saturate to (INPUT_WIDTH-3)
SatUnsigned # (
               .INPUT_WIDTH(INPUT_WIDTH),
               .OUTPUT_WIDTH(INPUT_WIDTH-3)
              ) U_SATUSG6 (
                           .InputData(RxDataDelAbsRnd),
                           .SatData(RxDataDelAbsSat)
                          );

//Register the rounded-saturated versions
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RegRnd_Blk
      if (nAGCRst == 1'b0) begin
         RxDataInt0 <= {{(INPUT_WIDTH-3)}{1'b0}};
         RxDataInt1 <= {{(INPUT_WIDTH-3)}{1'b0}};
      end
      else if (AGCEn == 1'b0) begin
         RxDataInt0 <= {{(INPUT_WIDTH-3)}{1'b0}};
         RxDataInt1 <= {{(INPUT_WIDTH-3)}{1'b0}};
      end
      else begin
         RxDataInt0 <= RxDataInAbsSat;
         RxDataInt1 <= RxDataDelAbsSat;
      end
   end //RegRnd_Blk

//Find the square of the registered values
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Mult1_Blk
      if (nAGCRst == 1'b0) begin
         RxDataPow0 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
         RxDataPow1 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
      end
      else if (AGCEn == 1'b0) begin
         RxDataPow0 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
         RxDataPow1 <= {{(2*(INPUT_WIDTH-3))}{1'b0}};
      end
      else begin
         RxDataPow0 <= RxDataInt0 * RxDataInt0;
         RxDataPow1 <= RxDataInt1 * RxDataInt1;
      end
   end //Mult1_Blk

//Muxing the Output of RxDataPow1 with zero when AddZeroValue is set to 1
//This is to avoid invalid inputs from the delay line initially
assign MuxRxDataPow1 = (AddZeroValue || AddZeroValueDel) ? CONSTZERO[(2*(INPUT_WIDTH-3))-1:0] : RxDataPow1;

//Delay AddZeroValue for 5 clocks to qualify MuxRxDataPow1
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddZeroValueDel_Blk
      if (nAGCRst == 1'b0)
         AddZeroValueDel <= 1'b0;
      else if (AGCEn == 1'b0)
         AddZeroValueDel <= 1'b0;
      else if (RxDataValid5D == 1'b1)
         AddZeroValueDel <= AddZeroValue;
   end //AddZeroValueDel_Blk

//Delay RxDataValid for 5 clocks to qualify output of the multiplier
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RxDataValidDel_Blk
      if (nAGCRst == 1'b0)
         RxDataValidDel <= 5'b0;
      else if (AGCEn == 1'b0)
         RxDataValidDel <= 5'b0;
      else
         RxDataValidDel <= {RxDataValidDel[3:0], RxDataValid};
   end //RxDataValidDel_Blk

assign RxDataValid5D = RxDataValidDel[4];

//Power variation
assign RxDataPowDiff = {1'b0,RxDataPow0} - {1'b0,MuxRxDataPow1};

//Accumulator
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Acc_Blk
      if (nAGCRst == 1'b0)
         InBdPowLinInt <= {{(2*(INPUT_WIDTH-3)+6)}{1'b0}};
      else if (DelLineClr == 1'b1 || AGCEn == 1'b0)
         InBdPowLinInt <= {{(2*(INPUT_WIDTH-3)+6)}{1'b0}};
      else if (RxDataValid5D == 1'b1)
      begin
        //Accumulator must be always positive, so check diff before addition
        if (($signed({1'b0,InBdPowLinInt}) >= $signed({{6{RxDataPowDiff[(2*(INPUT_WIDTH-3))]}},RxDataPowDiff})) || !RxDataPowDiff[(2*(INPUT_WIDTH-3))])
          InBdPowLinInt <= InBdPowLinInt + {{5{RxDataPowDiff[(2*(INPUT_WIDTH-3))]}},RxDataPowDiff};
        else
          InBdPowLinInt <= {{(2*(INPUT_WIDTH-3)+6)}{1'b0}};
      end
   end //Acc_Blk

//Round accumulated value by 2 bits
USgnRound #(
            .INPUT_WIDTH((2*(INPUT_WIDTH-3))+6),
            .OUTPUT_WIDTH(OUTPUT_WIDTH)
           )
            U_ROUNDPOW7(
                        .InputData(InBdPowLinInt),
                        .RoundData(InBdPowLinRnd)
                       );

//Register the final output
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Output_Blk
      if (nAGCRst == 1'b0)
         InBdPowLin <= {{OUTPUT_WIDTH}{1'b0}};
      else
         InBdPowLin <= InBdPowLinRnd;
   end //Output_Blk

//Generate InBdPowLinValidforFSM & InBdPowLinValidforCCA
//This signal indicates to the FSM that the output power can be used for
//its decision making. This should take care of the delay line length as well
//as the hardware latency.
always @ (*)
   begin: TermDelaySel0_Blk
         case (InBdSWLConfig)
            3'b000:  TermCnt0 = 9'd29;
            3'b001:  TermCnt0 = 9'd61;
            3'b010:  TermCnt0 = 9'd125;
            3'b011:  TermCnt0 = 9'd189;
            default: TermCnt0 = 9'd253;
         endcase

   end //TermDelaySel0_Blk

always @ (posedge AGCClk or negedge nAGCRst)
   begin: Valid0_Blk
      if (nAGCRst == 1'b0) begin
         InBdPowLinValidforFSM <= 1'b0;
         InBdPowLinValidforCCA <= 1'b0;
         DelayCnt0             <= 9'd0;
      end
      else if (AGCEn == 1'b0 || DelLineClr == 1'b1) begin
         InBdPowLinValidforFSM <= 1'b0;
         InBdPowLinValidforCCA <= 1'b0;
         DelayCnt0             <= 9'd0;
      end
      else begin
        if ((DelayCnt0 < TermCnt0) || (DelayCnt0 < CCA_MIN_DELAY)) begin
           DelayCnt0 <= DelayCnt0 + 9'b1;
        end
        if (DelayCnt0 < TermCnt0) begin
           InBdPowLinValidforFSM <= 1'b0;
        end
        else begin
           InBdPowLinValidforFSM <= 1'b1;
        end
        if (DelayCnt0 < CCA_MIN_DELAY) begin
           InBdPowLinValidforCCA <= 1'b0;
        end
        else begin
           InBdPowLinValidforCCA <= 1'b1;
        end
      end
   end //Valid0_Blk

endmodule //InBdPowEstShDel

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
