////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12017 $
// $Date: 2013-11-28 18:43:44 +0100 (Thu, 28 Nov 2013) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Linear to dBVrms Conversion Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/PowerEst/verilog/rtl/Lin2dBVrms.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

module Lin2dBVrms #(parameter INPUT_WIDTH  = 64, //Input Data Width
                    parameter OUTPUT_WIDTH = 12, //Output Data Width
                    parameter FSB_ITER     = 6   //64 = 2^6
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active LOW Reset
            input    wire                                  AGCClk,  //PHY Clock

            //Control Signals
            input    wire                                  LinDataValidforFSM, //Qualifies Data In

            //Registers
            input    wire   signed     [9:0]               LSBLess, //LSBs removed during Pow Est
            input    wire              [3:0]               NBitADC, //Effective ADC Width
            input    wire   signed     [7:0]               VPeakADCqdBV, //ADC Peak Voltage in dB

            //Data
            input    wire              [INPUT_WIDTH-1:0]   LinValIn, //Linear Value Input

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signals
            output   wire                                  qdBVValidforFSM, //Qualifies Data Out

            //Data
            output   reg signed     [OUTPUT_WIDTH-1:0]     qdBVOut //Decibel Value Output
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [OUTPUT_WIDTH-1:0] CONST_N400 = -'sd400;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [(2**(FSB_ITER-1))-1:0] IntArr[FSB_ITER-1:0][1:0];
wire    signed     [FSB_ITER+4:0]      IntVal3;
wire    signed     [OUTPUT_WIDTH-1:0]  IntVal4;
wire               [4:0]               LSBIndex;
wire               [INPUT_WIDTH-1:0]   LinValIn5D;
wire               [INPUT_WIDTH+4:0]   LinValIn2DInt;
wire    signed     [4:0]               NBitADCInt;
wire    signed     [9:0]               NBitADCShift;
wire                                   Bypass;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [INPUT_WIDTH-1:0]   LinValInD;
reg                [(2**FSB_ITER)-1:0] InputData[FSB_ITER:0];
reg                [FSB_ITER-1:0]      FSBInt;
reg                [FSB_ITER-1:0]      FSB;
reg                [FSB_ITER-1:0]      FSBD;
reg     signed     [10:0]              IntVal0;
reg     signed     [FSB_ITER+4:0]      IntVal1;
reg     signed     [FSB_ITER+6:0]      IntVal2;
reg                [INPUT_WIDTH-1:0]   LinValIn2D;
reg                [4:0]               LSBLUTOut;

genvar i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module performs Linear to dBVrms conversion
//Find out the First Significant Bit in the InputData
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DataInReg_Blk
      if (nAGCRst == 1'b0)
         LinValInD <= {{INPUT_WIDTH}{1'b0}};
      else
         LinValInD <= LinValIn;

   end //DataIn_RegBlk

always @ (*)
   begin: InputData0_Blk
      InputData[0][INPUT_WIDTH-1:0]             = LinValInD;
      InputData[0][(2**FSB_ITER)-1:INPUT_WIDTH] = {{((2**FSB_ITER)-INPUT_WIDTH)}{1'b0}};
   end


generate
   for (i = 0; i< FSB_ITER; i=i+1) begin: FSBCalc 
      assign IntArr[i][1] = InputData[i][(2**(FSB_ITER-i))-1:
                                         (2**(FSB_ITER-i-1))];
      assign IntArr[i][0] = InputData[i][(2**(FSB_ITER-i-1))-1:0];

      always @ (*)
         begin: FSBInt_Blk
            if (|IntArr[i][1] == 1'b1) begin
               FSBInt[FSB_ITER-i-1] = 1'b1;
               InputData[i+1]       = IntArr[i][1];
            end
            else begin
               FSBInt[FSB_ITER-i-1] = 1'b0;
               InputData[i+1]       = IntArr[i][0];
            end

         end //FSBInt_Blk
   end //FSBCalc
endgenerate

always @ (posedge AGCClk or negedge nAGCRst)
   begin: FSB_Blk
      if (nAGCRst == 1'b0)
         FSB <= {{FSB_ITER}{1'b0}};
      else
         FSB <= FSBInt;

   end //FSB_Blk

assign NBitADCInt[4:0] = {1'b0,NBitADC} - 5'b1;
assign NBitADCShift[9:0] = {NBitADCInt[4:0],5'b0};

//Add LSBLess and 32*(NBitADC-1)
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IntVal0_Blk
      if (nAGCRst == 1'b0)
         IntVal0 <= $signed({11{1'b0}});
      else
         IntVal0 <= NBitADCShift + LSBLess;

   end //IntVal0_Blk

//Find LSBIndex using FSB value
//Since there is 2 clocks delay in finding FSB, the input data has to be
//delayed by 2 clocks as well. Also append 5 zeros at the LSB of the input to
//handle cases where FSB is less than 6
assign LinValIn2DInt = {LinValIn2D, 5'b00000};
assign LSBIndex      = LinValIn2DInt[(FSB+4)-:5];

//Use LSBIndex to find output of LUT which contains log2 values
always @ (posedge AGCClk or negedge nAGCRst)
   begin: LSBLUT_Blk
      if (nAGCRst == 1'b0) begin
         LSBLUTOut <= 5'b0;
      end
      else begin
         case (LSBIndex)
            5'd00   : LSBLUTOut <= 5'd00;
            5'd01   : LSBLUTOut <= 5'd01;
            5'd02   : LSBLUTOut <= 5'd01;
            5'd03   : LSBLUTOut <= 5'd02;
            5'd04   : LSBLUTOut <= 5'd03;
            5'd05   : LSBLUTOut <= 5'd03;
            5'd06   : LSBLUTOut <= 5'd04;
            5'd07   : LSBLUTOut <= 5'd05;
            5'd08   : LSBLUTOut <= 5'd05;
            5'd09   : LSBLUTOut <= 5'd06;
            5'd10   : LSBLUTOut <= 5'd06;
            5'd11   : LSBLUTOut <= 5'd07;
            5'd12   : LSBLUTOut <= 5'd07;
            5'd13   : LSBLUTOut <= 5'd08;
            5'd14   : LSBLUTOut <= 5'd08;
            5'd15   : LSBLUTOut <= 5'd09;
            5'd16   : LSBLUTOut <= 5'd09;
            5'd17   : LSBLUTOut <= 5'd10;
            5'd18   : LSBLUTOut <= 5'd10;
            5'd19   : LSBLUTOut <= 5'd11;
            5'd20   : LSBLUTOut <= 5'd11;
            5'd21   : LSBLUTOut <= 5'd12;
            5'd22   : LSBLUTOut <= 5'd12;
            5'd23   : LSBLUTOut <= 5'd13;
            5'd24   : LSBLUTOut <= 5'd13;
            5'd25   : LSBLUTOut <= 5'd13;
            5'd26   : LSBLUTOut <= 5'd14;
            5'd27   : LSBLUTOut <= 5'd14;
            5'd28   : LSBLUTOut <= 5'd15;
            5'd29   : LSBLUTOut <= 5'd15;
            5'd30   : LSBLUTOut <= 5'd15;
            default : LSBLUTOut <= 5'd16;
         endcase //LSBIndex
      end

   end //LSBLUT_Blk

//Add 16*FSB and LSBLUT; then subtract IntVal0
//Use one clock delayed version of FSB to align it with LSBLUTOut
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IntVal1_Blk
      if (nAGCRst == 1'b0)
         IntVal1 <= $signed({{(FSB_ITER+5)}{1'b0}});
      else
         IntVal1 <= $signed({FSBD,4'b0}) + LSBLUTOut - IntVal0;

   end //IntVal1_Blk

//Multiply IntVal1 by 3
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IntVal2_Blk
      if (nAGCRst == 1'b0)
         IntVal2 <= $signed({{(FSB_ITER+7)}{1'b0}});
      else
         IntVal2 <= $signed({IntVal1,1'b0}) + IntVal1;

   end //IntVal1_Blk

//Round IntVal2 by 2 bits
Round #(
        .INPUT_WIDTH(FSB_ITER+7),
        .OUTPUT_WIDTH(FSB_ITER+5)
       )
        U_ROUNDINT0(
                    .InputData(IntVal2),
                    .RoundData(IntVal3)
                   );

//Add VPeakADCqdBV to the rounded value to get the final output
assign IntVal4 = IntVal3 + $signed({{FSB_ITER-3{VPeakADCqdBV[7]}},VPeakADCqdBV});

always @ (posedge AGCClk or negedge nAGCRst)
   begin: Out_Blk
      if (nAGCRst == 1'b0)
         qdBVOut <= CONST_N400;
      else if (LinValIn5D == {{INPUT_WIDTH}{1'b0}})
         qdBVOut <= CONST_N400;
      else
         qdBVOut <= IntVal4;

   end //IntVal1_Blk

//Generate delayed versions of FSB and LinValIn
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Delay_Blk
      if (nAGCRst == 1'b0) begin
         FSBD       <= {{FSB_ITER}{1'b0}};
         LinValIn2D <= {{INPUT_WIDTH}{1'b0}};
      end
      else begin
         FSBD       <= FSB;
         LinValIn2D <= InputData[0][INPUT_WIDTH-1:0];
      end

   end //Delay_Blk

DelayLine # (
             .DATAWIDTH(INPUT_WIDTH), //Width of LinValIn2D
             .LATENCY(3)              //3 clocks latency
             )
              U_DelayLine1 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(Bypass),
                            .DataIn(LinValIn2D),
                            .DataOut(LinValIn5D)
                           );

//Generate qdBVValidforFSM by delaying the incoming LinDataValidforFSM
//by the latency of this block (6 clocks)
DelayLine # (
             .DATAWIDTH(1), //Width of LinDataValidforFSM
             .LATENCY(6)    //6 clocks latency
             )
              U_DelayLine2 (
                            .PhyClk(AGCClk),
                            .nPhyRst(nAGCRst),
                            .Bypass(Bypass),
                            .DataIn(LinDataValidforFSM),
                            .DataOut(qdBVValidforFSM)
                           );

assign Bypass = 1'b0;

endmodule //Lin2dBVrms

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
