//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission 
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 6846 $
// $Date: 2013-03-19 17:30:27 +0100 (Tue, 19 Mar 2013) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Vectoring CORDIC. This CORDIC computes the angle and the
//                    magnitude of a complex input sample.
//                    The computation is performed using rotations to align the
//                    input sample to with the X axis.
//                    This CORDIC is configurable in term of pipelines and
//                    number of microrotation per clock cycle.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/CordicVectoring.v $
//
//////////////////////////////////////////////////////////////////////////////

module CordicVectoring #(
           parameter DATAWIDTH  = 13, // number of bits for the complex data
           parameter ANGLEWIDTH = 10, // number of bits for the output angle AngleInn
           parameter NCOMBSTG   = 4,  // number of pipes (must be at least ANGLEWIDTH/2)
                                      // NOTE : the total number of microrotations is 2 * NCOMBSTG
           parameter SCALING    = 1)  // 1:Use all the amplitude (pi/2 = 2^errosize_g=~ 01111....)
                                      // (-pi/2 = -2^errosize_g= 100000....)
           (
           ///////////////////////////////////////////////
           // Clock and reset
           ///////////////////////////////////////////////
           input wire                         PhyClk,
           input wire                         nPhyRst,
           
           ///////////////////////////////////////////////
           // Input vector to be rotated
           ///////////////////////////////////////////////
           input wire signed [DATAWIDTH-1:0]  ReDataIn,
           input wire signed [DATAWIDTH-1:0]  ImDataIn,
           input wire                         DataInEn,
    
           ///////////////////////////////////////////////
           // Angle of the input vector
           ///////////////////////////////////////////////
           output reg signed [ANGLEWIDTH-1:0] AngleOut
           );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
  localparam PI_CT        = 32'b01100100100001111110110101010001;
  localparam PI_SCALED_CT = 32'b10000000000000000000000000000000;
  localparam NULL_CT      = 32'b0;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  wire signed [ANGLEWIDTH-1:0] Angle [NCOMBSTG:0];
    
  //intermediate rotated outputs of microrotation :
  wire signed [DATAWIDTH+1:0]  ReData [NCOMBSTG:0];
  wire signed [DATAWIDTH+1:0]  ImData [NCOMBSTG:0];
  
  // sign is added on the last Angle. 
  wire signed [ANGLEWIDTH-1:0] ext_AngleOut;
  
  // Arctangent reference values (32-bit).
  wire [31:0]                  arctan_array_ref [31:0];

  // Pi after scaling
  wire [31:0]                  pi;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
  // ReData and ImData sign. 0: positive ; 1 : negative
  reg [NCOMBSTG-1:0] ReData_sign;
  reg [NCOMBSTG-1:0] ImData_sign;

  // Check result of inputs are null
  reg [NCOMBSTG-1:0] input_eq_2_zero;
  
  integer i;
  genvar  j;
  
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // initial angle set to 0
  assign Angle[0] = {ANGLEWIDTH{1'b0}};
  
  // As the CORDIC algorithm computes angles between -PI/2 and PI/2,
  // the input vector is negged if ReData is negative. 
  // At the end of the CORDIC computation, the angle will be changed 
  // depending on the ReData and ImData initial signs.
  assign ReData[0] = (ReDataIn[DATAWIDTH-1] == 1'b0) ?  {{2{ReDataIn[DATAWIDTH-1]}}, ReDataIn} :
                     ~({{2{ReDataIn[DATAWIDTH-1]}}, ReDataIn}) + {{DATAWIDTH+1{1'b0}},1'b1};

  assign ImData[0] = (ReDataIn[DATAWIDTH-1] == 1'b0) ? {{2{ImDataIn[DATAWIDTH-1]}}, ImDataIn}  :
                     ~({{2{ImDataIn[DATAWIDTH-1]}}, ImDataIn}) + {{DATAWIDTH+1{1'b0}},1'b1};

  // pipeline the signs of the input samples
  always @ (posedge PhyClk or negedge nPhyRst) begin
    if (nPhyRst == 1'b0) begin
      for(i=0;i<=NCOMBSTG-1;i=i+1) begin
        ReData_sign[i] <= 1'b0;
        ImData_sign[i] <= 1'b0;
      end
    end
    else if (DataInEn) begin
      ReData_sign[0] <= ReDataIn[DATAWIDTH-1];
      ImData_sign[0] <= ImDataIn[DATAWIDTH-1];
      for(i=1;i<=NCOMBSTG-1;i=i+1) begin
        ReData_sign[i] <= ReData_sign[i-1];
        ImData_sign[i] <= ImData_sign[i-1];
      end
    end
  end

  // Pipeline the check: inputs are both null.
  always @ (posedge PhyClk or negedge nPhyRst) begin
    if (nPhyRst == 1'b0) begin
      input_eq_2_zero <= {NCOMBSTG{1'b0}};
    end
    else if (DataInEn) begin
      if ((ReDataIn == NULL_CT[DATAWIDTH-1:0]) && (ImDataIn == NULL_CT[DATAWIDTH-1:0]))
        input_eq_2_zero[0] <= 1'b1;
      else
        input_eq_2_zero[0] <= 1'b0;
      
      input_eq_2_zero[NCOMBSTG-1:1] <= input_eq_2_zero[NCOMBSTG-2:0];
    end
  end
  
  ////////////////////////////////////////////
  // Pipeline generation.
  // Each stage generates nbr_comb_stage microrotations.
  ////////////////////////////////////////////
  generate for (j=0; j<NCOMBSTG; j=j+1) begin : Gen_CordicVectoringComb
    CordicVectoringComb #(                                                         
        .DATAWIDTH   (DATAWIDTH+1),
        .ANGLEWIDTH  (ANGLEWIDTH),
        .STARTSTAGE  (j*2))
      U_CordicVectoringComb (                                                            
        .PhyClk      (PhyClk),
        .nPhyRst     (nPhyRst),
        .DataInEn    (DataInEn),                                                                   
        // angle with which the input has been rotated before this stage                          
        .AngleIn     (Angle[j]),
        // inputs to be rotated :                                         
        .ReDataIn    (ReData[j]),
        .ImDataIn    (ImData[j]),
        // Arctangent reference table                    
        .arctan_ref1 (arctan_array_ref[(j*2)+1]),
        .arctan_ref0 (arctan_array_ref[(j*2)]),
        // angle with which the input has been rotated after this stage                          
        .AngleOut    (Angle[j+1]),
        // rotated output. They have been rotated of (AngleInn-AngleOutut)
        .ReDataOut   (ReData[j+1]),
        .ImDataOut   (ImData[j+1])
      );
    end
  endgenerate
  
  ////////////////////////////////////////////
  // outputs
  ////////////////////////////////////////////
  assign ext_AngleOut = Angle[NCOMBSTG];

  always@(*)
  begin
    if (ReData_sign[NCOMBSTG-1] == 1'b1) begin
      if (ImData_sign[NCOMBSTG-1] == 1'b1)
        // if the sample is in the 3rd quadrant, the true angle is angle - PI.
        AngleOut = ext_AngleOut - pi[31:31-ANGLEWIDTH+1];
      else
        // if the sample is in the 2nd quadrant, the true angle is angle + PI.
        AngleOut = ext_AngleOut + pi[31:31-ANGLEWIDTH+1];
    end
    else if (input_eq_2_zero[NCOMBSTG-1] == 1'b1)
      AngleOut = {ANGLEWIDTH{1'b0}};
    else
      // if the sample is in the 1st or 4th quadrant, the computed angle is in [-PI/2; PI/2].
      AngleOut = ext_AngleOut;
  end


  generate if (SCALING == 0) begin
                
    assign pi = PI_CT;

    assign arctan_array_ref[0]  = 32'b11001001000011111101101010100010;
    assign arctan_array_ref[1]  = 32'b01110110101100011001110000010101;
    assign arctan_array_ref[2]  = 32'b00111110101101101110101111110010;
    assign arctan_array_ref[3]  = 32'b00011111110101011011101010011010;
    assign arctan_array_ref[4]  = 32'b00001111111110101010110111011011;
    assign arctan_array_ref[5]  = 32'b00000111111111110101010101101110;
    assign arctan_array_ref[6]  = 32'b00000011111111111110101010101011;
    assign arctan_array_ref[7]  = 32'b00000001111111111111110101010101;
    assign arctan_array_ref[8]  = 32'b00000000111111111111111110101010;
    assign arctan_array_ref[9]  = 32'b00000000011111111111111111110101;
    assign arctan_array_ref[10] = 32'b00000000001111111111111111111110;
    assign arctan_array_ref[11] = 32'b00000000000111111111111111111111;
    assign arctan_array_ref[12] = 32'b00000000000011111111111111111111;
    assign arctan_array_ref[13] = 32'b00000000000001111111111111111111;
    assign arctan_array_ref[14] = 32'b00000000000000111111111111111111;
    assign arctan_array_ref[15] = 32'b00000000000000011111111111111111;
    assign arctan_array_ref[16] = 32'b00000000000000001111111111111111;
    assign arctan_array_ref[17] = 32'b00000000000000000111111111111111;
    assign arctan_array_ref[18] = 32'b00000000000000000011111111111111;
    assign arctan_array_ref[19] = 32'b00000000000000000001111111111111;
    assign arctan_array_ref[20] = 32'b00000000000000000000111111111111;
    assign arctan_array_ref[21] = 32'b00000000000000000000011111111111;
    assign arctan_array_ref[22] = 32'b00000000000000000000001111111111;
    assign arctan_array_ref[23] = 32'b00000000000000000000000111111111;
    assign arctan_array_ref[24] = 32'b00000000000000000000000011111111;
    assign arctan_array_ref[25] = 32'b00000000000000000000000001111111;
    assign arctan_array_ref[26] = 32'b00000000000000000000000000111111;
    assign arctan_array_ref[27] = 32'b00000000000000000000000000100000;
    assign arctan_array_ref[28] = 32'b00000000000000000000000000010000;
    assign arctan_array_ref[29] = 32'b00000000000000000000000000001000;
    assign arctan_array_ref[30] = 32'b00000000000000000000000000000100;
    assign arctan_array_ref[31] = 32'b00000000000000000000000000000010;

    end
  endgenerate

  ////////////////////////////////////////////////////////////////////////////-
  // SCALING = pi/4 => 111111111111111  
  ////////////////////////////////////////////////////////////////////////////-
  generate if (SCALING == 1) begin

    assign pi = PI_SCALED_CT;

    assign arctan_array_ref[0]  = 32'b11111111111111111111111111111111; // 2^32 * arctan(2^-0)*4/pi 
    assign arctan_array_ref[1]  = 32'b10010111001000000010100011101101; // 2^32 * arctan(2^-1)*4/pi 
    assign arctan_array_ref[2]  = 32'b01001111110110011100001011011011; // 2^32 * arctan(2^-2)*4/pi 
    assign arctan_array_ref[3]  = 32'b00101000100010001000111010100001; // 2^32 * arctan(2^-3)*4/pi 
    assign arctan_array_ref[4]  = 32'b00010100010110000110101000011000; // 2^32 * arctan(2^-4)*4/pi 
    assign arctan_array_ref[5]  = 32'b00001010001011101011111100001011; // 2^32 * arctan(2^-5)*4/pi 
    assign arctan_array_ref[6]  = 32'b00000101000101111011000011110011; // ...                
    assign arctan_array_ref[7]  = 32'b00000010100010111110001010101001;                       
    assign arctan_array_ref[8]  = 32'b00000001010001011111001010011010;                       
    assign arctan_array_ref[9]  = 32'b00000000101000101111100101110110;                       
    assign arctan_array_ref[10] = 32'b00000000010100010111110011000000;                       
    assign arctan_array_ref[11] = 32'b00000000001010001011111001100001;                       
    assign arctan_array_ref[12] = 32'b00000000000101000101111100110000;                       
    assign arctan_array_ref[13] = 32'b00000000000010100010111110011000;                       
    assign arctan_array_ref[14] = 32'b00000000000001010001011111001100;                       
    assign arctan_array_ref[15] = 32'b00000000000000101000101111100110;                       
    assign arctan_array_ref[16] = 32'b00000000000000010100010111110011;                       
    assign arctan_array_ref[17] = 32'b00000000000000001010001011111010;                       
    assign arctan_array_ref[18] = 32'b00000000000000000101000101111101;                       
    assign arctan_array_ref[19] = 32'b00000000000000000010100010111110;                       
    assign arctan_array_ref[20] = 32'b00000000000000000001010001011111;                       
    assign arctan_array_ref[21] = 32'b00000000000000000000101000110000;                       
    assign arctan_array_ref[22] = 32'b00000000000000000000010100011000;                       
    assign arctan_array_ref[23] = 32'b00000000000000000000001010001100;                       
    assign arctan_array_ref[24] = 32'b00000000000000000000000101000110;                       
    assign arctan_array_ref[25] = 32'b00000000000000000000000010100011;                       
    assign arctan_array_ref[26] = 32'b00000000000000000000000001010001;                       
    assign arctan_array_ref[27] = 32'b00000000000000000000000000101001;                       
    assign arctan_array_ref[28] = 32'b00000000000000000000000000010100;                       
    assign arctan_array_ref[29] = 32'b00000000000000000000000000001010;                       
    assign arctan_array_ref[30] = 32'b00000000000000000000000000000101;                       
    assign arctan_array_ref[31] = 32'b00000000000000000000000000000011;                       
    
    end
  endgenerate

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
