//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission 
//  from RivieraWaves.
//////////////////////////////////////////////////////////////////////////////
// $Author: cvandebu $
// Company          : RivieraWaves
//////////////////////////////////////////////////////////////////////////////
// $Revision: 6846 $
// $Date: 2013-03-19 17:30:27 +0100 (Tue, 19 Mar 2013) $
// //////////////////////////////////////////////////////////////////////////-
// Dependencies     : None
// Description      : This block is composed of 2 microrotation
//                    stages. These microrotations are performed to align the
//                    input samle with the X axis.
//                    The microrotation stages are performed in a combinational
//                    way, without any flip-flop between stages.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// //////////////////////////////////////////////////////////////////////////-
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/CordicVectoringComb.v $
//
//////////////////////////////////////////////////////////////////////////////

module CordicVectoringComb #(parameter DATAWIDTH  = 14,
                             parameter ANGLEWIDTH = 10,
                             parameter STARTSTAGE = 0
                             )(
  ///////////////////////////////////////////////
  // Inputs
  ///////////////////////////////////////////////
  // Clock and Reset
  input wire                         PhyClk,
  input wire                         nPhyRst,
  
  // Data enable
  input wire                         DataInEn,
  
  // Angle with which the inputs must be rotated
  input wire signed [ANGLEWIDTH-1:0] AngleIn,
        
  // Inputs to be rotated
  input wire signed [DATAWIDTH:0]    ReDataIn,
  input wire signed [DATAWIDTH:0]    ImDataIn,
         
  // Arctangent reference table
  input wire        [31:0]           arctan_ref1,
  input wire        [31:0]           arctan_ref0,
        
  ///////////////////////////////////////////////
  // Outputs
  ///////////////////////////////////////////////
  // Remaining angle with which outputs have not been rotated
  output reg signed [ANGLEWIDTH-1:0] AngleOut,
        
  // Rotated output. They have been rotated of (AngleInn-AngleOutut)
  output reg signed [DATAWIDTH:0]    ReDataOut,
  output reg signed [DATAWIDTH:0]    ImDataOut
  );

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  // Remaining angle after each microrotation stage
  wire signed [ANGLEWIDTH-1:0] Angle [2:0];

  // ImData sign for each stage : 1 : neg ; 0 : pos
  wire [1:0]                   ImData_sign;
  
  // Intermediate rotated outputs of microrotation
  wire signed [DATAWIDTH:0]    ReData [2:0];
  wire signed [DATAWIDTH:0]    ImData [2:0];
  
  // Arctangent reference values (32-bit).
  wire signed [ANGLEWIDTH-1:0] arctan_array [1:0];

  genvar                       i;
  
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  assign Angle[0]  = AngleIn;
  assign ReData[0] = ReDataIn;
  assign ImData[0] = ImDataIn;
  
  assign arctan_array[0] = (ImData_sign[0] == 1'b0) ? {3'b0, arctan_ref0[31:31-ANGLEWIDTH+4]} :
                                                      (~({3'b0, arctan_ref0[31:31-ANGLEWIDTH+4]}) + {{ANGLEWIDTH-1{1'b0}},1'b1});
  assign arctan_array[1] = (ImData_sign[1] == 1'b0) ? {3'b0, arctan_ref1[31:31-ANGLEWIDTH+4]} :
                                                      (~({3'b0, arctan_ref1[31:31-ANGLEWIDTH+4]}) + {{ANGLEWIDTH-1{1'b0}},1'b1});

  generate for (i=0; i<2; i=i+1) begin : Gen_CordicVectoringRot
    ////////////////////////////////////////////
    // Angle computation
    ////////////////////////////////////////////
    assign ImData_sign[i] = ImData[i][DATAWIDTH];
    assign Angle[i+1]     = Angle[i] + arctan_array[i];
  
    ////////////////////////////////////////////
    // Stage of microrotations 
    ////////////////////////////////////////////
    CordicVectoringRot #(
        .DATAWIDTH (DATAWIDTH),
        .STAGE     (i+STARTSTAGE))
      U_CordicVectoringRot (
        .ReDataIn  (ReData[i]),
        .ImDataIn  (ImData[i]),
        .ReDataOut (ReData[i+1]),
        .ImDataOut (ImData[i+1])
        );
    end
  endgenerate

  ////////////////////////////////////////////
  // Samples the generated outputs
  ////////////////////////////////////////////
  always @ (posedge PhyClk or negedge nPhyRst) begin
    if (nPhyRst == 1'b0) begin
      AngleOut  <= {ANGLEWIDTH{1'b0}};
      ReDataOut <= {DATAWIDTH+1{1'b0}};
      ImDataOut <= {DATAWIDTH+1{1'b0}};
    end
    else if (DataInEn) begin
      AngleOut  <= Angle[2];
      ReDataOut <= ReData[2];
      ImDataOut <= ImData[2];
    end
  end

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
