//////////////////////////////////////////////////////////////////////////////
//  Copyright [C] by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 11677 $
// $Date: 2013-11-19 13:43:40 +0100 (Tue, 19 Nov 2013) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Unwrap phase from +Pi to -Pi or vice versa.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/PhaseUnwrap.v $
//
//////////////////////////////////////////////////////////////////////////////

module PhaseUnwrap
(
            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock
            input  wire               AGCClk,
            //Reset
            input  wire               nAGCRst,

            input  wire               PhaseUnwrapEn,
            input  wire signed [9:0]  Phi,        // Phase to unwrap
    
            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output reg  signed [17:0] PhiUnwrap
            );


/////////////////////////////////////////////////////////////////////////////
// Local parameter
/////////////////////////////////////////////////////////////////////////////
localparam [10:0] PHASE_SHIFT_CT    = 11'd512;  // Phase shift to detect for n=1
localparam [10:0] PHASE_SHIFT_N2_CT = 11'd492;  // Phase shift to detect for n>=2 : 512-20
localparam [17:0] TWO_PI_CT         = 18'd1024; // 2 x Pi
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire signed [10:0] PhiDelta;
wire        [10:0] PhiDeltaAbs;
wire signed [17:0] PhiToUnwrap;
wire signed [17:0] PhiUnwrapInt;
wire signed [17:0] Skip;
wire               SlopeOrientation;
wire               SlopeOrientationN2;
wire               SlopeCondition;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg signed [9 :0] Phi_ff1;
reg signed [17:0] SkipReg;
reg               PhaseUnwrapEn_ff1;
reg               PhaseUnwrapEn_ff2;
reg        [1:0]  Slope;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign PhiDelta    = Phi - Phi_ff1;
assign PhiDeltaAbs = (PhiDelta[10] == 1'b1) ? -PhiDelta : PhiDelta;

// Slope orientations
assign SlopeOrientation   = (PhiDeltaAbs >= PHASE_SHIFT_CT)    ? 1'b1 : 1'b0;
assign SlopeOrientationN2 = (PhiDeltaAbs >= PHASE_SHIFT_N2_CT) ? 1'b1 : 1'b0;

assign PhiToUnwrap = (PhiDelta[10] == 1'b1) ? TWO_PI_CT  : (~TWO_PI_CT) + 18'd1;

// Slope condition
assign SlopeCondition = (~Slope[1] & PhiDelta[10]) | (~Slope[0] & ~PhiDelta[10]);

// Values for unwrap are accumulated
assign Skip = ((SlopeOrientation   && !PhaseUnwrapEn_ff2) ||
               (SlopeOrientationN2 &&  PhaseUnwrapEn_ff2 && SlopeCondition)) ? SkipReg + PhiToUnwrap : SkipReg;

always @ (posedge AGCClk or negedge nAGCRst)
  begin
    if (nAGCRst == 1'b0) begin
      Phi_ff1           <= 10'b0;
      PhiUnwrap         <= 18'b0;
      SkipReg           <= 18'b0;
      PhaseUnwrapEn_ff1 <= 1'b0;
      PhaseUnwrapEn_ff2 <= 1'b0;
      Slope             <= 2'b0;
    end
    else if (PhaseUnwrapEn == 1'b0) begin
      Phi_ff1           <= 10'b0;
      PhiUnwrap         <= 18'b0;
      SkipReg           <= 18'b0;
      PhaseUnwrapEn_ff1 <= 1'b0;
      PhaseUnwrapEn_ff2 <= 1'b0;
      Slope             <= 2'b0;
    end
    else begin
      // Values for unwrap are accumulated
      SkipReg <= Skip;

      // Register output
      if (!PhaseUnwrapEn_ff1)
        PhiUnwrap <= {{8{Phi[9]}},Phi};
      else
        PhiUnwrap <= PhiUnwrapInt;

      // Register phase current value
      Phi_ff1   <= Phi;
      
      // Generate initial slope
      if (PhaseUnwrapEn_ff1 && !PhaseUnwrapEn_ff2) begin
        if (SlopeOrientation) begin
          if (PhiDelta[10])
            Slope <= 2'b01; // => +1
          else
            Slope <= 2'b10; // => -1
        end
        else if (PhiDeltaAbs >= {1'b0, PHASE_SHIFT_CT[10:1]}) begin
          if (PhiDelta[10])
            Slope <= 2'b10; // => -1
          else
            Slope <= 2'b01; // => +1
        end
        else
          Slope <= 2'b0; // => 0
      end

      // Delayed version of PhaseUnwrapEn
      PhaseUnwrapEn_ff2 <= PhaseUnwrapEn_ff1;
      PhaseUnwrapEn_ff1 <= PhaseUnwrapEn;
      
    end
  end

// Unwrapped phase
assign PhiUnwrapInt = {{8{Phi[9]}},Phi} + Skip;

endmodule

////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
