//////////////////////////////////////////////////////////////////////////////
//  Copyright [C] by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15445 $
// $Date: 2014-07-03 11:36:09 +0200 (Thu, 03 Jul 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Pulse storage FIFO
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/PulseFifo.v $
//
//////////////////////////////////////////////////////////////////////////////


module PulseFifo #(parameter FIFOSIZE      = 8, // FIFO size
                   parameter FIFOADDRWIDTH = 3, // FIFO address width
                   parameter DATASIZE      = 32 // Data size
                   )(
    ///////////////////////////////////////////////
    // Clock and reset
    ///////////////////////////////////////////////
    input    wire         nRadarTimRst,          // AGC Active Low Reset
    input    wire         RadarTimClk,           // AGC Clock - 80MHz

    input    wire         nHRst,                 // AHB Active Low Reset
    input    wire         HClk,                  // AHB Clock

    ///////////////////////////////////////////////
    // Tick every microseconds
    ///////////////////////////////////////////////
    input    wire         Tick1UsP,              // Tick 1us

    ///////////////////////////////////////////////
    // Registers
    ///////////////////////////////////////////////
    input    wire                     RegRadarDetEn,      // Enable radar detection
    input    wire [FIFOADDRWIDTH-1:0] RegPulseIrqThr,     // FIFO threshold for IRQ generation
    input    wire [7:0]               RegPulseIrqTimeOut, // Time-out for Radar IRQ generation (in 128us)
    input    wire                     RegPulseTimerClear, // Clear Radar time stamp timer
    // 
    output   wire                     RadarFifoEmpty,     // FIFO empty status

    ///////////////////////////////////////////////
    // Read interface with Pulse FIFO
    ///////////////////////////////////////////////
    input    wire         RadarFifoRdEn,         // Read signal
    // 
    output   wire [31:0]  RadarFifoRdData,       // Read data

    ///////////////////////////////////////////////
    // Pulse identification
    ///////////////////////////////////////////////
    input    wire [5:0]   PulseLengthTimer,      // Pulse length timer
    input    wire [3:0]   Fom,                   // Figure Of Merite
    input    wire [5:0]   Freq,                  // Frequency

    ///////////////////////////////////////////////
    // Commands from AGC FSM
    ///////////////////////////////////////////////
    input    wire         RadStoreFifoP,         // Store pulse to FIFO

    ///////////////////////////////////////////////
    // Interrupts
    ///////////////////////////////////////////////
    output   reg          RadarIrq
    );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
 wire [FIFOADDRWIDTH-1:0] fifoWrPtr;      // Write ptr
 wire [FIFOADDRWIDTH-1:0] fifoRdPtr;      // Read ptr
 wire                     GenTimeOutIrq;  // Time out interruption
 wire [FIFOADDRWIDTH:0]   fifoWrAddr;     // fifo write address at wrClk
 wire [FIFOADDRWIDTH:0]   fifoRdAddrSync; // fifo read address at wrClk
 wire [FIFOADDRWIDTH:0]   fifoCount;      // fifo counter

//////////////////////////////////////////////////////////////////////////////
// Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////

  // FIFO is made of FIFOSIZE data that are DATASIZE wide
  reg [DATASIZE-1:0]      PulseFifoMem [FIFOSIZE-1:0]; // FIFO content
  reg [FIFOADDRWIDTH:0]   fifoCountSamp;               // fifo counter sampling @ 1us
  reg [FIFOADDRWIDTH:0]   fifoCountSampDly;            // fifo counter sampling delayed by 1 cc
  reg [15:0]              PulseRepetitionTimer;        // Pulse repetition timer
  reg                     PulseRepetitionTimerEn;      // Pulse repetition timer enable

  integer i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
  
// FIFO memory
always @ (posedge RadarTimClk or negedge nRadarTimRst) begin
  if (nRadarTimRst == 1'b0) begin
    for(i=0;i<FIFOSIZE;i=i+1)
      PulseFifoMem[i] <= 32'b0;
  end
  else if (RadStoreFifoP)
    PulseFifoMem[fifoWrPtr] <= {PulseRepetitionTimer,PulseLengthTimer,Fom,Freq};
end

// Instanciation of PulseFifoControl
PulseFifoControl #(.ADDRWIDTH (FIFOADDRWIDTH))
  U_PulseFifoControl (
    .wrClk                  (RadarTimClk),
    .wrHardReset_n          (nRadarTimRst),
    .rdClk                  (HClk),
    .rdHardReset_n          (nHRst),
    .fifoWrite              (RadStoreFifoP),
    .fifoRead               (RadarFifoRdEn),
    .fifoWrPtr              (fifoWrPtr),
    .fifoRdPtr              (fifoRdPtr),
    .fifoWrAddr             (fifoWrAddr),
    .fifoRdAddrSync         (fifoRdAddrSync),
    .fifoFull               (),
    .fifoEmpty              (RadarFifoEmpty),
    .fifoPtrsNull           ()
    );


// AHB read data
assign RadarFifoRdData = PulseFifoMem[fifoRdPtr];

// Radar interruption when pulse FIFO reached Irq threshold or time out (in 128us)
always @ (posedge RadarTimClk or negedge nRadarTimRst) begin
  if (nRadarTimRst == 1'b0)
    RadarIrq <= 1'b0;
  else if (!RegRadarDetEn)
    RadarIrq <= 1'b0;
  else if (((fifoCountSamp >= {1'b0,RegPulseIrqThr}) && (fifoCountSamp != fifoCountSampDly)) || GenTimeOutIrq)
    RadarIrq <= 1'b1;
  else
    RadarIrq <= 1'b0;
end

assign fifoCount = (fifoWrAddr[FIFOADDRWIDTH] ^ fifoRdAddrSync[FIFOADDRWIDTH]) ?
  {1'b1, fifoWrAddr[(FIFOADDRWIDTH-1):0]} - {1'b0, fifoRdAddrSync[(FIFOADDRWIDTH-1):0]} :
  fifoWrAddr - fifoRdAddrSync;

// Delayed version of fifoCount for interrupt pulse generation
always @ (posedge RadarTimClk or negedge nRadarTimRst) begin
  if (nRadarTimRst == 1'b0) begin
    fifoCountSampDly <= {FIFOADDRWIDTH+1{1'b0}};
    fifoCountSamp    <= {FIFOADDRWIDTH+1{1'b0}};
  end
  else begin
    fifoCountSampDly <= fifoCountSamp;
    if (Tick1UsP)
      fifoCountSamp <= fifoCount;
  end
end

// Pulse repetition timer
always @ (posedge RadarTimClk or negedge nRadarTimRst) begin
  if (nRadarTimRst == 1'b0) begin
    PulseRepetitionTimer   <= 16'b0;
    PulseRepetitionTimerEn <= 1'b0;
  end
  else if (!RegRadarDetEn) begin
    PulseRepetitionTimer   <= 16'b0;
    PulseRepetitionTimerEn <= 1'b0;
  end
  else if (RadStoreFifoP) begin
    PulseRepetitionTimer   <= 16'b0;
    PulseRepetitionTimerEn <= 1'b1;
  end
  else if (!PulseRepetitionTimerEn || RegPulseTimerClear)
    PulseRepetitionTimer   <= 16'b0;
  else if (PulseRepetitionTimer == 16'hffff && Tick1UsP)
    PulseRepetitionTimerEn <= 1'b0;
  else if (Tick1UsP)
    PulseRepetitionTimer <= PulseRepetitionTimer + 16'd1;
end

// Time out irq: trig SW if a pulse is available in the FIFO
assign GenTimeOutIrq = ((PulseRepetitionTimer == {1'b0,RegPulseIrqTimeOut,7'b0}) && 
                        PulseRepetitionTimerEn && Tick1UsP && (fifoCountSamp != {FIFOADDRWIDTH+1{1'b0}})) ? 1'b1 : 1'b0;


endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
