//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15445 $
// $Date: 2014-07-03 11:36:09 +0200 (Thu, 03 Jul 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//     The read and write pointers for Pulse FIFO are controlled here.
//     Also the FIFO status such as, full, empty, 
//                    
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module PulseFifoControl #(parameter ADDRWIDTH = 4)(
          //$port_g Inputs are defined here..
          input  wire                       wrClk,            // Write clock
          input  wire                       wrHardReset_n,    // Write clock hard reset
          input  wire                       rdClk,            // Read clock
          input  wire                       rdHardReset_n,    // Read clock hard reset
          input  wire                       fifoWrite,        // Write enables
          input  wire                       fifoRead,         // Read enable

          //$port_g Outputs defined here ..
          output wire   [(ADDRWIDTH - 1):0] fifoWrPtr,        // Write ptr
          output wire   [(ADDRWIDTH - 1):0] fifoRdPtr,        // Read ptr
          
          output reg    [ADDRWIDTH:0]       fifoWrAddr,       // Write fifo address at wrClk
          output wire   [ADDRWIDTH:0]       fifoRdAddrSync,   // Read fifo address at wrClk
          
          output wire                       fifoFull,         // Fifo full indicator
          output wire                       fifoEmpty,        // Fifo empty indicator
          output wire                       fifoPtrsNull      // Indicates FIFO pointers null
          );


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// Internal variables of the module ..

// Internal variables of the module ..
wire   [ADDRWIDTH:0] fifoWrAddrGray;
wire   [ADDRWIDTH:0] fifoWrAddrSync;
wire   [ADDRWIDTH:0] fifoWrAddrGraySync;
wire   [ADDRWIDTH:0] fifoRdAddr;
wire   [ADDRWIDTH:0] fifoRdAddrGray;
wire   [ADDRWIDTH:0] fifoRdAddrGraySync;

`ifdef SIMU_ON
  integer fifoCnt;
`endif // SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
// This function converts Gray Code into Binary Code
function [ADDRWIDTH:0] gray2Binary;
input    [ADDRWIDTH:0] gray;
integer i;
begin
   gray2Binary[ADDRWIDTH] = gray[ADDRWIDTH];
   for(i = 1; i <= ADDRWIDTH; i = i + 1) 
   begin
      gray2Binary[ADDRWIDTH - i] =
           gray2Binary[ADDRWIDTH - i+1] ^ gray[ADDRWIDTH - i];
   end
end
endfunction

// fifoRdAddrGray converted to binary
assign fifoRdAddr = gray2Binary(fifoRdAddrGray);

// Gray to Binary conversion for the synchronised Read Address
assign fifoRdAddrSync = gray2Binary(fifoRdAddrGraySync);

// Gray to Binary conversion for the synchronised Write Address
assign fifoWrAddrSync = gray2Binary(fifoWrAddrGraySync);

// The MSB of the address is not used as a part of Pointer.
//fifoWrPtr
assign fifoWrPtr  = fifoWrAddr[(ADDRWIDTH-1):0];
//fifoRdPtr
assign fifoRdPtr  = fifoRdAddr[(ADDRWIDTH-1):0];

// The status of fifoFull is continuously driven based on the 
// fifo addresses
assign fifoFull = (fifoWrAddr[(ADDRWIDTH-1):0] == fifoRdAddrSync[(ADDRWIDTH-1):0]) &&
                  (fifoWrAddr[ADDRWIDTH] ^ fifoRdAddrSync[ADDRWIDTH]);

// fifoEmpty is asserted in rdClk domain after synchronising the
// fifoWrAddr to rdClk. 
assign fifoEmpty = (fifoRdAddr == fifoWrAddrSync);

// fifoPtrsNull is asserted when both write and read pointers are null
assign fifoPtrsNull = (fifoRdAddr == fifoWrAddrSync) && (fifoWrAddrSync == {(ADDRWIDTH+1){1'b0}});

// fifoRdAddrGray is synchronised to rdClk
// Following instantiation increments fifoRdAddr and
// converts this new address into corresponding Gray encoding
PulseFifoPtrGray #(.ADDRWIDTH(ADDRWIDTH)) U_fifoRdAddrGray(
                             .clk(rdClk),
                             .hardReset_n(rdHardReset_n),
                             .rdWrEn(fifoRead),
                             .rdWrDisable(fifoEmpty),
                             .fifoAddr(fifoRdAddr),
                             .fifoAddrGray(fifoRdAddrGray)
                            );

// fifoRdAddrGray is synchronised to wrClk
ClkSyncSimpleTop #(.SIZE(ADDRWIDTH+1)) U_fifoRdAddrSync (
                              .dstclk (wrClk),
                             .dstresetn (wrHardReset_n),
                             .srcdata (fifoRdAddrGray),
                             .dstdata (fifoRdAddrGraySync)
                             );

// fifoWrAddrGray is syncronised to rdClk
// Following instantiation increments fifoWrAddr and
// converts this new address into corresponding Gray encoding
PulseFifoPtrGray #(.ADDRWIDTH(ADDRWIDTH)) U_fifoWrAddrGray(
                             .clk(wrClk),
                             .hardReset_n(wrHardReset_n),
                             .rdWrEn(fifoWrite),
                             .rdWrDisable(fifoFull),
                             .fifoAddr(fifoWrAddr),
                             .fifoAddrGray(fifoWrAddrGray)
                            );

// fifoWrAddrGray is synchronised to rdClk
ClkSyncSimpleTop #(.SIZE(ADDRWIDTH+1)) U_fifoWrAddrSync (
                             .dstclk (rdClk),
                             .dstresetn (rdHardReset_n),
                             .srcdata (fifoWrAddrGray),
                             .dstdata (fifoWrAddrGraySync)
                             );

// The fifoWrAddr is incremented on every fifoWrite.
always @(posedge wrClk or negedge wrHardReset_n) 
begin
   if (wrHardReset_n == 1'b0) 
     fifoWrAddr <= {(ADDRWIDTH+1){1'b0}}; 
   else if(fifoWrite && (fifoFull == 1'b0)) 
     fifoWrAddr <= fifoWrAddr + {{(ADDRWIDTH){1'b0}},1'd1};
   else 
     fifoWrAddr <= fifoWrAddr;
end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef SIMU_ON

always @(posedge wrClk or negedge wrHardReset_n) 
begin
   if(wrHardReset_n == 1'b0) 
     fifoCnt = 0; 
   else if(fifoWrite && (fifoFull == 1'b0)) 
     fifoCnt = fifoCnt + 1; 
end

always @(posedge rdClk or negedge rdHardReset_n) 
begin
   if(rdHardReset_n == 1'b0) 
     fifoCnt = 0; 
   else if(fifoRead && (fifoEmpty == 1'b0)) 
     fifoCnt = fifoCnt - 1; 
end

`endif // SIMU_ON

endmodule

////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
