//////////////////////////////////////////////////////////////////////////////
//  Copyright [C] by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession || use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15445 $
// $Date: 2014-07-03 11:36:09 +0200 (Thu, 03 Jul 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Generate control signals for all radar detection sub-blocks.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/RadarController.v $
//
//////////////////////////////////////////////////////////////////////////////

module RadarController
(
            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and Reset
            input wire       nAGCRst,          // Active Low Reset
            input wire       AGCClk,           // AGC Clock
            
            // AGC FSM controls
            input wire       RadSWResetP,      // Radar SW reset
            input wire [7:0] N2Plus1Param,     // 2N+1 parameter

            // Radar trigger
            input wire       RadStartP,        // radar pulse rise/fall trigger

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            // Phase storage FIFO control
            output reg       FifoShiftEn,      // Enable FIFO shift
            output reg       FifoLoadEn,       // Enable FIFO load
            output reg       FifoInitPtr,      // Init FIFO read pointer
            
            // Blocks control
            output wire      PhaseUnwrapEn,    // Enable unwrap block
            output reg       CordicRotEn,      // Enable CORDIC rotator
            output reg       SlopeEstimEn,     // Enable slope estimation
            output reg       PhiRampEn,        // Enable ramp phase block
            output reg       PhiDistEn,        // Enable phi distance block

            output wire      TrigFreqP,        // Trigger frequency estimate
            output wire      TrigFomP,         // Trigger fom estimate
            
            // AGC status information
            output wire      Meas1Done,        // Indicates measure 1 is ended
            output wire      Meas2Done,        // Indicates measure 2 is ended
            output wire      Meas3Done,        // Indicates measure 3 is ended
            
            output reg [7:0] PhiIndex,         // Phi FIFO index

            // Debug fsm
            output reg [2:0] ControlState      // FSM for debug
            );

/////////////////////////////////////////////////////////////////////////////
// Local parameter
/////////////////////////////////////////////////////////////////////////////

localparam [7:0] FOM_LAT = 8'd7; // FOM latency = 7

localparam IDLE            = 3'd0,
           KEEP_WR_POINTER = 3'd1,
           INIT_POINTER1   = 3'd2,
           DATA_PROCESS    = 3'd3,
           INIT_POINTER2   = 3'd4,
           RAMP_PROCESS    = 3'd5,
           WAIT_PIPELINE   = 3'd6;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg [7:0] WaitCount;
reg [1:0] MeasCount;

// Triggers the counter to start incrementing for omega and psi estimation
reg       StartDataProcess;
// Triggers the counter to start incrementing for phi estimation
reg       StartRampProcess;

`ifdef SIMU_ON
reg   [50*8:0] RadarControlStateStr;
`endif //SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

always @ (posedge AGCClk or negedge nAGCRst)
begin
   if (nAGCRst == 1'b0) begin
      ControlState       <= IDLE;
      FifoShiftEn        <= 1'b0;     
      FifoLoadEn         <= 1'b0;
      FifoInitPtr        <= 1'b0;
      StartDataProcess   <= 1'b0;
      StartRampProcess   <= 1'b0;
      WaitCount          <= 8'b0;
   end
   else if (RadSWResetP == 1'b1) begin
      ControlState       <= IDLE;
      FifoShiftEn        <= 1'b0;     
      FifoLoadEn         <= 1'b0;
      FifoInitPtr        <= 1'b0;
      StartDataProcess   <= 1'b0;
      StartRampProcess   <= 1'b0;
      WaitCount          <= 8'b0;
   end
   else begin
      case (ControlState)
         // Idle state
         IDLE : begin
            FifoShiftEn <= 1'b1;
            FifoInitPtr <= 1'b0;
            FifoLoadEn  <= 1'b0;
            if (RadStartP == 1'b1) begin
              FifoShiftEn  <= 1'b0;
              ControlState <= KEEP_WR_POINTER;
            end
         end

         // Keep write pointer
         KEEP_WR_POINTER : begin
            FifoLoadEn   <= 1'b0;
            FifoShiftEn  <= 1'b0;
            FifoInitPtr  <= 1'b1;
            ControlState <= INIT_POINTER1;
         end

         // Init read pointer for phi estimation
         INIT_POINTER1 : begin
            FifoInitPtr      <= 1'b0;
            FifoLoadEn       <= 1'b1;
            ControlState     <= DATA_PROCESS;
         end

         // Phi estimation
         DATA_PROCESS : begin
            StartDataProcess <= FifoLoadEn;
            if (PhiIndex == N2Plus1Param-8'd3)
              FifoLoadEn     <= 1'b0;
            else if (PhiIndex == N2Plus1Param-8'd1) begin
              ControlState   <= INIT_POINTER2;
            end
         end

         // Init read pointer for measure estimation
         INIT_POINTER2 : begin
            FifoInitPtr      <= 1'b1;
            FifoLoadEn       <= 1'b0;
            ControlState     <= RAMP_PROCESS;
         end
         
         // Measure estimation
         RAMP_PROCESS : begin
            FifoInitPtr      <= 1'b0;
            StartRampProcess <= FifoLoadEn;
            if (PhiIndex == N2Plus1Param-8'd2)
              FifoLoadEn     <= 1'b0;
            else if (PhiIndex == N2Plus1Param-8'd1) begin  
              FifoShiftEn    <= 1'b1;
              ControlState   <= WAIT_PIPELINE;
            end
            else
              FifoLoadEn     <= 1'b1;
         end
         
         // Wait end of measure pipeline
         WAIT_PIPELINE : begin
            if (WaitCount == FOM_LAT)  begin 
               WaitCount    <= 8'b0;                                   
               ControlState <= IDLE;                                
            end                                                              
            else                                                                 
               WaitCount <= WaitCount + 8'd1;
         end

         // pragma coverage block = off
         default : begin
            ControlState <= IDLE;
         end
         // pragma coverage block = on

      endcase
   end
end

// Unwrap phase control
assign PhaseUnwrapEn = StartDataProcess | StartRampProcess;

always @ (posedge AGCClk or negedge nAGCRst)
begin
  if (nAGCRst == 1'b0) begin
    CordicRotEn       <= 1'b0;
    SlopeEstimEn      <= 1'b0;
    PhiRampEn         <= 1'b0;
    PhiDistEn         <= 1'b0;
  end
  else if (RadSWResetP == 1'b1) begin
    CordicRotEn       <= 1'b0;
    SlopeEstimEn      <= 1'b0;
    PhiRampEn         <= 1'b0;
    PhiDistEn         <= 1'b0;
  end
  else begin
    // Slope estimator block
    if (StartDataProcess)
      SlopeEstimEn <= 1'b1;
    else if (PhiIndex == N2Plus1Param)
      SlopeEstimEn <= 1'b0;

    // Ramp phase control
    if (FifoLoadEn && ControlState == RAMP_PROCESS)
      PhiRampEn <= 1'b1;
    else if (ControlState == IDLE)
      PhiRampEn <= 1'b0;
    
    // Phi distance control
    if (PhiRampEn)
      PhiDistEn <= 1'b1;
    else if (ControlState == IDLE)
      PhiDistEn <= 1'b0;
    
    // Cordic rotator block
    if (PhiDistEn)
      CordicRotEn <= 1'b1;
    else if (ControlState == IDLE)
      CordicRotEn <= 1'b0;

   end
end

// Trig pulse measure
assign TrigFreqP = (ControlState == WAIT_PIPELINE) & (WaitCount == 8'b0);
assign TrigFomP  = (ControlState == WAIT_PIPELINE) & (WaitCount == FOM_LAT);

// Measure counter
always @ (posedge AGCClk or negedge nAGCRst)
begin
   if (nAGCRst == 1'b0)
      MeasCount <= 2'b0;
   else if (RadSWResetP == 1'b1)
      MeasCount <= 2'b0;
   else if (TrigFomP && MeasCount != 2'd3)
      MeasCount <= MeasCount + 2'd1;
end

// AGC status information
assign Meas1Done = (MeasCount >= 2'd1) ? 1'b1 : 1'b0; // Indicates measure 1 is ended
assign Meas2Done = (MeasCount >= 2'd2) ? 1'b1 : 1'b0; // Indicates measure 2 is ended
assign Meas3Done = (MeasCount >= 2'd3) ? 1'b1 : 1'b0; // Indicates measure 3 is ended


// Phi index counter
always @ (posedge AGCClk or negedge nAGCRst)
begin
  if (nAGCRst == 1'b0)
    PhiIndex <= 8'b0;
  else if ((RadSWResetP) || (!SlopeEstimEn && !PhiRampEn) ||
    (ControlState == INIT_POINTER1) || (ControlState == INIT_POINTER2))
    PhiIndex <= 8'b0;
  else
    PhiIndex <= PhiIndex + 8'd1;
end

//Additional Code to ease verification
//Display FSM State as a string in RTL simulation waveform
`ifdef SIMU_ON
 always @ (*)
   begin: RadarControlStateStr_Blk
      case (ControlState)
         IDLE              : RadarControlStateStr = {"IDLE"};
         KEEP_WR_POINTER   : RadarControlStateStr = {"KEEP_WR_POINTER"};
         DATA_PROCESS      : RadarControlStateStr = {"DATA_PROCESS"};
         INIT_POINTER1     : RadarControlStateStr = {"INIT_POINTER1"};
         RAMP_PROCESS      : RadarControlStateStr = {"RAMP_PROCESS"};
         INIT_POINTER2     : RadarControlStateStr = {"INIT_POINTER2"};
         WAIT_PIPELINE     : RadarControlStateStr = {"WAIT_PIPELINE"};
         default           : RadarControlStateStr = {"UNKNOWN"};
      endcase
 end
`endif


endmodule

////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
