//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15445 $
// $Date: 2014-07-03 11:36:09 +0200 (Thu, 03 Jul 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Radar detection module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/RadarDetection.v $
//
//////////////////////////////////////////////////////////////////////////////

module RadarDetection( 
            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input    wire                                  nAGCRst,      // AGC Active Low Reset
            input    wire                                  AGCClk,       // AGC Clock - 80MHz

            input    wire                                  nRadarTimRst, // Radar Timer Active Low Reset
            input    wire                                  RadarTimClk,  // Radar Timer Clock - 80MHz ungated

            input    wire                                  nHRst,        // AHB Active Low Reset
            input    wire                                  HClk,         // AHB Clock

            ///////////////////////////////////////////////
            // AGC interface
            ///////////////////////////////////////////////
            input    wire                                  RadarEn,      // Enable radar processing
            //
            input    wire [7:0]                            AGCCommand,   // Command from FSM Core
            input    wire [23:0]                           AGCParam,     // Command Parameter
            input    wire                                  AGCCmdValid,  // AGC Command Valid
            //
            output   wire                                  FomSing,      // FOM singuliar status
            output   wire                                  FomHigh,      // FOM high status
            output   wire                                  FomMed,       // FOM low status

            output   wire                                  FreqSing,     // Frequency singuliar status
            output   wire                                  FreqDC,       // Frequency DC status
            output   wire                                  FreqLow,      // Frequency low status
            output   wire                                  FreqHigh,     // Frequency high status

            output   wire                                  LengthLow,    // Timer length low status
            output   wire                                  LengthHigh,   // Timer length high status

            output   wire                                  Meas1Done,    // Measure 1 done status
            output   wire                                  Meas2Done,    // Measure 2 done status
            output   wire                                  Meas3Done,    // Measure 3 done status
                        
            ///////////////////////////////////////////////
            // Registers
            ///////////////////////////////////////////////
            input    wire                                  RegRadarDetEn,      // Enable radar detection
            input    wire [7:0]                            RegFreqSing,        // Radar Frequency Singuliar value
            input    wire [3:0]                            RegFreqSingGap,     // Radar Frequency Singuliar Gap
            input    wire [7:0]                            RegFomSing,         // Radar Figure Of Merite Singuliar value
            input    wire [3:0]                            RegFomSingGap,      // Radar Figure Of Merite Singuliar Gap
            input    wire [2:0]                            RegPulseIrqThr,     // FIFO Threshold for Radar IRQ generation
            input    wire [7:0]                            RegPulseIrqTimeOut, // Time-out for Radar IRQ generation (in 128us)
            input    wire                                  RegPulseTimerClear, // Clear Radar time stamp timer
            input    wire                                  RegForce20,         // Force 20 Mode
            input    wire                                  RegForce40,         // Force 40 Mode
            // 
            output   wire                                  RadarFifoEmpty,     // FIFO empty status

            ///////////////////////////////////////////////
            // From FE 20/40/80MHz
            ///////////////////////////////////////////////
            input    wire [12:0]                           RxRadI,        // Real Component
            input    wire [12:0]                           RxRadQ,        // Imaginary Component
            input    wire                                  RxRadValid,    // Radar data valid

            ///////////////////////////////////////////////
            // Read interface with pulse FIFO
            ///////////////////////////////////////////////
            input    wire                                  RadarFifoRdEn,   // Read signal
            // 
            output   wire [31:0]                           RadarFifoRdData, // Read data

            ///////////////////////////////////////////////
            // Radar memory
            ///////////////////////////////////////////////
            input    wire [9:0]                            RadarMemRdData,   // Data from Radar RAM
            //
            output   wire [9:0]                            RadarMemWrData,   // Data to Radar RAM
            output   wire [8:0]                            RadarMemAddr,     // Address to Radar RAM
            output   wire                                  RadarMemWriteEn,  // Write enable to Radar RAM
            output   wire                                  RadarMemEn,       // Enable to Radar RAM
    
            ///////////////////////////////////////////////
            // Interrupt
            ///////////////////////////////////////////////
            output   wire                                  RadarIrq,         // Radar interrupt
    
            ///////////////////////////////////////////////
            // Debug port
            ///////////////////////////////////////////////
            output   wire      [15:0]                      DbgRad
            );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
localparam [9:0] HUNDRED_TWENTY_SIX = 10'sd126;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers declarations
//////////////////////////////////////////////////////////////////////////////
reg signed        [18:0]              PhiDist;               // Phase distance
reg signed        [16:0]              CRAccu;                // Cordic rotator accumulator
reg               [7:0]               Fom;                   // Figure Of Merite
reg signed        [7:0]               Freq;                  // Frequency
reg signed        [18:0]              PhiRampEst;            // Phase ramp estimated

reg               [6:0]               DivCount1Us;           // 1us counter
reg               [7:0]               PulseLengthTimer;      // Pulse length timer
reg                                   RadTimerEnD1;          // Timer enable delayed by 1cc
reg                                   PhiDataEnableD1;       // Enable phase computation delayed by 1cc

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire                                  RadSWResetP;            // Radar SW reset
wire                                  RadStartP;              // Start pulse processing
wire              [2:0]               PulseProcConf;          // Pulse processing configuration
wire              [7:0]               TimeOffsetPhases;       // Time offset to select phases
wire                                  RadStoreFifoP;          // Store pulse information into storage FIFO
wire                                  RadTimerEn;             // Timer enable
wire              [7:0]               FomThrHigh;             // FOM threshold high
wire              [7:0]               FomThrMed;              // FOM threshold medium
wire              [7:0]               FreqThrInf;             // Frequency threshold inferior
wire              [7:0]               FreqThrSup;             // Frequency threshold superior
wire              [7:0]               FreqThrDC;              // Frequency threshold DC
wire              [7:0]               LengthThrInf;           // Timer length threshold inferior
wire              [7:0]               LengthThrSup;           // Timer length threshold superior

wire                                  FifoShiftEn;            // Enable FIFO shift
wire                                  FifoLoadEn;             // Enable FIFO load
wire                                  FifoInitPtr;            // Init read pointer
wire              [7:0]               PhiIndex;               // Phi index
wire                                  PhaseUnwrapEn;          // Enable unwrap block accu.
wire                                  SlopeEstimEn;           // Enable slope estimation
wire                                  PhiRampEn;              // Enable ramp phase block
wire                                  PhiDistEn;              // Enable phi distance block
wire                                  CordicRotEn;            // Enable CORDIC rotator

wire                                  TrigFreqP;              // Capture frequency value
wire                                  TrigFomP;               // Capture fom value
wire              [2:0]               ControlStateFsm;        // Radar controller FSM

wire              [7:0]               NRomParam       [4:0];  // N parameter
wire              [7:0]               N2Plus1RomParam [4:0];  // 2N+1 parameter
wire              [9:0]               D1RomParam      [4:0];  // D1 parameter
wire              [9:0]               D2RomParam      [4:0];  // D2 parameter
wire              [15:0]              SRomParam       [4:0];  // S parameter

wire                                  PhiDataEnable;          // Enable phase computation
wire signed       [9:0]               Phi;                    // Cordic phase
wire signed       [9:0]               PhiFifo;                // Fifo phase
wire signed       [17:0]              PhiUnwrap;              // Phase unwrapped
wire signed       [9:0]               AngleCordicRot;         // Cordic rotator angle
wire signed       [9:0]               CordicRotReDataIn;      // Cordic rotator Real Component input
wire signed       [9:0]               CordicRotRe;            // Cordic rotator Real Component output
wire                                  CordicRotReValid;       // Cordic rotator Real Component
wire              [16:0]              CRAccuAbs;              // Absolute of Cordic rotator accumulator
wire              [15:0]              CRAccuAbsTrunc;         // Accumulator truncate
wire              [23:0]              CRAccuAbsMultS;         // Accumulator multiply by S(N)


wire signed       [18:0]              PsiEst;                 // Psi estimate
wire signed       [10:0]              OmegaEst;               // Omega estimate
wire              [6:0]               BandwidthSel;           // Frame bandwidth
wire signed       [19:0]              OmegaEstMultBW;         // Omega multiply by BW
wire signed       [18:0]              PhiRampPreEst0;         // Phase ramp 1st intermediate calculation
wire signed       [18:0]              PhiRampPreEst1;         // Phase ramp 2nd intermediate calculation

wire                                  Tick1UsP;               // Tick 1us

wire              [5:0]               PulseLengthTimerToFifo; // Pulse length timer to FIFO

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


// Instanciation of RadarIFReg
///////////////////////////////////////////
RadarIFReg U_RadarIFReg (
  .nAGCRst            (nAGCRst),
  .AGCClk             (AGCClk),
  .AGCCommand         (AGCCommand),
  .AGCParam           (AGCParam),
  .AGCCmdValid        (AGCCmdValid),
  .RadSWResetP        (RadSWResetP),
  .RadStartP          (RadStartP),
  .PulseProcConf      (PulseProcConf),
  .TimeOffsetPhases   (TimeOffsetPhases),
  .RadStoreFifoP      (RadStoreFifoP),
  .RadTimerEn         (RadTimerEn),
  .FomThrHigh         (FomThrHigh),
  .FomThrMed          (FomThrMed),
  .FreqThrInf         (FreqThrInf),
  .FreqThrSup         (FreqThrSup),
  .FreqThrDC          (FreqThrDC),
  .LengthThrInf       (LengthThrInf),
  .LengthThrSup       (LengthThrSup)
  );

// ROM for storing N related parameters
///////////////////////////////////////////
assign NRomParam[0]       = 8'd4;
assign NRomParam[1]       = 8'd8;
assign NRomParam[2]       = 8'd18;
assign NRomParam[3]       = 8'd38;
assign NRomParam[4]       = 8'd78;

assign N2Plus1RomParam[0] = 8'd9;
assign N2Plus1RomParam[1] = 8'd17;
assign N2Plus1RomParam[2] = 8'd37;
assign N2Plus1RomParam[3] = 8'd77;
assign N2Plus1RomParam[4] = 8'd157;

assign D1RomParam[0]      = 10'd546;
assign D1RomParam[1]      = 10'd643;
assign D1RomParam[2]      = 10'd497;
assign D1RomParam[3]      = 10'd441;
assign D1RomParam[4]      = 10'd416;

assign D2RomParam[0]      = 10'd910;
assign D2RomParam[1]      = 10'd964;
assign D2RomParam[2]      = 10'd886;
assign D2RomParam[3]      = 10'd851;
assign D2RomParam[4]      = 10'd835;

assign SRomParam[0]       = 16'd228;
assign SRomParam[1]       = 16'd120;
assign SRomParam[2]       = 16'd55;
assign SRomParam[3]       = 16'd27;
assign SRomParam[4]       = 16'd13;

// Radar controller
///////////////////////////////////////////
RadarController U_RadarController (
  .nAGCRst                (nAGCRst),
  .AGCClk                 (AGCClk),
  .RadSWResetP            (RadSWResetP),
  .RadStartP              (RadStartP),
  .N2Plus1Param           (N2Plus1RomParam[PulseProcConf]),
  .FifoShiftEn            (FifoShiftEn),
  .FifoLoadEn             (FifoLoadEn),
  .FifoInitPtr            (FifoInitPtr),
  .PhaseUnwrapEn          (PhaseUnwrapEn),
  .CordicRotEn            (CordicRotEn),
  .SlopeEstimEn           (SlopeEstimEn),
  .PhiRampEn              (PhiRampEn),
  .PhiDistEn              (PhiDistEn),
  .TrigFreqP              (TrigFreqP),
  .TrigFomP               (TrigFomP),
  .Meas1Done              (Meas1Done),
  .Meas2Done              (Meas2Done),
  .Meas3Done              (Meas3Done),
  .PhiIndex               (PhiIndex),
  .ControlState           (ControlStateFsm)
  );

// Continuous phase computation
///////////////////////////////////////////

assign PhiDataEnable = RxRadValid & RadarEn;

// Instantiation of CordicVectTop for phase computation
CordicVectoring  #(.DATAWIDTH  (13),
                   .ANGLEWIDTH (10),
                   .NCOMBSTG   (4),
                   .SCALING    (1))
  U_CordicVectoring(
  //Inputs
  .PhyClk       (AGCClk),
  .nPhyRst      (nAGCRst),
  .DataInEn     (PhiDataEnable),
  .ReDataIn     (RxRadI),
  .ImDataIn     (RxRadQ),
  // Outputs
  .AngleOut     (Phi)
  );

always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    PhiDataEnableD1 <= 1'b0;
  else
    PhiDataEnableD1 <= PhiDataEnable;
end

// Phase storage memory control
///////////////////////////////////////////
PhaseFifo #(.DATASIZE (10))
  U_PhaseFifo ( 
  .AGCClk                   (AGCClk),
  .nAGCRst                  (nAGCRst),
  .Phi                      (Phi),
  .PhiValid                 (PhiDataEnableD1),
  .FifoShiftEn              (FifoShiftEn),
  .FifoLoadEn               (FifoLoadEn),
  .FifoInitPtr              (FifoInitPtr),
  .TimeOffsetPhases         (TimeOffsetPhases),
  .N2Plus1Param             (N2Plus1RomParam[PulseProcConf]),
  .RadarMemRdData           (RadarMemRdData),
  .RadarMemWrData           (RadarMemWrData),
  .RadarMemAddr             (RadarMemAddr),
  .RadarMemWriteEn          (RadarMemWriteEn),
  .RadarMemEn               (RadarMemEn),
  .PhiFifo                  (PhiFifo)
  );

// Pulse measure
///////////////////////////////////////////

// Instanciation of PhaseUnwrap
PhaseUnwrap U_PhaseUnwrap (
  .AGCClk                 (AGCClk),
  .nAGCRst                (nAGCRst),
  .PhaseUnwrapEn          (PhaseUnwrapEn),
  .Phi                    (PhiFifo),
  .PhiUnwrap              (PhiUnwrap)
  );

// Instanciation of SlopeEstim
SlopeEstim U_SlopeEstim (
  .AGCClk                 (AGCClk),
  .nAGCRst                (nAGCRst),
  .RadSWResetP            (RadSWResetP),
  .SlopeEstimEn           (SlopeEstimEn),
  .PhiIndex               (PhiIndex),
  .PhiUnwrap              (PhiUnwrap),
  .PulseProcConf          (PulseProcConf),
  .NParam                 (NRomParam[PulseProcConf]),
  .D1Param                (D1RomParam[PulseProcConf]),
  .D2Param                (D2RomParam[PulseProcConf]),
  .OmegaEst               (OmegaEst),
  .PsiEst                 (PsiEst)
  );

// Generate the reference ramp phase
assign PhiRampPreEst0 = $signed({11'b0,PhiIndex}) - $signed({11'b0,NRomParam[PulseProcConf]});
assign PhiRampPreEst1 = $signed(PhiRampPreEst0) * $signed({{8{OmegaEst[10]}},OmegaEst});

always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    PhiRampEst <= 19'b0;
  else if (!PhiRampEn)
    PhiRampEst <= 19'b0;
  else
    PhiRampEst <= $signed(PhiRampPreEst1) + $signed(PsiEst);
end


// Frequency computation
///////////////////////////////////////////

// Bandwidth selection
assign BandwidthSel = (RegForce20 == 1'b0 && RegForce40 == 1'b0) ? 7'd80 :
                      (RegForce20 == 1'b0) ? 7'd40 : 7'd20;

// Omega multiply by BW
assign OmegaEstMultBW = $signed(OmegaEst) * $signed({4'b0,BandwidthSel});

// Frequency
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    Freq <= 8'b0;
  else if (RadSWResetP)
    Freq <= 8'b0;
  else if (TrigFreqP)
    Freq <= $signed(OmegaEstMultBW[16:9]);
end

// Figure Of Merite computation
///////////////////////////////////////////

// Measures the distance between the current phase and the estimated ramp phase
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    PhiDist <= 19'b0;
  else if (!PhiDistEn)
    PhiDist <= 19'b0;
  else 
    PhiDist <= $signed({PhiUnwrap[17],PhiUnwrap}) - $signed(PhiRampEst);
end

// Overflow is needed for AngleCordicRot
assign AngleCordicRot = PhiDist[9:0];

// CordicRot input set to constant 126
assign CordicRotReDataIn = HUNDRED_TWENTY_SIX;

// Instantiation of CordicRot
CordicRotTop #(.DATAWIDTH  (10),
               .ANGLEWIDTH (10),
               .NCOMBSTG   (5),
               .NPIPE      (2),
               .NIP        (1),
               .SCALING    (1))
  U_CordicRot (
  //Inputs
  .PhyClk          (AGCClk),
  .nPhyRst         (nAGCRst),
  .Enable          (CordicRotEn),
  .DataInValid     (CordicRotEn),
  .AngleIn         (AngleCordicRot),
  .ReDataIn0       (CordicRotReDataIn),
  .ImDataIn0       (10'b0),
  .ReDataIn1       (10'b0),
  .ImDataIn1       (10'b0),
  .ReDataIn2       (10'b0),
  .ImDataIn2       (10'b0),
  .ReDataIn3       (10'b0),
  .ImDataIn3       (10'b0),

  //Outputs
  .DataOutValid    (CordicRotReValid),
  .ReDataOut0      (CordicRotRe),
  .ImDataOut0      (),
  .ReDataOut1      (),
  .ImDataOut1      (),
  .ReDataOut2      (),
  .ImDataOut2      (),
  .ReDataOut3      (),
  .ImDataOut3      ()
  );

// Accumulation of the 2N+1
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    CRAccu <= 17'b0;
  else if (!CordicRotEn)
    CRAccu <= 17'b0;
  else if (CordicRotReValid)
    CRAccu <= CRAccu + $signed({{7{CordicRotRe[9]}},CordicRotRe});
end

// Accumulator absolute value
assign CRAccuAbs = (CRAccu[16] == 1'b1) ? -CRAccu : CRAccu;

// Accumulator absolute value truncate
assign CRAccuAbsTrunc = CRAccuAbs[15:0];

// Accumulator absolute value truncate multiply by S(N)
assign CRAccuAbsMultS = CRAccuAbsTrunc * SRomParam[PulseProcConf];

// Figure Of Merite
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    Fom <= 8'b0;
  else if (RadSWResetP)
    Fom <= 8'b0;
  else if (TrigFomP)
    Fom <= CRAccuAbsMultS[17:10];
end


// Timers
///////////////////////////////////////////

// 1us counter
always @ (posedge RadarTimClk or negedge nRadarTimRst) begin
  if (nRadarTimRst == 1'b0)
    DivCount1Us <= 7'b0;
  else if (Tick1UsP)
    DivCount1Us <= 7'b0;
  else
    DivCount1Us <= DivCount1Us + 7'd1;
end

assign Tick1UsP = (DivCount1Us == 7'd79) ? 1'b1 : 1'b0;

// Pulse length timer
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0) begin
    PulseLengthTimer <= 8'b0;
    RadTimerEnD1     <= 1'b0;
  end
  else begin
    RadTimerEnD1 <= RadTimerEn;
    if (RadTimerEn && !RadTimerEnD1)
      PulseLengthTimer <= 8'b0;
    else if (Tick1UsP && RadTimerEn)
      PulseLengthTimer <= PulseLengthTimer + 8'd1;
  end
end

// Radar comparators
///////////////////////////////////////////
RadarComp U_RadarComp (
  .Fom               (Fom),
  .Freq              (Freq),
  .PulseLengthTimer  (PulseLengthTimer),
  .FomThrHigh        (FomThrHigh),
  .FomThrMed         (FomThrMed),
  .FreqThrInf        (FreqThrInf),
  .FreqThrSup        (FreqThrSup),
  .FreqThrDC         (FreqThrDC),
  .LengthThrInf      (LengthThrInf),
  .LengthThrSup      (LengthThrSup),
  .RegFreqSing       (RegFreqSing),
  .RegFreqSingGap    (RegFreqSingGap),
  .RegFomSing        (RegFomSing),
  .RegFomSingGap     (RegFomSingGap),
  .FomSing           (FomSing),
  .FomHigh           (FomHigh),
  .FomMed            (FomMed),
  .FreqSing          (FreqSing),
  .FreqDC            (FreqDC),
  .FreqLow           (FreqLow),
  .FreqHigh          (FreqHigh),
  .LengthLow         (LengthLow),
  .LengthHigh        (LengthHigh)
  );

// Pulse storage FIFO
///////////////////////////////////////////
PulseFifo U_PulseFifo (
  .nRadarTimRst         (nRadarTimRst),
  .RadarTimClk          (RadarTimClk),
  .nHRst                (nHRst),
  .HClk                 (HClk),
  .Tick1UsP             (Tick1UsP),
  .RegRadarDetEn        (RegRadarDetEn),
  .RegPulseIrqThr       (RegPulseIrqThr),
  .RegPulseIrqTimeOut   (RegPulseIrqTimeOut),
  .RegPulseTimerClear   (RegPulseTimerClear),
  .RadarFifoEmpty       (RadarFifoEmpty),
  .RadarFifoRdEn        (RadarFifoRdEn),
  .RadarFifoRdData      (RadarFifoRdData),
  .PulseLengthTimer     (PulseLengthTimerToFifo),
  .Fom                  (Fom[7:4]),
  .Freq                 (Freq[7:2]),
  .RadStoreFifoP        (RadStoreFifoP),
  .RadarIrq             (RadarIrq)
  );

// Saturate PulseLengthTimer before Pulse storage FIFO
SatUnsigned #(.INPUT_WIDTH  (7),
              .OUTPUT_WIDTH (6))
  U_SatPulseLength (
              .InputData    (PulseLengthTimer[7:1]),
              .SatData      (PulseLengthTimerToFifo)
              );

// Debug port
assign DbgRad = {RadStoreFifoP,
                 RadarIrq,
                 RadarFifoEmpty,
                 RadarFifoRdEn,
                 FomSing,
                 FomHigh,
                 FomMed,
                 FreqSing,
                 FreqDC,
                 FreqLow,
                 FreqHigh,
                 LengthLow,
                 LengthHigh,
                 ControlStateFsm  // Controller state FSM
                };
endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
