//////////////////////////////////////////////////////////////////////////////
//  Copyright [C] by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 10828 $
// $Date: 2013-10-15 11:55:21 +0200 (Tue, 15 Oct 2013) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : The block can function in two modes Data accumulation
//                    Each of them have a dedicated input but the same output
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/RadarDetection/verilog/rtl/SlopeEstim.v $
//
//////////////////////////////////////////////////////////////////////////////

module SlopeEstim
(
            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            input  wire               AGCClk,          // Clock
            input  wire               nAGCRst,         // Reset

            input  wire               RadSWResetP,     // Radar SW reset
            input  wire               SlopeEstimEn,    // Slope estimation enable

            input  wire        [7:0]  PhiIndex,        // Phase index

            input  wire signed [17:0] PhiUnwrap,       // Phase unwrapped

            input  wire        [2:0]  PulseProcConf,   // CP parameter
            input  wire        [7:0]  NParam,          // N parameter
            input  wire        [9:0]  D1Param,         // D1 parameter
            input  wire        [9:0]  D2Param,         // D2 parameter
    
            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output reg  signed [10:0] OmegaEst,        // Omega estimate
            output reg  signed [18:0] PsiEst           // Psi estimate
            );

//////////////////////////////////////////////////////////////////////////////
// Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire signed [25:0] PhiUnwrapMult;
wire signed [39:0] dot1AccuMultSel;
wire signed [35:0] dot2AccuMultSel;
wire signed [24:0] dot1AccuMultRnd;
wire signed [22:0] dot2AccuMultRnd;
wire signed [17:0] coefMult;
wire signed [10:0] OmegaEstInt;
wire signed [18:0] PsiEstInt;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg signed [29:0] dot1Accu;
reg signed [25:0] dot2Accu;
reg signed [39:0] dot1AccuMult;
reg signed [35:0] dot2AccuMult;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign coefMult      = $signed({10'b0,PhiIndex}) - $signed({10'b0,NParam});
assign PhiUnwrapMult = $signed(PhiUnwrap) * $signed(coefMult);

// Dot1 accumulation of the 2N+1
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    dot1Accu <= 30'b0;
  else if (!SlopeEstimEn || RadSWResetP)
    dot1Accu <= 30'b0;
  else
    dot1Accu <= dot1Accu + {{4{PhiUnwrapMult[25]}},PhiUnwrapMult};
end

// Dot2 accumulation of the 2N+1
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    dot2Accu <= 26'b0;
  else if (!SlopeEstimEn || RadSWResetP)
    dot2Accu <= 26'b0;
  else
    dot2Accu <= dot2Accu + {{8{PhiUnwrap[17]}},PhiUnwrap};
end

// Register out of accumulation multiply by D1 parameter
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    dot1AccuMult <= 40'b0;
  else if (RadSWResetP)
    dot1AccuMult <= 40'b0;
  else if (SlopeEstimEn)
    dot1AccuMult <= $signed(dot1Accu) * $signed({20'b0,D1Param});
end

// Register out of accumulation multiply by D2 parameter
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    dot2AccuMult <= 36'b0;
  else if (RadSWResetP)
    dot2AccuMult <= 36'b0;
  else if (SlopeEstimEn)
    dot2AccuMult <= $signed(dot2Accu) * $signed({16'b0,D2Param});
end

// Rounding select from CP parameter
assign dot1AccuMultSel = (PulseProcConf == 3'd0) ? dot1AccuMult :
                         (PulseProcConf == 3'd1) ? {{ 3{dot1AccuMult[39]}},dot1AccuMult[39:3]} :
                         (PulseProcConf == 3'd2) ? {{ 6{dot1AccuMult[39]}},dot1AccuMult[39:6]} :
                         (PulseProcConf == 3'd3) ? {{ 9{dot1AccuMult[39]}},dot1AccuMult[39:9]} :
                                                   {{12{dot1AccuMult[39]}},dot1AccuMult[39:12]};

assign dot2AccuMultSel = (PulseProcConf == 3'd0) ? dot2AccuMult                               :
                         (PulseProcConf == 3'd1) ? {dot2AccuMult[35],     dot2AccuMult[35:1]} :
                         (PulseProcConf == 3'd2) ? {{2{dot2AccuMult[35]}},dot2AccuMult[35:2]} :
                         (PulseProcConf == 3'd3) ? {{3{dot2AccuMult[35]}},dot2AccuMult[35:3]} :
                                                   {{4{dot2AccuMult[35]}},dot2AccuMult[35:4]};

// Rounding
Round #(
        .INPUT_WIDTH(40),
        .OUTPUT_WIDTH(25))
  U_dot1Rnd(
        .InputData(dot1AccuMultSel),
        .RoundData(dot1AccuMultRnd)
        );

Round #(
        .INPUT_WIDTH(36),
        .OUTPUT_WIDTH(23))
  U_dot2Rnd(
        .InputData(dot2AccuMultSel),
        .RoundData(dot2AccuMultRnd)
        );

// Symetric saturation
SatSymSigned #(.INPUT_WIDTH(25),
               .OUTPUT_WIDTH(11))
  U_dot1SatSym(
               .InputData(dot1AccuMultRnd),
               .SatSymData(OmegaEstInt)
               );

SatSymSigned #(.INPUT_WIDTH(23),
               .OUTPUT_WIDTH(19))
  U_dot2SatSym(
               .InputData(dot2AccuMultRnd),
               .SatSymData(PsiEstInt)
               );

// Register OmegaEst out of SatSymSigned
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    OmegaEst <= 11'b0;
  else if (RadSWResetP)
    OmegaEst <= 11'b0;
  else
    OmegaEst <= OmegaEstInt;
end

// Register PsiEst out of SatSymSigned
always @ (posedge AGCClk or negedge nAGCRst) begin
  if (nAGCRst == 1'b0)
    PsiEst <= 19'b0;
  else if (RadSWResetP)
    PsiEst <= 19'b0;
  else
    PsiEst <= PsiEstInt;
end

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
