//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 10202 $
// $Date: 2013-09-23 09:53:07 +0200 (Mon, 23 Sep 2013) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :IIR Filter DC Offset Compensation.
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/DCIIRFilterComp/verilog/rtl/DCIIRFilterComp.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module  DCIIRFilterComp #(parameter INPUTWIDTH =14,
                          parameter BW_PARAM   =80
                         )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock & Reset.
            input    wire                                  PhyClk, //DC Clock (max of 40/80/160MHz support)
            input    wire                                  nPhyRst,
            //Enable for valid DC Estimation value.
            input    wire                                  DCEstValid,
            //Rx input with valid signal.
            input    wire                                  RxDataInValid,
            input    wire   signed  [(INPUTWIDTH-1):0]     RxDataIn,
            //Block enable.
            input    wire                                  BlockEn,
            //DC Estimation value.
            input    wire    signed    [(INPUTWIDTH-1):0]  DCEst,
            //Hold time after which DC Estimation value is hold.
            input    wire                 [6:0]            dcHoldTime,
            //Compensation Type.
            //00-0 DC Compensation
            //01-Simple DC Comp,whatever is DC value,input is compensated.
            //10-Zero DC Compensation and then simple DC COmpensation
            //without holding DC Value.
            //11-Zero DC Compensation, then simple DC Compensation and
            // then hold the value after dcHold Time.
            input    wire                 [1:0]            dcCompType,
            //Zero compensation counter value.
            input    wire                 [6:0]            dcForceZeroCnt,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Rx Output Port with valid signal.
            output reg                                     DCOutValid,
            //Neutral Scale output.
            output reg  signed    [(INPUTWIDTH-3):0]       DCOutNeutralScale
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [(INPUTWIDTH+9):0] SIG_INPUTWIDTH_P10_PARAM_0 = {INPUTWIDTH+10{1'b0}};
localparam signed [(INPUTWIDTH-1):0] SIG_INPUTWIDTH_PARAM_0     = {INPUTWIDTH{1'b0}};
localparam signed [(INPUTWIDTH-3):0] SIG_INPUTWIDTH_N2_PARAM_0  = {INPUTWIDTH-2{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////

//Input DC Estimation value is upscaled by 1024 or Reg value.
//Before filter - Upscale by 1024
//After filter  - Upscale by Reg value.
reg         signed    [(INPUTWIDTH+9):0]    DCEstBeforeFilter;

//DC Value used for compensation after rounding.
reg         signed    [(INPUTWIDTH-1):0]    DC4CompBeforeFilter;

//Rx Data Input & Valid delayed values.
reg         signed    [(INPUTWIDTH-1):0]    RxDataInD1;
reg         signed    [(INPUTWIDTH-1):0]    RxDataInD2;
reg                                         RxDataInValidD1;
reg                                         RxDataInValidD2;

//Counter for samples.
reg                   [6:0]                 Counter;
//Counter for 50ns delay.
reg                   [2:0]                 DivCount50NS;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
//Round DC value.
wire        signed    [(INPUTWIDTH-1):0]    RndDCEstBeforeFilter;

//Compensated Output
wire        signed    [ INPUTWIDTH : 0 ]    CompOutBeforeFilter;

//Saturated Compensated Output.
wire        signed    [(INPUTWIDTH-3):0]    SatCompOutputBeforeFilter;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//DC Compensation.
//Input Data - DC Value.
assign CompOutBeforeFilter = $signed({RxDataInD2[INPUTWIDTH-1],RxDataInD2}) - 
                             $signed({DC4CompBeforeFilter[INPUTWIDTH-1],DC4CompBeforeFilter});

//Input DC value is scaled by 1024 and register value.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    DCEstBeforeFilter <= SIG_INPUTWIDTH_P10_PARAM_0;
  end
  else if (!DCEstValid) begin
    DCEstBeforeFilter <= SIG_INPUTWIDTH_P10_PARAM_0;
  end
  else begin
    DCEstBeforeFilter <= $signed({DCEst,10'b0});
  end
end

//Delay of input valid and data.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    RxDataInValidD1 <= 1'b0;
    RxDataInValidD2 <= 1'b0;
    RxDataInD1      <= SIG_INPUTWIDTH_PARAM_0;
    RxDataInD2      <= SIG_INPUTWIDTH_PARAM_0;
  end
  else begin
    if (!BlockEn) begin
      RxDataInValidD1 <= 1'b0;
      RxDataInValidD2 <= 1'b0;
      RxDataInD1      <= SIG_INPUTWIDTH_PARAM_0;
      RxDataInD2      <= SIG_INPUTWIDTH_PARAM_0;
    end
    else begin
      RxDataInD1      <= RxDataIn;
      RxDataInD2      <= RxDataInD1;
      RxDataInValidD1 <= RxDataInValid;
      RxDataInValidD2 <= RxDataInValidD1;
    end
  end
end

//Registering valid output and saturated outputs.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    DCOutValid        <= 1'b0;
    DCOutNeutralScale <= SIG_INPUTWIDTH_N2_PARAM_0;
  end
  else begin
    if (!BlockEn) begin
      DCOutValid        <= 1'b0;
      DCOutNeutralScale <= SIG_INPUTWIDTH_N2_PARAM_0;
    end
    else begin
      DCOutValid        <= RxDataInValidD2;
      if (RxDataInValidD2) begin
        DCOutNeutralScale <= SatCompOutputBeforeFilter;
      end
    end
  end
end

//Div Counter for 50 ns delay
generate
if (BW_PARAM == 80) begin
   //Clk = 160 MHz
   always @ (posedge PhyClk or negedge nPhyRst)
      begin: DivCount50NS_Blk
         if (nPhyRst == 1'b0)
            DivCount50NS <= 3'h0;
         else if (!BlockEn)
            DivCount50NS <= 3'h0;
         else if (DivCount50NS == 3'h7)
            DivCount50NS <= 3'h0;
         else
            DivCount50NS <= DivCount50NS + 3'h1;
      end //DivCount50NS_Blk
end
else if (BW_PARAM == 40) begin
   //Clk = 80 MHz
   always @ (posedge PhyClk or negedge nPhyRst)
      begin: DivCount50NS_Blk
         if (nPhyRst == 1'b0)
            DivCount50NS <= 3'h0;
         else if (!BlockEn)
            DivCount50NS <= 3'h0;
         else if (DivCount50NS == 3'h3)
            DivCount50NS <= 3'h0;
         else
            DivCount50NS <= DivCount50NS + 3'h1;
      end //DivCount50NS_Blk
end
else begin
   //Clk = 40 MHz
   always @ (posedge PhyClk or negedge nPhyRst)
      begin: DivCount50NS_Blk
         if (nPhyRst == 1'b0)
            DivCount50NS <= 3'h0;
         else if (!BlockEn)
            DivCount50NS <= 3'h0;
         else
            DivCount50NS[0] <= ~DivCount50NS[0];
      end //DivCount50NS_Blk
end
endgenerate

//Counter increment depending on DC Comp type.
//When no dc hold counter is incremented till dcForceZeroCnt
//otherwise till dcHoldTime+1.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    Counter <= 7'd0;
  else begin
    if (!BlockEn)
      Counter <= 7'd0;
    else begin
      if (DCEstValid && DivCount50NS == 3'h0)  begin
        if ((Counter <= (dcHoldTime+7'd1) && dcCompType == 2'd3) ||
            (Counter < dcForceZeroCnt && dcCompType == 2'd2))
            Counter <= Counter + 7'd1;
        else
            Counter <= Counter;
      end
    end
  end
end

//Depending on dcCompType mode is selected.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
     DC4CompBeforeFilter <= SIG_INPUTWIDTH_PARAM_0;
  end
  else if (!BlockEn) begin
     DC4CompBeforeFilter <= SIG_INPUTWIDTH_PARAM_0;
  end
  else if (RxDataInValidD1) begin
    case (dcCompType)
      //Mode 00 : Zero DC Compensation.
      2'b00: begin
        DC4CompBeforeFilter <= SIG_INPUTWIDTH_PARAM_0;
      end
      //Mode 01 : Simple DC Compensation.
      2'b01: begin
        DC4CompBeforeFilter <= RndDCEstBeforeFilter;
      end
      //Mode 10 : Counter < dcForceZeroCnt:Zero Compenstion
      //          else  Simple DC Compensation.
      2'b10: begin
         if (Counter < dcForceZeroCnt)  begin
           DC4CompBeforeFilter <= SIG_INPUTWIDTH_PARAM_0;
         end
         else begin
           DC4CompBeforeFilter <= RndDCEstBeforeFilter;
         end
      end
      //Mode 11:  Counter < dcForceZeroCnt:Zero Compenstion
      //          Counter < (dcHoldTime+1):Simple Compensation
      //          else Hold the last DC Value for compensation.
      default: begin
        if (Counter <= dcForceZeroCnt)  begin
          DC4CompBeforeFilter <= SIG_INPUTWIDTH_PARAM_0;
        end
        else if (Counter <= (dcHoldTime+7'd1)) begin
          DC4CompBeforeFilter <= RndDCEstBeforeFilter;
        end
        else begin
          DC4CompBeforeFilter <= DC4CompBeforeFilter;
        end
      end
    endcase
  end
end

//Round by 10 bits.
Round #(
            .INPUT_WIDTH(INPUTWIDTH +10),
            .OUTPUT_WIDTH(INPUTWIDTH)
           )
            U1_ROUND(
                        .InputData(DCEstBeforeFilter),
                        .RoundData(RndDCEstBeforeFilter)
                       );

//Instantiation of Saturation component
SatSymSigned #(
               .INPUT_WIDTH(INPUTWIDTH +1),
               .OUTPUT_WIDTH(INPUTWIDTH-2)
              )
               U1_SAT(
                        .InputData(CompOutBeforeFilter),
                        .SatSymData(SatCompOutputBeforeFilter)
                        );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
