//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: rblanc $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 14670 $
// $Date: 2014-04-23 18:45:38 +0200 (Wed, 23 Apr 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : DC IIR Filter offset Estimation.                                                  
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/DCIIRFilterEst/verilog/rtl/DCIIRFilterEst.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module  DCIIRFilterEst #(parameter INPUTWIDTH = 12 //Input Data Width
                        )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Input Ports
            //Clock & Reset.
            input    wire                                  PhyClk,
            input    wire                                  nPhyRst,

            // Data and Control
            //Enable input for valid data in.
            input    wire                                  EnableIn,
            //Rx input.
            input    wire  signed [(INPUTWIDTH-1):0]       RxDataIn,
            input    wire         [1:0]                    Mode,
            //Register to read index of M Table.
            input    wire         [6:0]                    DelStableDC,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Rx Output Port with valid signal.
            output  reg                                    DCEstValid,
            output  reg signed    [(INPUTWIDTH-1):0]       DCEst
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [(INPUTWIDTH -1):0] SIG_INPUTWIDTH_PARAM_0    = {INPUTWIDTH{1'b0}};
localparam    signed   [(INPUTWIDTH +6):0] SIG_INPUTWIDTH_P7_PARAM_0 = {INPUTWIDTH+7{1'b0}};
localparam    signed   [(INPUTWIDTH +7):0] SIG_INPUTWIDTH_P8_PARAM_0 = {INPUTWIDTH+8{1'b0}};


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
//Delay of Enable signal.
reg                                         EnableD;
//Counter for input samples.
reg                   [7:0]                 CountSample;
//Index for M array.
reg                   [2:0]                 Index;
reg                   [2:0]                 IndexD;
//Register value of IIR output.
reg         signed    [(INPUTWIDTH +7):0]   IIROutReg;
//Rx Input first stage output.
reg         signed    [(INPUTWIDTH +7):0]   ShftRxData;
//saturated IIR output.
reg         signed    [(INPUTWIDTH +6):0]   SatIIROutD;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire        signed    [(INPUTWIDTH +6):0]   SatIIROut;
//Rounded IIR Output.
wire        signed    [(INPUTWIDTH -1):0]   RndIIROut;
//IIR Output.
wire        signed    [(INPUTWIDTH +7):0]   IIROut;
//M Table.
wire                  [2:0]                 MTable[3:0][5:0];
wire                  [2:0]                 MTableShift;
//Temp for shifted first stage output.
wire        signed    [INPUTWIDTH :0]       Temp;
wire        signed    [(INPUTWIDTH +7):0]   TempShifted;



//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//MTable
assign  MTable[0][0] = 3'd7;
assign  MTable[0][1] = 3'd6;
assign  MTable[0][2] = 3'd5;
assign  MTable[0][3] = 3'd4;
assign  MTable[0][4] = 3'd3;
assign  MTable[0][5] = 3'd2;
assign  MTable[1][0] = 3'd6;
assign  MTable[1][1] = 3'd5;
assign  MTable[1][2] = 3'd4;
assign  MTable[1][3] = 3'd3;
assign  MTable[1][4] = 3'd2;
assign  MTable[1][5] = 3'd1;
assign  MTable[2][0] = 3'd5;
assign  MTable[2][1] = 3'd4;
assign  MTable[2][2] = 3'd3;
assign  MTable[2][3] = 3'd2;
assign  MTable[2][4] = 3'd1;
assign  MTable[2][5] = 3'd1;
assign  MTable[3][0] = 3'd4;
assign  MTable[3][1] = 3'd3;
assign  MTable[3][2] = 3'd2;
assign  MTable[3][3] = 3'd1;
assign  MTable[3][4] = 3'd1;
assign  MTable[3][5] = 3'd1;


//STEP1:First stage output = Rx Data In shifted by 7 bits and
//subtracted from previous IIR output.
always@(posedge PhyClk or negedge nPhyRst)
begin:ShiftInputData
  if (nPhyRst == 1'b0)begin
    ShftRxData <= SIG_INPUTWIDTH_P8_PARAM_0;
  end
  else begin
    if (EnableIn)
      ShftRxData <= $signed({RxDataIn, 7'b0}) - SatIIROut;
    else
     ShftRxData <= SIG_INPUTWIDTH_P8_PARAM_0;
  end
end

//STEP2:Truncated first stage output.
assign Temp = $signed(ShftRxData[(INPUTWIDTH +7):7]);

//STEP3:Previous IIR O/P + Shifted first stage output.
assign MTableShift = MTable[Mode][IndexD];
assign TempShifted = Temp <<< MTableShift;
assign IIROut = EnableD ? (TempShifted + $signed({SatIIROutD[INPUTWIDTH+6],SatIIROutD})) : IIROutReg;

//STEP4:Registering final output.
always@(posedge PhyClk or negedge nPhyRst)
begin:DCOutput
  if (nPhyRst == 1'b0)begin
    DCEst      <= SIG_INPUTWIDTH_PARAM_0;
    DCEstValid <= 1'b0;
  end
  else begin
    DCEst      <= RndIIROut;
    DCEstValid <= EnableD;
  end
end


//Registering intermediate outputs.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    IndexD     <= 3'h0;
    EnableD    <= 1'b0;
    SatIIROutD <= SIG_INPUTWIDTH_P7_PARAM_0;
  end
  else begin 
    SatIIROutD <= SatIIROut;
    EnableD    <= EnableIn;
    IndexD     <= Index;
  end
end

//Registering IIR Output.
always@(posedge PhyClk or negedge nPhyRst)
begin:IIROutput
  if (nPhyRst == 1'b0)
    IIROutReg <= SIG_INPUTWIDTH_P8_PARAM_0;
  else if (!EnableD)
    IIROutReg <= SIG_INPUTWIDTH_P8_PARAM_0;
  else
    IIROutReg <= IIROut;
end

//CONTROL LOGIC STARTS HERE:
//Increment counter whenever EnableIn is high 
//and index value is less then max M Table index.
always@(posedge PhyClk or negedge nPhyRst)
begin:IncrementCount
  if (nPhyRst == 1'b0)
    CountSample <= 8'd0;
  else if (!EnableIn)
    CountSample <= 8'd0;
  else if (Index < 3'd5)
    CountSample <= CountSample + 8'd1;
end

//Assignment of index for reading value from M array.
always@(posedge PhyClk or negedge nPhyRst)
begin:IncrementIndex
  if (nPhyRst == 1'b0)begin
    Index <= 3'h0;
  end
  else if (EnableIn) begin
    if ((CountSample == {1'b0,DelStableDC}+8'd1)  || (CountSample == {1'b0,DelStableDC}+8'd5)  ||
        (CountSample == {1'b0,DelStableDC}+8'd13) || (CountSample == {1'b0,DelStableDC}+8'd29) ||
        (CountSample == {1'b0,DelStableDC}+8'd61))  //Final RTL add 1 to numbers
      Index <= Index + 3'h1;
  end
  else
    Index <= 3'h0;
end


//Instantiation of Saturation component
SatSigned #(
               .INPUT_WIDTH(INPUTWIDTH+8),
               .OUTPUT_WIDTH(INPUTWIDTH+7)
              )
               U1_SAT(
                        .InputData(IIROut),
                        .SatData(SatIIROut)
                      );

//Round by 7 bits.
Round #(
            .INPUT_WIDTH(INPUTWIDTH+7),
            .OUTPUT_WIDTH(INPUTWIDTH)
           )
            U1_ROUND(
                        .InputData(SatIIROutD),
                        .RoundData(RndIIROut)
                       );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
