//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of freqshift_fs8
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module freqshift_fs8
#(
  parameter g_pipeline = 1
)
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire  rst_n,
  input  wire  clk,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire  cfsen,   // spectrum shift enable
  input  wire  cfs,     // direction 0=left, 1=right
  
  input  wire        in_valid,
  input  wire [11:0] in_i,
  input  wire [11:0] in_q,
  
  output reg         out_valid,
  output reg  [11:0] out_i,
  output reg  [11:0] out_q
);
  /*****************************************************************************
  * declaration
  *****************************************************************************/
  reg         ii_en,iq_en,qi_en,qq_en;
  reg         ii_not,iq_not,qi_not,qq_not;
  reg  [ 1:0] if_w,qf_w;
  reg  [ 2:0] phase;
  
  wire [11:0] i0,i1,q0,q1;
  wire [13:0] is,qs;
  reg  [13:0] is_1t,qs_1t;
  reg         coef_1t,valid_1t;
  wire [21:0] ix256,ix181,i;
  wire [21:0] qx256,qx181,q;

  /*****************************************************************************
  * datapath
  *****************************************************************************/
  /* gate / 1's complement input operands */
  assign i0 = (ii_en)?(in_i ^ {12{ii_not}}):12'b0;
  assign i1 = (iq_en)?(in_q ^ {12{iq_not}}):12'b0;
  assign q0 = (qi_en)?(in_i ^ {12{qi_not}}):12'b0;
  assign q1 = (qq_en)?(in_q ^ {12{qq_not}}):12'b0;
  
  /* sum operands */
  assign is = {{2{i0[11]}},i0} + {{2{i1[11]}},i1} + {12'b0,if_w};
  assign qs = {{2{q0[11]}},q0} + {{2{q1[11]}},q1} + {12'b0,qf_w};
  
  /* weighted operand sum with rounding fix at weight 2^8*/
  assign ix256 = {is_1t,8'b0}             + 
                 {14'b0,1'b1,7'b0};
 
  assign ix181 = {is_1t[13],is_1t,7'b0}      +
                 {{3{is_1t[13]}},is_1t,5'b0} +
                 {{4{is_1t[13]}},is_1t,4'b0} +
                 {{6{is_1t[13]}},is_1t,2'b0} +
                 {{8{is_1t[13]}},is_1t}      + 
                 {14'b0,1'b1,7'b0};
   
  assign qx256 = {qs_1t,8'b0}             + 
                 {14'b0,1'b1,7'b0};
                 
  assign qx181 = {qs_1t[13],qs_1t,7'b0}      +
                 {{3{qs_1t[13]}},qs_1t,5'b0} +
                 {{4{qs_1t[13]}},qs_1t,4'b0} +
                 {{6{qs_1t[13]}},qs_1t,2'b0} +
                 {{8{qs_1t[13]}},qs_1t}      + 
                 {14'b0,1'b1,7'b0};
                 
  generate
    /* no pipelining */
    if(g_pipeline==0)
    begin 
      always @(*)
      begin
        valid_1t = in_valid;
        is_1t    = is;  
        qs_1t    = qs;
        coef_1t  = ~phase[0]; 
      end  
    end
    
    /* pipelining */
    if(g_pipeline==1)
    begin
      always @(posedge clk, negedge rst_n)
      begin
        if(rst_n==1'b0)
        begin
          valid_1t <= 1'b0;
          is_1t    <= 14'b0;  
          qs_1t    <= 14'b0;
          coef_1t  <= 1'b0; 
        end
        else
        begin
          valid_1t <= in_valid;
          is_1t    <= is;          
          qs_1t    <= qs;          
          coef_1t  <= ~phase[0];   
        end
      end  
    end
  
  endgenerate

  // weigthed i/q sum + rounding fix 
  assign i = (coef_1t)?ix256:ix181;
  assign q = (coef_1t)?qx256:qx181;
 
  always @(*)
  begin:b_decoder
    reg [7:0] k;
    /*************************************************************************            
    * datapath decoder                                                                    
    **************************************************************************            
                    out_i generation                                                   
                    ================                                                   
                    in_i enable                                                        
                    |in_i not                                                          
                    ||in_q enable                                                      
                    |||in_q not                                                        
                    ||||                                                               
                    ||||out_q generation                                               
                    ||||================                                               
                    ||||in_i enable                                                    
                    |||||in_i not                                                      
                    ||||||in_q enable                                                  
                    |||||||in_q not                                                    
                    ||||||||                                                 */                                                         
    case(phase)                                                                         
      3'd0:   k={8'b10000010}; // 256 * { [+1*in_i+0*in_q] + j*[ 0*in_i+1*in_q]}     0     
      3'd1:   k={8'b10111010}; // 181 * { [+1*in_i-1*in_q] + j*[+1*in_i+1*in_q]}    45deg
      3'd2:   k={8'b00111000}; // 256 * { [ 0*in_i-1*in_q] + j*[+1*in_i+0*in_q]}    90deg
      3'd3:   k={8'b11111011}; // 181 * { [-1*in_i-1*in_q] + j*[+1*in_i-1*in_q]}   135deg
      3'd4:   k={8'b11000011}; // 256 * { [-1*in_i+0*in_q] + j*[ 0*in_i-1*in_q]}   180deg
      3'd5:   k={8'b11101111}; // 181 * { [-1*in_i+1*in_q] + j*[-1*in_i-1*in_q]}   225deg  
      3'd6:   k={8'b00101100}; // 256 * { [ 0*in_i+1*in_q] + j*[-1*in_i+0*in_q]}   270deg
     default: k={8'b10101110}; // 181 * { [+1*in_i+1*in_q] + j*[-1*in_i+1*in_q]}   315deg    
    endcase 
    
    ii_en  = k[7];                                         
    ii_not = k[6];                                         
    iq_en  = k[5];                                         
    iq_not = k[4];                                         
    qi_en  = k[3];                                         
    qi_not = k[2];                                         
    qq_en  = k[1];                                         
    qq_not = k[0];                                         
 
    if_w   = {k[6]&k[4],k[6]^k[4]};               
    qf_w   = {k[2]&k[0],k[2]^k[0]};               

  end

  /*****************************************************************************
  * 45deg phase increment 
  *
  *  cfs=0, shift spectrum left
  *  cfs=1, shift spectrum right
  *
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      phase           <=  3'b0;
      out_i           <= 12'b0;
      out_q           <= 12'b0;
      out_valid       <=  1'b0;  
    end
    else
    begin
      /*************************************************************************
      * datapath control
      *************************************************************************/
      if(cfsen)
      begin
        if(in_valid)
          phase  <=  phase + {{2{~cfs}},1'b1};
        else
          phase <= 3'b0;
          
        if(valid_1t)
        begin
          /*************************************************************************  
          * rounding saturation                                                       
          *************************************************************************/  
          out_valid <= 1'b1;                                                          
          if((i[21:19]==3'b111) || (i[21:19]==3'b000))                                  
            out_i <= i[19:8]; // rounding                                             
          else                                                                        
            out_i <= {i[21],{11{~i[21]}}}; // saturation                              
          if((q[21:19]==3'b111) || (q[21:19]==3'b000))                                  
            out_q <= q[19:8]; // rounding                                             
          else                                                                        
            out_q <= {q[21],{11{~q[21]}}}; // saturation                              
        end
        else
        begin       
          out_valid <= 1'b0;
          out_i     <= 12'b0;
          out_q     <= 12'b0;
        end
      end
      else
      begin
        out_valid <= in_valid;
        out_i     <= in_i;
        out_q     <= in_q;
      end
    end  
  end
endmodule

`default_nettype wire

