//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of hbf20
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module hbf20_lpf4040
#(
  parameter lpf4040_present=1 // enable instance of lpf4040 
)
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire  rst_n,
  input  wire  clk,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire  blocken,  // global enable
  input  wire  rxmode,   // 1=rx; 0=tx
  input  wire  cfsen,    // frequency shift enable (ignored in tx)
  input  wire  cfs,      // 0=primary is left, 1=primary is right 
  input  wire  lpf4040en,// 1=enable lpf4040 filtering, 0=disable
  
  /*****************************************************************************
  * RX
  *****************************************************************************/
  input  wire        in_rx_valid,
  input  wire [11:0] in_rx_i,
  input  wire [11:0] in_rx_q,
  input  wire [11:0] in_rx_dc_i,
  input  wire [11:0] in_rx_dc_q,
  
  output wire        out_rx_valid,
  
  output wire [12:0] out_rx_hbf20_pri_i,
  output wire [12:0] out_rx_hbf20_pri_q,
  output wire [12:0] out_rx_hbf20_sec_i,
  output wire [12:0] out_rx_hbf20_sec_q,
  output wire [12:0] out_rx_hbf20_dc_i,
  output wire [12:0] out_rx_hbf20_dc_q,
  
  output wire [12:0] out_rx_lpf4040_i,
  output wire [12:0] out_rx_lpf4040_q,
  output wire [12:0] out_rx_lpf4040_dc_i,
  output wire [12:0] out_rx_lpf4040_dc_q,

  /*****************************************************************************
  * TX
  *****************************************************************************/
  input  wire        in_tx_valid,
  input  wire [11:0] in_tx_i,
  input  wire [11:0] in_tx_q,
 
  output wire        out_tx_valid,
  output wire [11:0] out_tx_i,
  output wire [11:0] out_tx_q
);
  /*****************************************************************************
  * HBF20
  ******************************************************************************
  * impulse response :
  *
  *   hbf20=[20 21 -20 -40 16 63 -13 -109 12 333 512 333 12 -109 -13 63 16 -40 -20 21 20];
  *
  * maximum range:
  *   sum(abs(hbf20))*[-2048 2047]+512 = [ -3698176   3697394]
  *  [-2^22 2^22-1] = [-4194304   4194303] => fits 23 bits
  *****************************************************************************/
 
  /*****************************************************************************
  * S0
  *****************************************************************************/
  /* declaration */
  localparam  HBF20_0_20 = 20, HBF20_1_19 =  21, HBF20_2_18 = -20, HBF20_3_17 =  -40,
              HBF20_4_16 = 16, HBF20_5_15 =  63, HBF20_6_14 = -13, HBF20_7_13 = -109,
              HBF20_8_12 = 12, HBF20_9_11 = 333, HBF20_10   = 512;
  /* lpf4040 specific */
  localparam  LPF4040_0_20 =  -2, LPF4040_1_19 =   -2, LPF4040_2_18 =   4, LPF4040_3_17 =  12,    
              LPF4040_4_16 =   6, LPF4040_5_15 =  -20, LPF4040_6_14 = -40, LPF4040_7_13 = -10,    
              LPF4040_8_12 =  88, LPF4040_9_11 =  204, LPF4040_10   = 256;                        
  
  /* wire  */
  wire        in_valid;
  reg  [11:0] n_s0_swp_i,n_s0_swp_q,n_s0_hold_i;
  /* flops */
  reg         in_cycle;
  reg         s0_lpf4040en;
  reg         s0_rxmode;
  reg         s0_blocken;
  reg         s0_cfsen;
  reg         s0_cfs;
  reg  [20:0] s0_valid;
  reg         s0_even;
  reg         s0_odd;
  reg  [ 1:0] s0_cycle;
  reg  [11:0] s0_hold_i;
  reg  [11:0] s0_swp_q;
  reg  [11:0] s0_dc_i,s0_dc_q;
  reg  [11:0] s0_tap0_i,s0_tap1_i,s0_tap2_i,s0_tap3_i,s0_tap4_i,s0_tap5_i;
  reg  [11:0] s0_tap6_i,s0_tap7_i,s0_tap8_i,s0_tap9_i,s0_tap10_i,s0_tap11_i;
  reg  [11:0] s0_tap12_i,s0_tap13_i,s0_tap14_i,s0_tap15_i,s0_tap16_i,s0_tap17_i; 
  reg  [11:0] s0_tap18_i,s0_tap19_i,s0_tap20_i;
  reg  [11:0] s0_tap0_q,s0_tap1_q,s0_tap2_q,s0_tap3_q,s0_tap4_q,s0_tap5_q;
  reg  [11:0] s0_tap6_q,s0_tap7_q,s0_tap8_q,s0_tap9_q,s0_tap10_q,s0_tap11_q;
  reg  [11:0] s0_tap12_q,s0_tap13_q,s0_tap14_q,s0_tap15_q,s0_tap16_q,s0_tap17_q; 
  reg  [11:0] s0_tap18_q,s0_tap19_q,s0_tap20_q;
 
  /* RX i/q swapping for spectral shift, TX hold samples for interpolation */
  assign in_valid   = rxmode?in_rx_valid:in_tx_valid;
 
  always @(*)
  begin
    if(rxmode)
    begin
      n_s0_hold_i = 12'b0;
      if(!in_rx_valid)
      begin
        n_s0_swp_i = 12'b0;
        n_s0_swp_q = 12'b0;
      end
      else if(cfsen)
      begin
        if(!in_cycle)
        begin
          n_s0_swp_i = in_rx_i;
          n_s0_swp_q = in_rx_q;
        end
        else
        begin
          n_s0_swp_i = in_rx_q;
          n_s0_swp_q = in_rx_i;
        end
      end
      else
      begin
        n_s0_swp_i = in_rx_i;
        n_s0_swp_q = in_rx_q;
      end
    end
    else
    begin
      if(!in_tx_valid)
      begin
        n_s0_hold_i = 12'b0;
        n_s0_swp_i  = 12'b0;
        n_s0_swp_q  = 12'b0;
      end
      else if(!in_cycle)
      begin
        n_s0_hold_i = in_tx_i;
        n_s0_swp_i  = in_tx_i;
        n_s0_swp_q  = in_tx_q;
      end
      else
      begin
        n_s0_hold_i = s0_hold_i;
        n_s0_swp_i  = s0_hold_i;
        n_s0_swp_q  = s0_swp_q;
      end
    end
  end
  
  /* 21 tap */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      in_cycle                <= 1'b0;
      s0_lpf4040en            <= 1'b0;
      s0_rxmode               <= 1'b0;
      s0_blocken              <= 1'b0;
      s0_cfsen                <= 1'b0;
      s0_cfs                  <= 1'b0; 
      s0_cycle                <= 2'b0;
      s0_valid                <= 21'b0;
      s0_even                 <= 1'b0;
      s0_odd                  <= 1'b0;
      s0_dc_i                 <= 12'd0;
      s0_dc_q                 <= 12'd0;    
      
      s0_hold_i               <= 12'b0;
      s0_swp_q                <= 12'b0;
      { s0_tap0_i, s0_tap0_q} <= 24'b0;
      { s0_tap1_i, s0_tap1_q} <= 24'b0;
      { s0_tap2_i, s0_tap2_q} <= 24'b0;
      { s0_tap3_i, s0_tap3_q} <= 24'b0;
      { s0_tap4_i, s0_tap4_q} <= 24'b0;
      { s0_tap5_i, s0_tap5_q} <= 24'b0;
      { s0_tap6_i, s0_tap6_q} <= 24'b0;
      { s0_tap7_i, s0_tap7_q} <= 24'b0;
      { s0_tap8_i, s0_tap8_q} <= 24'b0;
      { s0_tap9_i, s0_tap9_q} <= 24'b0;
      {s0_tap10_i,s0_tap10_q} <= 24'b0;
      {s0_tap11_i,s0_tap11_q} <= 24'b0;
      {s0_tap12_i,s0_tap12_q} <= 24'b0;
      {s0_tap13_i,s0_tap13_q} <= 24'b0;
      {s0_tap14_i,s0_tap14_q} <= 24'b0;
      {s0_tap15_i,s0_tap15_q} <= 24'b0;
      {s0_tap16_i,s0_tap16_q} <= 24'b0;
      {s0_tap17_i,s0_tap17_q} <= 24'b0;
      {s0_tap18_i,s0_tap18_q} <= 24'b0;
      {s0_tap19_i,s0_tap19_q} <= 24'b0;
      {s0_tap20_i,s0_tap20_q} <= 24'b0;
    end
    else
    begin
      if(blocken)
      begin
        /* param */
        s0_lpf4040en <= lpf4040en;
        s0_rxmode    <= rxmode;
        s0_blocken   <= 1'b1;
        s0_cfsen     <= cfsen & rxmode;
        s0_cfs       <= cfs; 

        /* valid tag */                                    
        if(in_valid)
          in_cycle <= ~in_cycle;
        else
          in_cycle <= 1'b0;
       
        s0_valid   <= {s0_valid[19:0],in_valid};
     
        /* cycle tag */
        if(s0_valid!=21'd0)
        begin
          s0_cycle <= s0_cycle +  2'd1;                       
          if(rxmode)                                               
          begin
            s0_even <= ~s0_cycle[0];
            s0_odd  <= ~s0_cycle[0];
          end
          else
          begin
            s0_even <= ~s0_cycle[0];
            s0_odd  <=  s0_cycle[0];
          end
        end
        else
        begin
          s0_cycle <= 2'd0;                             
          s0_even  <= 1'b0;                                  
          s0_odd   <= 1'b0;
        end
      
        s0_swp_q  <= n_s0_swp_q;
        s0_hold_i <= n_s0_hold_i;
        { s0_tap0_i, s0_tap0_q} <= {n_s0_swp_i, s0_swp_q}; 
      
        if(rxmode)
        begin
          if(in_cycle)
          begin
            s0_dc_i <= in_rx_dc_i;
            s0_dc_q <= in_rx_dc_q;                 
          end
        end
        else
        begin
          s0_dc_i <= 12'd0;
          s0_dc_q <= 12'd0;                 
        end
      end
      else
      begin
        in_cycle                <= 1'b0;
        s0_rxmode               <= 1'b0;
        s0_lpf4040en            <= 1'b0;
        s0_blocken              <= 1'b0;
        s0_cfsen                <= 1'b0;
        s0_cfs                  <= 1'b0; 
        s0_cycle                <= 2'b0;
        s0_valid                <= 21'b0;
        s0_even                 <= 1'b0;
        s0_odd                  <= 1'b0;

        s0_hold_i               <= 12'b0;
        s0_swp_q                <= 12'b0;
        s0_dc_i                 <= 12'd0;
        s0_dc_q                 <= 12'd0;    
        { s0_tap0_i, s0_tap0_q} <= 24'b0;
      end                
      
      if(s0_blocken)
      begin
        { s0_tap1_i, s0_tap1_q} <= { s0_tap0_i, s0_tap0_q};            
        { s0_tap2_i, s0_tap2_q} <= { s0_tap1_i, s0_tap1_q};            
        { s0_tap3_i, s0_tap3_q} <= { s0_tap2_i, s0_tap2_q};            
        { s0_tap4_i, s0_tap4_q} <= { s0_tap3_i, s0_tap3_q};            
        { s0_tap5_i, s0_tap5_q} <= { s0_tap4_i, s0_tap4_q};            
        { s0_tap6_i, s0_tap6_q} <= { s0_tap5_i, s0_tap5_q};            
        { s0_tap7_i, s0_tap7_q} <= { s0_tap6_i, s0_tap6_q};            
        { s0_tap8_i, s0_tap8_q} <= { s0_tap7_i, s0_tap7_q};            
        { s0_tap9_i, s0_tap9_q} <= { s0_tap8_i, s0_tap8_q};            
        {s0_tap10_i,s0_tap10_q} <= { s0_tap9_i, s0_tap9_q};            
        {s0_tap11_i,s0_tap11_q} <= {s0_tap10_i,s0_tap10_q};            
        {s0_tap12_i,s0_tap12_q} <= {s0_tap11_i,s0_tap11_q};            
        {s0_tap13_i,s0_tap13_q} <= {s0_tap12_i,s0_tap12_q};            
        {s0_tap14_i,s0_tap14_q} <= {s0_tap13_i,s0_tap13_q};            
        {s0_tap15_i,s0_tap15_q} <= {s0_tap14_i,s0_tap14_q};            
        {s0_tap16_i,s0_tap16_q} <= {s0_tap15_i,s0_tap15_q};            
        {s0_tap17_i,s0_tap17_q} <= {s0_tap16_i,s0_tap16_q};            
        {s0_tap18_i,s0_tap18_q} <= {s0_tap17_i,s0_tap17_q};            
        {s0_tap19_i,s0_tap19_q} <= {s0_tap18_i,s0_tap18_q};            
        {s0_tap20_i,s0_tap20_q} <= {s0_tap19_i,s0_tap19_q};
      end
      else
      begin
        { s0_tap1_i, s0_tap1_q} <= 24'b0;
        { s0_tap2_i, s0_tap2_q} <= 24'b0;
        { s0_tap3_i, s0_tap3_q} <= 24'b0;
        { s0_tap4_i, s0_tap4_q} <= 24'b0;
        { s0_tap5_i, s0_tap5_q} <= 24'b0;
        { s0_tap6_i, s0_tap6_q} <= 24'b0;
        { s0_tap7_i, s0_tap7_q} <= 24'b0;
        { s0_tap8_i, s0_tap8_q} <= 24'b0;
        { s0_tap9_i, s0_tap9_q} <= 24'b0;
        {s0_tap10_i,s0_tap10_q} <= 24'b0;
        {s0_tap11_i,s0_tap11_q} <= 24'b0;
        {s0_tap12_i,s0_tap12_q} <= 24'b0;
        {s0_tap13_i,s0_tap13_q} <= 24'b0;
        {s0_tap14_i,s0_tap14_q} <= 24'b0;
        {s0_tap15_i,s0_tap15_q} <= 24'b0;
        {s0_tap16_i,s0_tap16_q} <= 24'b0;
        {s0_tap17_i,s0_tap17_q} <= 24'b0;
        {s0_tap18_i,s0_tap18_q} <= 24'b0;
        {s0_tap19_i,s0_tap19_q} <= 24'b0;
        {s0_tap20_i,s0_tap20_q} <= 24'b0;
      end            
    end
  end
      
  /*****************************************************************************
  * S1
  *****************************************************************************/
  /* wire */
  reg  [ 3:0] n_s1_primary,n_s1_secondary;
  wire [11:0] n_s1_mux0 ,n_s1_mux1 ,n_s1_mux2 ,n_s1_mux3,n_s1_mux4,n_s1_mux5;
  wire [11:0] n_s1_mux6 ,n_s1_mux7 ,n_s1_mux8 ,n_s1_mux9,n_s1_mux10,n_s1_mux11;
  wire [11:0] n_s1_mux12,n_s1_mux13,n_s1_mux14,n_s1_mux15,n_s1_mux16,n_s1_mux17;
  wire [11:0] n_s1_mux18,n_s1_mux19,n_s1_mux20;
  wire [12:0] n_s1_sum0_20,n_s1_sum1_19,n_s1_sum2_18,n_s1_sum3_17,n_s1_sum4_16;
  wire [12:0] n_s1_sum5_15,n_s1_sum6_14,n_s1_sum7_13,n_s1_sum8_12,n_s1_sum9_11;
  wire [12:0] n_s1_sum10;   
  wire [21:0] n_s1_mul0_20,n_s1_mul1_19,n_s1_mul2_18,n_s1_mul3_17,n_s1_mul4_16;
  wire [21:0] n_s1_mul5_15,n_s1_mul6_14,n_s1_mul7_13,n_s1_mul8_12,n_s1_mul9_11;
  wire [21:0] n_s1_mul10;   
  wire [22:0] n_s1_sum0,n_s1_sum1,n_s1_sum2,n_s1_sum3;
  /* flops */
  reg         s1_rxmode;
  reg         s1_cfsen;
  reg         s1_valid;
  reg         s1_lpf4040en;
  reg         s1_cycle;
  reg  [ 3:0] s1_primary,s1_secondary;
  reg  [22:0] s1_sum0,s1_sum1,s1_sum2,s1_sum3;

  /* i/q arithmetic ressource sharing (12 bits [-2048,+2047]) */
  assign n_s1_mux0     = s0_even ?  s0_tap0_q :  s0_tap0_i;
  assign n_s1_mux2     = s0_even ?  s0_tap2_q :  s0_tap2_i;
  assign n_s1_mux4     = s0_even ?  s0_tap4_q :  s0_tap4_i;
  assign n_s1_mux6     = s0_even ?  s0_tap6_q :  s0_tap6_i;
  assign n_s1_mux8     = s0_even ?  s0_tap8_q :  s0_tap8_i;
  assign n_s1_mux10    = s0_even ? s0_tap10_q : s0_tap10_i;
  assign n_s1_mux12    = s0_even ? s0_tap12_q : s0_tap12_i;
  assign n_s1_mux14    = s0_even ? s0_tap14_q : s0_tap14_i;
  assign n_s1_mux16    = s0_even ? s0_tap16_q : s0_tap16_i;
  assign n_s1_mux18    = s0_even ? s0_tap18_q : s0_tap18_i;
  assign n_s1_mux20    = s0_even ? s0_tap20_q : s0_tap20_i;
   
  assign n_s1_mux1     =  s0_odd ?  s0_tap1_q :  s0_tap1_i;
  assign n_s1_mux3     =  s0_odd ?  s0_tap3_q :  s0_tap3_i;
  assign n_s1_mux5     =  s0_odd ?  s0_tap5_q :  s0_tap5_i;
  assign n_s1_mux7     =  s0_odd ?  s0_tap7_q :  s0_tap7_i;
  assign n_s1_mux9     =  s0_odd ?  s0_tap9_q :  s0_tap9_i;
  assign n_s1_mux11    =  s0_odd ? s0_tap11_q : s0_tap11_i;
  assign n_s1_mux13    =  s0_odd ? s0_tap13_q : s0_tap13_i;
  assign n_s1_mux15    =  s0_odd ? s0_tap15_q : s0_tap15_i;
  assign n_s1_mux17    =  s0_odd ? s0_tap17_q : s0_tap17_i;
  assign n_s1_mux19    =  s0_odd ? s0_tap19_q : s0_tap19_i;

  /* factoring coefficients by index parity (13 bits [-4096,+4095]) */
  assign n_s1_sum0_20 = { n_s1_mux0[11], n_s1_mux0} +  {n_s1_mux20[11],n_s1_mux20};
  assign n_s1_sum2_18 = { n_s1_mux2[11], n_s1_mux2} +  {n_s1_mux18[11],n_s1_mux18};
  assign n_s1_sum4_16 = { n_s1_mux4[11], n_s1_mux4} +  {n_s1_mux16[11],n_s1_mux16};
  assign n_s1_sum6_14 = { n_s1_mux6[11], n_s1_mux6} +  {n_s1_mux14[11],n_s1_mux14};
  assign n_s1_sum8_12 = { n_s1_mux8[11], n_s1_mux8} +  {n_s1_mux12[11],n_s1_mux12};
  assign n_s1_sum10   = {n_s1_mux10[11],n_s1_mux10};
  
  assign n_s1_sum1_19 = { n_s1_mux1[11], n_s1_mux1} + ({n_s1_mux19[11],n_s1_mux19}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum3_17 = { n_s1_mux3[11], n_s1_mux3} + ({n_s1_mux17[11],n_s1_mux17}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum5_15 = { n_s1_mux5[11], n_s1_mux5} + ({n_s1_mux15[11],n_s1_mux15}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum7_13 = { n_s1_mux7[11], n_s1_mux7} + ({n_s1_mux13[11],n_s1_mux13}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum9_11 = { n_s1_mux9[11], n_s1_mux9} + ({n_s1_mux11[11],n_s1_mux11}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
                                                     
  /* coefficient weighting 
  *  max values are -4096*512 and +4095*512 => [-2097152,2096640]
  *  which fit a 22 bits range ([-2097152   2097151])
  */
  assign n_s1_mul0_20  =  $signed(HBF20_0_20) * $signed(n_s1_sum0_20);
  assign n_s1_mul1_19  =  $signed(HBF20_1_19) * $signed(n_s1_sum1_19);
  assign n_s1_mul2_18  =  $signed(HBF20_2_18) * $signed(n_s1_sum2_18);
  assign n_s1_mul3_17  =  $signed(HBF20_3_17) * $signed(n_s1_sum3_17);
  assign n_s1_mul4_16  =  $signed(HBF20_4_16) * $signed(n_s1_sum4_16);
  assign n_s1_mul5_15  =  $signed(HBF20_5_15) * $signed(n_s1_sum5_15);
  assign n_s1_mul6_14  =  $signed(HBF20_6_14) * $signed(n_s1_sum6_14);
  assign n_s1_mul7_13  =  $signed(HBF20_7_13) * $signed(n_s1_sum7_13);
  assign n_s1_mul8_12  =  $signed(HBF20_8_12) * $signed(n_s1_sum8_12);
  assign n_s1_mul9_11  =  $signed(HBF20_9_11) * $signed(n_s1_sum9_11);
  assign n_s1_mul10    =  $signed(  HBF20_10) * $signed(  n_s1_sum10);

  /* sum all weighted samples which have the same modulo-4 index */
  assign n_s1_sum0 = {n_s1_mul0_20[21], n_s1_mul0_20} +
                     {n_s1_mul4_16[21], n_s1_mul4_16} +
                     {n_s1_mul8_12[21], n_s1_mul8_12};

  assign n_s1_sum1 = {n_s1_mul1_19[21], n_s1_mul1_19} +
                     {n_s1_mul5_15[21], n_s1_mul5_15} +
                     {n_s1_mul9_11[21], n_s1_mul9_11};

  assign n_s1_sum2 = {n_s1_mul2_18[21], n_s1_mul2_18} +
                     {n_s1_mul6_14[21], n_s1_mul6_14} +
                     {  n_s1_mul10[21],   n_s1_mul10};

  assign n_s1_sum3 = {n_s1_mul3_17[21], n_s1_mul3_17} +
                     {n_s1_mul7_13[21], n_s1_mul7_13};
  
  always @(*)
  begin
    if(s0_rxmode && s0_cfsen)
    begin
      if(!s0_cfs)
      begin
        /* shift right to center primary */
        case(s0_cycle)
          2'd0:   {n_s1_primary,n_s1_secondary} = {4'b1100,4'b0110};
          2'd1:   {n_s1_primary,n_s1_secondary} = {4'b0110,4'b1100}; 
          2'd2:   {n_s1_primary,n_s1_secondary} = {4'b0011,4'b1001}; 
          default:{n_s1_primary,n_s1_secondary} = {4'b1001,4'b0011};
        endcase
      end
      else
      begin
        /* shift left to center primary */
        case(s0_cycle)
          2'd0:   {n_s1_primary,n_s1_secondary} = {4'b0110,4'b1100};
          2'd1:   {n_s1_primary,n_s1_secondary} = {4'b1100,4'b0110}; 
          2'd2:   {n_s1_primary,n_s1_secondary} = {4'b1001,4'b0011}; 
          default:{n_s1_primary,n_s1_secondary} = {4'b0011,4'b1001};
        endcase
      end
    end
    else
    begin                
      n_s1_primary   = 4'b0000;
      n_s1_secondary = 4'b0000;
    end  
  end

  /* scale DC 
  * [-2048 2047]*1078 + 512 = [-2207232   2207178]
  * [-2^22 2^22-1] = [ -4194304   4194303 ]   => 23 bits range, no saturation required
  */
  wire [11:0] n_s1_dc;
  wire [22:0] n_s1_dc_scaled;
  wire [22:0] n_s1_dc_fix;
  reg  [12:0] s1_dc_i;
  assign n_s1_dc        = s0_cycle[0]?s0_dc_i:s0_dc_q;
  assign n_s1_dc_scaled = $signed(12'd1078) * $signed(n_s1_dc);
  assign n_s1_dc_fix    = n_s1_dc_scaled + 23'd512;

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_lpf4040en <= 1'b0;   
      s1_rxmode    <= 1'b0;   
      s1_cfsen     <= 1'b0;   
      s1_cycle     <= 1'b0;   
      s1_valid     <= 1'b0;   
      s1_primary   <= 4'b0;    
      s1_secondary <= 4'b0;    
      s1_sum0      <= 23'b0;   
      s1_sum1      <= 23'b0;   
      s1_sum2      <= 23'b0;   
      s1_sum3      <= 23'b0;   
      s1_dc_i      <= 13'd0;  
    end
    else if(!s0_blocken)
    begin
      s1_lpf4040en <= 1'b0;   
      s1_rxmode    <= 1'b0;    
      s1_cfsen     <= 1'b0;    
      s1_cycle     <= 1'b0;    
      s1_valid     <= 1'b0;    
      s1_primary   <= 4'b0;    
      s1_secondary <= 4'b0;    
      s1_sum0      <= 23'b0;   
      s1_sum1      <= 23'b0;   
      s1_sum2      <= 23'b0;   
      s1_sum3      <= 23'b0;   
      s1_dc_i      <= 13'd0;  
    end
    else
    begin
      s1_lpf4040en <= s0_lpf4040en;   
      s1_rxmode    <= s0_rxmode;      
      s1_cfsen     <= s0_cfsen;       
      s1_cycle     <= s0_cycle[0];       
      s1_valid     <= s0_valid!=21'd0;      
      s1_primary   <= n_s1_primary;       
      s1_secondary <= n_s1_secondary;       
      s1_sum0      <= n_s1_sum0;      
      s1_sum1      <= n_s1_sum1;      
      s1_sum2      <= n_s1_sum2;      
      s1_sum3      <= n_s1_sum3;      
      if(s0_cycle[0])
        s1_dc_i <= n_s1_dc_fix[22:10];
    end
  end
 
  /*****************************************************************************
  * S2
  *****************************************************************************/
  /* wires */
  wire [22:0] n_s2_sum_pri,n_s2_sum_sec;
  wire [12:0] n_s2_sum_pri_rnd,n_s2_sum_sec_rnd;
  wire [22:0] n_s2_sum_even,n_s2_sum_odd;
  wire [13:0] n_s2_sum_even_trc,n_s2_sum_odd_trc;
  reg  [11:0] n_s2_sum_even_sat,n_s2_sum_odd_sat;
  /* flops */
  reg  [12:0] s2_hold_pri_i,s2_hold_sec_i;
  reg  [12:0] s2_pri_i,s2_pri_q;
  reg  [12:0] s2_sec_i,s2_sec_q;
  reg  [ 1:0] s2_rx_valid;
  reg         s2_tx_valid;
  reg  [ 1:0] s2_hold_valid;
  reg  [11:0] s2_tx_i,s2_tx_q;
  reg  [11:0] s2_hold_0;
  reg  [12:0] s2_dc_i,s2_dc_q;
  
  /* RX */
  assign n_s2_sum_pri = 
    (s1_sum0 ^ {23{s1_primary[0]}})  +
    (s1_sum1 ^ {23{s1_primary[1]}})  +                                      
    (s1_sum2 ^ {23{s1_primary[2]}})  +                                      
    (s1_sum3 ^ {23{s1_primary[3]}})  +                                      
    {13'b0,1'b1,7'b0,s1_cfsen,1'b0}; // rndfix + c2fix                      

  assign n_s2_sum_sec = 
    (s1_sum0 ^ {23{s1_secondary[0]}}) +
    (s1_sum1 ^ {23{s1_secondary[1]}}) +                            
    (s1_sum2 ^ {23{s1_secondary[2]}}) +                            
    (s1_sum3 ^ {23{s1_secondary[3]}}) +                            
    {13'b0,1'b1,7'b0,s1_cfsen,1'b0}; // rndfix + c2fix                    

  /* rnd 10 */
  assign n_s2_sum_pri_rnd = n_s2_sum_pri[22:10];
  assign n_s2_sum_sec_rnd = n_s2_sum_sec[22:10]; 
  
  /* TX */
  assign n_s2_sum_even     = s1_sum0 + s1_sum2;                      
  assign n_s2_sum_odd      = s1_sum1 + s1_sum3;      

  /* trunc 9 */
  assign n_s2_sum_even_trc = n_s2_sum_even[22:9];
  assign n_s2_sum_odd_trc  = n_s2_sum_odd[22:9];
  
  /* sat 12 */
  always @(*)
  begin
    if((n_s2_sum_even_trc[13:11]==3'b000) || (n_s2_sum_even_trc[13:11]==3'b111))
       n_s2_sum_even_sat = n_s2_sum_even_trc[11:0];
     else
       n_s2_sum_even_sat = {n_s2_sum_even_trc[13],{11{~n_s2_sum_even_trc[13]}}};
  end  
  
  always @(*)
  begin
    if((n_s2_sum_odd_trc[13:11]==3'b000) || (n_s2_sum_odd_trc[13:11]==3'b111))
       n_s2_sum_odd_sat = n_s2_sum_odd_trc[11:0];
     else
       n_s2_sum_odd_sat = {n_s2_sum_odd_trc[13],{11{~n_s2_sum_odd_trc[13]}}};
  end
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_dc_i        <= 13'b0;
      s2_dc_q        <= 13'b0;
      s2_hold_pri_i  <= 13'b0;
      s2_hold_sec_i  <= 13'b0;
      s2_hold_valid  <= 2'b0;
      s2_pri_i       <= 13'b0;
      s2_pri_q       <= 13'b0;
      s2_sec_i       <= 13'b0;
      s2_sec_q       <= 13'b0;
      s2_rx_valid    <= 2'b0;
      s2_dc_i       <= 13'b0;
      s2_dc_q       <= 13'b0;
      s2_hold_0      <= 12'b0;
      s2_tx_i        <= 12'b0; 
      s2_tx_q        <= 12'b0; 
      s2_tx_valid    <= 1'b0; 
    end                    
    else if(!blocken)
    begin
      s2_dc_i        <= 13'b0;
      s2_dc_q        <= 13'b0;
      s2_hold_pri_i  <= 13'b0;
      s2_hold_sec_i  <= 13'b0;
      s2_hold_valid  <= 2'b0;
      s2_pri_i       <= 13'b0;
      s2_pri_q       <= 13'b0;
      s2_sec_i       <= 13'b0;
      s2_sec_q       <= 13'b0;
      s2_rx_valid    <= 2'b0;
      s2_dc_i       <= 13'b0;
      s2_dc_q       <= 13'b0;
      s2_hold_0      <= 12'b0;
      s2_tx_i        <= 12'b0; 
      s2_tx_q        <= 12'b0; 
      s2_tx_valid    <= 1'b0; 
    end
    else
    begin
      if(s1_rxmode)                                                 
      begin
        s2_tx_valid <= 1'b0;
        s2_tx_i     <= 12'b0;
        s2_tx_q     <= 12'b0;
        s2_rx_valid <= {1'b0,s2_rx_valid[1]};
        
        if(s1_valid)
        begin
          if(!s1_cycle)
          begin
            s2_hold_pri_i <= n_s2_sum_pri_rnd;
            s2_hold_sec_i <= n_s2_sum_sec_rnd;
          end
          else
          begin
            s2_rx_valid  <= 2'b11;
            s2_pri_i     <= s2_hold_pri_i;
            s2_pri_q     <= n_s2_sum_pri_rnd;
            if(s1_cfsen)
            begin
              s2_sec_i <= s2_hold_sec_i;
              s2_sec_q <= n_s2_sum_sec_rnd;
            end
            else
            begin
              s2_sec_i <= 13'b0;
              s2_sec_q <= 13'b0;
            end
          end
          
          if(s1_cycle)
          begin
            s2_dc_i     <= s1_dc_i;
            s2_dc_q     <= n_s1_dc_fix[22:10];
          end
        end
      end                                                      
      else                                                     
      begin
        s2_pri_i       <= 13'b0;  
        s2_pri_q       <= 13'b0;  
        s2_sec_i       <= 13'b0;  
        s2_sec_q       <= 13'b0;  
        s2_dc_i        <= 13'b0;
        s2_dc_q        <= 13'b0;
        s2_rx_valid    <= 2'b0;   
      
        if(s1_valid && !s1_cycle)
          s2_hold_valid <= 2'b11;
        else
          s2_hold_valid <= {1'b0,s2_hold_valid[1]};
        
        if(s2_hold_valid==2'b11)
        begin
          s2_hold_0   <= n_s2_sum_odd_sat;
          s2_tx_valid <= 1'b1;
          s2_tx_i     <= s2_hold_0;
          s2_tx_q     <= n_s2_sum_even_sat;
        end
        else if(s2_hold_valid==2'b01)
        begin
          s2_hold_0   <= n_s2_sum_even_sat;
          s2_tx_valid <= 1'b1;
          s2_tx_i     <= s2_hold_0;
          s2_tx_q     <= n_s2_sum_odd_sat;
        end  
        else
        begin
          s2_hold_0   <= n_s2_sum_even_sat;
          s2_tx_valid <= 1'b0;
          s2_tx_i     <= 12'b0;
          s2_tx_q     <= 12'b0;
        end
      end                                                      
    end
  end
  
  assign out_rx_valid       = s2_rx_valid[0];
  assign out_rx_hbf20_pri_i = s2_pri_i;
  assign out_rx_hbf20_pri_q = s2_pri_q;
  assign out_rx_hbf20_sec_i = s2_sec_i;
  assign out_rx_hbf20_sec_q = s2_sec_q;
  assign out_rx_hbf20_dc_i  = s2_dc_i;
  assign out_rx_hbf20_dc_q  = s2_dc_q;

  assign out_tx_valid       = s2_tx_valid;
  assign out_tx_i           = s2_tx_i;
  assign out_tx_q           = s2_tx_q;
  
  generate if(lpf4040_present==1)
  begin:g_lpf4040_present                                                                                                    
    /*****************************************************************************
    * LPF4040
    ******************************************************************************
    * impulse response:
    *   lpf4040 =  [-2 -2 4 12 6 -20 -40 -10 88 204 256 204 88 -10 -40 -20 6 12 4 -2 -2]
    *
    * maximum range:
    *  sum(abs(lpf4040))*[-2048 2047] +256 = [-2113280   2112760]
    * [-2^22 2^22-1] = [ -4194304   4194303] which fits 23 bits
    *
    *****************************************************************************/

    /***************************************************************************
    * S0
    ***************************************************************************/
    /* wires */
    wire [11:0] s0_tapg0_i,s0_tapg1_i,s0_tapg2_i,s0_tapg3_i,s0_tapg4_i,s0_tapg5_i;
    wire [11:0] s0_tapg6_i,s0_tapg7_i,s0_tapg8_i,s0_tapg9_i,s0_tapg10_i,s0_tapg11_i;                 
    wire [11:0] s0_tapg12_i,s0_tapg13_i,s0_tapg14_i,s0_tapg15_i,s0_tapg16_i,s0_tapg17_i;                  
    wire [11:0] s0_tapg18_i,s0_tapg19_i,s0_tapg20_i;                 
    wire [11:0] s0_tapg0_q,s0_tapg1_q,s0_tapg2_q,s0_tapg3_q,s0_tapg4_q,s0_tapg5_q;
    wire [11:0] s0_tapg6_q,s0_tapg7_q,s0_tapg8_q,s0_tapg9_q,s0_tapg10_q,s0_tapg11_q;                
    wire [11:0] s0_tapg12_q,s0_tapg13_q,s0_tapg14_q,s0_tapg15_q,s0_tapg16_q,s0_tapg17_q;                 
    wire [11:0] s0_tapg18_q,s0_tapg19_q,s0_tapg20_q;                 
    
    /* because tap registers are shared between HBF20 and LPF4040, tap outputs are
     * gated to prevent useless activity when LFP4040 is unused
     */
    assign { s0_tapg0_i, s0_tapg0_q} = { s0_tap0_i, s0_tap0_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg1_i, s0_tapg1_q} = { s0_tap1_i, s0_tap1_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg2_i, s0_tapg2_q} = { s0_tap2_i, s0_tap2_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg3_i, s0_tapg3_q} = { s0_tap3_i, s0_tap3_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg4_i, s0_tapg4_q} = { s0_tap4_i, s0_tap4_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg5_i, s0_tapg5_q} = { s0_tap5_i, s0_tap5_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg6_i, s0_tapg6_q} = { s0_tap6_i, s0_tap6_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg7_i, s0_tapg7_q} = { s0_tap7_i, s0_tap7_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg8_i, s0_tapg8_q} = { s0_tap8_i, s0_tap8_q} & {24{s0_lpf4040en}};                                               
    assign { s0_tapg9_i, s0_tapg9_q} = { s0_tap9_i, s0_tap9_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg10_i,s0_tapg10_q} = {s0_tap10_i,s0_tap10_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg11_i,s0_tapg11_q} = {s0_tap11_i,s0_tap11_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg12_i,s0_tapg12_q} = {s0_tap12_i,s0_tap12_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg13_i,s0_tapg13_q} = {s0_tap13_i,s0_tap13_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg14_i,s0_tapg14_q} = {s0_tap14_i,s0_tap14_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg15_i,s0_tapg15_q} = {s0_tap15_i,s0_tap15_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg16_i,s0_tapg16_q} = {s0_tap16_i,s0_tap16_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg17_i,s0_tapg17_q} = {s0_tap17_i,s0_tap17_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg18_i,s0_tapg18_q} = {s0_tap18_i,s0_tap18_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg19_i,s0_tapg19_q} = {s0_tap19_i,s0_tap19_q} & {24{s0_lpf4040en}};                                               
    assign {s0_tapg20_i,s0_tapg20_q} = {s0_tap20_i,s0_tap20_q} & {24{s0_lpf4040en}};                                               

    /***************************************************************************
    * S1
    ***************************************************************************/
    /* wires */
    wire [12:0] n_s1_lpf4040_sum0_20_i,n_s1_lpf4040_sum1_19_i,n_s1_lpf4040_sum2_18_i,n_s1_lpf4040_sum3_17_i;
    wire [12:0] n_s1_lpf4040_sum4_16_i,n_s1_lpf4040_sum5_15_i,n_s1_lpf4040_sum6_14_i,n_s1_lpf4040_sum7_13_i;
    wire [12:0] n_s1_lpf4040_sum8_12_i,n_s1_lpf4040_sum9_11_i,n_s1_lpf4040_sum10_i;
    wire [12:0] n_s1_lpf4040_sum0_20_q,n_s1_lpf4040_sum1_19_q,n_s1_lpf4040_sum2_18_q,n_s1_lpf4040_sum3_17_q;
    wire [12:0] n_s1_lpf4040_sum4_16_q,n_s1_lpf4040_sum5_15_q,n_s1_lpf4040_sum6_14_q,n_s1_lpf4040_sum7_13_q;
    wire [12:0] n_s1_lpf4040_sum8_12_q,n_s1_lpf4040_sum9_11_q,n_s1_lpf4040_sum10_q;
 
    wire [20:0] n_s1_lpf4040_mul0_20_i,n_s1_lpf4040_mul1_19_i,n_s1_lpf4040_mul2_18_i,n_s1_lpf4040_mul3_17_i;                              
    wire [20:0] n_s1_lpf4040_mul4_16_i,n_s1_lpf4040_mul5_15_i,n_s1_lpf4040_mul6_14_i,n_s1_lpf4040_mul7_13_i;                                
    wire [20:0] n_s1_lpf4040_mul8_12_i,n_s1_lpf4040_mul9_11_i,n_s1_lpf4040_mul10_i;                                                         
    wire [20:0] n_s1_lpf4040_mul0_20_q,n_s1_lpf4040_mul1_19_q,n_s1_lpf4040_mul2_18_q,n_s1_lpf4040_mul3_17_q;                                  
    wire [20:0] n_s1_lpf4040_mul4_16_q,n_s1_lpf4040_mul5_15_q,n_s1_lpf4040_mul6_14_q,n_s1_lpf4040_mul7_13_q;                                 
    wire [20:0] n_s1_lpf4040_mul8_12_q,n_s1_lpf4040_mul9_11_q,n_s1_lpf4040_mul10_q;                                                         
   
    wire [22:0] n_s1_lpf4040_sum0_i,n_s1_lpf4040_sum1_i,n_s1_lpf4040_sum2_i,n_s1_lpf4040_sum3_i;
    wire [22:0] n_s1_lpf4040_sum0_q,n_s1_lpf4040_sum1_q,n_s1_lpf4040_sum2_q,n_s1_lpf4040_sum3_q;
    reg  [ 3:0] n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q;
    
    /* flops */
    reg  [22:0] s1_lpf4040_sum0_i,s1_lpf4040_sum1_i,s1_lpf4040_sum2_i,s1_lpf4040_sum3_i;                                 
    reg  [22:0] s1_lpf4040_sum0_q,s1_lpf4040_sum1_q,s1_lpf4040_sum2_q,s1_lpf4040_sum3_q; 
    reg  [ 3:0] s1_lpf4040_sh_i,s1_lpf4040_sh_q,s1_lpf4040_sh_q_1t;                              
 
    /* factoring coefficients by index parity                                                                        
    *  [-2048 2047] + [-2048 2047] + [ 1 1] =  [-4095   4095] -> 13 bits                                                      
    */                                           
    assign n_s1_lpf4040_sum0_20_i = { s0_tapg0_i[11], s0_tapg0_i} +  {s0_tapg20_i[11],s0_tapg20_i};                                   
    assign n_s1_lpf4040_sum1_19_i = { s0_tapg1_i[11], s0_tapg1_i} + ({s0_tapg19_i[11],s0_tapg19_i} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum2_18_i = { s0_tapg2_i[11], s0_tapg2_i} +  {s0_tapg18_i[11],s0_tapg18_i};                                   
    assign n_s1_lpf4040_sum3_17_i = { s0_tapg3_i[11], s0_tapg3_i} + ({s0_tapg17_i[11],s0_tapg17_i} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum4_16_i = { s0_tapg4_i[11], s0_tapg4_i} +  {s0_tapg16_i[11],s0_tapg16_i};                                   
    assign n_s1_lpf4040_sum5_15_i = { s0_tapg5_i[11], s0_tapg5_i} + ({s0_tapg15_i[11],s0_tapg15_i} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum6_14_i = { s0_tapg6_i[11], s0_tapg6_i} +  {s0_tapg14_i[11],s0_tapg14_i};                                   
    assign n_s1_lpf4040_sum7_13_i = { s0_tapg7_i[11], s0_tapg7_i} + ({s0_tapg13_i[11],s0_tapg13_i} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum8_12_i = { s0_tapg8_i[11], s0_tapg8_i} +  {s0_tapg12_i[11],s0_tapg12_i};                                   
    assign n_s1_lpf4040_sum9_11_i = { s0_tapg9_i[11], s0_tapg9_i} + ({s0_tapg11_i[11],s0_tapg11_i} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum10_i   = {s0_tapg10_i[11],s0_tapg10_i};                                                                 

    assign n_s1_lpf4040_sum0_20_q = { s0_tapg0_q[11], s0_tapg0_q} +  {s0_tapg20_q[11],s0_tapg20_q};                                   
    assign n_s1_lpf4040_sum1_19_q = { s0_tapg1_q[11], s0_tapg1_q} + ({s0_tapg19_q[11],s0_tapg19_q} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum2_18_q = { s0_tapg2_q[11], s0_tapg2_q} +  {s0_tapg18_q[11],s0_tapg18_q};                                   
    assign n_s1_lpf4040_sum3_17_q = { s0_tapg3_q[11], s0_tapg3_q} + ({s0_tapg17_q[11],s0_tapg17_q} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum4_16_q = { s0_tapg4_q[11], s0_tapg4_q} +  {s0_tapg16_q[11],s0_tapg16_q};                                   
    assign n_s1_lpf4040_sum5_15_q = { s0_tapg5_q[11], s0_tapg5_q} + ({s0_tapg15_q[11],s0_tapg15_q} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum6_14_q = { s0_tapg6_q[11], s0_tapg6_q} +  {s0_tapg14_q[11],s0_tapg14_q};                                   
    assign n_s1_lpf4040_sum7_13_q = { s0_tapg7_q[11], s0_tapg7_q} + ({s0_tapg13_q[11],s0_tapg13_q} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum8_12_q = { s0_tapg8_q[11], s0_tapg8_q} +  {s0_tapg12_q[11],s0_tapg12_q};                                   
    assign n_s1_lpf4040_sum9_11_q = { s0_tapg9_q[11], s0_tapg9_q} + ({s0_tapg11_q[11],s0_tapg11_q} ^ {13{s0_cfsen}}) + {12'b0,s0_cfsen};     
    assign n_s1_lpf4040_sum10_q   = {s0_tapg10_q[11],s0_tapg10_q};                                                                 

    /* coefficient weighting                                                                                         
    *  max values are [-4095 4095]*256 => [-1048320 1048320]                                                                  
    *  which fit a 21 bits range ([-1048576   1048577])                                                                       
    */                                           
    assign n_s1_lpf4040_mul0_20_i  =  $signed(LPF4040_0_20) * $signed(n_s1_lpf4040_sum0_20_i);                                        
    assign n_s1_lpf4040_mul1_19_i  =  $signed(LPF4040_1_19) * $signed(n_s1_lpf4040_sum1_19_i);                                        
    assign n_s1_lpf4040_mul2_18_i  =  $signed(LPF4040_2_18) * $signed(n_s1_lpf4040_sum2_18_i);                                        
    assign n_s1_lpf4040_mul3_17_i  =  $signed(LPF4040_3_17) * $signed(n_s1_lpf4040_sum3_17_i);                                        
    assign n_s1_lpf4040_mul4_16_i  =  $signed(LPF4040_4_16) * $signed(n_s1_lpf4040_sum4_16_i);                                        
    assign n_s1_lpf4040_mul5_15_i  =  $signed(LPF4040_5_15) * $signed(n_s1_lpf4040_sum5_15_i);                                        
    assign n_s1_lpf4040_mul6_14_i  =  $signed(LPF4040_6_14) * $signed(n_s1_lpf4040_sum6_14_i);                                        
    assign n_s1_lpf4040_mul7_13_i  =  $signed(LPF4040_7_13) * $signed(n_s1_lpf4040_sum7_13_i);                                        
    assign n_s1_lpf4040_mul8_12_i  =  $signed(LPF4040_8_12) * $signed(n_s1_lpf4040_sum8_12_i);                                        
    assign n_s1_lpf4040_mul9_11_i  =  $signed(LPF4040_9_11) * $signed(n_s1_lpf4040_sum9_11_i);                                        
    assign n_s1_lpf4040_mul10_i    =  $signed(LPF4040_10)   * $signed(n_s1_lpf4040_sum10_i);                                          

    assign n_s1_lpf4040_mul0_20_q  =  $signed(LPF4040_0_20) * $signed(n_s1_lpf4040_sum0_20_q);                                        
    assign n_s1_lpf4040_mul1_19_q  =  $signed(LPF4040_1_19) * $signed(n_s1_lpf4040_sum1_19_q);                                        
    assign n_s1_lpf4040_mul2_18_q  =  $signed(LPF4040_2_18) * $signed(n_s1_lpf4040_sum2_18_q);                                        
    assign n_s1_lpf4040_mul3_17_q  =  $signed(LPF4040_3_17) * $signed(n_s1_lpf4040_sum3_17_q);                                        
    assign n_s1_lpf4040_mul4_16_q  =  $signed(LPF4040_4_16) * $signed(n_s1_lpf4040_sum4_16_q);                                        
    assign n_s1_lpf4040_mul5_15_q  =  $signed(LPF4040_5_15) * $signed(n_s1_lpf4040_sum5_15_q);                                        
    assign n_s1_lpf4040_mul6_14_q  =  $signed(LPF4040_6_14) * $signed(n_s1_lpf4040_sum6_14_q);                                        
    assign n_s1_lpf4040_mul7_13_q  =  $signed(LPF4040_7_13) * $signed(n_s1_lpf4040_sum7_13_q);                                        
    assign n_s1_lpf4040_mul8_12_q  =  $signed(LPF4040_8_12) * $signed(n_s1_lpf4040_sum8_12_q);                                        
    assign n_s1_lpf4040_mul9_11_q  =  $signed(LPF4040_9_11) * $signed(n_s1_lpf4040_sum9_11_q);                                        
    assign n_s1_lpf4040_mul10_q    =  $signed(LPF4040_10)   * $signed(n_s1_lpf4040_sum10_q);                                          


    /* sum all weighted samples which have the same modulo-4 index                                                   
    *  max values are 3*[-1048576   1048320] = [-3145728   3144960] which                                                     
    *  fit a 23 bits range ([-4194304   4194303])                                                                             
    */     
    assign n_s1_lpf4040_sum0_i = {{2{ n_s1_lpf4040_mul0_20_i[20]}}, n_s1_lpf4040_mul0_20_i} +                                              
                                 {{2{ n_s1_lpf4040_mul4_16_i[20]}}, n_s1_lpf4040_mul4_16_i} +                                              
                                 {{2{ n_s1_lpf4040_mul8_12_i[20]}}, n_s1_lpf4040_mul8_12_i};                                               

    assign n_s1_lpf4040_sum1_i = {{2{ n_s1_lpf4040_mul1_19_i[20]}}, n_s1_lpf4040_mul1_19_i} +                                              
                                 {{2{ n_s1_lpf4040_mul5_15_i[20]}}, n_s1_lpf4040_mul5_15_i} +                                              
                                 {{2{ n_s1_lpf4040_mul9_11_i[20]}}, n_s1_lpf4040_mul9_11_i};                                               

    assign n_s1_lpf4040_sum2_i = {{2{ n_s1_lpf4040_mul2_18_i[20]}}, n_s1_lpf4040_mul2_18_i} +                                              
                                 {{2{ n_s1_lpf4040_mul6_14_i[20]}}, n_s1_lpf4040_mul6_14_i} +                                              
                                 {{2{ n_s1_lpf4040_mul10_i[20]}},     n_s1_lpf4040_mul10_i};                                               

    assign n_s1_lpf4040_sum3_i = {{2{ n_s1_lpf4040_mul3_17_i[20]}}, n_s1_lpf4040_mul3_17_i} +                                              
                                 {{2{ n_s1_lpf4040_mul7_13_i[20]}}, n_s1_lpf4040_mul7_13_i};                                               


    assign n_s1_lpf4040_sum0_q = {{2{ n_s1_lpf4040_mul0_20_q[20]}}, n_s1_lpf4040_mul0_20_q} +                                              
                                 {{2{ n_s1_lpf4040_mul4_16_q[20]}}, n_s1_lpf4040_mul4_16_q} +                                              
                                 {{2{ n_s1_lpf4040_mul8_12_q[20]}}, n_s1_lpf4040_mul8_12_q};                                               
 
    assign n_s1_lpf4040_sum1_q = {{2{ n_s1_lpf4040_mul1_19_q[20]}}, n_s1_lpf4040_mul1_19_q} +                                              
                                 {{2{ n_s1_lpf4040_mul5_15_q[20]}}, n_s1_lpf4040_mul5_15_q} +                                              
                                 {{2{ n_s1_lpf4040_mul9_11_q[20]}}, n_s1_lpf4040_mul9_11_q};                                               

    assign n_s1_lpf4040_sum2_q = {{2{ n_s1_lpf4040_mul2_18_q[20]}}, n_s1_lpf4040_mul2_18_q} +                                              
                                 {{2{ n_s1_lpf4040_mul6_14_q[20]}}, n_s1_lpf4040_mul6_14_q} +                                              
                                 {{2{ n_s1_lpf4040_mul10_q[20]}},     n_s1_lpf4040_mul10_q};                                               

    assign n_s1_lpf4040_sum3_q = {{2{ n_s1_lpf4040_mul3_17_q[20]}}, n_s1_lpf4040_mul3_17_q} +                                              
                                 {{2{ n_s1_lpf4040_mul7_13_q[20]}}, n_s1_lpf4040_mul7_13_q};                                               
  
    always @(*)
    begin
      if(s0_cfsen)
      begin
        if(!s0_cfs)
        begin
          case(s0_cycle)
            2'd0:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b1100,4'b0110};
            2'd1:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b1001,4'b1100}; 
            2'd2:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b0011,4'b1001}; 
            default:  {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b0110,4'b0011};
          endcase
        end
        else
        begin
          case(s0_cycle)
            2'd0:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b0110,4'b1100};
            2'd1:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b1100,4'b1001}; 
            2'd2:     {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b1001,4'b0011}; 
            default:  {n_s1_lpf4040_sh_i,n_s1_lpf4040_sh_q} = {4'b0011,4'b0110};
          endcase
        end
      end
      else
      begin
        n_s1_lpf4040_sh_i = 4'b0000;
        n_s1_lpf4040_sh_q = 4'b0000;
      end
    end
    
    /* scale DC (sum(lpf4040)/512*1024=1472)
    * [-2048 2047] * 1472 + 512 = [-3014144   3013696]
    * [-2^22 2^22-1] = [ -4194304   4194303] => 23 bits range, no saturation required
    */
    wire [22:0] n_s1_lpf4040_dc_scaled_i,n_s1_lpf4040_dc_scaled_q;
    wire [22:0] n_s1_lpf4040_dc_fix_i,n_s1_lpf4040_dc_fix_q;
    reg  [12:0] s1_lpf4040_dc_i,s1_lpf4040_dc_q;
    
    assign n_s1_lpf4040_dc_scaled_i = $signed(12'd1472) * $signed(s0_dc_i);
    assign n_s1_lpf4040_dc_fix_i    = n_s1_lpf4040_dc_scaled_i + 23'd512;
  
    assign n_s1_lpf4040_dc_scaled_q = $signed(12'd1472) * $signed(s0_dc_q);
    assign n_s1_lpf4040_dc_fix_q    = n_s1_lpf4040_dc_scaled_q + 23'd512;

    always @(posedge clk, negedge rst_n)                                                                                     
    begin                                                                                                                    
      if(!rst_n)                                                                                                             
      begin                                                                                                                  
        s1_lpf4040_sum0_i <= 23'b0;                                                                                         
        s1_lpf4040_sum1_i <= 23'b0;                                                                                         
        s1_lpf4040_sum2_i <= 23'b0;                                                                                         
        s1_lpf4040_sum3_i <= 23'b0;                                                                                         
        s1_lpf4040_sum0_q <= 23'b0;                                                                                         
        s1_lpf4040_sum1_q <= 23'b0;                                                                                         
        s1_lpf4040_sum2_q <= 23'b0;                                                                                         
        s1_lpf4040_sum3_q <= 23'b0;
        s1_lpf4040_sh_i   <= 4'b0;
        s1_lpf4040_sh_q   <= 4'b0;
        s1_lpf4040_sh_q_1t<= 4'b0;
      end
      else if(!s0_lpf4040en || !s0_rxmode)
      begin
        s1_lpf4040_sum0_i <= 23'b0;                                                                                         
        s1_lpf4040_sum1_i <= 23'b0;                                                                                         
        s1_lpf4040_sum2_i <= 23'b0;                                                                                         
        s1_lpf4040_sum3_i <= 23'b0;                                                                                         
        s1_lpf4040_sum0_q <= 23'b0;                                                                                         
        s1_lpf4040_sum1_q <= 23'b0;                                                                                         
        s1_lpf4040_sum2_q <= 23'b0;                                                                                         
        s1_lpf4040_sum3_q <= 23'b0;
        s1_lpf4040_sh_i   <= 4'b0;
        s1_lpf4040_sh_q   <= 4'b0;
        s1_lpf4040_sh_q_1t<= 4'b0;
      end
      else
      begin
        s1_lpf4040_sum0_i <= n_s1_lpf4040_sum0_i;                                                                            
        s1_lpf4040_sum1_i <= n_s1_lpf4040_sum1_i;                                                                            
        s1_lpf4040_sum2_i <= n_s1_lpf4040_sum2_i;                                                                            
        s1_lpf4040_sum3_i <= n_s1_lpf4040_sum3_i;                                                                            
        s1_lpf4040_sum0_q <= n_s1_lpf4040_sum0_q;                                                                            
        s1_lpf4040_sum1_q <= n_s1_lpf4040_sum1_q;                                                                            
        s1_lpf4040_sum2_q <= n_s1_lpf4040_sum2_q;                                                                            
        s1_lpf4040_sum3_q <= n_s1_lpf4040_sum3_q;
        s1_lpf4040_sh_i   <= n_s1_lpf4040_sh_i;
        s1_lpf4040_sh_q   <= n_s1_lpf4040_sh_q;
        s1_lpf4040_sh_q_1t<= s1_lpf4040_sh_q;
      end
    end                                                                           
   
    always @(posedge clk, negedge rst_n)                                                                                     
    begin                                                                                                                    
      if(!rst_n)                                                                                                             
      begin                                                                                                                  
        s1_lpf4040_dc_i   <= 13'b0;
        s1_lpf4040_dc_q   <= 13'b0;
      end
      else if(!s0_blocken || !s0_rxmode)
      begin
        s1_lpf4040_dc_i   <= 13'b0;
        s1_lpf4040_dc_q   <= 13'b0;
      end
      else
      begin
        s1_lpf4040_dc_i   <= n_s1_lpf4040_dc_fix_i[22:10];
        s1_lpf4040_dc_q   <= n_s1_lpf4040_dc_fix_q[22:10];
      end
    end                                                                           

    /***************************************************************************
    * S2
    ***************************************************************************/
    /* wires */
    wire [22:0] n_s2_lpf4040_sum_i,n_s2_lpf4040_sum_q;
    wire [13:0] n_s2_lpf4040_sum_rnd_i,n_s2_lpf4040_sum_rnd_q; 
    reg  [12:0] n_s2_lpf4040_sum_sat_i,n_s2_lpf4040_sum_sat_q;
    /* reg */
    reg [12:0]  s2_lpf4040_i,s2_lpf4040_hold_i,s2_lpf4040_q;
    reg [12:0]  s2_lpf4040_dc_i,s2_lpf4040_dc_q;
    
    /* final summing */                                           
    assign n_s2_lpf4040_sum_i = (s1_lpf4040_sum0_i ^ {23{s1_lpf4040_sh_i[0]}}) +                                                         
                                (s1_lpf4040_sum1_i ^ {23{s1_lpf4040_sh_i[1]}}) +                                                    
                                (s1_lpf4040_sum2_i ^ {23{s1_lpf4040_sh_i[2]}}) +                                                    
                                (s1_lpf4040_sum3_i ^ {23{s1_lpf4040_sh_i[3]}}) +                                                    
                                {14'b0,1'b1,6'b0,s1_cfsen,1'b0}; // rndfix + c2fix                                            

    assign n_s2_lpf4040_sum_q = (s1_lpf4040_sum0_q ^ {23{s1_lpf4040_sh_q_1t[0]}}) +                                                         
                                (s1_lpf4040_sum1_q ^ {23{s1_lpf4040_sh_q_1t[1]}}) +                                                    
                                (s1_lpf4040_sum2_q ^ {23{s1_lpf4040_sh_q_1t[2]}}) +                                                    
                                (s1_lpf4040_sum3_q ^ {23{s1_lpf4040_sh_q_1t[3]}}) +                                                    
                                {14'b0,1'b1,6'b0,s1_cfsen,1'b0};  // rndfix + c2fix                                           
                                                                                                                             
    /* rnd 9 */                                               
    assign n_s2_lpf4040_sum_rnd_i = n_s2_lpf4040_sum_i[22:9];                                                                   
    assign n_s2_lpf4040_sum_rnd_q = n_s2_lpf4040_sum_q[22:9];                                                                   
                                                                                                                             
    /* saturate 13 bits */                                                                                                   
    always @(*)                                                                                                              
    begin                                                                                                                    
      if((n_s2_lpf4040_sum_rnd_i[13:12]==2'b00) || (n_s2_lpf4040_sum_rnd_i[13:12]==2'b11))                                       
        n_s2_lpf4040_sum_sat_i = n_s2_lpf4040_sum_rnd_i[12:0];                                                                   
      else                                                                                                                   
        n_s2_lpf4040_sum_sat_i = {n_s2_lpf4040_sum_rnd_i[13],{12{~n_s2_lpf4040_sum_rnd_i[13]}}};                                   
    end
                                                                                                                            
    always @(*)                                                                                                              
    begin                                                                                                                    
      if((n_s2_lpf4040_sum_rnd_q[13:12]==2'b00) || (n_s2_lpf4040_sum_rnd_q[13:12]==2'b11))                                       
        n_s2_lpf4040_sum_sat_q = n_s2_lpf4040_sum_rnd_q[12:0];                                                                   

      else                                                                                                                   
        n_s2_lpf4040_sum_sat_q = {n_s2_lpf4040_sum_rnd_q[13],{12{~n_s2_lpf4040_sum_rnd_q[13]}}};                                   
    end 
    
    always @(posedge clk, negedge rst_n)                                                                                     
    begin                                                                                                                    
      if(!rst_n)                                                                                                             
      begin                                                                                                                  
        s2_lpf4040_i      <= 13'b0;                                                                                         
        s2_lpf4040_hold_i <= 13'b0;                                                                                         
        s2_lpf4040_q      <= 13'b0;                                                                                         
      end
      else if(!blocken || !s1_lpf4040en || !s1_rxmode)
      begin
        s2_lpf4040_i      <= 13'b0;                                                                                         
        s2_lpf4040_hold_i <= 13'b0;                                                                                         
        s2_lpf4040_q      <= 13'b0;                                                                                         
      end
      else
      begin
        s2_lpf4040_i      <= n_s2_lpf4040_sum_sat_i;                                                                                         
        s2_lpf4040_hold_i <= s2_lpf4040_i;                                                                                         
        s2_lpf4040_q      <= n_s2_lpf4040_sum_sat_q;                                                                                         
      end
    end                                                                           
  
    always @(posedge clk, negedge rst_n)                                                                                     
    begin                                                                                                                    
      if(!rst_n)                                                                                                             
      begin                                                                                                                  
        s2_lpf4040_dc_i   <= 13'b0;                                                                                         
        s2_lpf4040_dc_q   <= 13'b0;                                                                                         
      end
      else if(!blocken || !s1_rxmode)
      begin
        s2_lpf4040_dc_i   <= 13'b0;                                                                                         
        s2_lpf4040_dc_q   <= 13'b0;                                                                                         
      end
      else
      begin
        s2_lpf4040_dc_i   <= s1_lpf4040_dc_i;                                                                                         
        s2_lpf4040_dc_q   <= s1_lpf4040_dc_q;                                                                                         
      end
    end                                                                           
        
    assign out_rx_lpf4040_i    = s2_lpf4040_hold_i; 
    assign out_rx_lpf4040_q    = s2_lpf4040_q;
    assign out_rx_lpf4040_dc_i = s2_lpf4040_dc_i;
    assign out_rx_lpf4040_dc_q = s2_lpf4040_dc_q;
                                                                                                                            
  end  
  endgenerate

endmodule
`default_nettype wire

