//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of hbf40
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module hbf40
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        rst_n,
  input  wire        clk,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire        blocken,  // global enable 
  input  wire        rxmode,   // 1=RX; 0=TX
  input  wire        cfsen,    // enable spectrum shift
  input  wire        cfs,      // 0=primary is left, 1=primary is right
  
  /*****************************************************************************
  * RX
  *****************************************************************************/
  input  wire        in_rx_valid,
  input  wire [11:0] in_rx_dc_i,
  input  wire [11:0] in_rx_dc_q,
  input  wire [11:0] in_rx_i,
  input  wire [11:0] in_rx_q,
  
  output wire        out_rx_valid,
  output wire [12:0] out_rx_dc_i,
  output wire [12:0] out_rx_dc_q,
  output wire [12:0] out_rx_pri_i,
  output wire [12:0] out_rx_pri_q,
  output wire [12:0] out_rx_sec_i,
  output wire [12:0] out_rx_sec_q,
  
  /*****************************************************************************
  * TX
  *****************************************************************************/
  input  wire        in_tx_valid,
  input  wire [11:0] in_tx_i,
  input  wire [11:0] in_tx_q,
  
  output wire        out_tx_valid,
  output wire [11:0] out_tx_i,
  output wire [11:0] out_tx_q
  
);

  /*****************************************************************************
  * impulse response :
  *   hbf40=[16 43 27 -33 -45 27 63 -24 -87 22 127 -21 -218 20 663 1024 663 ...
  *        20 -218 -21 127 22 -87 -24 63 27 -45 -33 27 43 16]
  * rx maximum range:
  * 
  *   input samples are 12 bits [-2048 2047]
  *   
  *   sum(abs(hbf40))*[-2048 2047] + 1024(c2fix) = [-7977984   7976136]
  *
  * which fits 24 bits range :
  *  
  *   [-2^23 2^23-1]= [-8388608   8388607] 
  *
  *
  *****************************************************************************/
  
  /*****************************************************************************
  * S0
  *****************************************************************************/
  /* declaration */
  localparam  H_0_30  =   16, H_1_29  =   43, H_2_28  =   27, H_3_27  =  -33,
              H_4_26  =  -45, H_5_25  =   27, H_6_24  =   63, H_7_23  =  -24,
              H_8_22  =  -87, H_9_21  =   22, H_10_20 =  127, H_11_19 =  -21,
              H_12_18 = -218, H_13_17 =   20, H_14_16 =  663, H_15    = 1024;
  
  /* wire */
  wire        in_valid;
  reg  [11:0] n_s0_swp_i,n_s0_swp_q,n_s0_hold_i;
  /* flops */
  reg         in_cycle;
  reg         s0_rxmode;
  reg         s0_blocken;
  reg         s0_cfsen;
  reg         s0_cfs;
  reg  [30:0] s0_valid;
  reg         s0_even,s0_odd;
  reg  [ 1:0] s0_cycle;
  reg  [11:0] s0_hold_i;
  reg  [11:0] s0_swp_q;
  reg  [11:0] s0_dc_i,s0_dc_q;
  reg  [11:0] s0_tap0_i,s0_tap1_i,s0_tap2_i,s0_tap3_i,s0_tap4_i,s0_tap5_i;
  reg  [11:0] s0_tap6_i,s0_tap7_i,s0_tap8_i,s0_tap9_i,s0_tap10_i,s0_tap11_i;
  reg  [11:0] s0_tap12_i,s0_tap13_i,s0_tap14_i,s0_tap15_i,s0_tap16_i,s0_tap17_i; 
  reg  [11:0] s0_tap18_i,s0_tap19_i,s0_tap20_i,s0_tap21_i,s0_tap22_i,s0_tap23_i;
  reg  [11:0] s0_tap24_i,s0_tap25_i,s0_tap26_i,s0_tap27_i,s0_tap28_i,s0_tap29_i;
  reg  [11:0] s0_tap30_i;
  reg  [11:0] s0_tap0_q,s0_tap1_q,s0_tap2_q,s0_tap3_q,s0_tap4_q,s0_tap5_q;
  reg  [11:0] s0_tap6_q,s0_tap7_q,s0_tap8_q,s0_tap9_q,s0_tap10_q,s0_tap11_q;
  reg  [11:0] s0_tap12_q,s0_tap13_q,s0_tap14_q,s0_tap15_q,s0_tap16_q,s0_tap17_q; 
  reg  [11:0] s0_tap18_q,s0_tap19_q,s0_tap20_q,s0_tap21_q,s0_tap22_q,s0_tap23_q;
  reg  [11:0] s0_tap24_q,s0_tap25_q,s0_tap26_q,s0_tap27_q,s0_tap28_q,s0_tap29_q;
  reg  [11:0] s0_tap30_q;


  /* RX i/q swapping for spectral shift, TX hold samples for interpolation */
  assign in_valid   = rxmode?in_rx_valid:in_tx_valid;
  
  always @(*)
  begin
    if(rxmode)
    begin
      n_s0_hold_i = 12'b0;
      if(!in_rx_valid)
      begin
        n_s0_swp_i = 12'b0;
        n_s0_swp_q = 12'b0;
      end
      else if(cfsen)
      begin
        if(!in_cycle)
        begin
          n_s0_swp_i = in_rx_i;
          n_s0_swp_q = in_rx_q;
        end
        else
        begin
          n_s0_swp_i = in_rx_q;
          n_s0_swp_q = in_rx_i;
        end
      end
      else
      begin
        n_s0_swp_i = in_rx_i;
        n_s0_swp_q = in_rx_q;
      end
    end
    else
    begin
      if(!in_cycle)
      begin
        if(!in_tx_valid)
        begin
          n_s0_hold_i = 12'b0;
          n_s0_swp_i  = 12'b0;
          n_s0_swp_q  = 12'b0;
        end
        else
        begin
          n_s0_hold_i = in_tx_i;
          n_s0_swp_i  = in_tx_i;
          n_s0_swp_q  = in_tx_q;
        end
      end
      else
      begin
        n_s0_hold_i = s0_hold_i;
        n_s0_swp_i  = s0_hold_i;
        n_s0_swp_q  = s0_swp_q;
      end
    end
  end
  
  
  /* s0 data path */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      in_cycle                <= 1'b0;
      s0_rxmode               <= 1'b0;
      s0_blocken              <= 1'b0;
      s0_cfsen                <= 1'b0;
      s0_cfs                  <= 1'b0; 
      s0_cycle                <= 2'b0;
      s0_valid                <= 31'b0;
      s0_even                 <= 1'b0;
      s0_odd                  <= 1'b0;

      s0_hold_i               <= 12'b0;
      s0_swp_q                <= 12'b0;
      s0_dc_i                 <= 12'b0;
      s0_dc_q                 <= 12'b0;
      { s0_tap0_i, s0_tap0_q} <= 24'b0;
      { s0_tap1_i, s0_tap1_q} <= 24'b0;
      { s0_tap2_i, s0_tap2_q} <= 24'b0;
      { s0_tap3_i, s0_tap3_q} <= 24'b0;
      { s0_tap4_i, s0_tap4_q} <= 24'b0;
      { s0_tap5_i, s0_tap5_q} <= 24'b0;
      { s0_tap6_i, s0_tap6_q} <= 24'b0;
      { s0_tap7_i, s0_tap7_q} <= 24'b0;
      { s0_tap8_i, s0_tap8_q} <= 24'b0;
      { s0_tap9_i, s0_tap9_q} <= 24'b0;
      {s0_tap10_i,s0_tap10_q} <= 24'b0;
      {s0_tap11_i,s0_tap11_q} <= 24'b0;
      {s0_tap12_i,s0_tap12_q} <= 24'b0;
      {s0_tap13_i,s0_tap13_q} <= 24'b0;
      {s0_tap14_i,s0_tap14_q} <= 24'b0;
      {s0_tap15_i,s0_tap15_q} <= 24'b0;
      {s0_tap16_i,s0_tap16_q} <= 24'b0;
      {s0_tap17_i,s0_tap17_q} <= 24'b0;
      {s0_tap18_i,s0_tap18_q} <= 24'b0;
      {s0_tap19_i,s0_tap19_q} <= 24'b0;
      {s0_tap20_i,s0_tap20_q} <= 24'b0;
      {s0_tap21_i,s0_tap21_q} <= 24'b0;
      {s0_tap22_i,s0_tap22_q} <= 24'b0;
      {s0_tap23_i,s0_tap23_q} <= 24'b0;
      {s0_tap24_i,s0_tap24_q} <= 24'b0;
      {s0_tap25_i,s0_tap25_q} <= 24'b0;
      {s0_tap26_i,s0_tap26_q} <= 24'b0;
      {s0_tap27_i,s0_tap27_q} <= 24'b0;
      {s0_tap28_i,s0_tap28_q} <= 24'b0;
      {s0_tap29_i,s0_tap29_q} <= 24'b0;
      {s0_tap30_i,s0_tap30_q} <= 24'b0;
    end
    else
    begin
      if(blocken)
      begin
        /* param */
        s0_rxmode    <= rxmode;
        s0_blocken   <= 1'b1;
        s0_cfsen     <= cfsen & rxmode;
        s0_cfs       <= cfs; 
      
        /* valid tag */                                    
        if(in_valid)
          in_cycle <= ~in_cycle;
        else
          in_cycle <= 1'b0;
      
        s0_valid     <= {s0_valid[29:0],in_valid};
      
        /* cycle tag */
        if(s0_valid!=31'd0)
        begin
          s0_cycle <= s0_cycle + 2'd1;                                  
          if(rxmode)
          begin
            s0_even <= ~s0_cycle[0];
            s0_odd  <= ~s0_cycle[0];
          end
          else
          begin
            s0_even <= ~s0_cycle[0];
            s0_odd  <=  s0_cycle[0];
          end
        end
        else
        begin
          s0_cycle <= 2'd0;                                    
          s0_even  <= 1'b0;
          s0_odd   <= 1'b0;
        end
      
        s0_swp_q  <= n_s0_swp_q;
        s0_hold_i <= n_s0_hold_i;
        { s0_tap0_i, s0_tap0_q} <= {n_s0_swp_i, s0_swp_q};                 
     
        if(rxmode)
        begin
          if(in_cycle)
          begin
            s0_dc_i <= in_rx_dc_i;
            s0_dc_q <= in_rx_dc_q;                 
          end
        end
        else
        begin
          s0_dc_i <= 12'd0;
          s0_dc_q <= 12'd0;                 
        end
      end
      else
      begin
        in_cycle                <= 1'b0;
        s0_rxmode               <= 1'b0;
        s0_blocken              <= 1'b0;
        s0_cfsen                <= 1'b0;
        s0_cfs                  <= 1'b0; 
        s0_cycle                <= 2'b0;
        s0_valid                <= 31'b0;
        s0_even                 <= 1'b0;
        s0_odd                  <= 1'b0;

        s0_hold_i               <= 12'b0;
        s0_swp_q                <= 12'b0;
        s0_dc_i                 <= 12'd0;
        s0_dc_q                 <= 12'd0;  
        { s0_tap0_i, s0_tap0_q} <= 24'b0;
      end
    
      if(s0_blocken)
      begin
        { s0_tap1_i, s0_tap1_q} <= { s0_tap0_i, s0_tap0_q};             
        { s0_tap2_i, s0_tap2_q} <= { s0_tap1_i, s0_tap1_q};             
        { s0_tap3_i, s0_tap3_q} <= { s0_tap2_i, s0_tap2_q};             
        { s0_tap4_i, s0_tap4_q} <= { s0_tap3_i, s0_tap3_q};             
        { s0_tap5_i, s0_tap5_q} <= { s0_tap4_i, s0_tap4_q};             
        { s0_tap6_i, s0_tap6_q} <= { s0_tap5_i, s0_tap5_q};             
        { s0_tap7_i, s0_tap7_q} <= { s0_tap6_i, s0_tap6_q};             
        { s0_tap8_i, s0_tap8_q} <= { s0_tap7_i, s0_tap7_q};             
        { s0_tap9_i, s0_tap9_q} <= { s0_tap8_i, s0_tap8_q};             
        {s0_tap10_i,s0_tap10_q} <= { s0_tap9_i, s0_tap9_q};             
        {s0_tap11_i,s0_tap11_q} <= {s0_tap10_i,s0_tap10_q};             
        {s0_tap12_i,s0_tap12_q} <= {s0_tap11_i,s0_tap11_q};             
        {s0_tap13_i,s0_tap13_q} <= {s0_tap12_i,s0_tap12_q};             
        {s0_tap14_i,s0_tap14_q} <= {s0_tap13_i,s0_tap13_q};             
        {s0_tap15_i,s0_tap15_q} <= {s0_tap14_i,s0_tap14_q};             
        {s0_tap16_i,s0_tap16_q} <= {s0_tap15_i,s0_tap15_q};             
        {s0_tap17_i,s0_tap17_q} <= {s0_tap16_i,s0_tap16_q};             
        {s0_tap18_i,s0_tap18_q} <= {s0_tap17_i,s0_tap17_q};             
        {s0_tap19_i,s0_tap19_q} <= {s0_tap18_i,s0_tap18_q};             
        {s0_tap20_i,s0_tap20_q} <= {s0_tap19_i,s0_tap19_q};             
        {s0_tap21_i,s0_tap21_q} <= {s0_tap20_i,s0_tap20_q};
        {s0_tap22_i,s0_tap22_q} <= {s0_tap21_i,s0_tap21_q};
        {s0_tap23_i,s0_tap23_q} <= {s0_tap22_i,s0_tap22_q};
        {s0_tap24_i,s0_tap24_q} <= {s0_tap23_i,s0_tap23_q};
        {s0_tap25_i,s0_tap25_q} <= {s0_tap24_i,s0_tap24_q};
        {s0_tap26_i,s0_tap26_q} <= {s0_tap25_i,s0_tap25_q};
        {s0_tap27_i,s0_tap27_q} <= {s0_tap26_i,s0_tap26_q};
        {s0_tap28_i,s0_tap28_q} <= {s0_tap27_i,s0_tap27_q};
        {s0_tap29_i,s0_tap29_q} <= {s0_tap28_i,s0_tap28_q};
        {s0_tap30_i,s0_tap30_q} <= {s0_tap29_i,s0_tap29_q};
      end
      else
      begin
        { s0_tap1_i, s0_tap1_q} <= 24'b0;
        { s0_tap2_i, s0_tap2_q} <= 24'b0;
        { s0_tap3_i, s0_tap3_q} <= 24'b0;
        { s0_tap4_i, s0_tap4_q} <= 24'b0;
        { s0_tap5_i, s0_tap5_q} <= 24'b0;
        { s0_tap6_i, s0_tap6_q} <= 24'b0;
        { s0_tap7_i, s0_tap7_q} <= 24'b0;
        { s0_tap8_i, s0_tap8_q} <= 24'b0;
        { s0_tap9_i, s0_tap9_q} <= 24'b0;
        {s0_tap10_i,s0_tap10_q} <= 24'b0;
        {s0_tap11_i,s0_tap11_q} <= 24'b0;
        {s0_tap12_i,s0_tap12_q} <= 24'b0;
        {s0_tap13_i,s0_tap13_q} <= 24'b0;
        {s0_tap14_i,s0_tap14_q} <= 24'b0;
        {s0_tap15_i,s0_tap15_q} <= 24'b0;
        {s0_tap16_i,s0_tap16_q} <= 24'b0;
        {s0_tap17_i,s0_tap17_q} <= 24'b0;
        {s0_tap18_i,s0_tap18_q} <= 24'b0;
        {s0_tap19_i,s0_tap19_q} <= 24'b0;
        {s0_tap20_i,s0_tap20_q} <= 24'b0;
        {s0_tap21_i,s0_tap21_q} <= 24'b0;
        {s0_tap22_i,s0_tap22_q} <= 24'b0;
        {s0_tap23_i,s0_tap23_q} <= 24'b0;
        {s0_tap24_i,s0_tap24_q} <= 24'b0;
        {s0_tap25_i,s0_tap25_q} <= 24'b0;
        {s0_tap26_i,s0_tap26_q} <= 24'b0;
        {s0_tap27_i,s0_tap27_q} <= 24'b0;
        {s0_tap28_i,s0_tap28_q} <= 24'b0;
        {s0_tap29_i,s0_tap29_q} <= 24'b0;
        {s0_tap30_i,s0_tap30_q} <= 24'b0;
      end
    end
  end 
  
  /*****************************************************************************
  * S1
  *****************************************************************************/
  /* wire */
  reg  [ 3:0] n_s1_primary,n_s1_secondary;
  wire [11:0] n_s1_mux0 ,n_s1_mux1 ,n_s1_mux2 ,n_s1_mux3,n_s1_mux4,n_s1_mux5;
  wire [11:0] n_s1_mux6 ,n_s1_mux7 ,n_s1_mux8 ,n_s1_mux9,n_s1_mux10,n_s1_mux11;
  wire [11:0] n_s1_mux12,n_s1_mux13,n_s1_mux14,n_s1_mux15,n_s1_mux16,n_s1_mux17;
  wire [11:0] n_s1_mux18,n_s1_mux19,n_s1_mux20,n_s1_mux21,n_s1_mux22,n_s1_mux23;
  wire [11:0] n_s1_mux24,n_s1_mux25,n_s1_mux26,n_s1_mux27,n_s1_mux28,n_s1_mux29;
  wire [11:0] n_s1_mux30;
  
  wire [12:0] n_s1_sum0_30,n_s1_sum1_29,n_s1_sum2_28,n_s1_sum3_27,n_s1_sum4_26;
  wire [12:0] n_s1_sum5_25,n_s1_sum6_24,n_s1_sum7_23,n_s1_sum8_22,n_s1_sum9_21;
  wire [12:0] n_s1_sum10_20,n_s1_sum11_19,n_s1_sum12_18,n_s1_sum13_17,n_s1_sum14_16; 
  wire [12:0] n_s1_sum15;  
    
  wire [22:0] n_s1_mul0_30,n_s1_mul1_29,n_s1_mul2_28,n_s1_mul3_27,n_s1_mul4_26;
  wire [22:0] n_s1_mul5_25,n_s1_mul6_24,n_s1_mul7_23,n_s1_mul8_22,n_s1_mul9_21;
  wire [22:0] n_s1_mul10_20,n_s1_mul11_19,n_s1_mul12_18,n_s1_mul13_17,n_s1_mul14_16; 
  wire [22:0] n_s1_mul15;  
  
  wire [23:0] n_s1_sum0,n_s1_sum1,n_s1_sum2,n_s1_sum3;
  /* flops */
  reg         s1_rxmode;
  reg         s1_cfsen;
  reg         s1_blocken;
  reg         s1_valid;
  reg         s1_cycle;
  reg  [ 3:0] s1_primary,s1_secondary;
  reg  [23:0] s1_sum0,s1_sum1,s1_sum2,s1_sum3;

  /* i/q arithmetic ressource sharing (12 bits [-2048,+2047]) */
  assign n_s1_mux0     = s0_even ?  s0_tap0_q :  s0_tap0_i;
  assign n_s1_mux2     = s0_even ?  s0_tap2_q :  s0_tap2_i;
  assign n_s1_mux4     = s0_even ?  s0_tap4_q :  s0_tap4_i;
  assign n_s1_mux6     = s0_even ?  s0_tap6_q :  s0_tap6_i;
  assign n_s1_mux8     = s0_even ?  s0_tap8_q :  s0_tap8_i;
  assign n_s1_mux10    = s0_even ? s0_tap10_q : s0_tap10_i;
  assign n_s1_mux12    = s0_even ? s0_tap12_q : s0_tap12_i;
  assign n_s1_mux14    = s0_even ? s0_tap14_q : s0_tap14_i;
  assign n_s1_mux16    = s0_even ? s0_tap16_q : s0_tap16_i;
  assign n_s1_mux18    = s0_even ? s0_tap18_q : s0_tap18_i;
  assign n_s1_mux20    = s0_even ? s0_tap20_q : s0_tap20_i;
  assign n_s1_mux22    = s0_even ? s0_tap22_q : s0_tap22_i;
  assign n_s1_mux24    = s0_even ? s0_tap24_q : s0_tap24_i;
  assign n_s1_mux26    = s0_even ? s0_tap26_q : s0_tap26_i;
  assign n_s1_mux28    = s0_even ? s0_tap28_q : s0_tap28_i;
  assign n_s1_mux30    = s0_even ? s0_tap30_q : s0_tap30_i;
   
  assign n_s1_mux1     =  s0_odd ?  s0_tap1_q :  s0_tap1_i;
  assign n_s1_mux3     =  s0_odd ?  s0_tap3_q :  s0_tap3_i;
  assign n_s1_mux5     =  s0_odd ?  s0_tap5_q :  s0_tap5_i;
  assign n_s1_mux7     =  s0_odd ?  s0_tap7_q :  s0_tap7_i;
  assign n_s1_mux9     =  s0_odd ?  s0_tap9_q :  s0_tap9_i;
  assign n_s1_mux11    =  s0_odd ? s0_tap11_q : s0_tap11_i;
  assign n_s1_mux13    =  s0_odd ? s0_tap13_q : s0_tap13_i;
  assign n_s1_mux15    =  s0_odd ? s0_tap15_q : s0_tap15_i;
  assign n_s1_mux17    =  s0_odd ? s0_tap17_q : s0_tap17_i;
  assign n_s1_mux19    =  s0_odd ? s0_tap19_q : s0_tap19_i;
  assign n_s1_mux21    =  s0_odd ? s0_tap21_q : s0_tap21_i;
  assign n_s1_mux23    =  s0_odd ? s0_tap23_q : s0_tap23_i;
  assign n_s1_mux25    =  s0_odd ? s0_tap25_q : s0_tap25_i;
  assign n_s1_mux27    =  s0_odd ? s0_tap27_q : s0_tap27_i;
  assign n_s1_mux29    =  s0_odd ? s0_tap29_q : s0_tap29_i;

  /* factoring coefficients by index parity (13 bits [-4096,+4095]) */
  assign n_s1_sum0_30  = { n_s1_mux0[11], n_s1_mux0} +  ({n_s1_mux30[11],n_s1_mux30}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum2_28  = { n_s1_mux2[11], n_s1_mux2} +  ({n_s1_mux28[11],n_s1_mux28}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum4_26  = { n_s1_mux4[11], n_s1_mux4} +  ({n_s1_mux26[11],n_s1_mux26}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum6_24  = { n_s1_mux6[11], n_s1_mux6} +  ({n_s1_mux24[11],n_s1_mux24}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum8_22  = { n_s1_mux8[11], n_s1_mux8} +  ({n_s1_mux22[11],n_s1_mux22}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum10_20 = {n_s1_mux10[11],n_s1_mux10} +  ({n_s1_mux20[11],n_s1_mux20}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum12_18 = {n_s1_mux12[11],n_s1_mux12} +  ({n_s1_mux18[11],n_s1_mux18}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  assign n_s1_sum14_16 = {n_s1_mux14[11],n_s1_mux14} +  ({n_s1_mux16[11],n_s1_mux16}^{13{s0_cfsen}}) + {12'b0,s0_cfsen};
  
  assign n_s1_sum1_29  = { n_s1_mux1[11], n_s1_mux1} + {n_s1_mux29[11],n_s1_mux29};
  assign n_s1_sum3_27  = { n_s1_mux3[11], n_s1_mux3} + {n_s1_mux27[11],n_s1_mux27};
  assign n_s1_sum5_25  = { n_s1_mux5[11], n_s1_mux5} + {n_s1_mux25[11],n_s1_mux25};
  assign n_s1_sum7_23  = { n_s1_mux7[11], n_s1_mux7} + {n_s1_mux23[11],n_s1_mux23};
  assign n_s1_sum9_21  = { n_s1_mux9[11], n_s1_mux9} + {n_s1_mux21[11],n_s1_mux21};
  assign n_s1_sum11_19 = {n_s1_mux11[11],n_s1_mux11} + {n_s1_mux19[11],n_s1_mux19};
  assign n_s1_sum13_17 = {n_s1_mux13[11],n_s1_mux13} + {n_s1_mux17[11],n_s1_mux17};
  assign n_s1_sum15    = {n_s1_mux15[11],n_s1_mux15};
  
  /* coefficient weighting */
  assign n_s1_mul0_30   =  $signed( H_0_30) * $signed(n_s1_sum0_30 );
  assign n_s1_mul1_29   =  $signed( H_1_29) * $signed(n_s1_sum1_29 );
  assign n_s1_mul2_28   =  $signed( H_2_28) * $signed(n_s1_sum2_28 );
  assign n_s1_mul3_27   =  $signed( H_3_27) * $signed(n_s1_sum3_27 );
  assign n_s1_mul4_26   =  $signed( H_4_26) * $signed(n_s1_sum4_26 );
  assign n_s1_mul5_25   =  $signed( H_5_25) * $signed(n_s1_sum5_25 );
  assign n_s1_mul6_24   =  $signed( H_6_24) * $signed(n_s1_sum6_24 );
  assign n_s1_mul7_23   =  $signed( H_7_23) * $signed(n_s1_sum7_23 );
  assign n_s1_mul8_22   =  $signed( H_8_22) * $signed(n_s1_sum8_22 );
  assign n_s1_mul9_21   =  $signed( H_9_21) * $signed(n_s1_sum9_21 );
  assign n_s1_mul10_20  =  $signed(H_10_20) * $signed(n_s1_sum10_20);
  assign n_s1_mul11_19  =  $signed(H_11_19) * $signed(n_s1_sum11_19);
  assign n_s1_mul12_18  =  $signed(H_12_18) * $signed(n_s1_sum12_18);
  assign n_s1_mul13_17  =  $signed(H_13_17) * $signed(n_s1_sum13_17);
  assign n_s1_mul14_16  =  $signed(H_14_16) * $signed(n_s1_sum14_16);
  assign n_s1_mul15     =  $signed(   H_15) * $signed(n_s1_sum15   );
  
  /* sum all weighted samples which have the same modulo-4 index */
  assign n_s1_sum0 = { n_s1_mul0_30[22], n_s1_mul0_30} +       
                     { n_s1_mul4_26[22], n_s1_mul4_26} +       
                     { n_s1_mul8_22[22], n_s1_mul8_22} +       
                     {n_s1_mul12_18[22],n_s1_mul12_18};        

  assign n_s1_sum1 = { n_s1_mul1_29[22], n_s1_mul1_29} +       
                     { n_s1_mul5_25[22], n_s1_mul5_25} +       
                     { n_s1_mul9_21[22], n_s1_mul9_21} +       
                     {n_s1_mul13_17[22],n_s1_mul13_17};        
 
  assign n_s1_sum2 = { n_s1_mul2_28[22], n_s1_mul2_28} +       
                     { n_s1_mul6_24[22], n_s1_mul6_24} +   
                     {n_s1_mul10_20[22],n_s1_mul10_20} +   
                     {n_s1_mul14_16[22],n_s1_mul14_16};    

  assign n_s1_sum3 = { n_s1_mul3_27[22], n_s1_mul3_27} +       
                     { n_s1_mul7_23[22], n_s1_mul7_23} +    
                     {n_s1_mul11_19[22],n_s1_mul11_19} +    
                     {   n_s1_mul15[22],   n_s1_mul15};     
  
  always @(*)
  begin
    if(s0_rxmode && s0_cfsen)
    begin
      if(!s0_cfs)
      begin
        /* shift right to center primary */
        case(s0_cycle)
          2'd0:   {n_s1_primary,n_s1_secondary} = {4'b1100,4'b0110};
          2'd1:   {n_s1_primary,n_s1_secondary} = {4'b0110,4'b1100}; 
          2'd2:   {n_s1_primary,n_s1_secondary} = {4'b0011,4'b1001}; 
          default:{n_s1_primary,n_s1_secondary} = {4'b1001,4'b0011};
        endcase
      end
      else
      begin
        /* shift left to center primary */
        case(s0_cycle)
          2'd0:   {n_s1_primary,n_s1_secondary} = {4'b0110,4'b1100};
          2'd1:   {n_s1_primary,n_s1_secondary} = {4'b1100,4'b0110}; 
          2'd2:   {n_s1_primary,n_s1_secondary} = {4'b1001,4'b0011}; 
          default:{n_s1_primary,n_s1_secondary} = {4'b0011,4'b1001};
        endcase
      end
    end
    else
    begin                
      n_s1_primary   = 4'b0000;
      n_s1_secondary = 4'b0000;
    end  
  end

  /* DC scaling 
  *
  * [-2048 2047]*1092 + 512 = [-2235904   2235836]
  * [-2^22 2^22-1] =  [-4194304   4194303] => 23 bits, not saturation required
  *
  */
  wire [11:0] n_s1_dc;
  wire [22:0] n_s1_dc_scaled;
  wire [22:0] n_s1_dc_fix;
  reg  [12:0] s1_dc_i;
  assign n_s1_dc        = s0_cycle[0]?s0_dc_i:s0_dc_q;
  assign n_s1_dc_scaled = $signed(12'd1092) * $signed(n_s1_dc);
  assign n_s1_dc_fix    = n_s1_dc_scaled + 23'd512;

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_rxmode    <= 1'b0;   
      s1_blocken   <= 1'b0;
      s1_cfsen     <= 1'b0;   
      s1_cycle     <= 1'b0;   
      s1_valid     <= 1'b0;   
      s1_primary   <= 4'b0;    
      s1_secondary <= 4'b0;    
      s1_sum0      <= 24'b0; 
      s1_sum1      <= 24'b0; 
      s1_sum2      <= 24'b0; 
      s1_sum3      <= 24'b0; 
      s1_dc_i      <= 13'd0;  
    end
    else if(!s0_blocken)
    begin
      s1_rxmode    <= 1'b0;    
      s1_blocken   <= 1'b0;
      s1_cfsen     <= 1'b0;    
      s1_cycle     <= 1'b0;    
      s1_valid     <= 1'b0;    
      s1_primary   <= 4'b0;    
      s1_secondary <= 4'b0;    
      s1_sum0      <= 24'b0;  
      s1_sum1      <= 24'b0;  
      s1_sum2      <= 24'b0;  
      s1_sum3      <= 24'b0;  
      s1_dc_i      <= 13'd0;  
    end
    else
    begin
      s1_rxmode    <= s0_rxmode;      
      s1_blocken   <= s0_blocken;
      s1_cfsen     <= s0_cfsen;       
      s1_cycle     <= s0_cycle[0];       
      s1_valid     <= s0_valid!=31'd0;      
      s1_primary   <= n_s1_primary;       
      s1_secondary <= n_s1_secondary;       
      s1_sum0      <= n_s1_sum0;    
      s1_sum1      <= n_s1_sum1;    
      s1_sum2      <= n_s1_sum2;    
      s1_sum3      <= n_s1_sum3;    
      if(s0_cycle[0])
        s1_dc_i <= n_s1_dc_fix[22:10];
    end
  end
  
  /*****************************************************************************
  * S2
  *****************************************************************************/
  /* wires */
  wire [23:0] n_s2_sum_pri,n_s2_sum_sec;
  wire [12:0] n_s2_sum_pri_rnd,n_s2_sum_sec_rnd;
  wire [23:0] n_s2_sum_even,n_s2_sum_odd;
  wire [13:0] n_s2_sum_even_trc,n_s2_sum_odd_trc;
  reg  [11:0] n_s2_sum_even_sat,n_s2_sum_odd_sat;
  /* flops */
  reg  [12:0] s2_hold_pri_i,s2_hold_sec_i;
  reg  [12:0] s2_pri_i,s2_pri_q;
  reg  [12:0] s2_sec_i,s2_sec_q;
  reg  [ 1:0] s2_rx_valid;
  reg         s2_tx_valid;
  reg  [ 1:0] s2_hold_valid;
  reg  [11:0] s2_tx_i,s2_tx_q;
  reg  [11:0] s2_hold_0;
  reg  [12:0] s2_dc_i,s2_dc_q;
  
  /* RX */
  assign n_s2_sum_pri = 
    (s1_sum0 ^ {24{s1_primary[0]}})  +
    (s1_sum1 ^ {24{s1_primary[1]}})  +                                      
    (s1_sum2 ^ {24{s1_primary[2]}})  +                                      
    (s1_sum3 ^ {24{s1_primary[3]}})  +                                      
    {13'b0,1'b1,8'b0,s1_cfsen,1'b0}; // rndfix + c2fix                      

  assign n_s2_sum_sec = 
    (s1_sum0 ^ {24{s1_secondary[0]}}) +
    (s1_sum1 ^ {24{s1_secondary[1]}}) +                            
    (s1_sum2 ^ {24{s1_secondary[2]}}) +                            
    (s1_sum3 ^ {24{s1_secondary[3]}}) +                            
    {13'b0,1'b1,8'b0,s1_cfsen,1'b0}; // rndfix + c2fix                    

  /* rnd 11 */
  assign n_s2_sum_pri_rnd = n_s2_sum_pri[23:11];
  assign n_s2_sum_sec_rnd = n_s2_sum_sec[23:11]; 

  /* TX */
  assign n_s2_sum_even     = s1_sum0 + s1_sum2;                      
  assign n_s2_sum_odd      = s1_sum1 + s1_sum3;      

  /* trunc 10 */
  assign n_s2_sum_even_trc = n_s2_sum_even[23:10];
  assign n_s2_sum_odd_trc  = n_s2_sum_odd[23:10];
  
  /* sat 12 */
  always @(*)
  begin
    if((n_s2_sum_even_trc[13:11]==3'b000) || (n_s2_sum_even_trc[13:11]==3'b111))
       n_s2_sum_even_sat = n_s2_sum_even_trc[11:0];
     else
       n_s2_sum_even_sat = {n_s2_sum_even_trc[13],{11{~n_s2_sum_even_trc[13]}}};
  end  
  
  always @(*)
  begin
    if((n_s2_sum_odd_trc[13:11]==3'b000) || (n_s2_sum_odd_trc[13:11]==3'b111))
       n_s2_sum_odd_sat = n_s2_sum_odd_trc[11:0];
     else
       n_s2_sum_odd_sat = {n_s2_sum_odd_trc[13],{11{~n_s2_sum_odd_trc[13]}}};
  end
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_hold_valid <= 2'b0;
      s2_rx_valid   <= 2'b0;
      s2_tx_valid   <= 1'b0; 
      
      s2_hold_pri_i <= 13'b0;
      s2_hold_sec_i <= 13'b0;
      s2_pri_i      <= 13'b0;
      s2_pri_q      <= 13'b0;
      s2_sec_i      <= 13'b0;
      s2_sec_q      <= 13'b0;
      s2_dc_i       <= 13'b0;
      s2_dc_q       <= 13'b0;
      s2_hold_0     <= 12'b0;
      s2_tx_i       <= 12'b0; 
      s2_tx_q       <= 12'b0; 
    end                    
    else if(!s1_blocken)
    begin
      s2_hold_valid <= 2'b0;
      s2_rx_valid   <= 2'b0;
      s2_tx_valid   <= 1'b0; 
      
      s2_hold_pri_i <= 13'b0;
      s2_hold_sec_i <= 13'b0;
      s2_pri_i      <= 13'b0;
      s2_pri_q      <= 13'b0;
      s2_sec_i      <= 13'b0;
      s2_sec_q      <= 13'b0;
      s2_dc_i       <= 13'b0;
      s2_dc_q       <= 13'b0;
      s2_hold_0     <= 12'b0;
      s2_tx_i       <= 12'b0; 
      s2_tx_q       <= 12'b0;
       
    end
    else
    begin
      if(s1_rxmode)                                                 
      begin
        s2_tx_valid <= 1'b0;
        s2_tx_i     <= 12'b0;
        s2_tx_q     <= 12'b0;
        
        s2_rx_valid <= {1'b0,s2_rx_valid[1]};
        
        if(s1_valid)
        begin
          if(!s1_cycle)
          begin
            s2_hold_pri_i <= n_s2_sum_pri_rnd;
            s2_hold_sec_i <= n_s2_sum_sec_rnd;
          end
          else
          begin
            s2_rx_valid <= 2'b11;
            s2_pri_i    <= s2_hold_pri_i;
            s2_pri_q    <= n_s2_sum_pri_rnd;
            if(s1_cfsen)
            begin
              s2_sec_i <= s2_hold_sec_i;
              s2_sec_q <= n_s2_sum_sec_rnd;
            end
            else
            begin
              s2_sec_i <= 13'b0;
              s2_sec_q <= 13'b0;
            end
          end
          
          if(s1_cycle)
          begin
            s2_dc_i     <= s1_dc_i;
            s2_dc_q     <= n_s1_dc_fix[22:10];
          end
        end
      end                                                      
      else                                                     
      begin
        s2_pri_i    <= 13'b0;  
        s2_pri_q    <= 13'b0;  
        s2_sec_i    <= 13'b0;  
        s2_sec_q    <= 13'b0;  
        s2_dc_i     <= 13'b0;
        s2_dc_q     <= 13'b0;
        s2_rx_valid <= 2'b0;   
      
        if(s1_valid && !s1_cycle)
          s2_hold_valid <= 2'b11;
        else
          s2_hold_valid <= {1'b0,s2_hold_valid[1]};
        
        if(s2_hold_valid==2'b11)
        begin
          s2_hold_0   <= n_s2_sum_odd_sat;
          s2_tx_valid <= 1'b1;
          s2_tx_i     <= s2_hold_0;
          s2_tx_q     <= n_s2_sum_even_sat;
        end
        else if(s2_hold_valid==2'b01)
        begin
          s2_hold_0   <= n_s2_sum_even_sat;
          s2_tx_valid <= 1'b1;
          s2_tx_i     <= s2_hold_0;
          s2_tx_q     <= n_s2_sum_odd_sat;
        end  
        else
        begin
          s2_hold_0   <= n_s2_sum_even_sat;
          s2_tx_valid <= 1'b0;
          s2_tx_i     <= 12'b0;
          s2_tx_q     <= 12'b0;
        end
      end                                                      
    end
  end
 
  assign out_rx_valid = s2_rx_valid[0];
  assign out_rx_pri_i = s2_pri_i;
  assign out_rx_pri_q = s2_pri_q;
  assign out_rx_sec_i = s2_sec_i;
  assign out_rx_sec_q = s2_sec_q;
  assign out_rx_dc_i  = s2_dc_i;
  assign out_rx_dc_q  = s2_dc_q;

  assign out_tx_valid = s2_tx_valid;
  assign out_tx_i     = s2_tx_i;
  assign out_tx_q     = s2_tx_q;
  
endmodule
`default_nettype wire
