//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of hbf80
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module hbf80
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        rst_n,
  input  wire        clk,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire        blocken,    // global enable
  input  wire        rxmode,     // 1=RX; 0=TX
  
  /*****************************************************************************
  * RX
  *****************************************************************************/
  input  wire        in_rx_valid,
  input  wire [11:0] in_rx_i,
  input  wire [11:0] in_rx_q,
  input  wire [11:0] in_rx_dc_i,
  input  wire [11:0] in_rx_dc_q,
  
  output wire        out_rx_valid,
  output wire [12:0] out_rx_i,
  output wire [12:0] out_rx_q,
  output wire [12:0] out_rx_dc_i,
  output wire [12:0] out_rx_dc_q,
  
  /*****************************************************************************
  * TX
  *****************************************************************************/
  input  wire        in_tx_valid, 
  input  wire [11:0] in_tx_i,
  input  wire [11:0] in_tx_q,
  
  output wire        out_tx_valid,
  output wire [11:0] out_tx_i,
  output wire [11:0] out_tx_q
);

  /*****************************************************************************
    impulse response :
      h80=[-5 -14 -8 11 12 -9 -15 7 18 -6 -21 6 24 -5 -29 5 36 -4 -46 4 65 -4 \
           -109 4 328 512 328 4 -109 -4 65 4 -46 -4 36 5 -29 -5 24 6 -21 -6 18 \
           7 -15 -9 12 11 -8 -14 -5];
    
    maximum range:
      sum(abs(h80))*2048
      > 4304896
    
    which fits 24 bits range :
      [-2^24/2 2^24/2-1]
      > -8388608   8388607
  
  *****************************************************************************/

  /*****************************************************************************
  * S0
  *****************************************************************************/
  /* wire */
  wire        in_valid;
  reg  [11:0] n_s0_hold_i,n_s0_hold_q;
  
  /* flops */
  reg         in_cycle;
  reg         s0_rxmode;
  reg         s0_blocken;
  reg  [50:0] s0_valid;
  reg         s0_even,s0_odd;
  reg         s0_cycle;
  reg  [11:0] s0_hold_i,s0_hold_q;
  reg  [11:0] s0_dc_i,s0_dc_q;
  reg  [11:0] s0_tap0_i,s0_tap1_i,s0_tap2_i,s0_tap3_i,s0_tap4_i,s0_tap5_i;
  reg  [11:0] s0_tap6_i,s0_tap7_i,s0_tap8_i,s0_tap9_i,s0_tap10_i,s0_tap11_i;
  reg  [11:0] s0_tap12_i,s0_tap13_i,s0_tap14_i,s0_tap15_i,s0_tap16_i,s0_tap17_i; 
  reg  [11:0] s0_tap18_i,s0_tap19_i,s0_tap20_i,s0_tap21_i,s0_tap22_i,s0_tap23_i;
  reg  [11:0] s0_tap24_i,s0_tap25_i,s0_tap26_i,s0_tap27_i,s0_tap28_i,s0_tap29_i;
  reg  [11:0] s0_tap30_i,s0_tap31_i,s0_tap32_i,s0_tap33_i,s0_tap34_i,s0_tap35_i;
  reg  [11:0] s0_tap36_i,s0_tap37_i,s0_tap38_i,s0_tap39_i,s0_tap40_i,s0_tap41_i;
  reg  [11:0] s0_tap42_i,s0_tap43_i,s0_tap44_i,s0_tap45_i,s0_tap46_i,s0_tap47_i; 
  reg  [11:0] s0_tap48_i,s0_tap49_i,s0_tap50_i;  
  reg  [11:0] s0_tap0_q,s0_tap1_q,s0_tap2_q,s0_tap3_q,s0_tap4_q,s0_tap5_q;
  reg  [11:0] s0_tap6_q,s0_tap7_q,s0_tap8_q,s0_tap9_q,s0_tap10_q,s0_tap11_q;
  reg  [11:0] s0_tap12_q,s0_tap13_q,s0_tap14_q,s0_tap15_q,s0_tap16_q,s0_tap17_q; 
  reg  [11:0] s0_tap18_q,s0_tap19_q,s0_tap20_q,s0_tap21_q,s0_tap22_q,s0_tap23_q;
  reg  [11:0] s0_tap24_q,s0_tap25_q,s0_tap26_q,s0_tap27_q,s0_tap28_q,s0_tap29_q;
  reg  [11:0] s0_tap30_q,s0_tap31_q,s0_tap32_q,s0_tap33_q,s0_tap34_q,s0_tap35_q;
  reg  [11:0] s0_tap36_q,s0_tap37_q,s0_tap38_q,s0_tap39_q,s0_tap40_q,s0_tap41_q;
  reg  [11:0] s0_tap42_q,s0_tap43_q,s0_tap44_q,s0_tap45_q,s0_tap46_q,s0_tap47_q; 
  reg  [11:0] s0_tap48_q,s0_tap49_q,s0_tap50_q;  

  assign in_valid = rxmode?in_rx_valid:in_tx_valid;
  
  always @(*)
  begin
    if(rxmode)
    begin
      if(in_rx_valid)
      begin
        n_s0_hold_i = in_rx_i;
        n_s0_hold_q = in_rx_q;
      end
      else
      begin
        n_s0_hold_i = 12'b0;
        n_s0_hold_q = 12'b0;
      end
    end
    else
    begin
      if(!in_cycle)               
      begin                       
        if(in_tx_valid)           
        begin                     
          n_s0_hold_i = in_tx_i;  
          n_s0_hold_q = in_tx_q;
        end
        else
        begin
          n_s0_hold_i = 12'b0;
          n_s0_hold_q = 12'b0;
        end  
      end                         
      else                        
      begin                       
        n_s0_hold_i = s0_hold_i;  
        n_s0_hold_q = s0_hold_q;  
      end                         
    end
  end
  
  /* 51 tap */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      in_cycle  <= 1'b0;
      s0_rxmode <= 1'b0;
      s0_blocken<= 1'b0;
      s0_cycle  <= 1'b0;
      s0_valid  <= 51'b0;
      s0_odd    <= 1'b1;
      s0_even   <= 1'b1;
      
      s0_hold_i               <= 12'b0;
      s0_hold_q               <= 12'b0;
      s0_dc_i                 <= 12'b0;
      s0_dc_q                 <= 12'b0;
      { s0_tap0_i, s0_tap0_q} <= 24'b0;
      { s0_tap1_i, s0_tap1_q} <= 24'b0;
      { s0_tap2_i, s0_tap2_q} <= 24'b0;
      { s0_tap3_i, s0_tap3_q} <= 24'b0;
      { s0_tap4_i, s0_tap4_q} <= 24'b0;
      { s0_tap5_i, s0_tap5_q} <= 24'b0;
      { s0_tap6_i, s0_tap6_q} <= 24'b0;
      { s0_tap7_i, s0_tap7_q} <= 24'b0;
      { s0_tap8_i, s0_tap8_q} <= 24'b0;
      { s0_tap9_i, s0_tap9_q} <= 24'b0;
      {s0_tap10_i,s0_tap10_q} <= 24'b0;
      {s0_tap11_i,s0_tap11_q} <= 24'b0;
      {s0_tap12_i,s0_tap12_q} <= 24'b0;
      {s0_tap13_i,s0_tap13_q} <= 24'b0;
      {s0_tap14_i,s0_tap14_q} <= 24'b0;
      {s0_tap15_i,s0_tap15_q} <= 24'b0;
      {s0_tap16_i,s0_tap16_q} <= 24'b0;
      {s0_tap17_i,s0_tap17_q} <= 24'b0;
      {s0_tap18_i,s0_tap18_q} <= 24'b0;
      {s0_tap19_i,s0_tap19_q} <= 24'b0;
      {s0_tap20_i,s0_tap20_q} <= 24'b0;
      {s0_tap21_i,s0_tap21_q} <= 24'b0;
      {s0_tap22_i,s0_tap22_q} <= 24'b0;
      {s0_tap23_i,s0_tap23_q} <= 24'b0;
      {s0_tap24_i,s0_tap24_q} <= 24'b0;
      {s0_tap25_i,s0_tap25_q} <= 24'b0;
      {s0_tap26_i,s0_tap26_q} <= 24'b0;
      {s0_tap27_i,s0_tap27_q} <= 24'b0;
      {s0_tap28_i,s0_tap28_q} <= 24'b0;
      {s0_tap29_i,s0_tap29_q} <= 24'b0;
      {s0_tap30_i,s0_tap30_q} <= 24'b0;
      {s0_tap31_i,s0_tap31_q} <= 24'b0;
      {s0_tap32_i,s0_tap32_q} <= 24'b0;
      {s0_tap33_i,s0_tap33_q} <= 24'b0;
      {s0_tap34_i,s0_tap34_q} <= 24'b0;
      {s0_tap35_i,s0_tap35_q} <= 24'b0;
      {s0_tap36_i,s0_tap36_q} <= 24'b0;
      {s0_tap37_i,s0_tap37_q} <= 24'b0;
      {s0_tap38_i,s0_tap38_q} <= 24'b0;
      {s0_tap39_i,s0_tap39_q} <= 24'b0;
      {s0_tap40_i,s0_tap40_q} <= 24'b0;
      {s0_tap41_i,s0_tap41_q} <= 24'b0;
      {s0_tap42_i,s0_tap42_q} <= 24'b0;
      {s0_tap43_i,s0_tap43_q} <= 24'b0;
      {s0_tap44_i,s0_tap44_q} <= 24'b0;
      {s0_tap45_i,s0_tap45_q} <= 24'b0;
      {s0_tap46_i,s0_tap46_q} <= 24'b0;
      {s0_tap47_i,s0_tap47_q} <= 24'b0;
      {s0_tap48_i,s0_tap48_q} <= 24'b0;
      {s0_tap49_i,s0_tap49_q} <= 24'b0;
      {s0_tap50_i,s0_tap50_q} <= 24'b0;
    end
    else
    begin
      if(blocken)
      begin
        /* param */
        s0_rxmode  <= rxmode;
        s0_blocken <= 1'b1;
      
        /* valid tag */                                   
        if(in_valid)
          in_cycle <= ~in_cycle;
        else
          in_cycle <= 1'b0;
          
        s0_valid <= {s0_valid[49:0], in_valid};
      
        /* datapath */
        if(s0_valid!=51'b0)                
        begin                              
          s0_cycle <= ~s0_cycle;  
          if(rxmode)                       
          begin                            
            s0_even <= ~s0_cycle;          
            s0_odd  <= ~s0_cycle;          
          end                              
          else                             
          begin                            
            s0_even <= ~s0_cycle;          
            s0_odd  <=  s0_cycle;          
          end                              
        end                                
        else                               
        begin                              
          s0_cycle <= 1'd0;                                  
          s0_even  <= 1'b0;                
          s0_odd   <= 1'b0;                
        end 
        
        s0_hold_i               <= n_s0_hold_i;                                     
        s0_hold_q               <= n_s0_hold_q;                                     
        { s0_tap0_i, s0_tap0_q} <= {n_s0_hold_i, s0_hold_q};
        
        if(rxmode)
        begin
          if(!s0_cycle)
          begin
            s0_dc_i <= in_rx_dc_i;
            s0_dc_q <= in_rx_dc_q;                 
          end
        end
        else
        begin
          s0_dc_i <= 12'd0;
          s0_dc_q <= 12'd0;                 
        end
      end
      else
      begin
        s0_rxmode  <= 1'b0;
        s0_blocken <= 1'b0;
        in_cycle   <= 1'b0;
        s0_valid   <= 51'd0;
        s0_cycle   <= 1'd0;                                
        s0_even    <= 1'b0;              
        s0_odd     <= 1'b0;              

        s0_hold_i               <= 12'b0;
        s0_hold_q               <= 12'b0;
        s0_dc_i                 <= 12'b0;
        s0_dc_q                 <= 12'b0;
        { s0_tap0_i, s0_tap0_q} <= 24'b0;
      end
       
      if(s0_blocken)
      begin
        { s0_tap1_i, s0_tap1_q} <= { s0_tap0_i, s0_tap0_q};               
        { s0_tap2_i, s0_tap2_q} <= { s0_tap1_i, s0_tap1_q};               
        { s0_tap3_i, s0_tap3_q} <= { s0_tap2_i, s0_tap2_q};               
        { s0_tap4_i, s0_tap4_q} <= { s0_tap3_i, s0_tap3_q};               
        { s0_tap5_i, s0_tap5_q} <= { s0_tap4_i, s0_tap4_q};               
        { s0_tap6_i, s0_tap6_q} <= { s0_tap5_i, s0_tap5_q};               
        { s0_tap7_i, s0_tap7_q} <= { s0_tap6_i, s0_tap6_q};               
        { s0_tap8_i, s0_tap8_q} <= { s0_tap7_i, s0_tap7_q};               
        { s0_tap9_i, s0_tap9_q} <= { s0_tap8_i, s0_tap8_q};               
        {s0_tap10_i,s0_tap10_q} <= { s0_tap9_i, s0_tap9_q};               
        {s0_tap11_i,s0_tap11_q} <= {s0_tap10_i,s0_tap10_q};               
        {s0_tap12_i,s0_tap12_q} <= {s0_tap11_i,s0_tap11_q};               
        {s0_tap13_i,s0_tap13_q} <= {s0_tap12_i,s0_tap12_q};               
        {s0_tap14_i,s0_tap14_q} <= {s0_tap13_i,s0_tap13_q};               
        {s0_tap15_i,s0_tap15_q} <= {s0_tap14_i,s0_tap14_q};               
        {s0_tap16_i,s0_tap16_q} <= {s0_tap15_i,s0_tap15_q};               
        {s0_tap17_i,s0_tap17_q} <= {s0_tap16_i,s0_tap16_q};               
        {s0_tap18_i,s0_tap18_q} <= {s0_tap17_i,s0_tap17_q};               
        {s0_tap19_i,s0_tap19_q} <= {s0_tap18_i,s0_tap18_q};               
        {s0_tap20_i,s0_tap20_q} <= {s0_tap19_i,s0_tap19_q};               
        {s0_tap21_i,s0_tap21_q} <= {s0_tap20_i,s0_tap20_q};            
        {s0_tap22_i,s0_tap22_q} <= {s0_tap21_i,s0_tap21_q};            
        {s0_tap23_i,s0_tap23_q} <= {s0_tap22_i,s0_tap22_q};            
        {s0_tap24_i,s0_tap24_q} <= {s0_tap23_i,s0_tap23_q};            
        {s0_tap25_i,s0_tap25_q} <= {s0_tap24_i,s0_tap24_q};            
        {s0_tap26_i,s0_tap26_q} <= {s0_tap25_i,s0_tap25_q};            
        {s0_tap27_i,s0_tap27_q} <= {s0_tap26_i,s0_tap26_q};            
        {s0_tap28_i,s0_tap28_q} <= {s0_tap27_i,s0_tap27_q};            
        {s0_tap29_i,s0_tap29_q} <= {s0_tap28_i,s0_tap28_q};            
        {s0_tap30_i,s0_tap30_q} <= {s0_tap29_i,s0_tap29_q};            
        {s0_tap31_i,s0_tap31_q} <= {s0_tap30_i,s0_tap30_q};            
        {s0_tap32_i,s0_tap32_q} <= {s0_tap31_i,s0_tap31_q};            
        {s0_tap33_i,s0_tap33_q} <= {s0_tap32_i,s0_tap32_q};            
        {s0_tap34_i,s0_tap34_q} <= {s0_tap33_i,s0_tap33_q};            
        {s0_tap35_i,s0_tap35_q} <= {s0_tap34_i,s0_tap34_q};            
        {s0_tap36_i,s0_tap36_q} <= {s0_tap35_i,s0_tap35_q};            
        {s0_tap37_i,s0_tap37_q} <= {s0_tap36_i,s0_tap36_q};            
        {s0_tap38_i,s0_tap38_q} <= {s0_tap37_i,s0_tap37_q};            
        {s0_tap39_i,s0_tap39_q} <= {s0_tap38_i,s0_tap38_q};            
        {s0_tap40_i,s0_tap40_q} <= {s0_tap39_i,s0_tap39_q};            
        {s0_tap41_i,s0_tap41_q} <= {s0_tap40_i,s0_tap40_q};            
        {s0_tap42_i,s0_tap42_q} <= {s0_tap41_i,s0_tap41_q};            
        {s0_tap43_i,s0_tap43_q} <= {s0_tap42_i,s0_tap42_q};            
        {s0_tap44_i,s0_tap44_q} <= {s0_tap43_i,s0_tap43_q};            
        {s0_tap45_i,s0_tap45_q} <= {s0_tap44_i,s0_tap44_q};            
        {s0_tap46_i,s0_tap46_q} <= {s0_tap45_i,s0_tap45_q};            
        {s0_tap47_i,s0_tap47_q} <= {s0_tap46_i,s0_tap46_q};            
        {s0_tap48_i,s0_tap48_q} <= {s0_tap47_i,s0_tap47_q};            
        {s0_tap49_i,s0_tap49_q} <= {s0_tap48_i,s0_tap48_q};            
        {s0_tap50_i,s0_tap50_q} <= {s0_tap49_i,s0_tap49_q};
      end
      else
      begin
        { s0_tap1_i, s0_tap1_q} <= 24'b0;
        { s0_tap2_i, s0_tap2_q} <= 24'b0;
        { s0_tap3_i, s0_tap3_q} <= 24'b0;
        { s0_tap4_i, s0_tap4_q} <= 24'b0;
        { s0_tap5_i, s0_tap5_q} <= 24'b0;
        { s0_tap6_i, s0_tap6_q} <= 24'b0;
        { s0_tap7_i, s0_tap7_q} <= 24'b0;
        { s0_tap8_i, s0_tap8_q} <= 24'b0;
        { s0_tap9_i, s0_tap9_q} <= 24'b0;
        {s0_tap10_i,s0_tap10_q} <= 24'b0;
        {s0_tap11_i,s0_tap11_q} <= 24'b0;
        {s0_tap12_i,s0_tap12_q} <= 24'b0;
        {s0_tap13_i,s0_tap13_q} <= 24'b0;
        {s0_tap14_i,s0_tap14_q} <= 24'b0;
        {s0_tap15_i,s0_tap15_q} <= 24'b0;
        {s0_tap16_i,s0_tap16_q} <= 24'b0;
        {s0_tap17_i,s0_tap17_q} <= 24'b0;
        {s0_tap18_i,s0_tap18_q} <= 24'b0;
        {s0_tap19_i,s0_tap19_q} <= 24'b0;
        {s0_tap20_i,s0_tap20_q} <= 24'b0;
        {s0_tap21_i,s0_tap21_q} <= 24'b0;
        {s0_tap22_i,s0_tap22_q} <= 24'b0;
        {s0_tap23_i,s0_tap23_q} <= 24'b0;
        {s0_tap24_i,s0_tap24_q} <= 24'b0;
        {s0_tap25_i,s0_tap25_q} <= 24'b0;
        {s0_tap26_i,s0_tap26_q} <= 24'b0;
        {s0_tap27_i,s0_tap27_q} <= 24'b0;
        {s0_tap28_i,s0_tap28_q} <= 24'b0;
        {s0_tap29_i,s0_tap29_q} <= 24'b0;
        {s0_tap30_i,s0_tap30_q} <= 24'b0;
        {s0_tap31_i,s0_tap31_q} <= 24'b0;
        {s0_tap32_i,s0_tap32_q} <= 24'b0;
        {s0_tap33_i,s0_tap33_q} <= 24'b0;
        {s0_tap34_i,s0_tap34_q} <= 24'b0;
        {s0_tap35_i,s0_tap35_q} <= 24'b0;
        {s0_tap36_i,s0_tap36_q} <= 24'b0;
        {s0_tap37_i,s0_tap37_q} <= 24'b0;
        {s0_tap38_i,s0_tap38_q} <= 24'b0;
        {s0_tap39_i,s0_tap39_q} <= 24'b0;
        {s0_tap40_i,s0_tap40_q} <= 24'b0;
        {s0_tap41_i,s0_tap41_q} <= 24'b0;
        {s0_tap42_i,s0_tap42_q} <= 24'b0;
        {s0_tap43_i,s0_tap43_q} <= 24'b0;
        {s0_tap44_i,s0_tap44_q} <= 24'b0;
        {s0_tap45_i,s0_tap45_q} <= 24'b0;
        {s0_tap46_i,s0_tap46_q} <= 24'b0;
        {s0_tap47_i,s0_tap47_q} <= 24'b0;
        {s0_tap48_i,s0_tap48_q} <= 24'b0;
        {s0_tap49_i,s0_tap49_q} <= 24'b0;
        {s0_tap50_i,s0_tap50_q} <= 24'b0;
      end            
    end
  end

  /*****************************************************************************
  * S1
  *****************************************************************************/
  /* wire */
  wire [11:0] n_s1_mux0 ,n_s1_mux1 , n_s1_mux2, n_s1_mux3, n_s1_mux4, n_s1_mux5;
  wire [11:0] n_s1_mux6 ,n_s1_mux7 , n_s1_mux8, n_s1_mux9,n_s1_mux10,n_s1_mux11;
  wire [11:0] n_s1_mux12,n_s1_mux13,n_s1_mux14,n_s1_mux15,n_s1_mux16,n_s1_mux17;
  wire [11:0] n_s1_mux18,n_s1_mux19,n_s1_mux20,n_s1_mux21,n_s1_mux22,n_s1_mux23;
  wire [11:0] n_s1_mux24,n_s1_mux25,n_s1_mux26,n_s1_mux27,n_s1_mux28,n_s1_mux29;
  wire [11:0] n_s1_mux30,n_s1_mux31,n_s1_mux32,n_s1_mux33,n_s1_mux34,n_s1_mux35;
  wire [11:0] n_s1_mux36,n_s1_mux37,n_s1_mux38,n_s1_mux39,n_s1_mux40,n_s1_mux41;
  wire [11:0] n_s1_mux42,n_s1_mux43,n_s1_mux44,n_s1_mux45,n_s1_mux46,n_s1_mux47;
  wire [11:0] n_s1_mux48,n_s1_mux49,n_s1_mux50;
  
  wire [12:0] n_s1_sum0_50,n_s1_sum1_49,n_s1_sum2_48,n_s1_sum3_47,n_s1_sum4_46;
  wire [12:0] n_s1_sum5_45,n_s1_sum6_44,n_s1_sum7_43,n_s1_sum8_42,n_s1_sum9_41;
  wire [12:0] n_s1_sum10_40,n_s1_sum11_39,n_s1_sum12_38,n_s1_sum13_37,n_s1_sum14_36; 
  wire [12:0] n_s1_sum15_35,n_s1_sum16_34,n_s1_sum17_33,n_s1_sum18_32,n_s1_sum19_31;  
  wire [12:0] n_s1_sum20_30,n_s1_sum21_29,n_s1_sum22_28,n_s1_sum23_27,n_s1_sum24_26;  
  wire [12:0] n_s1_sum25;  
   
  wire [21:0] n_s1_mul0_50,n_s1_mul1_49,n_s1_mul2_48,n_s1_mul3_47,n_s1_mul4_46;
  wire [21:0] n_s1_mul5_45,n_s1_mul6_44,n_s1_mul7_43,n_s1_mul8_42,n_s1_mul9_41;
  wire [21:0] n_s1_mul10_40,n_s1_mul11_39,n_s1_mul12_38,n_s1_mul13_37,n_s1_mul14_36; 
  wire [21:0] n_s1_mul15_35,n_s1_mul16_34,n_s1_mul17_33,n_s1_mul18_32,n_s1_mul19_31;  
  wire [21:0] n_s1_mul20_30,n_s1_mul21_29,n_s1_mul22_28,n_s1_mul23_27,n_s1_mul24_26;  
  wire [21:0] n_s1_mul25;  

  /* flops */
  reg         s1_rxmode;
  reg         s1_valid;
  reg         s1_cycle;
  reg         s1_blocken;
   
  reg  [21:0] s1_mul0_50,s1_mul1_49,s1_mul2_48,s1_mul3_47,s1_mul4_46;
  reg  [21:0] s1_mul5_45,s1_mul6_44,s1_mul7_43,s1_mul8_42,s1_mul9_41;
  reg  [21:0] s1_mul10_40,s1_mul11_39,s1_mul12_38,s1_mul13_37,s1_mul14_36; 
  reg  [21:0] s1_mul15_35,s1_mul16_34,s1_mul17_33,s1_mul18_32,s1_mul19_31;  
  reg  [21:0] s1_mul20_30,s1_mul21_29,s1_mul22_28,s1_mul23_27,s1_mul24_26;  
  reg  [21:0] s1_mul25;  

  /* i/q arithmetic ressource sharing (12 bits [-2048,+2047]) */
  assign n_s1_mux0     = s0_even ?  s0_tap0_q :  s0_tap0_i;
  assign n_s1_mux2     = s0_even ?  s0_tap2_q :  s0_tap2_i;
  assign n_s1_mux4     = s0_even ?  s0_tap4_q :  s0_tap4_i;
  assign n_s1_mux6     = s0_even ?  s0_tap6_q :  s0_tap6_i;
  assign n_s1_mux8     = s0_even ?  s0_tap8_q :  s0_tap8_i;
  assign n_s1_mux10    = s0_even ? s0_tap10_q : s0_tap10_i;
  assign n_s1_mux12    = s0_even ? s0_tap12_q : s0_tap12_i;
  assign n_s1_mux14    = s0_even ? s0_tap14_q : s0_tap14_i;
  assign n_s1_mux16    = s0_even ? s0_tap16_q : s0_tap16_i;
  assign n_s1_mux18    = s0_even ? s0_tap18_q : s0_tap18_i;
  assign n_s1_mux20    = s0_even ? s0_tap20_q : s0_tap20_i;
  assign n_s1_mux22    = s0_even ? s0_tap22_q : s0_tap22_i;
  assign n_s1_mux24    = s0_even ? s0_tap24_q : s0_tap24_i;
  assign n_s1_mux26    = s0_even ? s0_tap26_q : s0_tap26_i;
  assign n_s1_mux28    = s0_even ? s0_tap28_q : s0_tap28_i;
  assign n_s1_mux30    = s0_even ? s0_tap30_q : s0_tap30_i;
  assign n_s1_mux32    = s0_even ? s0_tap32_q : s0_tap32_i;
  assign n_s1_mux34    = s0_even ? s0_tap34_q : s0_tap34_i;
  assign n_s1_mux36    = s0_even ? s0_tap36_q : s0_tap36_i;
  assign n_s1_mux38    = s0_even ? s0_tap38_q : s0_tap38_i;
  assign n_s1_mux40    = s0_even ? s0_tap40_q : s0_tap40_i;
  assign n_s1_mux42    = s0_even ? s0_tap42_q : s0_tap42_i;
  assign n_s1_mux44    = s0_even ? s0_tap44_q : s0_tap44_i;
  assign n_s1_mux46    = s0_even ? s0_tap46_q : s0_tap46_i;
  assign n_s1_mux48    = s0_even ? s0_tap48_q : s0_tap48_i;
  assign n_s1_mux50    = s0_even ? s0_tap50_q : s0_tap50_i;
  
  assign n_s1_mux1     =  s0_odd ?  s0_tap1_q :  s0_tap1_i;
  assign n_s1_mux3     =  s0_odd ?  s0_tap3_q :  s0_tap3_i;
  assign n_s1_mux5     =  s0_odd ?  s0_tap5_q :  s0_tap5_i;
  assign n_s1_mux7     =  s0_odd ?  s0_tap7_q :  s0_tap7_i;
  assign n_s1_mux9     =  s0_odd ?  s0_tap9_q :  s0_tap9_i;
  assign n_s1_mux11    =  s0_odd ? s0_tap11_q : s0_tap11_i;
  assign n_s1_mux13    =  s0_odd ? s0_tap13_q : s0_tap13_i;
  assign n_s1_mux15    =  s0_odd ? s0_tap15_q : s0_tap15_i;
  assign n_s1_mux17    =  s0_odd ? s0_tap17_q : s0_tap17_i;
  assign n_s1_mux19    =  s0_odd ? s0_tap19_q : s0_tap19_i;
  assign n_s1_mux21    =  s0_odd ? s0_tap21_q : s0_tap21_i;
  assign n_s1_mux23    =  s0_odd ? s0_tap23_q : s0_tap23_i;
  assign n_s1_mux25    =  s0_odd ? s0_tap25_q : s0_tap25_i;
  assign n_s1_mux27    =  s0_odd ? s0_tap27_q : s0_tap27_i;
  assign n_s1_mux29    =  s0_odd ? s0_tap29_q : s0_tap29_i;
  assign n_s1_mux31    =  s0_odd ? s0_tap31_q : s0_tap31_i;
  assign n_s1_mux33    =  s0_odd ? s0_tap33_q : s0_tap33_i;
  assign n_s1_mux35    =  s0_odd ? s0_tap35_q : s0_tap35_i;
  assign n_s1_mux37    =  s0_odd ? s0_tap37_q : s0_tap37_i;
  assign n_s1_mux39    =  s0_odd ? s0_tap39_q : s0_tap39_i;
  assign n_s1_mux41    =  s0_odd ? s0_tap41_q : s0_tap41_i;
  assign n_s1_mux43    =  s0_odd ? s0_tap43_q : s0_tap43_i;
  assign n_s1_mux45    =  s0_odd ? s0_tap45_q : s0_tap45_i;
  assign n_s1_mux47    =  s0_odd ? s0_tap47_q : s0_tap47_i;
  assign n_s1_mux49    =  s0_odd ? s0_tap49_q : s0_tap49_i;

  /* factoring coefficients by index parity (13 bits [-4096,+4095]) */
  assign n_s1_sum0_50  = { n_s1_mux0[11], n_s1_mux0} + {n_s1_mux50[11],n_s1_mux50};
  assign n_s1_sum2_48  = { n_s1_mux2[11], n_s1_mux2} + {n_s1_mux48[11],n_s1_mux48};
  assign n_s1_sum4_46  = { n_s1_mux4[11], n_s1_mux4} + {n_s1_mux46[11],n_s1_mux46};
  assign n_s1_sum6_44  = { n_s1_mux6[11], n_s1_mux6} + {n_s1_mux44[11],n_s1_mux44};
  assign n_s1_sum8_42  = { n_s1_mux8[11], n_s1_mux8} + {n_s1_mux42[11],n_s1_mux42};
  assign n_s1_sum10_40 = {n_s1_mux10[11],n_s1_mux10} + {n_s1_mux40[11],n_s1_mux40};
  assign n_s1_sum12_38 = {n_s1_mux12[11],n_s1_mux12} + {n_s1_mux38[11],n_s1_mux38};
  assign n_s1_sum14_36 = {n_s1_mux14[11],n_s1_mux14} + {n_s1_mux36[11],n_s1_mux36};
  assign n_s1_sum16_34 = {n_s1_mux16[11],n_s1_mux16} + {n_s1_mux34[11],n_s1_mux34};
  assign n_s1_sum18_32 = {n_s1_mux18[11],n_s1_mux18} + {n_s1_mux32[11],n_s1_mux32};
  assign n_s1_sum20_30 = {n_s1_mux20[11],n_s1_mux20} + {n_s1_mux30[11],n_s1_mux30};
  assign n_s1_sum22_28 = {n_s1_mux22[11],n_s1_mux22} + {n_s1_mux28[11],n_s1_mux28};
  assign n_s1_sum24_26 = {n_s1_mux24[11],n_s1_mux24} + {n_s1_mux26[11],n_s1_mux26};
  
  assign n_s1_sum1_49  = { n_s1_mux1[11], n_s1_mux1} + {n_s1_mux49[11],n_s1_mux49};
  assign n_s1_sum3_47  = { n_s1_mux3[11], n_s1_mux3} + {n_s1_mux47[11],n_s1_mux47};
  assign n_s1_sum5_45  = { n_s1_mux5[11], n_s1_mux5} + {n_s1_mux45[11],n_s1_mux45};
  assign n_s1_sum7_43  = { n_s1_mux7[11], n_s1_mux7} + {n_s1_mux43[11],n_s1_mux43};
  assign n_s1_sum9_41  = { n_s1_mux9[11], n_s1_mux9} + {n_s1_mux41[11],n_s1_mux41};
  assign n_s1_sum11_39 = {n_s1_mux11[11],n_s1_mux11} + {n_s1_mux39[11],n_s1_mux39};
  assign n_s1_sum13_37 = {n_s1_mux13[11],n_s1_mux13} + {n_s1_mux37[11],n_s1_mux37};
  assign n_s1_sum15_35 = {n_s1_mux15[11],n_s1_mux15} + {n_s1_mux35[11],n_s1_mux35};
  assign n_s1_sum17_33 = {n_s1_mux17[11],n_s1_mux17} + {n_s1_mux33[11],n_s1_mux33};
  assign n_s1_sum19_31 = {n_s1_mux19[11],n_s1_mux19} + {n_s1_mux31[11],n_s1_mux31};
  assign n_s1_sum21_29 = {n_s1_mux21[11],n_s1_mux21} + {n_s1_mux29[11],n_s1_mux29};
  assign n_s1_sum23_27 = {n_s1_mux23[11],n_s1_mux23} + {n_s1_mux27[11],n_s1_mux27};
  assign n_s1_sum25    = {n_s1_mux25[11],n_s1_mux25};           

  /* coefficient weighting */
  /* declaration */
  localparam  H_0_50  =   -5, H_1_49  =  -14, H_2_48  =   -8, H_3_47  =   11,
              H_4_46  =   12, H_5_45  =   -9, H_6_44  =  -15, H_7_43  =   7,
              H_8_42  =   18, H_9_41  =   -6, H_10_40 =  -21, H_11_39 =   6,
              H_12_38 =   24, H_13_37 =   -5, H_14_36 =  -29, H_15_35 =   5,
              H_16_34 =   36, H_17_33 =   -4, H_18_32 =  -46, H_19_31 =   4,
              H_20_30 =   65, H_21_29 =   -4, H_22_28 = -109, H_23_27 =   4,
              H_24_26 =  328, H_25    =  512;

  assign n_s1_mul0_50  = $signed( H_0_50) * $signed( n_s1_sum0_50);
  assign n_s1_mul1_49  = $signed( H_1_49) * $signed( n_s1_sum1_49);
  assign n_s1_mul2_48  = $signed( H_2_48) * $signed( n_s1_sum2_48);
  assign n_s1_mul3_47  = $signed( H_3_47) * $signed( n_s1_sum3_47);
  assign n_s1_mul4_46  = $signed( H_4_46) * $signed( n_s1_sum4_46);
  assign n_s1_mul5_45  = $signed( H_5_45) * $signed( n_s1_sum5_45);
  assign n_s1_mul6_44  = $signed( H_6_44) * $signed( n_s1_sum6_44);
  assign n_s1_mul7_43  = $signed( H_7_43) * $signed( n_s1_sum7_43);
  assign n_s1_mul8_42  = $signed( H_8_42) * $signed( n_s1_sum8_42);
  assign n_s1_mul9_41  = $signed( H_9_41) * $signed( n_s1_sum9_41);
  assign n_s1_mul10_40 = $signed(H_10_40) * $signed(n_s1_sum10_40);
  assign n_s1_mul11_39 = $signed(H_11_39) * $signed(n_s1_sum11_39);
  assign n_s1_mul12_38 = $signed(H_12_38) * $signed(n_s1_sum12_38);
  assign n_s1_mul13_37 = $signed(H_13_37) * $signed(n_s1_sum13_37);
  assign n_s1_mul14_36 = $signed(H_14_36) * $signed(n_s1_sum14_36);
  assign n_s1_mul15_35 = $signed(H_15_35) * $signed(n_s1_sum15_35);
  assign n_s1_mul16_34 = $signed(H_16_34) * $signed(n_s1_sum16_34);
  assign n_s1_mul17_33 = $signed(H_17_33) * $signed(n_s1_sum17_33);
  assign n_s1_mul18_32 = $signed(H_18_32) * $signed(n_s1_sum18_32);
  assign n_s1_mul19_31 = $signed(H_19_31) * $signed(n_s1_sum19_31);
  assign n_s1_mul20_30 = $signed(H_20_30) * $signed(n_s1_sum20_30);
  assign n_s1_mul21_29 = $signed(H_21_29) * $signed(n_s1_sum21_29);
  assign n_s1_mul22_28 = $signed(H_22_28) * $signed(n_s1_sum22_28);
  assign n_s1_mul23_27 = $signed(H_23_27) * $signed(n_s1_sum23_27);
  assign n_s1_mul24_26 = $signed(H_24_26) * $signed(n_s1_sum24_26);
  assign n_s1_mul25    = $signed(   H_25) * $signed(   n_s1_sum25);  
                        
  /* DC scaling 
  *
  * [-2048 2047]*1002 + 512 = [-2051584   2051606]
  * [-2^21 2^21-1] = [-2097152   2097151] => 22 bits , no saturation required
  *
  */
  wire [11:0] n_s1_dc;
  wire [21:0] n_s1_dc_scaled;
  wire [21:0] n_s1_dc_fix;
  reg  [11:0] s1_dc_i,s1_dc_q;
  assign n_s1_dc        = s0_cycle?s0_dc_i:s0_dc_q;
  assign n_s1_dc_scaled = $signed(12'd1002) * $signed(n_s1_dc);
  assign n_s1_dc_fix    = n_s1_dc_scaled + 22'd512;
                        
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_rxmode  <= 1'b0;
      s1_cycle   <= 1'b0;
      s1_valid   <= 1'b0;
      s1_blocken <= 1'b0; 
      s1_mul0_50  <= 22'b0;
      s1_mul1_49  <= 22'b0; 
      s1_mul2_48  <= 22'b0; 
      s1_mul3_47  <= 22'b0; 
      s1_mul4_46  <= 22'b0; 
      s1_mul5_45  <= 22'b0; 
      s1_mul6_44  <= 22'b0; 
      s1_mul7_43  <= 22'b0; 
      s1_mul8_42  <= 22'b0; 
      s1_mul9_41  <= 22'b0; 
      s1_mul10_40 <= 22'b0;
      s1_mul11_39 <= 22'b0;
      s1_mul12_38 <= 22'b0;
      s1_mul13_37 <= 22'b0;
      s1_mul14_36 <= 22'b0;
      s1_mul15_35 <= 22'b0;
      s1_mul16_34 <= 22'b0;
      s1_mul17_33 <= 22'b0;
      s1_mul18_32 <= 22'b0;
      s1_mul19_31 <= 22'b0;
      s1_mul20_30 <= 22'b0;
      s1_mul21_29 <= 22'b0;
      s1_mul22_28 <= 22'b0;
      s1_mul23_27 <= 22'b0;
      s1_mul24_26 <= 22'b0;
      s1_mul25    <= 22'b0; 
      s1_dc_i     <= 12'd0;  
      s1_dc_q     <= 12'd0;  
    end
    else if(!s0_blocken)
    begin
      s1_rxmode  <= 1'b0;
      s1_cycle   <= 1'b0;
      s1_valid   <= 1'b0;
      s1_blocken <= 1'b0; 
      s1_mul0_50  <= 22'b0;
      s1_mul1_49  <= 22'b0; 
      s1_mul2_48  <= 22'b0; 
      s1_mul3_47  <= 22'b0; 
      s1_mul4_46  <= 22'b0; 
      s1_mul5_45  <= 22'b0; 
      s1_mul6_44  <= 22'b0; 
      s1_mul7_43  <= 22'b0; 
      s1_mul8_42  <= 22'b0; 
      s1_mul9_41  <= 22'b0; 
      s1_mul10_40 <= 22'b0;
      s1_mul11_39 <= 22'b0;
      s1_mul12_38 <= 22'b0;
      s1_mul13_37 <= 22'b0;
      s1_mul14_36 <= 22'b0;
      s1_mul15_35 <= 22'b0;
      s1_mul16_34 <= 22'b0;
      s1_mul17_33 <= 22'b0;
      s1_mul18_32 <= 22'b0;
      s1_mul19_31 <= 22'b0;
      s1_mul20_30 <= 22'b0;
      s1_mul21_29 <= 22'b0;
      s1_mul22_28 <= 22'b0;
      s1_mul23_27 <= 22'b0;
      s1_mul24_26 <= 22'b0;
      s1_mul25    <= 22'b0;   
      s1_dc_i     <= 12'd0;  
      s1_dc_q     <= 12'd0;  
    end
    else
    begin
      s1_rxmode  <= s0_rxmode;
      s1_cycle   <= s0_cycle;
      s1_valid   <= s0_valid!=51'b0;
      s1_blocken <= s0_blocken;
      s1_mul0_50  <= n_s1_mul0_50;   
      s1_mul1_49  <= n_s1_mul1_49;   
      s1_mul2_48  <= n_s1_mul2_48;   
      s1_mul3_47  <= n_s1_mul3_47;   
      s1_mul4_46  <= n_s1_mul4_46;   
      s1_mul5_45  <= n_s1_mul5_45;   
      s1_mul6_44  <= n_s1_mul6_44;   
      s1_mul7_43  <= n_s1_mul7_43;   
      s1_mul8_42  <= n_s1_mul8_42;   
      s1_mul9_41  <= n_s1_mul9_41;   
      s1_mul10_40 <= n_s1_mul10_40;  
      s1_mul11_39 <= n_s1_mul11_39;  
      s1_mul12_38 <= n_s1_mul12_38;  
      s1_mul13_37 <= n_s1_mul13_37;  
      s1_mul14_36 <= n_s1_mul14_36;  
      s1_mul15_35 <= n_s1_mul15_35;  
      s1_mul16_34 <= n_s1_mul16_34;  
      s1_mul17_33 <= n_s1_mul17_33;  
      s1_mul18_32 <= n_s1_mul18_32;  
      s1_mul19_31 <= n_s1_mul19_31;  
      s1_mul20_30 <= n_s1_mul20_30;  
      s1_mul21_29 <= n_s1_mul21_29;  
      s1_mul22_28 <= n_s1_mul22_28;  
      s1_mul23_27 <= n_s1_mul23_27;  
      s1_mul24_26 <= n_s1_mul24_26;  
      s1_mul25    <= n_s1_mul25;     
      
      if(s0_cycle)
        s1_dc_i <= n_s1_dc_fix[21:10];
      else
        s1_dc_q <= n_s1_dc_fix[21:10];
    end
  end
    
  /*****************************************************************************
  * S2
  *****************************************************************************/
  wire [23:0] n_s2_sum_even0,n_s2_sum_even1;
  wire [23:0] n_s2_sum_odd0,n_s2_sum_odd1;
  
  reg         s2_blocken;
  reg         s2_rxmode;
  reg         s2_valid;
  reg         s2_cycle;
  reg  [23:0] s2_sum_even0,s2_sum_even1;
  reg  [23:0] s2_sum_odd0,s2_sum_odd1;
  
  assign n_s2_sum_even0 = {{2{ s1_mul0_50[21]}}, s1_mul0_50} + 
                          {{2{ s1_mul2_48[21]}}, s1_mul2_48} + 
                          {{2{ s1_mul4_46[21]}}, s1_mul4_46} + 
                          {{2{ s1_mul6_44[21]}}, s1_mul6_44} +
                          {{2{ s1_mul8_42[21]}}, s1_mul8_42} +
                          {{2{s1_mul10_40[21]}},s1_mul10_40};
                        
  assign n_s2_sum_even1 = {{2{s1_mul12_38[21]}},s1_mul12_38} + 
                          {{2{s1_mul14_36[21]}},s1_mul14_36} +
                          {{2{s1_mul16_34[21]}},s1_mul16_34} + 
                          {{2{s1_mul18_32[21]}},s1_mul18_32} +
                          {{2{s1_mul20_30[21]}},s1_mul20_30} + 
                          {{2{s1_mul22_28[21]}},s1_mul22_28} + 
                          {{2{s1_mul24_26[21]}},s1_mul24_26};
       
  assign n_s2_sum_odd0  = {{2{ s1_mul1_49[21]}}, s1_mul1_49} + 
                          {{2{ s1_mul3_47[21]}}, s1_mul3_47} + 
                          {{2{ s1_mul5_45[21]}}, s1_mul5_45} + 
                          {{2{ s1_mul7_43[21]}}, s1_mul7_43} + 
                          {{2{ s1_mul9_41[21]}}, s1_mul9_41} +
                          {{2{s1_mul11_39[21]}},s1_mul11_39};

  assign n_s2_sum_odd1  = {{2{s1_mul13_37[21]}},s1_mul13_37} + 
                          {{2{s1_mul15_35[21]}},s1_mul15_35} + 
                          {{2{s1_mul17_33[21]}},s1_mul17_33} + 
                          {{2{s1_mul19_31[21]}},s1_mul19_31} +
                          {{2{s1_mul21_29[21]}},s1_mul21_29} + 
                          {{2{s1_mul23_27[21]}},s1_mul23_27} + 
                          {{2{   s1_mul25[21]}},   s1_mul25};

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_rxmode  <= 1'b0;
      s2_cycle   <= 1'b0;
      s2_valid   <= 1'b0;
      s2_blocken <= 1'b0;
      s2_sum_even0 <= 24'b0;
      s2_sum_even1 <= 24'b0; 
      s2_sum_odd0  <= 24'b0; 
      s2_sum_odd1  <= 24'b0; 
    end
    else if(!s1_blocken)
    begin
      s2_rxmode  <= 1'b0;
      s2_cycle   <= 1'b0;
      s2_valid   <= 1'b0;
      s2_blocken <= 1'b0;
      s2_sum_even0 <= 24'b0; 
      s2_sum_even1 <= 24'b0; 
      s2_sum_odd0  <= 24'b0; 
      s2_sum_odd1  <= 24'b0; 
    end
    else
    begin
      s2_rxmode  <= s1_rxmode;
      s2_cycle   <= s1_cycle;
      s2_valid   <= s1_valid;
      s2_blocken <= s1_blocken;
      s2_sum_even0 <= n_s2_sum_even0;
      s2_sum_even1 <= n_s2_sum_even1;
      s2_sum_odd0  <= n_s2_sum_odd0; 
      s2_sum_odd1  <= n_s2_sum_odd1; 
    end
  end
    
  /*****************************************************************************
  * S3
  *****************************************************************************/
  /* wires */
  wire [23:0]  n_s3_sum_all;
  wire [13:0]  n_s3_sum_all_rnd;
  reg  [12:0]  n_s3_sum_all_sat;
  
  wire [23:0]  n_s3_sum_even,n_s3_sum_odd;
  wire [14:0]  n_s3_sum_even_trc,n_s3_sum_odd_trc;
  reg  [11:0]  n_s3_sum_even_sat,n_s3_sum_odd_sat;
  
  /* flops */
  reg  [ 1:0] s3_rx_valid;
  reg  [12:0] s3_hold_i;
  reg  [ 1:0] s3_hold_valid;
  reg  [12:0] s3_i,s3_q;
  reg  [11:0] s3_hold_0;
  reg  [11:0] s3_tx_i,s3_tx_q; 
  reg         s3_tx_valid; 
  reg  [11:0] s3_dc_i,s3_dc_q; 
  
  /* RX */
  assign n_s3_sum_all     = s2_sum_even0 + s2_sum_even1 + 
                            s2_sum_odd0  + s2_sum_odd1  + 
                            {14'b0,10'b1000000000}; // rndfix

  /* rnd 10 */
  assign n_s3_sum_all_rnd = n_s3_sum_all[23:10];

  /* saturate 13 bits */
  always @(*)
  begin
    if((n_s3_sum_all_rnd[13:12]==2'b11) || (n_s3_sum_all_rnd[13:12]==2'b00))
      n_s3_sum_all_sat = n_s3_sum_all_rnd[12:0];
    else
      n_s3_sum_all_sat = {n_s3_sum_all_rnd[13],{12{~n_s3_sum_all_rnd[13]}}};
  end

  /* TX */
  assign n_s3_sum_even   = s2_sum_even0 + s2_sum_even1;
  assign n_s3_sum_odd    = s2_sum_odd0  + s2_sum_odd1;

  /* trc 9 */
  assign n_s3_sum_even_trc = n_s3_sum_even[23:9];
  assign n_s3_sum_odd_trc  = n_s3_sum_odd[23:9];
  
  /* sat 12 */
  always @(*)
  begin
    if((n_s3_sum_even_trc[14:11]==4'b1111) || (n_s3_sum_even_trc[14:11]==4'b0000))
      n_s3_sum_even_sat = n_s3_sum_even_trc[11:0];
    else
      n_s3_sum_even_sat = {n_s3_sum_even_trc[14],{11{~n_s3_sum_even_trc[14]}}};
  end
      
  always @(*)
  begin
    if((n_s3_sum_odd_trc[14:11]==4'b1111) || (n_s3_sum_odd_trc[14:11]==4'b0000))
      n_s3_sum_odd_sat = n_s3_sum_odd_trc[11:0];
    else
      n_s3_sum_odd_sat = {n_s3_sum_odd_trc[14],{11{~n_s3_sum_odd_trc[14]}}};
  end

  /* s3 data path */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s3_hold_valid <= 2'b0;
      s3_rx_valid   <= 2'b0;
      s3_tx_valid   <= 1'b0; 

      s3_hold_i     <= 13'b0;
      s3_i          <= 13'b0;
      s3_q          <= 13'b0;
      s3_dc_i       <= 12'b0;
      s3_dc_q       <= 12'b0;
      s3_hold_0     <= 12'b0;
      s3_tx_i       <= 12'b0; 
      s3_tx_q       <= 12'b0; 
    end
    else if(!s2_blocken)
    begin
      s3_hold_valid <= 2'b0;
      s3_rx_valid   <= 2'b0;
      s3_tx_valid   <= 1'b0; 

      s3_hold_i     <= 13'b0;
      s3_i          <= 13'b0;
      s3_q          <= 13'b0;
      //s3_dc_i       <= 12'b0;
      //s3_dc_q       <= 12'b0;
      s3_hold_0     <= 12'b0;
      s3_tx_i       <= 12'b0; 
      s3_tx_q       <= 12'b0; 
    end
    else
    begin
      if(s2_rxmode)                                                 
      begin
        s3_tx_valid <= 1'b0;
        s3_rx_valid <= {1'b0,s3_rx_valid[1]};  
        s3_tx_i     <= 12'b0;
        s3_tx_q     <= 12'b0;
        if(s2_valid)                           
        begin                                  
          if(!s2_cycle)                     
          begin                                
            s3_hold_i <= n_s3_sum_all_sat;     
          end                                  
          else                                 
          begin                                
            s3_rx_valid <= 2'b11;              
            s3_i    <= s3_hold_i;          
            s3_q    <= n_s3_sum_all_sat;   
            s3_dc_i <= s1_dc_i;   
            s3_dc_q <= s1_dc_q;
          end                                  
        end
      end                                                      
      else                                                     
      begin
        s3_rx_valid <= 2'b0; 
          
        s3_i     <= 13'b0;  
        s3_q     <= 13'b0;  
        s3_dc_i  <= 12'b0;   
        s3_dc_q  <= 12'b0;
       
        if(s2_valid && !s2_cycle)
          s3_hold_valid <= 2'b11;
        else
          s3_hold_valid <= {1'b0,s3_hold_valid[1]};

        if(s3_hold_valid==2'b11)
        begin
          s3_tx_valid <= 1'b1;
          s3_hold_0   <= n_s3_sum_odd_sat;
          s3_tx_i     <= s3_hold_0;
          s3_tx_q     <= n_s3_sum_even_sat;
        end
        else if(s3_hold_valid==2'b01)
        begin
          s3_tx_valid <= 1'b1;
          s3_hold_0   <= n_s3_sum_even_sat;
          s3_tx_i     <= s3_hold_0;
          s3_tx_q     <= n_s3_sum_odd_sat;
        end  
        else
        begin
          s3_tx_valid <= 1'b0;
          s3_hold_0   <= n_s3_sum_even_sat;
          s3_tx_i     <= 12'b0;
          s3_tx_q     <= 12'b0;
        end
      end
    end
  end
  
  assign out_rx_valid = s3_rx_valid[0];
  assign out_rx_i     = s3_i;
  assign out_rx_q     = s3_q;
  assign out_rx_dc_i  = {s3_dc_i[11],s3_dc_i};
  assign out_rx_dc_q  = {s3_dc_q[11],s3_dc_q};

  assign out_tx_valid = s3_tx_valid;
  assign out_tx_i     = s3_tx_i;
  assign out_tx_q     = s3_tx_q;

endmodule
`default_nettype wire
