`default_nettype wire
module rsf440_rx_fifo
(
  /*****************************************************************************
  * write port 
  *****************************************************************************/
  input  wire        fe40_rst_n,
  input  wire        fe40_clk,
  
  input  wire        fe40_valid,
  input  wire [ 7:0] fe40_i,     
  input  wire [ 7:0] fe40_q,     
  
  /*****************************************************************************
  * read port
  *****************************************************************************/
  input  wire        phy44_rst_n, 
  input  wire        phy44_clk, 
  
  input  wire        phy44_ready,
  output reg         phy44_valid,
  output reg  [ 7:0] phy44_i, 
  output reg  [ 7:0] phy44_q
);

  /*****************************************************************************
  * fe40 domain / write port
  *****************************************************************************/
  reg  [15:0] w_d0,w_d1,w_d2,w_d3,w_d4,w_d5,w_d6,w_d7;
  reg         w_v0,w_v1,w_v2,w_v3,w_v4,w_v5,w_v6,w_v7;
  reg  [ 2:0] w_ptr;
  reg         w_state;
  wire        w_wen;
  
  assign w_wen = !w_state & fe40_valid | w_state;
  
  always @(posedge fe40_clk,negedge fe40_rst_n)
  begin
    if(!fe40_rst_n)
    begin
      {w_v0,w_d0}    <= 17'b0;
      {w_v1,w_d1}    <= 17'b0;
      {w_v2,w_d2}    <= 17'b0;
      {w_v3,w_d3}    <= 17'b0;
      {w_v4,w_d4}    <= 17'b0;
      {w_v5,w_d5}    <= 17'b0;
      {w_v6,w_d6}    <= 17'b0;
      {w_v7,w_d7}    <= 17'b0;
      w_ptr          <= 3'd0;
      w_state        <= 1'd0;
    end
    else
    begin
      if(!w_state)
      begin
        w_ptr <= 3'd0;
        if(fe40_valid)
        begin
          /* first write */
          w_state           <= 1'b1;
          {w_ptr,w_v0,w_d0} <= {3'd1,1'd1,fe40_i,fe40_q};
        end
      end
      else
      begin
        /* fifo write */
        if(w_wen)
        begin
          case(w_ptr)
            3'd0:    {w_ptr,w_v0,w_d0} <= {3'd1,fe40_valid,fe40_i,fe40_q};
            3'd1:    {w_ptr,w_v1,w_d1} <= {3'd2,fe40_valid,fe40_i,fe40_q};
            3'd2:    {w_ptr,w_v2,w_d2} <= {3'd3,fe40_valid,fe40_i,fe40_q};
            3'd3:    {w_ptr,w_v3,w_d3} <= {3'd4,fe40_valid,fe40_i,fe40_q};
            3'd4:    {w_ptr,w_v4,w_d4} <= {3'd5,fe40_valid,fe40_i,fe40_q};
            3'd5:    {w_ptr,w_v5,w_d5} <= {3'd6,fe40_valid,fe40_i,fe40_q};
            3'd6:    {w_ptr,w_v6,w_d6} <= {3'd7,fe40_valid,fe40_i,fe40_q};
            default: {w_ptr,w_v7,w_d7} <= {3'd0,fe40_valid,fe40_i,fe40_q};
          endcase
        end
        
        /* stop condition */
        if(w_wen && !fe40_valid)
          w_state <= 1'b0;
      end
    end
  end

  /*****************************************************************************
  * phy44 domain / read port
  *****************************************************************************/
  wire       r_start;
  reg        r_state;
  reg  [2:0] r_ptr; 
  wire       r_ren;
  
  /* resync */
  ClkSyncSimple u_fe2phy44_start_resync
  ( 
    .dstclk(         phy44_clk),
    .dstresetn(      phy44_rst_n),
    .srcdata(        w_state),   
    .dstdata(        r_start)   
  );
  
  assign r_ren = ~phy44_valid | phy44_ready;
  
  always @(posedge phy44_clk,negedge phy44_rst_n)
  begin
    if(!phy44_rst_n)
    begin
      {phy44_valid,phy44_i,phy44_q} <= 17'd0;
      r_state        <= 1'b0;
      r_ptr          <= 3'd0;
    end
    else
    begin
      if(!r_state)
      begin
        r_ptr <= 3'd0;
        if(r_start)
        begin
          r_state <= 1'b1;
          {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd1,w_v0,w_d0};
        end
      end
      else
      begin
        if(r_ren)
        begin
          case(r_ptr)                                                       
            3'd0:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd1,w_v0,w_d0};  
            3'd1:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd2,w_v1,w_d1};  
            3'd2:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd3,w_v2,w_d2};  
            3'd3:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd4,w_v3,w_d3};  
            3'd4:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd5,w_v4,w_d4};  
            3'd5:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd6,w_v5,w_d5};  
            3'd6:    {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd7,w_v6,w_d6};  
            default: {r_ptr,phy44_valid,phy44_i,phy44_q} <= {3'd0,w_v7,w_d7};  
          endcase
        end                                                           
        
        /* stop condition */
        if(!r_start)
        begin
          {phy44_valid,phy44_i,phy44_q} <= 17'd0;
          r_state                       <= 1'b0;
        end
      end
    end
  end

`ifdef RW_SIMU_ON
  /*****************************************************************************
  * timing checks
  *****************************************************************************/
  real hold_min,setup_min;
  real last_write[7:0],last_read[7:0];
  reg  hold_min_valid,setup_min_valid;
 
  always @(posedge fe40_clk)
  begin:b_hold_min
    real v;
    
    if(!w_state && !fe40_valid)
    begin
      hold_min_valid <= 1'b0;
    end
    else if(w_wen)
    begin
      /* update last_write */
      last_write[w_ptr] = $realtime;
      
      /* compute corresponding hold */
      hold_min_valid<=1'b1;
      v=($realtime - last_read[w_ptr]);
     
      if( (v<hold_min) || !hold_min_valid)
      begin
        hold_min <= v;
        if(v<22.727)
        begin
          
          $display("error:%m: hold violation detected %2.3f < 22.727 ns !",v);
          $stop();
        end
      end
    end
  end
  
  always @(posedge phy44_clk)
  begin:b_setup_min
    real v;
    
    if(!r_state && !r_start)
    begin
      setup_min_valid <= 1'b0;
    end
    else if(r_ren)
    begin
      /* update last_read */
      last_read[r_ptr] = $realtime;
      
      /* compute corresponding setup */
      setup_min_valid <= 1'b1;
      v=($realtime - last_write[r_ptr]);
     
      if((v<setup_min) ||!setup_min_valid)
      begin
        setup_min <= v;
        if(v<22.727)
        begin
          $display("error:%m: setup violation detected %2.3f < 22.727 ns !",v);
          $stop();
        end
      end
    end 
  end
`endif

endmodule
`default_nettype wire

