//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15202 $
// $Date: 2014-06-06 11:10:00 +0200 (Fri, 06 Jun 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of dc_comp
// Simulation Notes : 
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/TxRxFrontEndX40/verilog/rtl/rx_dc_comp.v $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module rx_dc_comp 
#(
  /* clock period: 0=40M, 1=80M, 2=160M */
  parameter g_clk_period=0,
  /* output operands width : 12 or 13 */
  parameter g_width=12
)
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire                rst_n,
  input  wire                clk,
  
  /*****************************************************************************
  * control parameter
  *****************************************************************************/
  input  wire                blocken,            
  input  wire [ 1:0]         dc_comp_type,
  input  wire [ 6:0]         dc_hold_time,
  input  wire [ 6:0]         dc_zero_time,

  /*****************************************************************************
  * data
  *****************************************************************************/
  input  wire                in_valid,
  input  wire [12: 0]        in_dc_i,
  input  wire [12: 0]        in_dc_q,
  input  wire [12: 0]        in_i,
  input  wire [12: 0]        in_q,
  
  output reg  [g_width-1:0]  out_i,
  output reg  [g_width-1:0]  out_q
);
  
  /* prescaled delay */
  wire [9:0] zero_delay,hold_delay;
  generate if(g_clk_period==2)
  begin
    assign zero_delay = {dc_zero_time,3'b0};
    assign hold_delay = {dc_hold_time,3'b0};
  end
  endgenerate
  
  generate if(g_clk_period==1)
  begin
    assign zero_delay = {1'b0,dc_zero_time,2'b0};
    assign hold_delay = {1'b0,dc_hold_time,2'b0};
  end
  endgenerate
  
  generate if(g_clk_period==0)
  begin
    assign zero_delay = {2'b0,dc_zero_time,1'b0};
    assign hold_delay = {2'b0,dc_hold_time,1'b0};
  end
  endgenerate
  
  
  /* counter */
  localparam [1:0] IDLE=2'd0,ZEROTIME=2'd1,HOLDTIME=2'd2,DONE=2'd3;
  reg  [12: 0] hold_dc_i,hold_dc_q;
  reg  [ 1: 0] dc_mux_sel;
  reg  [ 1: 0] state;
  reg  [ 9: 0] counter;
  wire [ 9: 0] n_counter;
  
  assign n_counter = counter + 10'd1;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      hold_dc_i  <= 13'b0;
      hold_dc_q  <= 13'b0;
      dc_mux_sel <=  2'b0;
      counter    <= 10'd0;
      state      <=  IDLE;
    end  
    else if(!blocken)
    begin
      hold_dc_i  <= 13'b0;
      hold_dc_q  <= 13'b0;
      dc_mux_sel <=  2'b0;
      counter    <= 10'd0;
      state      <=  IDLE;
    end
    else
    begin
      case(dc_comp_type)
        /* NO COMPENSATION */
        2'd0: dc_mux_sel <= 2'd0;
        /* CONTINUOUS COMPENSATION */
        2'd1: dc_mux_sel <= 2'd1;
        /* NO COMPENSATION DURING ZEROTIME, COMPENSATION UP TO HOLDTIME */
        default:
        begin
          case(state)
            IDLE:
            begin
              dc_mux_sel <= 2'd0;
              if(in_valid)
              begin
                counter <= 10'd1;
                state   <= ZEROTIME;
              end
            end
            ZEROTIME:
            begin
              counter <= n_counter;
              if(n_counter==zero_delay)
              begin
                dc_mux_sel <= 2'd1;
                if(dc_comp_type==2'd3)
                  state <= HOLDTIME;
                else
                  state <= DONE;
              end
            end
            HOLDTIME:
            begin
              counter <= n_counter;
              if(n_counter==hold_delay)
              begin
                hold_dc_i  <= in_dc_i;
                hold_dc_q  <= in_dc_q;
                dc_mux_sel <= 2'd2;
                state <= DONE;
              end
            end
            default: /* DONE */ ;
          endcase
        end
      endcase
    end
  end
  
  /* dc estimate selection */
  reg  [13:0] dc_mux_i,dc_mux_q;
  
  always @(*)
    case(dc_mux_sel)
      2'd0:    {dc_mux_q,dc_mux_i} = {                  14'd0,                   14'd0};
      2'd1:    {dc_mux_q,dc_mux_i} = {    in_dc_q[12],in_dc_q,     in_dc_i[12],in_dc_i};
      default: {dc_mux_q,dc_mux_i} = {hold_dc_q[12],hold_dc_q, hold_dc_i[12],hold_dc_i};
    endcase
  
  /* dc compensation */
  wire [13:0]        comp_i,comp_q;
 
  assign comp_i      = {in_i[12],in_i} - dc_mux_i;  
  assign comp_q      = {in_q[12],in_q} - dc_mux_q; 
  
  /* output saturation */
  always @(*)                                                           
  begin                                                                 
    if(!in_valid)
    begin
      /* output gated */
      out_i = {g_width{1'b0}};
      out_q = {g_width{1'b0}};
    end
    else
    begin
      /* i saturation */
      if(comp_i[13:g_width-1]=={(15-g_width){1'b1}} || comp_i[13:g_width-1]=={(15-g_width){1'b0}})    
        out_i = comp_i[g_width-1:0];                                        
      else                                                                
        out_i = {comp_i[13],{(g_width-1){~comp_i[13]}}};                 
      
      /* q saturation */
      if(comp_q[13:g_width-1]=={(15-g_width){1'b1}} || comp_q[13:g_width-1]=={(15-g_width){1'b0}})    
        out_q = comp_q[g_width-1:0];                                        
      else                                                                
        out_q = {comp_q[13],{(g_width-1){~comp_q[13]}}};
    end
  end

`ifdef RW_SIMU_ON
  reg [32*8-1:0] state_str;
  always @(*)
    case(state)
      2'd0:    state_str = "IDLE";
      2'd1:    state_str = "ZEROTIME";
      2'd2:    state_str = "HOLDTIME";
      default: state_str = "DONE";
    endcase
  
`endif  
                                                                     
endmodule
`default_nettype wire

