//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 21173 $
// $Date: 2015-09-07 17:32:14 +0200 (Mon, 07 Sep 2015) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Filter at the beginning of the reception path to filter
//                    out the adjacent channel interferers. 
//                    It is implemented as an 11-taps filter.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://oringot@svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/MDMCOMMON/RIUCORE/TxRxFrontEndX40/verilog/rtl/rx_filter4corr.v $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rx_filter4corr
(
  // Clock and reset
  input  wire       clk44,   // DSSS-CCK Clock (44 MHz)
  input  wire       reset_n, // Global asynchronous reset (active LOW)

  // Control
  input  wire       blocken,
    
  // Input data.
  input  wire [6:0] filter_in_i,
  input  wire [6:0] filter_in_q,
  input  wire       filter_in_en,

  // For correlator in AGC  
  // These outputs are only used by the correlator in the AGC block
  output reg  [9:0] filt_out_4_corr_i,
  output reg  [9:0] filt_out_4_corr_q,
  output reg        filter_out_tog
  );

//////////////////////////////////////////////////////////////////////////////
// Local Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
  localparam F0_CT = 6'b000010;  // coefficient f0
  localparam F1_CT = 6'b111101;  // coefficient f1
  localparam F2_CT = 6'b001010;  // coefficient f2
  localparam F3_CT = 6'b010000;  // coefficient f3
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  // Control
  reg       blocken_1t;
  reg       blocken_2t;

  // filter_in_i input delayed
  reg [6:0] filter_in_i_ff1;
  reg [6:0] filter_in_i_ff2;
  reg [6:0] filter_in_i_ff3;
  reg [6:0] filter_in_i_ff4;
  reg [6:0] filter_in_i_ff5;
  reg [6:0] filter_in_i_ff6;
  reg [6:0] filter_in_i_ff7;
  reg [6:0] filter_in_i_ff8;
  reg [6:0] filter_in_i_ff9;
  reg [6:0] filter_in_i_ff10;
  reg [6:0] filter_in_i_ff11;

  // filter_in_q input delayed
  reg [6:0] filter_in_q_ff1;
  reg [6:0] filter_in_q_ff2;
  reg [6:0] filter_in_q_ff3;
  reg [6:0] filter_in_q_ff4;
  reg [6:0] filter_in_q_ff5;
  reg [6:0] filter_in_q_ff6;
  reg [6:0] filter_in_q_ff7;
  reg [6:0] filter_in_q_ff8;
  reg [6:0] filter_in_q_ff9;
  reg [6:0] filter_in_q_ff10;
  reg [6:0] filter_in_q_ff11;
  reg [6:0] filter_in_q_ff12;

  reg [14:0] sum_iq_ff1;  // sum of q values

  reg tick_22M_en;        // 22 MHz clock enable

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

  // Multiplexed data to compute multiplication by the coefficient
  wire [6:0] filter_in_ff1;
  wire [6:0] filter_in_ff3;
  wire [6:0] filter_in_ff5;
  wire [6:0] filter_in_ff6;
  wire [6:0] filter_in_ff7;
  wire [6:0] filter_in_ff9;
  wire [6:0] filter_in_ff11;

  // Sum data before multiplication
  wire [7:0] sum_f0;
  wire [7:0] sum_f1;
  wire [7:0] sum_f2;

  // Multiplication results
  wire [13:0] filter_in_mul0;
  wire [13:0] filter_in_mul1;
  wire [13:0] filter_in_mul2;
  wire [12:0] filter_in_mul3;
  wire [14:0] sum_iq;  // sum of q values

  wire [9:0]  sum_sat_i;  // sum of i values
  wire [9:0]  sum_sat_q;  // sum of q values

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  /* Control */
  always @(posedge clk44, negedge reset_n)
  begin
    if(!reset_n) begin
      /* control */
      blocken_1t <= 1'b0;
      blocken_2t <= 1'b0;
    end
    else begin
      /* resynchronization of static signals */
      blocken_1t <= blocken;
      blocken_2t <= blocken_1t;
    end
  end

  // Shift registers
  always @ (posedge clk44 or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      filter_in_i_ff1  <= 7'b0;
      filter_in_i_ff2  <= 7'b0;
      filter_in_i_ff3  <= 7'b0;
      filter_in_i_ff4  <= 7'b0;
      filter_in_i_ff5  <= 7'b0;
      filter_in_i_ff6  <= 7'b0;
      filter_in_i_ff7  <= 7'b0;
      filter_in_i_ff8  <= 7'b0;
      filter_in_i_ff9  <= 7'b0;
      filter_in_i_ff10 <= 7'b0;
      filter_in_i_ff11 <= 7'b0;
    end
    else begin
      if (!blocken_2t) begin
        filter_in_i_ff1  <= 7'b0;
        filter_in_i_ff2  <= 7'b0;
        filter_in_i_ff3  <= 7'b0;
        filter_in_i_ff4  <= 7'b0;
        filter_in_i_ff5  <= 7'b0;
        filter_in_i_ff6  <= 7'b0;
        filter_in_i_ff7  <= 7'b0;
        filter_in_i_ff8  <= 7'b0;
        filter_in_i_ff9  <= 7'b0;
        filter_in_i_ff10 <= 7'b0;
        filter_in_i_ff11 <= 7'b0;
      end
      else if (filter_in_en) begin
        filter_in_i_ff1  <= filter_in_i;
        filter_in_i_ff2  <= filter_in_i_ff1;
        filter_in_i_ff3  <= filter_in_i_ff2;
        filter_in_i_ff4  <= filter_in_i_ff3;
        filter_in_i_ff5  <= filter_in_i_ff4;
        filter_in_i_ff6  <= filter_in_i_ff5;
        filter_in_i_ff7  <= filter_in_i_ff6;
        filter_in_i_ff8  <= filter_in_i_ff7;
        filter_in_i_ff9  <= filter_in_i_ff8;
        filter_in_i_ff10 <= filter_in_i_ff9;
        filter_in_i_ff11 <= filter_in_i_ff10;
      end
    end
  end

  always @ (posedge clk44 or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      filter_in_q_ff1  <= 7'b0;
      filter_in_q_ff2  <= 7'b0;
      filter_in_q_ff3  <= 7'b0;
      filter_in_q_ff4  <= 7'b0;
      filter_in_q_ff5  <= 7'b0;
      filter_in_q_ff6  <= 7'b0;
      filter_in_q_ff7  <= 7'b0;
      filter_in_q_ff8  <= 7'b0;
      filter_in_q_ff9  <= 7'b0;
      filter_in_q_ff10 <= 7'b0;
      filter_in_q_ff11 <= 7'b0;
      filter_in_q_ff12 <= 7'b0;
    end
    else begin
      if (!blocken_2t) begin
        filter_in_q_ff1  <= 7'b0;
        filter_in_q_ff2  <= 7'b0;
        filter_in_q_ff3  <= 7'b0;
        filter_in_q_ff4  <= 7'b0;
        filter_in_q_ff5  <= 7'b0;
        filter_in_q_ff6  <= 7'b0;
        filter_in_q_ff7  <= 7'b0;
        filter_in_q_ff8  <= 7'b0;
        filter_in_q_ff9  <= 7'b0;
        filter_in_q_ff10 <= 7'b0;
        filter_in_q_ff11 <= 7'b0;
        filter_in_q_ff12 <= 7'b0;
      end
      else if (filter_in_en) begin
        filter_in_q_ff1  <= filter_in_q;
        filter_in_q_ff2  <= filter_in_q_ff1;
        filter_in_q_ff3  <= filter_in_q_ff2;
        filter_in_q_ff4  <= filter_in_q_ff3;
        filter_in_q_ff5  <= filter_in_q_ff4;
        filter_in_q_ff6  <= filter_in_q_ff5;
        filter_in_q_ff7  <= filter_in_q_ff6;
        filter_in_q_ff8  <= filter_in_q_ff7;
        filter_in_q_ff9  <= filter_in_q_ff8;
        filter_in_q_ff10 <= filter_in_q_ff9;
        filter_in_q_ff11 <= filter_in_q_ff10;
        filter_in_q_ff12 <= filter_in_q_ff11;
      end
    end
  end

  // Inputs for sum are multiplexed as there is a downsample on the block output
  assign filter_in_ff1  = (tick_22M_en) ? filter_in_i_ff1  : filter_in_q_ff2;
  assign filter_in_ff3  = (tick_22M_en) ? filter_in_i_ff3  : filter_in_q_ff4;
  assign filter_in_ff5  = (tick_22M_en) ? filter_in_i_ff5  : filter_in_q_ff6;
  assign filter_in_ff6  = (tick_22M_en) ? filter_in_i_ff6  : filter_in_q_ff7;
  assign filter_in_ff7  = (tick_22M_en) ? filter_in_i_ff7  : filter_in_q_ff8;
  assign filter_in_ff9  = (tick_22M_en) ? filter_in_i_ff9  : filter_in_q_ff10;
  assign filter_in_ff11 = (tick_22M_en) ? filter_in_i_ff11 : filter_in_q_ff12;

  // Coefficients
  // As the filter coefficient are symetric, the FFs are first added and then
  // multiplied. All bits are kept for sum accuracy.
  assign sum_f0 = $signed({filter_in_ff1[6],filter_in_ff1}) + $signed({filter_in_ff11[6],filter_in_ff11});
  assign sum_f1 = $signed({filter_in_ff3[6],filter_in_ff3}) + $signed({filter_in_ff9[6],filter_in_ff9});
  assign sum_f2 = $signed({filter_in_ff5[6],filter_in_ff5}) + $signed({filter_in_ff7[6],filter_in_ff7});

  assign filter_in_mul0 = $signed(sum_f0)        * $signed(F0_CT);
  assign filter_in_mul1 = $signed(sum_f1)        * $signed(F1_CT);
  assign filter_in_mul2 = $signed(sum_f2)        * $signed(F2_CT);
  assign filter_in_mul3 = $signed(filter_in_ff6) * $signed(F3_CT);
  
  assign sum_iq = $signed({filter_in_mul0[13],filter_in_mul0}) + $signed({filter_in_mul1[13],filter_in_mul1}) + 
                  $signed({filter_in_mul2[13],filter_in_mul2}) + $signed({filter_in_mul3[12],filter_in_mul3[12],filter_in_mul3});

  // Outputs
  always @ (posedge clk44 or negedge reset_n)
  begin
    if (reset_n == 1'b0) begin
      tick_22M_en       <= 1'b0;
      filter_out_tog    <= 1'b0;
      sum_iq_ff1        <= 15'b0;
      filt_out_4_corr_i <= 10'b0;
      filt_out_4_corr_q <= 10'b0;
    end
    else begin
      if (!blocken_2t) begin
        tick_22M_en       <= 1'b0;
        filter_out_tog    <= 1'b0;
        sum_iq_ff1        <= 15'b0;
        filt_out_4_corr_i <= 10'b0;
        filt_out_4_corr_q <= 10'b0;
      end
      else begin

        // Generate enable to downsample by 2
        if (filter_in_en) begin
          tick_22M_en <= ~ tick_22M_en;
          sum_iq_ff1  <= sum_iq;
        end

        // Outputs are downsampled to 22 MHz
        if (!tick_22M_en && filter_in_en) begin
          filter_out_tog    <= ~ filter_out_tog;
          filt_out_4_corr_i <= sum_sat_i;
          filt_out_4_corr_q <= sum_sat_q;
        end
      end
    end
  end

  // Saturate and trunc on sum: 3 Msb and 2 lsb removed
  SatSigned # (
    .INPUT_WIDTH(13),
    .OUTPUT_WIDTH(10)
    )
    u_sat_I (
      .InputData(sum_iq_ff1[14:2]),
      .SatData(sum_sat_i)
      );

  SatSigned # (
    .INPUT_WIDTH(13),
    .OUTPUT_WIDTH(10)
    )
    u_sat_Q (
      .InputData(sum_iq[14:2]),
      .SatData(sum_sat_q)
      );

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////

`default_nettype wire
