//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Resampler module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module Resampler 
# (
  parameter integer g_bandwidth_max = 32'd1,
  parameter DWIDTH = 13
) 
(
  // System 
  input  wire              Clk,
  input  wire              nRst,
  
  // Controls
  input  wire              stall,
  input  wire              Enable,
  input  wire              ResamplingBypass, // when high, no resampling. Data is forwarded unchanged.

  // Configuration registers (static)
  input  wire [26:0]       CfgDeltaT,
  input  wire [1:0]        CfgBw,

  // Data path
  input  wire [DWIDTH-1:0] DataInI,
  input  wire [DWIDTH-1:0] DataInQ,
  input  wire              DataInValid,
  
  output reg  [DWIDTH-1:0] DataOutI,
  output reg  [DWIDTH-1:0] DataOutQ,
  output reg               DataOutValid
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
localparam BW20  = 2'd0;

//parameter NTAPS  = 8 // Share TAPS between 40 MHz: Lagange 8 taps and 20 MHz: Parabolic 4 taps
//parameter NTAPS  = 4 // Only 20 MHz: Parabolic 4 taps

localparam NTAPS =  4*g_bandwidth_max + 4;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// Controls
wire               SkipMode;
wire               SkipSample;
wire               InsertSample;
wire               UpdateU;
reg          [3:0] Cpt;
reg                CptInit;

// Prescaler
reg         [27:0] UStep;      
wire        [27:0] UStepDelta; 
wire        [27:0] UStepMinus1;
wire        [27:0] UStepDeltaMinus1;
wire        [27:0] UStepDeltaMinus2;

// Data delay line
reg   [DWIDTH-1:0] DelayLineI[0:15], DelayLineQ[0:15];
reg         [15:0] DelayLineValid;

// Data delay line
reg   [DWIDTH-1:0] memSigI[0:NTAPS-1], memSigQ[0:NTAPS-1];
reg    [NTAPS-1:0] memSigValid;

// Coefficients LUT
wire         [7:0] LUTAddr;

// Multipliers
wire  [DWIDTH+8:0] NextMultI[0:NTAPS-1], NextMultQ[0:NTAPS-1];
reg   [DWIDTH+8:0] MultI[0:NTAPS-1], MultQ[0:NTAPS-1];
reg                MultValid;

// Delay data path by 1 flop to add register at multipliers inputs
reg          [9:0] Coeff[0:NTAPS-1];
reg   [DWIDTH-1:0] memSigI_1t, memSigQ_1t;
wire  [DWIDTH-1:0] memSigID[0:NTAPS-1], memSigQD[0:NTAPS-1];
reg                CoeffValid;
reg                InsertSampleD;
reg                SkipSampleD;

// Sum
wire [DWIDTH+10:0] NextSumI0, NextSumI1;
reg  [DWIDTH+10:0] SumI0, SumI1;
wire [DWIDTH+10:0] NextSumQ0, NextSumQ1;
reg  [DWIDTH+10:0] SumQ0, SumQ1;
reg                SumValid;

// DataOut
wire [DWIDTH+11:0] NextDataOutI, NextDataOutQ;
wire  [DWIDTH+2:0] NextDataOutRoundI, NextDataOutRoundQ;
reg   [DWIDTH-1:0] NextDataOutSatI, NextDataOutSatQ;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// In Skip mode, fill delay line in advance to be ready to skip one sample,
// by directly taking the next one from the delay line.
assign SkipMode = CfgDeltaT[26];
// Else (InsertMode), fill the delay line with the unprocessed input data when
// an interpolated sample is inserted in the output flow.

// When to skip one sample:
assign SkipSample = (ResamplingBypass==1'b0)              // Resampler is not in bypass mode
                 && (UStep[27:26]==2'd0)                  // u < 1: compute an output sample
                 && (UStepDelta[27:26]!=2'd0)             // UStepDelta >= 2: need to do two steps in one, skip one sample
                 && (UStepDeltaMinus1[27:26]!=2'd0);      // Note that the condition on UStepDeltaMinus1 alone is not enough,
                                                          // because UStepDeltaMinus1 can wrap ( <0 ) in InsertMode

assign InsertSample   = (UpdateU==1'b1) && (ResamplingBypass==1'b0) && (UStep[27:26]==2'd0) && (UStepDelta[27:26]==2'd0);

always @(posedge Clk or negedge nRst)
begin: memSigP
  integer idx;
  if (nRst == 1'b0) begin
    DelayLineValid <= 16'd0;
    for (idx=0;idx<16;idx=idx+1) begin
      DelayLineI[idx] <= {DWIDTH{1'b0}};
      DelayLineQ[idx] <= {DWIDTH{1'b0}};
    end
    for (idx=0;idx<NTAPS;idx=idx+1) begin
      memSigI[idx] <= {DWIDTH{1'b0}};
      memSigQ[idx] <= {DWIDTH{1'b0}};
    end
    memSigValid <= {NTAPS{1'b0}};
    Cpt         <= 4'd15;
    CptInit     <= 1'b0;
  end else begin
    if (Enable==1'b0) begin
      DelayLineValid <= 16'd0;
      for (idx=0;idx<16;idx=idx+1) begin
        DelayLineI[idx] <= {DWIDTH{1'b0}};
        DelayLineQ[idx] <= {DWIDTH{1'b0}};
      end
      for (idx=0;idx<NTAPS;idx=idx+1) begin
        memSigI[idx] <= {DWIDTH{1'b0}};
        memSigQ[idx] <= {DWIDTH{1'b0}};
      end
      memSigValid <= {NTAPS{1'b0}};
      Cpt         <= 4'd15;
      CptInit     <= 1'b0;
    end else if(!stall) begin

      // Set counter init value on first valid data. 
      if ((SkipMode==1'b0) && (CptInit==1'd0)) begin
        Cpt         <= 4'd0;
        CptInit     <= 1'b1; // init done
      end

      // Always store incoming data in the Mem line - no input flow control to the previous blocks
      // Do not update memSig[0] when DataInValid = 0, Matlab model reuses last valid input data
      if (DataInValid==1'b1)
      begin
        DelayLineI[0] <= DataInI;
        DelayLineQ[0] <= DataInQ;
      end
      
      for (idx=1;idx<16;idx=idx+1) begin
        DelayLineI[idx] <= DelayLineI[idx-1];
        DelayLineQ[idx] <= DelayLineQ[idx-1];
      end
     
      DelayLineValid <= {DelayLineValid[14:0], DataInValid};

      if (InsertSampleD==1'b1) 
      begin
        // Use the same memSig0 ... memSig3 data to compute the inserted sample.
        // The current input sample is stored in the Mem Line.
        // so increase Cpt to use it at next cycle.
        Cpt <= Cpt + 4'd1;
      end 
      else if (SkipSampleD==1'b1) 
      begin
        // Decrease Cpt to skip one sample
        Cpt <= Cpt - 4'd1;
        // Shift by 2 to skip one sample in memSig0 ... memSig3
        for (idx=0;idx<2;idx=idx+1) 
        begin
          memSigI[idx] <= memSigI[idx+2];
          memSigQ[idx] <= memSigQ[idx+2];
        end
        if (CfgBw==BW20) 
        begin
          memSigI[2]       <= DelayLineI[Cpt];
          memSigI[3]       <= DelayLineI[Cpt-1];
          memSigQ[2]       <= DelayLineQ[Cpt];
          memSigQ[3]       <= DelayLineQ[Cpt-1];
          memSigValid[3:0] <= {DelayLineValid[Cpt-1], DelayLineValid[Cpt], memSigValid[3:2]};
//ifdef RW_NX_DERIV_CHBW4020ONLY
        end else 
        if(g_bandwidth_max!=32'd0) 
        begin
          for (idx=2;idx<6;idx=idx+1) 
          begin
            memSigI[idx] <= memSigI[idx+2];
            memSigQ[idx] <= memSigQ[idx+2];
          end
          memSigI[6]  <= DelayLineI[Cpt];
          memSigI[7]  <= DelayLineI[Cpt-1];
          memSigQ[6]  <= DelayLineQ[Cpt];
          memSigQ[7]  <= DelayLineQ[Cpt-1];
          memSigValid <= {DelayLineValid[Cpt-1], DelayLineValid[Cpt], memSigValid[7:2]};
//endif // RW_NX_DERIV_CHBW4020ONLY
        end
      end 
      else 
      begin
        // Process input data. Shift memSig0 ... memSig3 by one sample
        for (idx=0;idx<3;idx=idx+1) begin
          memSigI[idx] <= memSigI[idx+1];
          memSigQ[idx] <= memSigQ[idx+1];
        end
        if (CfgBw==BW20) begin
          memSigI[3]       <= DelayLineI[Cpt];
          memSigQ[3]       <= DelayLineQ[Cpt];
          memSigValid[3:0] <= {DelayLineValid[Cpt], memSigValid[3:1]};
//ifdef RW_NX_DERIV_CHBW4020ONLY
        end else if(g_bandwidth_max!=32'd0) begin
          for (idx=3;idx<7;idx=idx+1) begin
            memSigI[idx] <= memSigI[idx+1];
            memSigQ[idx] <= memSigQ[idx+1];
          end
          memSigI[7]  <= DelayLineI[Cpt];
          memSigQ[7]  <= DelayLineQ[Cpt];
          memSigValid <= {DelayLineValid[Cpt], memSigValid[7:1]};
//endif // RW_NX_DERIV_CHBW4020ONLY
        end
      end

    end
  end
end


// Control
assign UStepDelta       = UStep + {1'b0,CfgDeltaT};
assign UStepMinus1      = UStep - {2'd1,26'd0};
assign UStepDeltaMinus1 = UStepDelta - {2'd1,26'd0};
assign UStepDeltaMinus2 = UStepDeltaMinus1 - {2'd1,26'd0};

assign UpdateU = (CfgBw==BW20) ? memSigValid[1] : memSigValid[3];

always @(posedge Clk or negedge nRst)
begin
  if (nRst == 1'b0) begin
    UStep           <= 28'd0;

  end else begin
    
    if (Enable==1'b0) begin
      UStep           <= 28'd0;

    end else if(!stall) begin

      // Processing
      if ((UpdateU==1'b1) && (ResamplingBypass==1'b0)) begin

        if (UStep[27:26]!=2'd0) begin // u >=1
          UStep     <= UStepMinus1;
        end else begin // compute u = u + delta, if >=1, substract 1
          if (UStepDelta[27:26]!=2'd0) begin // >=1
            if (UStepDeltaMinus1[27:26]!=2'd0) begin // still >=1. Skip one sample
              UStep <= UStepDeltaMinus2;
            end else begin
              UStep <= UStepDeltaMinus1;
            end
          end else begin // freeze intput and add an output sample
            UStep   <= UStepDelta;
          end
        end
      end
      
    end
  end
end


///////////////////////////////////////////////////////////////////////////////
// Coefficients
///////////////////////////////////////////////////////////////////////////////

// LUT addressing
assign LUTAddr = {1'b0,UStep[25:19]}+{7'd0,UStep[18]};

// Coefficients LUT
wire         [9:0] c1P1;
wire         [9:0] c1P0;
wire         [9:0] c1M1;

ResamplerLUT U_ResamplerLUT ( 
  .UAddr  (LUTAddr),
  .c1P1   (c1P1),
  .c1P0   (c1P0),
  .c1M1   (c1M1)
);


wire [9:0] NextCoeff[0:7];

generate if(g_bandwidth_max!=32'd0)
begin:g_lagrange_lut
  
  wire [9:0] coeff0, coeff1, coeff2, coeff3, coeff4, coeff5, coeff6, coeff7;

  LagrangeLUT U_LagrangeLUT ( 
    .UAddr  (LUTAddr),
    .coeff0 (coeff0),
    .coeff1 (coeff1),
    .coeff2 (coeff2),
    .coeff3 (coeff3),
    .coeff4 (coeff4),
    .coeff5 (coeff5),
    .coeff6 (coeff6),
    .coeff7 (coeff7)
  );

  assign NextCoeff[0] = (CfgBw==BW20)? c1P1:coeff0;
  assign NextCoeff[1] = (CfgBw==BW20)? c1P0:coeff1;
  assign NextCoeff[2] = (CfgBw==BW20)? c1M1:coeff2;
  assign NextCoeff[3] = (CfgBw==BW20)? c1P1:coeff3;
  assign NextCoeff[4] = (CfgBw==BW20)?10'd0:coeff4;
  assign NextCoeff[5] = (CfgBw==BW20)?10'd0:coeff5; 
  assign NextCoeff[6] = (CfgBw==BW20)?10'd0:coeff6;
  assign NextCoeff[7] = (CfgBw==BW20)?10'd0:coeff7;

end
endgenerate

generate if(g_bandwidth_max==32'd0)
begin:g_lagrange_lut_absent
  // 20 MHz only
  assign NextCoeff[0] = c1P1;
  assign NextCoeff[1] = c1P0;
  assign NextCoeff[2] = c1M1;
  assign NextCoeff[3] = c1P1; // same coefficient as for memSig0
  assign NextCoeff[4] = 10'd0;
  assign NextCoeff[5] = 10'd0; 
  assign NextCoeff[6] = 10'd0;
  assign NextCoeff[7] = 10'd0;
end
endgenerate

///////////////////////////////////////////////////////////////////////////////
// Filtering
///////////////////////////////////////////////////////////////////////////////

// Multipliers
genvar tap_idx;
generate
for (tap_idx=0;tap_idx<NTAPS;tap_idx=tap_idx+1) begin : NextMultGen
  if (tap_idx>0) begin : memSigDGen
    assign memSigID[tap_idx] = memSigI[tap_idx-1];
    assign memSigQD[tap_idx] = memSigQ[tap_idx-1];
  end else begin : memSigD0Gen
    assign memSigID[tap_idx] = memSigI_1t;
    assign memSigQD[tap_idx] = memSigQ_1t;
  end
  assign NextMultI[tap_idx] = $signed({{9{memSigID[tap_idx][DWIDTH-1]}},memSigID[tap_idx]}) * $signed({{12{Coeff[tap_idx][9]}},Coeff[tap_idx]});
  assign NextMultQ[tap_idx] = $signed({{9{memSigQD[tap_idx][DWIDTH-1]}},memSigQD[tap_idx]}) * $signed({{12{Coeff[tap_idx][9]}},Coeff[tap_idx]});
end
endgenerate

always @(posedge Clk or negedge nRst)
begin: MultP
  integer idx;
  if (nRst == 1'b0) 
  begin
    for (idx=0;idx<NTAPS;idx=idx+1) 
    begin
      MultI[idx] <= {(DWIDTH+9){1'b0}};
      MultQ[idx] <= {(DWIDTH+9){1'b0}};
      Coeff[idx] <= 10'd0;
    end
    CoeffValid    <= 1'b0;
    // Add 1 flop stage in path
    memSigI_1t    <= {DWIDTH{1'b0}};
    memSigQ_1t    <= {DWIDTH{1'b0}};
    MultValid     <= 1'b0;
    SkipSampleD   <= 1'b0;
    InsertSampleD <= 1'b0;
  end 
  else if(!Enable)
  begin
    for (idx=0;idx<NTAPS;idx=idx+1) 
    begin
      MultI[idx] <= {(DWIDTH+9){1'b0}};
      MultQ[idx] <= {(DWIDTH+9){1'b0}};
      Coeff[idx] <= 10'd0;
    end
    CoeffValid    <= 1'b0;
    // Add 1 flop stage in path
    memSigI_1t    <= {DWIDTH{1'b0}};
    memSigQ_1t    <= {DWIDTH{1'b0}};
    MultValid     <= 1'b0;
    SkipSampleD   <= 1'b0;
    InsertSampleD <= 1'b0;
  end
  else
  begin
    if(!stall) 
    begin

      // These flops (used to cut the timing path) are also used in bypass mode,
      // in order to have the same latency with and without bypass
      memSigI_1t    <= memSigI[0];
      memSigQ_1t    <= memSigQ[0];
      if (CfgBw==BW20)       // Output TAP is tap 1 of [0:3]
        CoeffValid  <= memSigValid[1];
//ifdef RW_NX_DERIV_CHBW4020ONLY
      else if(g_bandwidth_max!=32'd0)                  // Output TAP is tap 3 of [0:7]
        CoeffValid  <= memSigValid[3];
//endif // RW_NX_DERIV_CHBW4020ONLY
      MultValid     <= CoeffValid;
    
      if (ResamplingBypass==1'b0) 
      begin
        for (idx=0;idx<NTAPS;idx=idx+1) 
        begin
          MultI[idx] <= NextMultI[idx];
          MultQ[idx] <= NextMultQ[idx];
          Coeff[idx] <= NextCoeff[idx];
        end
        // Add 1 flop stage in path
        SkipSampleD   <= SkipSample;
        InsertSampleD <= InsertSample;
      end 
      else 
      begin
        SkipSampleD   <= 1'b0;
        InsertSampleD <= 1'b0;
      end
    end
  end
end

// Sum and rounding
/* 20M only */
generate
if(g_bandwidth_max==32'd0) 
begin:g_sum_round_20only

  assign NextSumI0 = {{2{MultI[0][DWIDTH+8]}},MultI[0]}
                   + {{2{MultI[1][DWIDTH+8]}},MultI[1]};

  assign NextSumI1 = {{2{MultI[2][DWIDTH+8]}},MultI[2]}
                   + {{2{MultI[3][DWIDTH+8]}},MultI[3]};

  assign NextSumQ0 = {{2{MultQ[0][DWIDTH+8]}},MultQ[0]}
                   + {{2{MultQ[1][DWIDTH+8]}},MultQ[1]};

  assign NextSumQ1 = {{2{MultQ[2][DWIDTH+8]}},MultQ[2]}
                   + {{2{MultQ[3][DWIDTH+8]}},MultQ[3]};
end
endgenerate

/* 20 and 40M only */
generate
if(g_bandwidth_max!=32'd0)
begin:g_sum_round_2040only
    assign NextSumI0 = {{2{MultI[0][DWIDTH+8]}},MultI[0]}
                     + {{2{MultI[1][DWIDTH+8]}},MultI[1]}
                     + {{2{MultI[4][DWIDTH+8]}},MultI[4]} 
                     + {{2{MultI[5][DWIDTH+8]}},MultI[5]};

    assign NextSumI1 = {{2{MultI[2][DWIDTH+8]}},MultI[2]}
                     + {{2{MultI[3][DWIDTH+8]}},MultI[3]}
                     + {{2{MultI[6][DWIDTH+8]}},MultI[6]} 
                     + {{2{MultI[7][DWIDTH+8]}},MultI[7]};

    assign NextSumQ0 = {{2{MultQ[0][DWIDTH+8]}},MultQ[0]}
                     + {{2{MultQ[1][DWIDTH+8]}},MultQ[1]}
                     + {{2{MultQ[4][DWIDTH+8]}},MultQ[4]} 
                     + {{2{MultQ[5][DWIDTH+8]}},MultQ[5]};

    assign NextSumQ1 = {{2{MultQ[2][DWIDTH+8]}},MultQ[2]}
                     + {{2{MultQ[3][DWIDTH+8]}},MultQ[3]}
                     + {{2{MultQ[6][DWIDTH+8]}},MultQ[6]} 
                     + {{2{MultQ[7][DWIDTH+8]}},MultQ[7]};
end
endgenerate

assign NextDataOutI = {SumI0[DWIDTH+10],SumI0} + 
                      {SumI1[DWIDTH+10],SumI1} + 
                      {{(DWIDTH+3){1'b0}},1'b1,8'd0}; // rounding

assign NextDataOutRoundI = NextDataOutI[DWIDTH+11:9];


assign NextDataOutQ = {SumQ0[DWIDTH+10],SumQ0} + 
                      {SumQ1[DWIDTH+10],SumQ1} + 
                      {{(DWIDTH+3){1'b0}},1'b1,8'd0}; // rounding

assign NextDataOutRoundQ = NextDataOutQ[DWIDTH+11:9];

// Saturation
always@(*)
begin
  if (NextDataOutRoundI[DWIDTH+2]==1'b0) begin // positive number
    if (NextDataOutRoundI[DWIDTH+2:DWIDTH-1]!=4'd0)
      NextDataOutSatI = {1'b0,{(DWIDTH-1){1'b1}}}; // saturate to positive value
    else
      NextDataOutSatI = NextDataOutRoundI[DWIDTH-1:0];
  end else begin // negative number
    if ((NextDataOutRoundI[DWIDTH+2:DWIDTH-1]==4'b1111) && (NextDataOutRoundI[DWIDTH-2:0]=={(DWIDTH-1){1'b0}}))
      NextDataOutSatI = {2'b1,{(DWIDTH-2){1'b0}},1'b1}; // saturate to negative value
    else if (NextDataOutRoundI[DWIDTH+2:DWIDTH-1]==4'b1111)
      NextDataOutSatI = NextDataOutRoundI[DWIDTH-1:0];
    else
      NextDataOutSatI = {2'b1,{(DWIDTH-2){1'b0}},1'b1}; // saturate to negative value
  end
end

always@(*)
begin
  if (NextDataOutRoundQ[DWIDTH+2]==1'b0) begin // positive number
    if (NextDataOutRoundQ[DWIDTH+2:DWIDTH-1]!=4'd0)
      NextDataOutSatQ = {1'b0,{(DWIDTH-1){1'b1}}}; // saturate to positive value
    else
      NextDataOutSatQ = NextDataOutRoundQ[DWIDTH-1:0];
  end else begin // negative number
    if ((NextDataOutRoundQ[DWIDTH+2:DWIDTH-1]==4'b1111) && (NextDataOutRoundQ[DWIDTH-2:0]=={(DWIDTH-1){1'b0}}))
      NextDataOutSatQ = {2'b1,{(DWIDTH-2){1'b0}},1'b1}; // saturate to negative value
    else if (NextDataOutRoundQ[DWIDTH+2:DWIDTH-1]==4'b1111)
      NextDataOutSatQ = NextDataOutRoundQ[DWIDTH-1:0];
    else
      NextDataOutSatQ = {2'b1,{(DWIDTH-2){1'b0}},1'b1}; // saturate to negative value
  end
end

always @(posedge Clk or negedge nRst)
begin
  if (nRst == 1'b0) begin
    SumI0          <= {(DWIDTH+11){1'b0}};
    SumI1          <= {(DWIDTH+11){1'b0}};
    SumQ0          <= {(DWIDTH+11){1'b0}};
    SumQ1          <= {(DWIDTH+11){1'b0}};
    SumValid       <= 1'b0;
    DataOutI       <= {DWIDTH{1'b0}};
    DataOutQ       <= {DWIDTH{1'b0}};
    DataOutValid   <= 1'b0;
  end else begin
    if (Enable==1'b0) begin
      SumI0          <= {(DWIDTH+11){1'b0}};
      SumI1          <= {(DWIDTH+11){1'b0}};
      SumQ0          <= {(DWIDTH+11){1'b0}};
      SumQ1          <= {(DWIDTH+11){1'b0}};
      SumValid       <= 1'b0;
      DataOutI       <= {DWIDTH{1'b0}};
      DataOutQ       <= {DWIDTH{1'b0}};
      DataOutValid   <= 1'b0;
    end else if(!stall) begin
      if (ResamplingBypass==1'b0) begin
        SumI0          <= NextSumI0;
        SumI1          <= NextSumI1;
        SumQ0          <= NextSumQ0;
        SumQ1          <= NextSumQ1;
        SumValid       <= MultValid;
        DataOutI       <= NextDataOutSatI;
        DataOutQ       <= NextDataOutSatQ;
        DataOutValid   <= SumValid;
      end else begin
        if (CfgBw==BW20) begin
          DataOutI     <= memSigI_1t;
          DataOutQ     <= memSigQ_1t;
//ifdef RW_NX_DERIV_CHBW4020ONLY
        end else if(g_bandwidth_max!=32'd0) begin
          DataOutI     <= memSigI[1];
          DataOutQ     <= memSigQ[1];
//endif // RW_NX_DERIV_CHBW4020ONLY
        end
        DataOutValid   <= MultValid;
      end
    end
  end
end


endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
