//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 7840 $
// $Date: 2013-05-15 15:05:40 +0200 (Wed, 15 May 2013) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Top Level File for Cordic Rotator
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       : 
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL:  $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module TxTdFoCordicRotTop #(parameter NPIPE       =  7, //Number of Pipeline stages
                            parameter NCOMBSTG    =  2, //Number of Combinatorial stages
                            parameter SCALING     =  1, //Scaling to [-4,4] range 
                            parameter DATAWIDTH   = 13, //Width of Input Data
                            parameter ANGLEWIDTH  = 25  //Width of Input Angle
                           )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock
            //Control Signals
            input    wire                                  Enable,  //Block Enable 
            input    wire                                  stall,
            input    wire                                  DataInValid, //Qualifies Input
            input    wire      [ANGLEWIDTH-1:0]            AngleIn, //Angle for Rotation
            //Data
            input    wire      [DATAWIDTH-1:0]             ReDataIn0, //Real Comp of Data 0
            input    wire      [DATAWIDTH-1:0]             ImDataIn0, //Imag Comp of Data 0

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control
            output reg                                     DataOutValid, //Qualifies Output
            //Data
            output reg  signed [DATAWIDTH-1:0]             ReDataOut0, //Real Comp of Data 0
            output reg  signed [DATAWIDTH-1:0]             ImDataOut0  //Imag Comp of Data 0
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [DATAWIDTH-1:0]  CONST_ZERO_DATAWIDTH  = {{DATAWIDTH}{1'b0}};
localparam signed [DATAWIDTH+1:0]  CONST_1_DATAWIDTH_P2  = {{{(DATAWIDTH+1)}{1'b0}},1'b1};
localparam signed [DATAWIDTH+1:0]  CONST_2_DATAWIDTH_P2  = 'd2;
localparam signed [DATAWIDTH+1:0]  CONST_4_DATAWIDTH_P2  = 'd4;
localparam signed [DATAWIDTH+1:0]  CONST_5_DATAWIDTH_P2  = 'd5;
localparam signed [DATAWIDTH+1:0]  CONST_7_DATAWIDTH_P2  = 'd7;
localparam signed [DATAWIDTH+1:0]  CONST_8_DATAWIDTH_P2  = 'd8;
localparam signed [DATAWIDTH+1:0]  CONST_11_DATAWIDTH_P2 = 'd11;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DATAWIDTH+1:0]     ReDataOut0Int;
wire    signed     [DATAWIDTH+1:0]     ImDataOut0Int;
wire    signed     [DATAWIDTH+1:0]     ReDataOut0Norm;
wire    signed     [DATAWIDTH+1:0]     ImDataOut0Norm;
wire    signed     [DATAWIDTH-1:0]     ReDataOut0Sat;
wire    signed     [DATAWIDTH-1:0]     ImDataOut0Sat;
wire                                   DataOutValidInt;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This block instantiates the CordicRot module. It also performs
//normalization on the outputs. It then saturates the output to DATAWIDTH. 

//Instantiate the CordicRot module
TxTdFoCordicRot #(
            //Parameters
           .DATAWIDTH(DATAWIDTH),
           .ANGLEWIDTH(ANGLEWIDTH),
           .NCOMBSTG(NCOMBSTG),
           .NPIPE(NPIPE),
           .SCALING(SCALING)
           )U_CR (
                  //Inputs
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Enable(Enable),
                  .stall(stall),
                  .DataInValid(DataInValid),
                  .AngleIn(AngleIn),
                  .ReDataIn0(ReDataIn0),
                  .ImDataIn0(ImDataIn0),

                  //Outputs
                  .DataOutValid(DataOutValidInt),
                  .ReDataOut0(ReDataOut0Int),
                  .ImDataOut0(ImDataOut0Int)
                 );

//Normalize the outputs
assign ReDataOut0Norm = $signed(Normalize(ReDataOut0Int));
assign ImDataOut0Norm = $signed(Normalize(ImDataOut0Int));


//Saturate the Normalized Outputs
SatSymSigned #(
               .INPUT_WIDTH(DATAWIDTH+2),
               .OUTPUT_WIDTH(DATAWIDTH)
              )
               U_SATCRT0(
                        .InputData(ReDataOut0Norm),
                        .SatSymData(ReDataOut0Sat)
                        );

SatSymSigned #(
               .INPUT_WIDTH(DATAWIDTH+2),
               .OUTPUT_WIDTH(DATAWIDTH)
              )
               U_SATCRT1(
                        .InputData(ImDataOut0Norm),
                        .SatSymData(ImDataOut0Sat)
                        );


//Register the final outputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: OpReg_Blk
      if (nPhyRst == 1'b0) begin
         ReDataOut0 <= CONST_ZERO_DATAWIDTH;
         ImDataOut0 <= CONST_ZERO_DATAWIDTH;
         DataOutValid <= 1'b0;
      end
      else if (Enable == 1'b1) begin
         if(!stall)
         begin
            if (DataOutValidInt==1'b1) begin
              ReDataOut0 <= ReDataOut0Sat;
              ImDataOut0 <= ImDataOut0Sat;
              DataOutValid <= 1'b1;
            end else begin
              DataOutValid <= 1'b0;
            end
         end
      end else begin //Enable == 1'b1
         DataOutValid <= 1'b0;
      end

   end //OpReg_Blk


//Function to generate the Normalized Value of the output
//Multiply by normalization factor 2487/4096

function           [DATAWIDTH+1:0]                Normalize;

//Inputs
input    signed    [DATAWIDTH+1:0]                DataIn;

//Internal Registers
reg      signed    [DATAWIDTH+13:0]               Normal;

   begin
      //Multiply by 2487
      //2487 = 2048 + 256 + 128 + 32 + 16 + 4 + 2 + 1
      Normal =   (DataIn <<< CONST_11_DATAWIDTH_P2) + (DataIn <<< CONST_8_DATAWIDTH_P2) + (DataIn <<< CONST_7_DATAWIDTH_P2)
               + (DataIn <<< CONST_5_DATAWIDTH_P2)  + (DataIn <<< CONST_4_DATAWIDTH_P2) + (DataIn <<< CONST_2_DATAWIDTH_P2)
               + (DataIn <<< CONST_1_DATAWIDTH_P2)  + (DataIn);

      // Divide by 4096
      Normalize = Normal[DATAWIDTH+13:12];

   end

endfunction  //Normalize;

endmodule //CordicRotTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
