//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of fIQComp module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

// This module computes the next Coefficient value based on the current coefficient
// value, the input sample and the error at the output of the CompFilter.
// The coefficient value itself is stored outside so that the module can be shared
// between Tx and Rx. 
module CoeffUpdate ( 
  // System
  input  wire        Clk,
  input  wire        nRst,
  
  // Control
  input  wire [3:0]  CfgRegfIQEstStep,

  // Update from estimation
  input  wire [11:0] SampleXk,
  input  wire        EstErrorValidk,
  input  wire        EstSignErrork,
  input  wire        EstNullErrork,

  // Coefficient value
  input  wire [15:0] CoeffToUpdate,
  output reg  [15:0] UpdatedCoeff,
  output reg         UpdatedCoeffValid
);


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  reg         SignErrorD;
  reg         NullErrorD;
  reg  [ 3:0] EstStep;
  reg  [12:0] SampleXkCorr;
  reg  [12:0] SampleXkCorrRnd, SampleXkCorrCut;
  wire [16:0] CoeffUpdatedNext;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // Register locally control signals with big load (going to all CoeffUpdate blocks)
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      UpdatedCoeffValid <= 1'b0;
      SignErrorD        <= 1'b0;
      NullErrorD        <= 1'b0;
      EstStep           <= 4'd0;
    end else begin
      // No need to reset these signals when block is disabled because
      // - EstErrorValidk is a pulse,
      // - EstSignErrork and EstNullErrork are sampled according to EstErrorValidk 
      UpdatedCoeffValid <= EstErrorValidk;
      SignErrorD        <= EstSignErrork;
      NullErrorD        <= EstNullErrork;
      EstStep           <= CfgRegfIQEstStep;
    end
  end

  // Select correction value before rounding
  always @(*)
  begin
    // No correction when error is null
    if (NullErrorD==1'b1)
      SampleXkCorr = 13'd0;
    // Negative correction
    else if (SignErrorD==1'd1)
      SampleXkCorr = ~{SampleXk[11],SampleXk}+13'd1;
    // Positive correction
    else
      SampleXkCorr = {SampleXk[11],SampleXk};
  end

  // cut 0 to 10 bits
  always @(*)
  begin
    case (EstStep)
      4'd0   : begin
        SampleXkCorrRnd = SampleXkCorr;
        SampleXkCorrCut = SampleXkCorrRnd; 
      end
      4'd1   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000000001;
        SampleXkCorrCut = {SampleXkCorrRnd[12],SampleXkCorrRnd[12:1]}; 
      end
      4'd2   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000000010;
        SampleXkCorrCut = {{2{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:2]}; 
      end
      4'd3   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000000100;
        SampleXkCorrCut = {{3{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:3]}; 
      end
      4'd4   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000001000;
        SampleXkCorrCut = {{4{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:4]}; 
      end
      4'd5   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000010000;
        SampleXkCorrCut = {{5{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:5]}; 
      end
      4'd6   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000000100000;
        SampleXkCorrCut = {{6{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:6]}; 
      end
      4'd7   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000001000000;
        SampleXkCorrCut = {{7{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:7]}; 
      end
      4'd8   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000010000000;
        SampleXkCorrCut = {{8{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:8]}; 
      end
      4'd9   : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b000100000000;
        SampleXkCorrCut = {{9{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:9]}; 
      end
      4'd10  : begin
        SampleXkCorrRnd = SampleXkCorr + 13'b001000000000;
        SampleXkCorrCut = {{10{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:10]}; 
      end
      default: begin
        SampleXkCorrRnd = SampleXkCorr + 13'b010000000000;
        SampleXkCorrCut = {{11{SampleXkCorrRnd[12]}},SampleXkCorrRnd[12:11]}; 
      end

    endcase
  end
  
  // Update CoeffToUpdate based on estimation
  assign CoeffUpdatedNext = {CoeffToUpdate[15],CoeffToUpdate} + {{4{SampleXkCorrCut[12]}},SampleXkCorrCut};

  // Saturate to 16 bits
  always @(*)
  begin
    if((CoeffUpdatedNext[16:15]==2'b11) || (CoeffUpdatedNext[16:15]==2'b00))
      UpdatedCoeff = CoeffUpdatedNext[15:0];
    else
      UpdatedCoeff = {CoeffUpdatedNext[16],{15{~CoeffUpdatedNext[16]}}};
  end

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
