//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module CompDelay # (
  parameter LENGTH    = 30, // between 9 and 30
  parameter FILTERLAT = 4   // Additional filter latency due to registers added on the output path for timing reasons
) (
  input  wire        Clk,
  input  wire        nRst,
  
  input  wire        Enable,
  input  wire        CfgRegfIQCalEn, // enable for Sample Collection and Estimation
  input  wire [4:0]  fIQDel,         // minimum value = 2

  input  wire        ValidXkp1,
  input  wire [11:0] SampleXkp1,
  output reg  [11:0] SampleYk
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  reg   [11:0]  SampleLine     [LENGTH+FILTERLAT-1:0];
  wire  [11:0]  SampleLineWire [30+FILTERLAT-1:0];
  reg   [5:0]   OutSel;
  reg   [2:0]   FlushCnt;
  reg           FlushDone;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
  genvar j;
  generate 
    for (j=0; j<LENGTH+FILTERLAT; j=j+1) begin: CompDelayUsedGen
      assign SampleLineWire[j] = SampleLine[j];
    end //CompDelayUsedGen
  endgenerate

  genvar k;
  generate 
    for (k=LENGTH+FILTERLAT; k<30+FILTERLAT; k=k+1) begin: CompDelayUnusedGen
      assign SampleLineWire[k] = 12'd0;
    end //CompDelayUnusedGen
  endgenerate
  
  // Select line output. Matlab spec is to output sample[fIQDel], but:
  // - Matlab counts from 1 to fIQDel, HW counts from 0 to fIQDel-1
  // - The output process adds one register to the delay line
  // So the correct output compared to Matlab is actually sample[fIQDel-2]. Additionally,
  // - in Estimation mode, the delay line must match the filter delay line length: output sample[fIQDel-2]
  // - in Compensation mode, the delay line must match the complete filter processing delay, including output pipe
  // after the delay line, so add FILTERLAT flops
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      OutSel                 <= 6'd36; // max value
    end else begin
      if  (CfgRegfIQCalEn==1'b1)
        OutSel <= {1'b0,fIQDel};
      else
        OutSel <= {1'b0,fIQDel} + FILTERLAT[5:0]; // Add 4 for S1, S2, S3, and output stages in CompFilter
    end
  end


  // Delay line and output selection
  always @(posedge Clk or negedge nRst)
  begin : DelayLine
    integer i;
    if (nRst == 1'b0)
    begin
      for (i=0;i<LENGTH+FILTERLAT;i=i+1) begin : CompDelayInitLoop
        SampleLine[i]        <= 12'd0;
      end // CompDelayInitLoop
      SampleYk               <= 12'd0;
      FlushCnt               <= 3'd0;
      FlushDone              <= 1'd1;

    end else begin

      // Reset line (e.g. between tx and Rx)
      if (Enable==1'b0) begin
        for (i=0;i<LENGTH+FILTERLAT;i=i+1) begin : CompDelayEnableLoop
          SampleLine[i]        <= 12'd0;
        end // CompDelayEnableLoop
        SampleYk               <= 12'd0;
        FlushCnt               <= 3'd0;
        FlushDone              <= 1'd1;

      // Shift data in line
      end else begin
        if ((ValidXkp1==1'b1) || (FlushDone==1'b0)) begin
          if (CfgRegfIQCalEn==1'b0)
            FlushDone            <= 1'd0; // Flush is needed only in compensation mode
          FlushCnt               <= FILTERLAT; // By default, init flush counter and reset FlushDone flag
          SampleLine[0]          <= SampleXkp1;
          for (i=0;i<LENGTH+FILTERLAT-1;i=i+1) begin : CompDelayShiftLoop
            SampleLine[i+1] <= SampleLine[i];
          end // CompDelayShiftLoop

          // Output selection
          case (OutSel)
            6'd35   : SampleYk  <= SampleLineWire[33];
            6'd34   : SampleYk  <= SampleLineWire[32];
            6'd33   : SampleYk  <= SampleLineWire[31];
            6'd32   : SampleYk  <= SampleLineWire[30];
            6'd31   : SampleYk  <= SampleLineWire[29];
            6'd30   : SampleYk  <= SampleLineWire[28];
            6'd29   : SampleYk  <= SampleLineWire[27];
            6'd28   : SampleYk  <= SampleLineWire[26];
            6'd27   : SampleYk  <= SampleLineWire[25];
            6'd26   : SampleYk  <= SampleLineWire[24];
            6'd25   : SampleYk  <= SampleLineWire[23];
            6'd24   : SampleYk  <= SampleLineWire[22];
            6'd23   : SampleYk  <= SampleLineWire[21];
            6'd22   : SampleYk  <= SampleLineWire[20];
            6'd21   : SampleYk  <= SampleLineWire[19];
            6'd20   : SampleYk  <= SampleLineWire[18];
            6'd19   : SampleYk  <= SampleLineWire[17];
            6'd18   : SampleYk  <= SampleLineWire[16];
            6'd17   : SampleYk  <= SampleLineWire[15];
            6'd16   : SampleYk  <= SampleLineWire[14];
            6'd15   : SampleYk  <= SampleLineWire[13];
            6'd14   : SampleYk  <= SampleLineWire[12];
            6'd13   : SampleYk  <= SampleLineWire[11];
            6'd12   : SampleYk  <= SampleLineWire[10];
            6'd11   : SampleYk  <= SampleLineWire[9] ;
            6'd10   : SampleYk  <= SampleLineWire[8] ;
            6'd9    : SampleYk  <= SampleLineWire[7] ;
            6'd8    : SampleYk  <= SampleLineWire[6] ;
            6'd7    : SampleYk  <= SampleLineWire[5] ;
            6'd6    : SampleYk  <= SampleLineWire[4] ;
            6'd5    : SampleYk  <= SampleLineWire[3] ;
            6'd4    : SampleYk  <= SampleLineWire[2] ;
            6'd3    : SampleYk  <= SampleLineWire[1] ;
            default : SampleYk  <= SampleLineWire[0] ;
          endcase
          // The if was entered on ValidXkp1=0, FlushDone=0: flush the line
          if ((ValidXkp1==1'b0) && (CfgRegfIQCalEn==1'b0)) begin // Compensation mode only
            if (FlushCnt>3'd0)
              FlushCnt <= FlushCnt-3'd1;
            else
              FlushDone <= 1'b1;
          end
        end
      end
    end
  end

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
