//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module fIQEstComp # (
  parameter NTAPS = 20, // between 12 and 40
  parameter NDEL  = 15  // between 9 and 30, typ. 3/4*NTAPS
                        // This is the length of the delay line to match max fiqdel value.
                        // Flops to match the filter latency are added to this number. 
) ( 
  input wire         Clk,
  input wire         nRst,
  input wire         nCoeffRst,      // Not asserted in case of watchdog reset

  input  wire        TxEn,           // Tx/Rx control in HW-driven mode 
  input  wire        RxIQCompEn,     // Rx Compensation filter is reset and block is in bypass mode when RxIQCompEn=0
  input  wire [1:0]  FeRamFreqRatio, // Indicates the ratio between RAM clock and fIQ clock
                                     // 2: fIQ clock is twice RAM clock    (typ. 80 MHz operation)
                                     // 1: fIQ clock = RAM clock           (typ. 40 MHz operation)
                                     // 0: fIQ clock is half of RAM clock  (typ. 20 MHz operation)
  input  wire        CollStartP,
  output reg         CollDone,
  input  wire        EstStartP,
  output reg         EstDone,

  // Control
  input  wire        CfgRegfIQCalEn,         // enable for Sample Collection and Estimation
`ifdef RW_NX_DERIV_FE_PATH1
  input  wire        CfgRegfIQPathSel,       // Selects the path to calibrate or estimate
`endif  
  input  wire        CfgRegfIQTxRx,          // Tx/Rx control in SW driven mode
  input  wire        CfgRegfIQTxBypass,
  input  wire        CfgRegfIQRxBypass,
  // For sample Collection
  input  wire        CfgRegfIQCollPath,
  input  wire [ 7:0] CfgRegfIQCollWrOffset,
  input  wire [ 7:0] CfgRegfIQCollRdOffset,
  input  wire [ 7:0] CfgRegfIQCollSize,
  // For estimation
  input  wire [ 7:0] CfgRegfIQEstIOffset,
  input  wire [ 7:0] CfgRegfIQEstQOffset,
  input  wire [ 7:0] CfgRegfIQEstSize,

  // Settings
  input  wire [ 4:0] CfgRegfIQTxDel,
  input  wire [ 4:0] CfgRegfIQRxDel,
  input  wire [ 7:0] CfgRegfIQCollDel,
  input  wire [ 3:0] CfgRegfIQEstStep,
  input  wire [15:0] CfgRegfIQEstNite,

  // Data path: Rx mode
  input  wire [11:0] rxInDataI0,   // Rx Data to be compensated or collected for estimation
  input  wire [11:0] rxInDataQ0,
  input  wire        rxInValid0, 
  output wire [11:0] rxCompDataI0, // Compensated data
  output wire [11:0] rxCompDataQ0,
  output wire        rxCompValid0,
  // Tx mode
  input  wire        txInValid0,   // Tx data to be compensated
  input  wire [11:0] txInDataI0,
  input  wire [11:0] txInDataQ0,
  output reg         txOutValid0,  // Tx compensated data, or collection pattern 
  output reg  [11:0] txOutDataI0,
  output reg  [11:0] txOutDataQ0,
`ifdef RW_NX_DERIV_FE_PATH1
  // Data path: Rx mode
  input  wire [11:0] rxInDataI1,   // Rx Data to be compensated or collected for estimation
  input  wire [11:0] rxInDataQ1,
  input  wire        rxInValid1, 
  output wire [11:0] rxCompDataI1, // Compensated data
  output wire [11:0] rxCompDataQ1,
  output wire        rxCompValid1,
  // Tx mode
  input  wire        txInValid1,   // Tx data to be compensated
  input  wire [11:0] txInDataI1,
  input  wire [11:0] txInDataQ1,
  output reg         txOutValid1,  // Tx compensated data, or collection pattern 
  output reg  [11:0] txOutDataI1,
  output reg  [11:0] txOutDataQ1,
`endif
  // RAM interface
  output wire [7:0]  RamAddr,
  output wire        RamRdEn,
  output wire        RamWrEn,
  output wire [63:0] RamWrData,
  input  wire [63:0] RamRdData,

  // Estimation error for status
  output reg         StatfIQToggle0, // Toggles with each new Error stat value
  output reg  [12:0] StatfIQRxEstError0,
  output reg  [12:0] StatfIQTxEstError0,
`ifdef RW_NX_DERIV_FE_PATH1
  output reg         StatfIQToggle1,
  output reg  [12:0] StatfIQRxEstError1,
  output reg  [12:0] StatfIQTxEstError1,
`endif
  // Coefficients values for status
  output wire [15:0] Coeff0,
  output wire [15:0] Coeff1,
  output wire [15:0] Coeff2,
  output wire [15:0] Coeff3,
  output wire [15:0] Coeff4,
  output wire [15:0] Coeff5,
  output wire [15:0] Coeff6,
  output wire [15:0] Coeff7,
  output wire [15:0] Coeff8,
  output wire [15:0] Coeff9,
  output wire [15:0] Coeff10,
  output wire [15:0] Coeff11,
  output wire [15:0] Coeff12,
  output wire [15:0] Coeff13,
  output wire [15:0] Coeff14,
  output wire [15:0] Coeff15,
  output wire [15:0] Coeff16,
  output wire [15:0] Coeff17,
  output wire [15:0] Coeff18,
  output wire [15:0] Coeff19,
  output wire [15:0] Coeff20,
  output wire [15:0] Coeff21,
  output wire [15:0] Coeff22,
  output wire [15:0] Coeff23,
  output wire [15:0] Coeff24,
  output wire [15:0] Coeff25,
  output wire [15:0] Coeff26,
  output wire [15:0] Coeff27,
  output wire [15:0] Coeff28,
  output wire [15:0] Coeff29,
  output wire [15:0] Coeff30,
  output wire [15:0] Coeff31,
  output wire [15:0] Coeff32,
  output wire [15:0] Coeff33,
  output wire [15:0] Coeff34,
  output wire [15:0] Coeff35,
  output wire [15:0] Coeff36,
  output wire [15:0] Coeff37,
  output wire [15:0] Coeff38,
  output wire [15:0] Coeff39,

  // Coefficients (Update from regs)
  input  wire [15:0] CfgRegfIQCoeff0,
  input  wire [15:0] CfgRegfIQCoeff1,
  input  wire [15:0] CfgRegfIQCoeff2,
  input  wire [15:0] CfgRegfIQCoeff3,
  input  wire [15:0] CfgRegfIQCoeff4,
  input  wire [15:0] CfgRegfIQCoeff5,
  input  wire [15:0] CfgRegfIQCoeff6,
  input  wire [15:0] CfgRegfIQCoeff7,
  input  wire [15:0] CfgRegfIQCoeff8,
  input  wire [15:0] CfgRegfIQCoeff9,
  input  wire [15:0] CfgRegfIQCoeff10,
  input  wire [15:0] CfgRegfIQCoeff11,
  input  wire [15:0] CfgRegfIQCoeff12,
  input  wire [15:0] CfgRegfIQCoeff13,
  input  wire [15:0] CfgRegfIQCoeff14,
  input  wire [15:0] CfgRegfIQCoeff15,
  input  wire [15:0] CfgRegfIQCoeff16,
  input  wire [15:0] CfgRegfIQCoeff17,
  input  wire [15:0] CfgRegfIQCoeff18,
  input  wire [15:0] CfgRegfIQCoeff19,
  input  wire [15:0] CfgRegfIQCoeff20,
  input  wire [15:0] CfgRegfIQCoeff21,
  input  wire [15:0] CfgRegfIQCoeff22,
  input  wire [15:0] CfgRegfIQCoeff23,
  input  wire [15:0] CfgRegfIQCoeff24,
  input  wire [15:0] CfgRegfIQCoeff25,
  input  wire [15:0] CfgRegfIQCoeff26,
  input  wire [15:0] CfgRegfIQCoeff27,
  input  wire [15:0] CfgRegfIQCoeff28,
  input  wire [15:0] CfgRegfIQCoeff29,
  input  wire [15:0] CfgRegfIQCoeff30,
  input  wire [15:0] CfgRegfIQCoeff31,
  input  wire [15:0] CfgRegfIQCoeff32,
  input  wire [15:0] CfgRegfIQCoeff33,
  input  wire [15:0] CfgRegfIQCoeff34,
  input  wire [15:0] CfgRegfIQCoeff35,
  input  wire [15:0] CfgRegfIQCoeff36,
  input  wire [15:0] CfgRegfIQCoeff37,
  input  wire [15:0] CfgRegfIQCoeff38,
  input  wire [15:0] CfgRegfIQCoeff39,
  input  wire        CfgRegfIQCoeffUpdate,
  
  output wire [15:0] Dbg0

);



//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
localparam FIQSTATE_TXCOMP = 2'd0, FIQSTATE_RXCOMP = 2'd1, FIQSTATE_EST = 2'd2,
           FIQSTATE_COLL = 2'd3;
localparam FIQTXRX_TX = 1'b0, FIQTXRX_RX = 1'b1; 
localparam FIQPATH0 = 1'b0, FIQPATH1 = 1'b1; 
localparam FIQMODE_COLL = 1'b0, FIQMODE_EST = 1'b1;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  // Debug port
  wire   TxCoeff9Array0_IsNull;
  wire   RxCoeff9Array0_IsNull;
  wire   SelCoeff9Array0_IsNull;

  // FSM
  reg  [ 1:0] fIQState;
  reg         CollEstStartP;
  reg         CollEstMode;
  wire        CollEstDone;
  wire        CollEstDoneGate;
  reg         CompFilterEn;

  // rx Data out
  wire [11:0] rxCompI0;
  wire [11:0] rxCompQ0;
  wire        rxValid0;

  // Data from/to fIQRAMIf, in Sample collection or Estimation mode
  wire [11:0] RAMIfInDataI;
  wire [11:0] RAMIfInDataQ;
  wire [11:0] CollEstDataI;
  wire [11:0] CollEstDataQ;
  wire        CollEstValid;

  // fIQComp control and data
  reg         TxEnD,CalEnD;
  wire        TxEnEdge, CalEnEdge;
  reg  [4:0]  fIQDel;
  wire        fIQCompEnable0;
  reg  [11:0] SampleXikp1Path0;
  reg  [11:0] SampleXqkp1Path0;
  reg         ValidXkp1Path0;
  wire [11:0] SampleYik0;
  wire [11:0] SampleYqk0;
  wire        ValidYk0;
  wire [11:0] SampleXqArray  [NTAPS-1:0];
  wire [11:0] SampleXqArray0 [39:0];
  wire        EstLineFilledP0;
  wire        EstLineFilledP;  

  // Error computation
  wire [12:0] EstErrork0;
  reg         EstSignErrork;
  reg         EstNullErrork;
  reg         EstErrorValidk;

  // Flow control for fIQComp in Estimation mode
  reg         EstUpdateOnGoing;
  wire        EstReady;
  wire        EstOnGoing;

  // Coefficients
  reg  [15:0]      TxCoeffArray0       [NTAPS-1:0]; // Flops for Tx coefficients path0
  reg  [15:0]      RxCoeffArray0       [NTAPS-1:0]; // Flops for Rx coefficients path0
  reg  [15:0]      CoeffKArray0        [39:0];      // Coefficient going to fIQComp for path0
                                                    // Tx/Rx mux is HW controlled in compensation,
                                                    // and SW controlled in Calibration 
  // Coefficients to/from update logic: CoeffKArray0/1, path selection is SW controlled
  wire [15:0]      CoeffToUpdateKArray [39:0];
  wire [15:0]      UpdatedCoeffKArray  [NTAPS-1:0];
  // Coefficient going to regbus stats. Tx/Rx and path selection is SW controlled.
  wire [15:0]      CoeffStatArray      [39:0];
  // Coefficient update from regbus
  wire [15:0]      CfgRegCoeffArray    [39:0];
  reg              CoeffUpdateDoneP;
  wire             DoneOnNextCycle;
  wire [NTAPS-1:0] UpdatedCoeffValid;
  wire [NTAPS-1:0] UpdatedCoeffValidEn;
  reg              CfgRegCoeffUpdateD, CfgRegCoeffValid;

  // Second path
`ifdef RW_NX_DERIV_FE_PATH1
  wire [11:0] rxCompI1;
  wire [11:0] rxCompQ1;
  wire        rxValid1;
  wire        fIQCompEnable1;
  wire [11:0] SampleYik1;
  wire [11:0] SampleYqk1;
  wire        ValidYk1;
  reg  [11:0] SampleXikp1Path1;
  reg  [11:0] SampleXqkp1Path1;
  reg         ValidXkp1Path1;
  wire [11:0] SampleXqArray1 [39:0];
  wire        EstLineFilledP1;
  wire [12:0] EstErrork1;
  reg  [15:0] TxCoeffArray1 [NTAPS-1:0];
  reg  [15:0] RxCoeffArray1 [NTAPS-1:0];
  reg  [15:0] CoeffKArray1  [39:0];
`endif


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // Debug port
  assign TxCoeff9Array0_IsNull = (TxCoeffArray0[9]==16'd0);
  assign RxCoeff9Array0_IsNull = (RxCoeffArray0[9]==16'd0);
  assign SelCoeff9Array0_IsNull = (CoeffKArray0[9]==16'd0);
  assign Dbg0 = {CalEnEdge,
                 CollEstStartP,  
                 CollEstMode,
                 CollDone,
                 EstDone,
                 RamRdEn,
                 RamWrEn,
                 CoeffUpdateDoneP,
                 RxIQCompEn,
                 TxEn,
                 fIQCompEnable0,
                 CfgRegCoeffValid,
                 UpdatedCoeffValidEn[9],
                 TxCoeff9Array0_IsNull,
                 RxCoeff9Array0_IsNull,
                 SelCoeff9Array0_IsNull
  };

  // Choose Tx or Rx values for Coefficients and Delay used by the HW
  // Note: Tx and Rx Coefficients, when written from CfgReg, are selected by CfgRegfIQTxRx
  // independentely from CfgRegfIQCalEn and txEn (see far below)
  always @(*)
  begin : TxRxCoeff
    integer i;
    for (i=NTAPS;i<40;i=i+1) begin : UnusedCoeffLoop
      CoeffKArray0[i] = 16'd0;
`ifdef RW_NX_DERIV_FE_PATH1
      CoeffKArray1[i] = 16'd0;
`endif
    end
    // During calibration procedure, Tx/Rx choice is under SW control 
    // (setting is used in Estimation, don't care in Sample collection)
    if (CfgRegfIQCalEn==1'b1) begin
      if (CfgRegfIQTxRx==FIQTXRX_TX) begin
        for (i=0;i<NTAPS;i=i+1) begin : TxCalCoeffLoop
          CoeffKArray0[i] = TxCoeffArray0[i];
`ifdef RW_NX_DERIV_FE_PATH1
          CoeffKArray1[i] = TxCoeffArray1[i];
`endif
        end
        fIQDel      = CfgRegfIQTxDel;
      end else begin // FIQTXRX_RX
        for (i=0;i<NTAPS;i=i+1) begin : RxCalCoeffLoop
          CoeffKArray0[i] = RxCoeffArray0[i];
`ifdef RW_NX_DERIV_FE_PATH1
          CoeffKArray1[i] = RxCoeffArray1[i];
`endif
        end
        fIQDel      = CfgRegfIQRxDel;
      end
    // During compensation, it is under HW control
    end else begin
      if (TxEn==1'b1) begin
        for (i=0;i<NTAPS;i=i+1) begin : TxCompCoeffLoop
          CoeffKArray0[i] = TxCoeffArray0[i];
`ifdef RW_NX_DERIV_FE_PATH1
          CoeffKArray1[i] = TxCoeffArray1[i];
`endif
        end
        fIQDel      = CfgRegfIQTxDel;
      end else begin
        for (i=0;i<NTAPS;i=i+1) begin : RxCompCoeffLoop
          CoeffKArray0[i] = RxCoeffArray0[i];
`ifdef RW_NX_DERIV_FE_PATH1
          CoeffKArray1[i] = RxCoeffArray1[i];
`endif
        end
        fIQDel      = CfgRegfIQRxDel;
      end
    end
  end


  // FSM part to interface with commands from registers.
  ////////////////////////////////////////////////////////////////////
    
  // It ensures that:
  // - when CfgRegfIQCalEn=0, only HW mode is available. Estimation and Collection starts are ignored.
  // - when CfgRegfIQCalEn=1, the FSM answers only one start at a time. This avoids conflicts between
  // Estimation and Collection in case the SW does a wrong programming.

  // Ignore the CollEstDone flag while the CollEstStartP control is high
  assign CollEstDoneGate = CollEstDone && !CollEstStartP;

  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      fIQState        <= FIQSTATE_RXCOMP;
      CollEstStartP   <= 1'b0;
      CollEstMode     <= FIQMODE_EST;
      EstDone         <= 1'b1;
      CollDone        <= 1'b1;
    end else begin
      CollEstStartP   <= 1'b0; // RTZ

      // SW driven modes, i.e. Sample collection and Estimation, have priority over HW driven modes
      if (CfgRegfIQCalEn == 1'b1) begin

        case (fIQState)
          // Sample collection
          FIQSTATE_COLL : begin
            if (CollEstDoneGate==1'b1)
              fIQState  <= FIQSTATE_RXCOMP;
          end

          // Filter used for estimation
          FIQSTATE_EST : begin
            // After fIQRamIf operation is done, wait for end of Coefficient update to exit FIQSTATE_EST state.
            if ((CollEstDoneGate==1'b1) && (EstUpdateOnGoing==1'b0))
              fIQState  <= FIQSTATE_RXCOMP;
          end

          // Filter used for compensation
          default : begin // FIQSTATE_RXCOMP, FIQSTATE_TXCOMP 
            CollDone          <= 1'b1;
            EstDone           <= 1'b1;
            // Priority is:
            // - SW driven: 1. Collection, 2. Estimation, 3. HW driven
            // - HW driven: Tx and Rx mutually exclusive, Rx by default
            if (CollStartP==1'b1) begin
              fIQState         <= FIQSTATE_COLL;
              CollEstStartP    <= 1'b1;
              CollEstMode      <= FIQMODE_COLL;
              CollDone         <= 1'b0;
            end else if (EstStartP==1'b1) begin
              fIQState        <= FIQSTATE_EST;
              CollEstStartP   <= 1'b1;
              CollEstMode     <= FIQMODE_EST;
              EstDone         <= 1'b0;
            end else if (TxEn==1'd1) begin
              fIQState  <= FIQSTATE_TXCOMP;
            end else begin
              fIQState  <= FIQSTATE_RXCOMP;
            end
          end

        endcase
        
      // CfgRegfIQCalEn=0. HW driven mode (compensation) driven by TxEn, Bypass controls
      // Estimation and Collection starts are ignored.
      end else begin
        EstDone     <= 1'b1;
        CollDone    <= 1'b1;
        if (TxEn==1'd1)
          fIQState  <= FIQSTATE_TXCOMP;
        else
          fIQState  <= FIQSTATE_RXCOMP;
      end
    end
  end

  
  // Tx and Rx output ports
  ////////////////////////////////////////////////////////////////////
  // txOutData data: used during:
  // - Sample collection : coming from fIQRAMIf
  // - Compensation      : coming from fIQComp, with possible FW bypass
  always @(*)
  begin
    case(fIQState)
      FIQSTATE_COLL   : begin
`ifdef RW_NX_DERIV_FE_PATH1
        // Collection module is shared between paths
        if (CfgRegfIQPathSel==FIQPATH0) begin
`endif        
          txOutDataI0 = CollEstDataI;
          txOutDataQ0 = CollEstDataQ;
          txOutValid0 = CollEstValid;
`ifdef RW_NX_DERIV_FE_PATH1
          txOutDataI1 = 12'd0;
          txOutDataQ1 = 12'd0;
          txOutValid1 = 1'b0;
        end else begin
          txOutDataI0 = 12'd0;
          txOutDataQ0 = 12'd0;
          txOutValid0 = 1'b0;
          txOutDataI1 = CollEstDataI;
          txOutDataQ1 = CollEstDataQ;
          txOutValid1 = CollEstValid;
        end
`endif        
      end
      FIQSTATE_TXCOMP : begin
       if (CfgRegfIQTxBypass==1'b1) begin
          txOutDataI0 = txInDataI0;
          txOutDataQ0 = txInDataQ0;
          txOutValid0 = txInValid0;
`ifdef RW_NX_DERIV_FE_PATH1
          txOutDataI1 = txInDataI1;
          txOutDataQ1 = txInDataQ1;
          txOutValid1 = txInValid1;
`endif        
        end else begin
          txOutDataI0 = SampleYik0;
          txOutDataQ0 = SampleYqk0;
          txOutValid0 = ValidYk0;
`ifdef RW_NX_DERIV_FE_PATH1
          txOutDataI1 = SampleYik1;
          txOutDataQ1 = SampleYqk1;
          txOutValid1 = ValidYk1;
`endif        
        end  
      end
      default         : begin // txOutData not used
        txOutDataI0 = 12'd0;
        txOutDataQ0 = 12'd0;
        txOutValid0 = 1'b0;
`ifdef RW_NX_DERIV_FE_PATH1
        txOutDataI1 = 12'd0;
        txOutDataQ1 = 12'd0;
        txOutValid1 = 1'b0;
`endif        
      end
    endcase
  end

  // Compensated data output: set to zero when not used (fIQComp in Tx or Estimation modes)
  assign rxCompI0 = (fIQState==FIQSTATE_RXCOMP) ? SampleYik0 : 12'd0;
  assign rxCompQ0 = (fIQState==FIQSTATE_RXCOMP) ? SampleYqk0 : 12'd0;
  assign rxValid0 = (fIQState==FIQSTATE_RXCOMP) ? ValidYk0   : 1'b0;

  // Compensation is bypassed when AGC is unlocked, or by SW
  assign rxCompDataI0 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInDataI0 : rxCompI0;
  assign rxCompDataQ0 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInDataQ0 : rxCompQ0;
  assign rxCompValid0 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInValid0 : rxValid0;

`ifdef RW_NX_DERIV_FE_PATH1
  assign rxCompI1 = (fIQState==FIQSTATE_RXCOMP) ? SampleYik1 : 12'd0;
  assign rxCompQ1 = (fIQState==FIQSTATE_RXCOMP) ? SampleYqk1 : 12'd0;
  assign rxValid1 = (fIQState==FIQSTATE_RXCOMP) ? ValidYk1   : 1'b0;

  assign rxCompDataI1 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInDataI1 : rxCompI1;
  assign rxCompDataQ1 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInDataQ1 : rxCompQ1;
  assign rxCompValid1 = ((CfgRegfIQRxBypass==1'b1) || (RxIQCompEn==1'b0)) ? rxInValid1 : rxValid1;
`endif        


  // RAM interface
  ////////////////////////////////////////////////////////////////////
  // During sample collection,
  // - reads the pattern from the RAM and sends it to CollEstData ports
  // - writes the collected samples from rxInData to the RAM
  // During Estimation
  // - reads the pattern from the RAM and sends it to CollEstData ports
  // Module is shared between paths
`ifdef RW_NX_DERIV_FE_PATH1
  assign RAMIfInDataI = (CfgRegfIQPathSel==FIQPATH0) ? rxInDataI0 : rxInDataI1;
  assign RAMIfInDataQ = (CfgRegfIQPathSel==FIQPATH0) ? rxInDataQ0 : rxInDataQ1;
`else 
  assign RAMIfInDataI = rxInDataI0;
  assign RAMIfInDataQ = rxInDataQ0;
`endif

  fIQRAMIf UfIQRAMIf (
    .Clk                   (Clk),
    .nRst                  (nRst),
    .Enable                (CfgRegfIQCalEn),
    .FeRamFreqRatio        (FeRamFreqRatio),
    //Sample collection
    .CollEstStartP         (CollEstStartP),
    .CollEstMode           (CollEstMode),
    .CollEstDone           (CollEstDone),
    .CoeffUpdateDoneP      (CoeffUpdateDoneP),
    .EstReady              (EstReady),
    //Control registers
    .CfgRegfIQCollRdOffset (CfgRegfIQCollRdOffset),
    .CfgRegfIQCollWrOffset (CfgRegfIQCollWrOffset),
    .CfgRegfIQCollSize     (CfgRegfIQCollSize),
    .CfgRegfIQCollDel      (CfgRegfIQCollDel),
    .CfgRegfIQCollPath     (CfgRegfIQCollPath),
    //For estimation
    .CfgRegfIQEstIOffset   (CfgRegfIQEstIOffset),
    .CfgRegfIQEstQOffset   (CfgRegfIQEstQOffset),
    .CfgRegfIQEstSize      (CfgRegfIQEstSize),
    .CfgRegfIQEstNite      (CfgRegfIQEstNite),
    //Data interface
    .rxInDataI             (RAMIfInDataI),
    .rxInDataQ             (RAMIfInDataQ),
    .CollEstValid          (CollEstValid),
    .CollEstDataI          (CollEstDataI),
    .CollEstDataQ          (CollEstDataQ),
    //RAM interface
    .RamAddr               (RamAddr),
    .RamRdEn               (RamRdEn),
    .RamWrEn               (RamWrEn),
    .RamWrData             (RamWrData),
    .RamRdData             (RamRdData)
  );


  // fIQComp instance used in Compensation (Tx and Rx) and in Estimation
  ////////////////////////////////////////////////////////////////////

  // Reset delay line and filter taps at estimation start, Tx start and Tx end
  // to ensure that samples from a former operating mode are not shifted out when
  // a new mode is started.
  // Reset it at Calibration start and end to handle SW abort cases.
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      TxEnD  <= 1'b0;
      CalEnD <= 1'b0;
    end else begin
      TxEnD  <= TxEn;
      CalEnD <= CfgRegfIQCalEn;
    end
  end

  assign TxEnEdge       = TxEn ^ TxEnD;
  assign CalEnEdge      = CfgRegfIQCalEn ^ CalEnD;
  assign fIQCompEnable0 = CompFilterEn && !(EstStartP || TxEnEdge || CalEnEdge
`ifdef RW_NX_DERIV_FE_PATH1
                          || (((fIQState==FIQSTATE_COLL) || (fIQState==FIQSTATE_EST)) 
                            && (CfgRegfIQPathSel==FIQPATH1))
`endif
                          );
`ifdef RW_NX_DERIV_FE_PATH1
  assign fIQCompEnable1 = CompFilterEn && !(EstStartP || TxEnEdge || CalEnEdge
                          || (((fIQState==FIQSTATE_COLL) || (fIQState==FIQSTATE_EST)) 
                            && (CfgRegfIQPathSel==FIQPATH0))
                          );
`endif

  // Compensation block is shared between Tx, Rx and estimation
  // Choose correct input
  always @(*)
  begin
    case (fIQState)
      // Estimation of ceofficients used for Tx/Rx
      FIQSTATE_EST: begin
        CompFilterEn     = 1'b1;
`ifdef RW_NX_DERIV_FE_PATH1
        if (CfgRegfIQPathSel==FIQPATH0) begin
`endif
          SampleXikp1Path0 = CollEstDataI;
          SampleXqkp1Path0 = CollEstDataQ;
          ValidXkp1Path0   = CollEstValid;
`ifdef RW_NX_DERIV_FE_PATH1
          SampleXikp1Path1 = 12'd0;
          SampleXqkp1Path1 = 12'd0;
          ValidXkp1Path1   = 1'b0;
        end else begin
          SampleXikp1Path0 = 12'd0;
          SampleXqkp1Path0 = 12'd0;
          ValidXkp1Path0   = 1'b0;
          SampleXikp1Path1 = CollEstDataI;
          SampleXqkp1Path1 = CollEstDataQ;
          ValidXkp1Path1   = CollEstValid;
        end
`endif
      end
      // fIQ compensation on Tx path
      FIQSTATE_TXCOMP: begin
        CompFilterEn     = 1'b1;
        SampleXikp1Path0 = txInDataI0;
        SampleXqkp1Path0 = txInDataQ0;
        ValidXkp1Path0   = txInValid0;
`ifdef RW_NX_DERIV_FE_PATH1
        SampleXikp1Path1 = txInDataI1;
        SampleXqkp1Path1 = txInDataQ1;
        ValidXkp1Path1   = txInValid1;
`endif
      end
      // fIQ compensation on Rx path
      FIQSTATE_RXCOMP : begin 
        CompFilterEn     = RxIQCompEn; // from AGC FSM
        SampleXikp1Path0 = rxInDataI0;
        SampleXqkp1Path0 = rxInDataQ0;
        ValidXkp1Path0   = rxInValid0;
`ifdef RW_NX_DERIV_FE_PATH1
        SampleXikp1Path1 = rxInDataI1;
        SampleXqkp1Path1 = rxInDataQ1;
        ValidXkp1Path1   = rxInValid1;
`endif
      end
      // Sample collection: compensation is bypassed
      default: begin // FIQSTATE_COLL
        CompFilterEn     = 1'b0;
        SampleXikp1Path0 = 12'd0;
        SampleXqkp1Path0 = 12'd0;
        ValidXkp1Path0   = 1'd0;
`ifdef RW_NX_DERIV_FE_PATH1
        SampleXikp1Path1 = 12'd0;
        SampleXqkp1Path1 = 12'd0;
        ValidXkp1Path1   = 1'd0;
`endif
      end
    endcase
  end

  // fIQComp module contains the Compensation Filter and Delay line
  // The filter tap values are output to be used in Coefficient update during Estimation 
  fIQComp #(
    .NTAPS(NTAPS),
    .NDEL (NDEL)
  ) UfIQCompPath0 (
    //System
    .Clk            (Clk),
    .nRst           (nRst),
    //Control
    .Enable         (fIQCompEnable0),
    .CfgRegfIQCalEn (CfgRegfIQCalEn),
    .fIQDel         (fIQDel),
    //Data path I and Q
    .ValidXkp1      (ValidXkp1Path0),
    .SampleXikp1    (SampleXikp1Path0),
    .SampleXqkp1    (SampleXqkp1Path0),
    .ValidYk        (ValidYk0),
    .SampleYik      (SampleYik0),
    .SampleYqk      (SampleYqk0),
    .EstLineFilledP (EstLineFilledP0),
    //Coefficients
    .Coeffk0        (CoeffKArray0[0]),
    .Coeffk1        (CoeffKArray0[1]),
    .Coeffk2        (CoeffKArray0[2]),
    .Coeffk3        (CoeffKArray0[3]),
    .Coeffk4        (CoeffKArray0[4]),
    .Coeffk5        (CoeffKArray0[5]),
    .Coeffk6        (CoeffKArray0[6]),
    .Coeffk7        (CoeffKArray0[7]),
    .Coeffk8        (CoeffKArray0[8]),
    .Coeffk9        (CoeffKArray0[9]),
    .Coeffk10       (CoeffKArray0[10]),
    .Coeffk11       (CoeffKArray0[11]),
    .Coeffk12       (CoeffKArray0[12]),
    .Coeffk13       (CoeffKArray0[13]),
    .Coeffk14       (CoeffKArray0[14]),
    .Coeffk15       (CoeffKArray0[15]),
    .Coeffk16       (CoeffKArray0[16]),
    .Coeffk17       (CoeffKArray0[17]),
    .Coeffk18       (CoeffKArray0[18]),
    .Coeffk19       (CoeffKArray0[19]),
    .Coeffk20       (CoeffKArray0[20]),
    .Coeffk21       (CoeffKArray0[21]),
    .Coeffk22       (CoeffKArray0[22]),
    .Coeffk23       (CoeffKArray0[23]),
    .Coeffk24       (CoeffKArray0[24]),
    .Coeffk25       (CoeffKArray0[25]),
    .Coeffk26       (CoeffKArray0[26]),
    .Coeffk27       (CoeffKArray0[27]),
    .Coeffk28       (CoeffKArray0[28]),
    .Coeffk29       (CoeffKArray0[29]),
    .Coeffk30       (CoeffKArray0[30]),
    .Coeffk31       (CoeffKArray0[31]),
    .Coeffk32       (CoeffKArray0[32]),
    .Coeffk33       (CoeffKArray0[33]),
    .Coeffk34       (CoeffKArray0[34]),
    .Coeffk35       (CoeffKArray0[35]),
    .Coeffk36       (CoeffKArray0[36]),
    .Coeffk37       (CoeffKArray0[37]),
    .Coeffk38       (CoeffKArray0[38]),
    .Coeffk39       (CoeffKArray0[39]),
    //Samples out to coefficient update
    .SampleXqk      (SampleXqArray0[0 ]), // SampleXq(k])
    .SampleXqkm1    (SampleXqArray0[1 ]), // SampleXq(k-1])
    .SampleXqkm2    (SampleXqArray0[2 ]), // SampleXq(k-2]) ... 
    .SampleXqkm3    (SampleXqArray0[3 ]),
    .SampleXqkm4    (SampleXqArray0[4 ]),
    .SampleXqkm5    (SampleXqArray0[5 ]),
    .SampleXqkm6    (SampleXqArray0[6 ]),
    .SampleXqkm7    (SampleXqArray0[7 ]),
    .SampleXqkm8    (SampleXqArray0[8 ]),
    .SampleXqkm9    (SampleXqArray0[9 ]),
    .SampleXqkm10   (SampleXqArray0[10]),
    .SampleXqkm11   (SampleXqArray0[11]),
    .SampleXqkm12   (SampleXqArray0[12]),
    .SampleXqkm13   (SampleXqArray0[13]),
    .SampleXqkm14   (SampleXqArray0[14]),
    .SampleXqkm15   (SampleXqArray0[15]),
    .SampleXqkm16   (SampleXqArray0[16]),
    .SampleXqkm17   (SampleXqArray0[17]),
    .SampleXqkm18   (SampleXqArray0[18]),
    .SampleXqkm19   (SampleXqArray0[19]),
    .SampleXqkm20   (SampleXqArray0[20]),
    .SampleXqkm21   (SampleXqArray0[21]),
    .SampleXqkm22   (SampleXqArray0[22]),
    .SampleXqkm23   (SampleXqArray0[23]),
    .SampleXqkm24   (SampleXqArray0[24]),
    .SampleXqkm25   (SampleXqArray0[25]),
    .SampleXqkm26   (SampleXqArray0[26]),
    .SampleXqkm27   (SampleXqArray0[27]),
    .SampleXqkm28   (SampleXqArray0[28]),
    .SampleXqkm29   (SampleXqArray0[29]),
    .SampleXqkm30   (SampleXqArray0[30]),
    .SampleXqkm31   (SampleXqArray0[31]),
    .SampleXqkm32   (SampleXqArray0[32]),
    .SampleXqkm33   (SampleXqArray0[33]),
    .SampleXqkm34   (SampleXqArray0[34]),
    .SampleXqkm35   (SampleXqArray0[35]),
    .SampleXqkm36   (SampleXqArray0[36]),
    .SampleXqkm37   (SampleXqArray0[37]),
    .SampleXqkm38   (SampleXqArray0[38]),
    .SampleXqkm39   (SampleXqArray0[39])
  );

`ifdef RW_NX_DERIV_FE_PATH1
  fIQComp #(
    .NTAPS(NTAPS),
    .NDEL (NDEL)
  )
  UfIQCompPath1 (
    //System
    .Clk            (Clk),
    .nRst           (nRst),
    //Control
    .Enable         (fIQCompEnable1),
    .CfgRegfIQCalEn (CfgRegfIQCalEn),
    .fIQDel         (fIQDel),
    //Data path I and Q
    .ValidXkp1      (ValidXkp1Path1),
    .SampleXikp1    (SampleXikp1Path1),
    .SampleXqkp1    (SampleXqkp1Path1),
    .ValidYk        (ValidYk1),
    .SampleYik      (SampleYik1),
    .SampleYqk      (SampleYqk1),
    .EstLineFilledP (EstLineFilledP1),
    //Coefficients
    .Coeffk0        (CoeffKArray1[0]),
    .Coeffk1        (CoeffKArray1[1]),
    .Coeffk2        (CoeffKArray1[2]),
    .Coeffk3        (CoeffKArray1[3]),
    .Coeffk4        (CoeffKArray1[4]),
    .Coeffk5        (CoeffKArray1[5]),
    .Coeffk6        (CoeffKArray1[6]),
    .Coeffk7        (CoeffKArray1[7]),
    .Coeffk8        (CoeffKArray1[8]),
    .Coeffk9        (CoeffKArray1[9]),
    .Coeffk10       (CoeffKArray1[10]),
    .Coeffk11       (CoeffKArray1[11]),
    .Coeffk12       (CoeffKArray1[12]),
    .Coeffk13       (CoeffKArray1[13]),
    .Coeffk14       (CoeffKArray1[14]),
    .Coeffk15       (CoeffKArray1[15]),
    .Coeffk16       (CoeffKArray1[16]),
    .Coeffk17       (CoeffKArray1[17]),
    .Coeffk18       (CoeffKArray1[18]),
    .Coeffk19       (CoeffKArray1[19]),
    .Coeffk20       (CoeffKArray1[20]),
    .Coeffk21       (CoeffKArray1[21]),
    .Coeffk22       (CoeffKArray1[22]),
    .Coeffk23       (CoeffKArray1[23]),
    .Coeffk24       (CoeffKArray1[24]),
    .Coeffk25       (CoeffKArray1[25]),
    .Coeffk26       (CoeffKArray1[26]),
    .Coeffk27       (CoeffKArray1[27]),
    .Coeffk28       (CoeffKArray1[28]),
    .Coeffk29       (CoeffKArray1[29]),
    .Coeffk30       (CoeffKArray1[30]),
    .Coeffk31       (CoeffKArray1[31]),
    .Coeffk32       (CoeffKArray1[32]),
    .Coeffk33       (CoeffKArray1[33]),
    .Coeffk34       (CoeffKArray1[34]),
    .Coeffk35       (CoeffKArray1[35]),
    .Coeffk36       (CoeffKArray1[36]),
    .Coeffk37       (CoeffKArray1[37]),
    .Coeffk38       (CoeffKArray1[38]),
    .Coeffk39       (CoeffKArray1[39]),
    //Samples out to coefficient update
    .SampleXqk      (SampleXqArray1[0 ]), // SampleXq(k])
    .SampleXqkm1    (SampleXqArray1[1 ]), // SampleXq(k-1])
    .SampleXqkm2    (SampleXqArray1[2 ]), // SampleXq(k-2]) ... 
    .SampleXqkm3    (SampleXqArray1[3 ]),
    .SampleXqkm4    (SampleXqArray1[4 ]),
    .SampleXqkm5    (SampleXqArray1[5 ]),
    .SampleXqkm6    (SampleXqArray1[6 ]),
    .SampleXqkm7    (SampleXqArray1[7 ]),
    .SampleXqkm8    (SampleXqArray1[8 ]),
    .SampleXqkm9    (SampleXqArray1[9 ]),
    .SampleXqkm10   (SampleXqArray1[10]),
    .SampleXqkm11   (SampleXqArray1[11]),
    .SampleXqkm12   (SampleXqArray1[12]),
    .SampleXqkm13   (SampleXqArray1[13]),
    .SampleXqkm14   (SampleXqArray1[14]),
    .SampleXqkm15   (SampleXqArray1[15]),
    .SampleXqkm16   (SampleXqArray1[16]),
    .SampleXqkm17   (SampleXqArray1[17]),
    .SampleXqkm18   (SampleXqArray1[18]),
    .SampleXqkm19   (SampleXqArray1[19]),
    .SampleXqkm20   (SampleXqArray1[20]),
    .SampleXqkm21   (SampleXqArray1[21]),
    .SampleXqkm22   (SampleXqArray1[22]),
    .SampleXqkm23   (SampleXqArray1[23]),
    .SampleXqkm24   (SampleXqArray1[24]),
    .SampleXqkm25   (SampleXqArray1[25]),
    .SampleXqkm26   (SampleXqArray1[26]),
    .SampleXqkm27   (SampleXqArray1[27]),
    .SampleXqkm28   (SampleXqArray1[28]),
    .SampleXqkm29   (SampleXqArray1[29]),
    .SampleXqkm30   (SampleXqArray1[30]),
    .SampleXqkm31   (SampleXqArray1[31]),
    .SampleXqkm32   (SampleXqArray1[32]),
    .SampleXqkm33   (SampleXqArray1[33]),
    .SampleXqkm34   (SampleXqArray1[34]),
    .SampleXqkm35   (SampleXqArray1[35]),
    .SampleXqkm36   (SampleXqArray1[36]),
    .SampleXqkm37   (SampleXqArray1[37]),
    .SampleXqkm38   (SampleXqArray1[38]),
    .SampleXqkm39   (SampleXqArray1[39])

    );
  `endif


  // Estimation
  ////////////////////////////////////////////////////////////////////
  
  // Falg indicating when Estimation is taking place
  assign EstOnGoing     = (fIQState==FIQSTATE_EST);  
  // Compute Estimation error
  assign EstErrork0     = {SampleYik0[11],SampleYik0} - {SampleYqk0[11],SampleYqk0};
`ifdef RW_NX_DERIV_FE_PATH1
  assign EstErrork1     = {SampleYik1[11],SampleYik1} - {SampleYqk1[11],SampleYqk1};
`endif

  // Estimation flow control: once the fIQComp delay lines are filled, they must be frozen
  // between samples until the coefficients have been updated.
  assign EstLineFilledP = EstLineFilledP0
`ifdef RW_NX_DERIV_FE_PATH1
                          || EstLineFilledP1
`endif
                          ;
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      EstUpdateOnGoing <= 1'b0;

    end else begin
      // Keep ready low from CompFilter delay line filled to update done.
      if (EstOnGoing==1'b1) begin
        if (EstLineFilledP == 1'b1)
          EstUpdateOnGoing <= 1'b1;
        else if (DoneOnNextCycle == 1'b1)
          EstUpdateOnGoing <= 1'b0;
      end else begin
        EstUpdateOnGoing   <= 1'b0;
      end
    end
  end
  assign EstReady       = !(EstLineFilledP || EstUpdateOnGoing);

  // Register controls to CoeffUpdate modules (big load)
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      EstSignErrork    <= 1'b0;
      EstNullErrork    <= 1'b0;
      EstErrorValidk   <= 1'b0;

    end else begin
`ifdef RW_NX_DERIV_FE_PATH1
      if (CfgRegfIQPathSel==FIQPATH0) begin
`endif      
        EstErrorValidk <= ValidYk0 && EstOnGoing;

        // 0 means EstErrork>0, 1 means EstErrork<0
        EstSignErrork  <= EstErrork0[12];

        // 0 means EstErrork!=0, 1 means EstErrork==0
        if (EstErrork0==13'd0) 
          EstNullErrork  <= 1'b1;
        else
          EstNullErrork  <= 1'b0;
`ifdef RW_NX_DERIV_FE_PATH1
      end else begin
        EstErrorValidk <= ValidYk1 && EstOnGoing;

        // 0 means EstErrork>0, 1 means EstErrork<0
        EstSignErrork  <= EstErrork1[12];

        // 0 means EstErrork!=0, 1 means EstErrork==0
        if (EstErrork1==13'd0) 
          EstNullErrork  <= 1'b1;
        else
          EstNullErrork  <= 1'b0;
      end
`endif      

    end
  end
  // Gate EstErrorValidk with CfgRegfIQCalEn to freeze coefficient update in case of abort
  // (error data from CompFilter is no more valid)
  wire EstErrorValidkEn;
  assign EstErrorValidkEn = EstErrorValidk && CfgRegfIQCalEn;

  // Save error status at each step
  // StatfIQToggle can be used as flag for update in another clock domain
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      StatfIQRxEstError0 <= 13'd0;
      StatfIQTxEstError0 <= 13'd0;
      StatfIQToggle0     <= 1'd0;
`ifdef RW_NX_DERIV_FE_PATH1
      StatfIQRxEstError1 <= 13'd0;
      StatfIQTxEstError1 <= 13'd0;
      StatfIQToggle1     <= 1'd0;
`endif
      
    end else begin

      if (EstErrorValidkEn==1'b1) begin
        // During estimation, Tx/Rx switch is under SW control
`ifdef RW_NX_DERIV_FE_PATH1
        if (CfgRegfIQPathSel==FIQPATH0) begin
`endif
          StatfIQToggle0 <= !StatfIQToggle0;
          if (CfgRegfIQTxRx==FIQTXRX_TX)
            StatfIQTxEstError0 <= EstErrork0;
          else
            StatfIQRxEstError0 <= EstErrork0;
`ifdef RW_NX_DERIV_FE_PATH1
        end else begin
          StatfIQToggle1 <= !StatfIQToggle1;
          if (CfgRegfIQTxRx==FIQTXRX_TX)
            StatfIQTxEstError1 <= EstErrork1;
          else
            StatfIQRxEstError1 <= EstErrork1;
        end
`endif
      end

    end
  end


  // Coefficients
  //////////////////////////////////////////////////////////////////////////////
  // Generate the NTAPS coefficients for Tx and Rx
  genvar m;
  generate 
    for (m = NTAPS; m < 40; m = m+1) begin: UnusedCoeffGen
      assign CoeffToUpdateKArray[m] = 16'd0;
    end // UnusedCoeffGen
  endgenerate

  genvar n;
  generate 
    for (n = 0; n < NTAPS; n = n+1) begin: CoeffUpdateGen
      
`ifdef RW_NX_DERIV_FE_PATH1
  assign CoeffToUpdateKArray[n] = (CfgRegfIQPathSel==FIQPATH0) ? CoeffKArray0[n]   : CoeffKArray1[n];
  assign SampleXqArray[n]       = (CfgRegfIQPathSel==FIQPATH0) ? SampleXqArray0[n] : SampleXqArray1[n];
`else
  assign CoeffToUpdateKArray[n] = CoeffKArray0[n];
  assign SampleXqArray[n]       = SampleXqArray0[n];
`endif

      // Instantiate the CoeffUpdate module shared betweeen Tx and for Rx, for each filter tap 
      CoeffUpdate UCoeffUpdate (
        //System
        .Clk              (Clk),
        .nRst             (nRst),
        //Control
        .CfgRegfIQEstStep (CfgRegfIQEstStep),
        //Update from estimation
        .EstErrorValidk   (EstErrorValidkEn),
        .EstSignErrork    (EstSignErrork),
        .EstNullErrork    (EstNullErrork),
        .SampleXk         (SampleXqArray[n]), // SampleXq[k-n]
        //Coefficient value
        .UpdatedCoeffValid(UpdatedCoeffValid[n]),
        .CoeffToUpdate    (CoeffToUpdateKArray[n]),
        .UpdatedCoeff     (UpdatedCoeffKArray[n])
      );

      // Update coefficient value: from estimation or from CfgReg
      // Gate UpdatedCoeffValid when the block is disabled (abort case)
      assign UpdatedCoeffValidEn[n] = UpdatedCoeffValid[n] && CfgRegfIQCalEn;
      always @(posedge Clk or negedge nCoeffRst)
      begin
        if (nCoeffRst == 1'b0) begin
          RxCoeffArray0[n]  <= 16'd0;
          TxCoeffArray0[n]  <= 16'd0;
        end else begin
 `ifdef RW_NX_DERIV_FE_PATH1
          if (CfgRegfIQPathSel==FIQPATH0) begin
 `endif
            // Priority to update from estimation
            if (UpdatedCoeffValidEn[n] == 1'b1) begin
              // Tx/Rx mux under SW control
              if (CfgRegfIQTxRx==FIQTXRX_TX)
                TxCoeffArray0[n] <= UpdatedCoeffKArray[n];
              else
                RxCoeffArray0[n] <= UpdatedCoeffKArray[n];
            end else if (CfgRegCoeffValid == 1'b1) begin
              // Tx/Rx mux under SW control
              if (CfgRegfIQTxRx==FIQTXRX_TX)
                TxCoeffArray0[n] <= CfgRegCoeffArray[n];
              else
                RxCoeffArray0[n] <= CfgRegCoeffArray[n];
            end
`ifdef RW_NX_DERIV_FE_PATH1
          end
`endif
        end
      end

`ifdef RW_NX_DERIV_FE_PATH1
      always @(posedge Clk or negedge nCoeffRst)
      begin
        if (nCoeffRst == 1'b0) begin
          RxCoeffArray1[n]  <= 16'd0;
          TxCoeffArray1[n]  <= 16'd0;
        end else begin
          if (CfgRegfIQPathSel==FIQPATH1) begin
            // Priority to update from estimation
            if (UpdatedCoeffValidEn[n] == 1'b1) begin
              // Tx/Rx mux under SW control
              if (CfgRegfIQTxRx==FIQTXRX_TX)
                TxCoeffArray1[n] <= UpdatedCoeffKArray[n];
              else
                RxCoeffArray1[n] <= UpdatedCoeffKArray[n];
            end else if (CfgRegCoeffValid == 1'b1) begin
              // Tx/Rx mux under SW control
              if (CfgRegfIQTxRx==FIQTXRX_TX)
                TxCoeffArray1[n] <= CfgRegCoeffArray[n];
              else
                RxCoeffArray1[n] <= CfgRegCoeffArray[n];
            end
          end
        end
      end
`endif

    end //CoeffUpdateGen
  endgenerate

  // All CoeffUpdate modules work at the same pace; Estimation process is controlled by instance 0
//  reg [15:0] DBGCount;
  assign DoneOnNextCycle  = UpdatedCoeffValid[0];
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
//      DBGCount <= 'd0;
      CoeffUpdateDoneP <= 1'b0;
    end else begin
      CoeffUpdateDoneP <= 1'b0; // Pulse to estimation control
      // Priority to update from estimation
      if (DoneOnNextCycle == 1'b1) begin
        CoeffUpdateDoneP <= 1'b1;
//        DBGCount <= DBGCount+1;
      end
    end
  end

  // Update from SW:
  // Detect rising edge on CfgRegCoeffUpdate to update coefficients
  // SW shall first reset CfgRegCoeffUpdate, then write all coefficient values,
  // then set CfgRegCoeffUpdate=1
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      CfgRegCoeffUpdateD  <= 1'b0;
      CfgRegCoeffValid    <= 1'b0;
    end else begin
      CfgRegCoeffUpdateD  <= CfgRegfIQCoeffUpdate;
      CfgRegCoeffValid    <= CfgRegfIQCoeffUpdate & !CfgRegCoeffUpdateD;
    end
  end
  

  // Arraies to ports conversion
  /////////////////////////////////

  genvar o;
  generate 
    for (o = 0; o < NTAPS; o = o+1) begin: StatCoeffGen
`ifdef RW_NX_DERIV_FE_PATH1
      // Need to assign in a wire for compatibility with UnusedStatCoeffGen
      assign CoeffStatArray[o] = ((CfgRegfIQPathSel==FIQPATH0) && (CfgRegfIQTxRx==FIQTXRX_TX)) ? TxCoeffArray0[o] :
                                 (
                                 ((CfgRegfIQPathSel==FIQPATH0) && (CfgRegfIQTxRx==FIQTXRX_RX)) ? RxCoeffArray0[o] :
                                 (
                                 ((CfgRegfIQPathSel==FIQPATH1) && (CfgRegfIQTxRx==FIQTXRX_TX)) ? TxCoeffArray1[o] :
                                 RxCoeffArray1[o]
                                 )
                                 );
`else
      assign CoeffStatArray[o] = (CfgRegfIQTxRx==FIQTXRX_TX) ? TxCoeffArray0[o] : RxCoeffArray0[o];
`endif
    end // StatCoeffGen
  endgenerate

  genvar p;
  generate 
    for (p = NTAPS; p < 40; p = p+1) begin: UnusedStatCoeffGen
      assign CoeffStatArray[p] = 16'd0;
    end // UnusedStatCoeffGen
  endgenerate
  
  assign Coeff0    = CoeffStatArray[0 ];
  assign Coeff1    = CoeffStatArray[1 ];
  assign Coeff2    = CoeffStatArray[2 ];
  assign Coeff3    = CoeffStatArray[3 ];
  assign Coeff4    = CoeffStatArray[4 ];
  assign Coeff5    = CoeffStatArray[5 ];
  assign Coeff6    = CoeffStatArray[6 ];
  assign Coeff7    = CoeffStatArray[7 ];
  assign Coeff8    = CoeffStatArray[8 ];
  assign Coeff9    = CoeffStatArray[9 ];
  assign Coeff10   = CoeffStatArray[10];
  assign Coeff11   = CoeffStatArray[11];
  assign Coeff12   = CoeffStatArray[12];
  assign Coeff13   = CoeffStatArray[13];
  assign Coeff14   = CoeffStatArray[14];
  assign Coeff15   = CoeffStatArray[15];
  assign Coeff16   = CoeffStatArray[16];
  assign Coeff17   = CoeffStatArray[17];
  assign Coeff18   = CoeffStatArray[18];
  assign Coeff19   = CoeffStatArray[19];
  assign Coeff20   = CoeffStatArray[20];
  assign Coeff21   = CoeffStatArray[21];
  assign Coeff22   = CoeffStatArray[22];
  assign Coeff23   = CoeffStatArray[23];
  assign Coeff24   = CoeffStatArray[24];
  assign Coeff25   = CoeffStatArray[25];
  assign Coeff26   = CoeffStatArray[26];
  assign Coeff27   = CoeffStatArray[27];
  assign Coeff28   = CoeffStatArray[28];
  assign Coeff29   = CoeffStatArray[29];
  assign Coeff30   = CoeffStatArray[30];
  assign Coeff31   = CoeffStatArray[31];
  assign Coeff32   = CoeffStatArray[32];
  assign Coeff33   = CoeffStatArray[33];
  assign Coeff34   = CoeffStatArray[34];
  assign Coeff35   = CoeffStatArray[35];
  assign Coeff36   = CoeffStatArray[36];
  assign Coeff37   = CoeffStatArray[37];
  assign Coeff38   = CoeffStatArray[38];
  assign Coeff39   = CoeffStatArray[39];

  assign CfgRegCoeffArray[0 ] = CfgRegfIQCoeff0;
  assign CfgRegCoeffArray[1 ] = CfgRegfIQCoeff1;
  assign CfgRegCoeffArray[2 ] = CfgRegfIQCoeff2;
  assign CfgRegCoeffArray[3 ] = CfgRegfIQCoeff3;
  assign CfgRegCoeffArray[4 ] = CfgRegfIQCoeff4;
  assign CfgRegCoeffArray[5 ] = CfgRegfIQCoeff5;
  assign CfgRegCoeffArray[6 ] = CfgRegfIQCoeff6;
  assign CfgRegCoeffArray[7 ] = CfgRegfIQCoeff7;
  assign CfgRegCoeffArray[8 ] = CfgRegfIQCoeff8;
  assign CfgRegCoeffArray[9 ] = CfgRegfIQCoeff9;
  assign CfgRegCoeffArray[10] = CfgRegfIQCoeff10;
  assign CfgRegCoeffArray[11] = CfgRegfIQCoeff11;
  assign CfgRegCoeffArray[12] = CfgRegfIQCoeff12;
  assign CfgRegCoeffArray[13] = CfgRegfIQCoeff13;
  assign CfgRegCoeffArray[14] = CfgRegfIQCoeff14;
  assign CfgRegCoeffArray[15] = CfgRegfIQCoeff15;
  assign CfgRegCoeffArray[16] = CfgRegfIQCoeff16;
  assign CfgRegCoeffArray[17] = CfgRegfIQCoeff17;
  assign CfgRegCoeffArray[18] = CfgRegfIQCoeff18;
  assign CfgRegCoeffArray[19] = CfgRegfIQCoeff19;
  assign CfgRegCoeffArray[20] = CfgRegfIQCoeff20;
  assign CfgRegCoeffArray[21] = CfgRegfIQCoeff21;
  assign CfgRegCoeffArray[22] = CfgRegfIQCoeff22;
  assign CfgRegCoeffArray[23] = CfgRegfIQCoeff23;
  assign CfgRegCoeffArray[24] = CfgRegfIQCoeff24;
  assign CfgRegCoeffArray[25] = CfgRegfIQCoeff25;
  assign CfgRegCoeffArray[26] = CfgRegfIQCoeff26;
  assign CfgRegCoeffArray[27] = CfgRegfIQCoeff27;
  assign CfgRegCoeffArray[28] = CfgRegfIQCoeff28;
  assign CfgRegCoeffArray[29] = CfgRegfIQCoeff29;
  assign CfgRegCoeffArray[30] = CfgRegfIQCoeff30;
  assign CfgRegCoeffArray[31] = CfgRegfIQCoeff31;
  assign CfgRegCoeffArray[32] = CfgRegfIQCoeff32;
  assign CfgRegCoeffArray[33] = CfgRegfIQCoeff33;
  assign CfgRegCoeffArray[34] = CfgRegfIQCoeff34;
  assign CfgRegCoeffArray[35] = CfgRegfIQCoeff35;
  assign CfgRegCoeffArray[36] = CfgRegfIQCoeff36;
  assign CfgRegCoeffArray[37] = CfgRegfIQCoeff37;
  assign CfgRegCoeffArray[38] = CfgRegfIQCoeff38;
  assign CfgRegCoeffArray[39] = CfgRegfIQCoeff39;
  
  
// debug print in waves
`ifdef RW_SIMU_ON
  // pragma coverage block = off, expr = off, toggle = off
  reg [20*8-1:0] strfIQState;
  always @(*)
    case(fIQState)
      FIQSTATE_TXCOMP : strfIQState = "TxComp";
      FIQSTATE_RXCOMP : strfIQState = "RxComp";
      FIQSTATE_EST    : strfIQState = "Est";
      FIQSTATE_COLL   : strfIQState = "SampleColl";
      default         : strfIQState = "Error";
    endcase

  // pragma coverage block = on, expr = on, toggle = on
`endif

  
endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
