//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module fIQRAMIf (
  input  wire        Clk,
  input  wire        nRst,
  
  input  wire        Enable,
  input  wire [1:0]  FeRamFreqRatio,
  input  wire        CollEstMode,
  input  wire        CollEstStartP,
  output reg         CollEstDone,
  input  wire        EstReady,
  input  wire        CoeffUpdateDoneP, 

  // Control registers
  // for sample collection
  input  wire [ 7:0] CfgRegfIQCollRdOffset,
  input  wire [ 7:0] CfgRegfIQCollWrOffset,
  input  wire [ 7:0] CfgRegfIQCollSize,
  input  wire [ 7:0] CfgRegfIQCollDel,
  input  wire        CfgRegfIQCollPath,
  // For estimation 
  input  wire [ 7:0] CfgRegfIQEstIOffset,
  input  wire [ 7:0] CfgRegfIQEstQOffset,
  input  wire [ 7:0] CfgRegfIQEstSize,
  input  wire [15:0] CfgRegfIQEstNite,

  // Input data: typ. from ADC for Sample Collection
  input  wire [11:0] rxInDataI,
  input  wire [11:0] rxInDataQ,
  // Output data: typ. to DAC for sample collection, else to CompFilter for estimation
  output wire        CollEstValid,
  output reg  [11:0] CollEstDataI,
  output reg  [11:0] CollEstDataQ,

  // RAM interface
  output reg  [7:0]  RamAddr,
  output reg         RamRdEn,
  output reg         RamWrEn,
  output reg  [63:0] RamWrData,
  input  wire [63:0] RamRdData

);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////
localparam FIQMODE_COLL = 1'b0, FIQMODE_EST = 1'b1;
localparam RATIO_TWICE = 2'd2, RATIO_ONE = 2'd1, RATIO_HALF = 2'd0;
localparam COLLPATH_I = 1'b0, COLLPATH_Q = 1'b1;
localparam EST_STATE_IDLE=3'd0,  EST_STATE_ADDRI=3'd1, EST_STATE_ADDRQ=3'd2,
           EST_STATE_DATA1=3'd3, EST_STATE_DATA2=3'd4;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  // Shared between Estimation and Sample Collection
  reg  [47:0] ShiftData1,ShiftData2; // Shift registers for I and Q data
  reg  [ 7:0] RamRdAddr;      // Sample collection: saved read address for interleaved rd/wr accesse
                              // Estimation: read address offset
  reg  [ 1:0] CntMod4;        // Sample collection: counter for RAM access cycles 
                              // Estimation: counter for samples per RAM data word
  reg  [15:0] DelayIterCount; // Sample collection (10 bits used): counts the delay before collection triggering
                              // Estimation: counts the number of iterations
  reg         CollEstBusy;
  reg         CollEstBusyD;
  wire        CollEstBusyRise;// internal start
  // Sample collection
  reg  [ 7:0] RamWrAddr; // Save write addresses for interleaved rd/wr accesses during sample collection
  wire [ 7:0] CollEndWrAddr;
  wire        CollLoadRdData;
  wire        CollRdEnNext;
  wire        CollWrEnNext;
  wire [ 7:0] CollWrapRdAddr;
  // Sample collection trig delay
  reg [ 9:0]  CollEndCount;
  reg         CollDelayDone;
  // Estimation
  reg  [ 2:0] EstState;
  reg         CollEstValidInt;
  wire [ 7:0] EstSizeWrap;
  reg         Toggle;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  // Sample collection trigger
  // Adapt CfgRegfIQCollDel value according to FeRamFreqRatio
  // Actual value used is CfgRegfIQCollDel*2^FeRamFreqRatio
  // so that the delay is the same when Clk frequency goes up or down
  always @(*)
  begin
    case (FeRamFreqRatio)
      RATIO_HALF : CollEndCount = {2'b0,CfgRegfIQCollDel};
      RATIO_ONE  : CollEndCount = {1'b0,CfgRegfIQCollDel,1'b0};
      default    : CollEndCount = {CfgRegfIQCollDel,2'b0}; //RATIO_TWICE
    endcase
  end
  
  // Detect sample collection end when last address is written
  assign CollEndWrAddr = CfgRegfIQCollWrOffset+CfgRegfIQCollSize-8'd1;

  // Sample collection and Estimation control
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      DelayIterCount    <= 16'd0;
      CollEstDone       <= 1'b1;
      CollEstBusy       <= 1'b0;
      CollEstBusyD      <= 1'b0;
      CollDelayDone     <= 1'b0;

    end else if (Enable==1'b0) begin
      DelayIterCount    <= 16'd0;
      CollEstDone       <= 1'b1;
      CollEstBusy       <= 1'b0;
      CollEstBusyD      <= 1'b0;
      CollDelayDone     <= 1'b0;

    end else begin
      CollEstBusyD      <= CollEstBusy;

      if (CollEstStartP == 1'b1) begin
        CollEstBusy     <= 1'd1;
        CollEstDone     <= 1'b0;
        CollDelayDone   <= 1'b0;
        // Delay counter init value depends on the mode
        if (CollEstMode==FIQMODE_COLL) 
          DelayIterCount  <= {6'd0, CollEndCount};
        else
          // -1 because module finishes the started iteration when DelayIterCount reaches 0, so assert stop one iteration before end  
          DelayIterCount  <= CfgRegfIQEstNite-16'd1;
      end else if (CollEstDone == 1'b1) begin 
        CollEstBusy     <= 1'd0;
      end else if (CollEstBusy == 1'b1) begin
        // Sample coollection
        if (CollEstMode==FIQMODE_COLL) begin
          // Count trigger delay
          if (DelayIterCount[9:0]==10'd0) begin
            // Align trig on possible write access
            if (CntMod4==2'd0)
              CollDelayDone  <= 1'b1;
          end else begin
            DelayIterCount <= DelayIterCount-16'd1;
            CollDelayDone  <= 1'b0;
          end
          // Detect end of sample collection on last write access
          if ((RamWrAddr==CollEndWrAddr) && (CntMod4==2'd3) && (CollDelayDone==1'b1))
            CollEstDone    <= 1'b1;

        // Estimation
        end else begin
          if (CoeffUpdateDoneP==1'b1)
            DelayIterCount <= DelayIterCount-16'd1;
          if (DelayIterCount==16'd0)
            CollEstDone   <= 1'b1;

        end
      end

    end
  end    
  assign CollEstBusyRise = CollEstBusy && !CollEstBusyD;

  
  // Sample collection: RAM accesses follow a fixed pattern driven by the FeRamFreqRatio, which indicates the RAM/FE
  // clock ratio. When FE clock is twice RAM clock (80 MHz), there is no margin and the read and write 
  // accesses must be exactly interleaved always in the same way.
  
  // RAM read access start is aligned with CntMod4=0.
  assign CollRdEnNext = (CntMod4==2'd3) ||
                        ((FeRamFreqRatio==RATIO_TWICE) && (CntMod4==2'd0)); //  FE clock is 2x RAM clock: RamRdEn must stay high 2 cycles
                         
  // When to sample RAM read data:
  assign CollLoadRdData = ((FeRamFreqRatio==RATIO_HALF) && (CntMod4==2'd0)) || // RAM clock is 2x FE clock so RdData already available
                          ((FeRamFreqRatio==RATIO_ONE) && (CntMod4==2'd1)) || // RAM clock = FE clock so RdData available 1cc after RdEn
                          ((FeRamFreqRatio==RATIO_TWICE) && (CntMod4==2'd2));   // FE clock is 2x RAM clock: wait CntMod4=2 
                                                                 // so that RdData is available for all clock alignments
                                                                     
  // RAM write access start is aligned with CntMod4=2.
  assign CollWrEnNext = (CntMod4==2'd1) ||
                        ((FeRamFreqRatio==RATIO_TWICE) && (CntMod4==2'd2)); //  FE clock is 2x RAM clock: RamWrEn must stay high 2 cycles

  assign CollWrapRdAddr = CfgRegfIQCollRdOffset+CfgRegfIQCollSize-8'd1;

  // RAM control
  always @(posedge Clk or negedge nRst)
  begin
    if (nRst == 1'b0) begin
      // RAM interface
      RamAddr          <= 8'd0;
      RamRdEn          <= 1'b0;
      RamWrEn          <= 1'b0;
      RamWrData        <= 64'd0;
      // Common
      RamRdAddr        <= 8'd0;
      ShiftData1       <= 48'd0;
      ShiftData2       <= 48'd0;
      CntMod4          <= 2'd3;
      CollEstDataI     <= 12'd0;
      CollEstDataQ     <= 12'd0;
      CollEstValidInt  <= 1'b0;
      // Sample collection
      RamWrAddr        <= 8'd0;
      // Estimation
      EstState         <= EST_STATE_IDLE;
      Toggle           <= 1'b0;

    end else if ((Enable==1'b0) || (CollEstBusy==1'b0)) begin
      // RAM interface
      RamAddr          <= 8'd0;
      RamRdEn          <= 1'b0;
      RamWrEn          <= 1'b0;
      RamWrData        <= 64'd0;
      // Common
      RamRdAddr        <= 8'd0;
      ShiftData1       <= 48'd0;
      ShiftData2       <= 48'd0;
      CntMod4          <= 2'd3;
      CollEstDataI     <= 12'd0;
      CollEstDataQ     <= 12'd0;
      CollEstValidInt  <= 1'b0;
      // Sample collection
      RamWrAddr        <= 8'd0;
      // Estimation
      EstState         <= EST_STATE_IDLE;
      Toggle           <= 1'b0;

    end else begin
      Toggle           <= !Toggle;
      
      // Sample Collection
      ///////////////////////////////////////////////////////
      if (CollEstMode==FIQMODE_COLL) begin
        CntMod4      <= CntMod4+2'd1;
        // Read cycle
        RamRdEn <= CollRdEnNext;

        // Read and Write address
        if (CollEstBusyRise==1'b1) begin
          RamAddr   <= CfgRegfIQCollRdOffset;
          RamRdAddr <= CfgRegfIQCollRdOffset;
          RamWrAddr <= CfgRegfIQCollWrOffset-8'd1;
        end else begin

          // Read access is done on cycles 0 (and 1 in 80 MHz mode) of CntMod4
          if (CntMod4==2'd3) begin
            if (RamRdAddr==CollWrapRdAddr) begin 
              RamAddr   <= CfgRegfIQCollRdOffset;
              RamRdAddr <= CfgRegfIQCollRdOffset;
            end else begin 
              RamAddr   <= RamRdAddr+8'd1;
              RamRdAddr <= RamRdAddr+8'd1;
            end
          // Write access is done on cycles 2 (and 3 in 80 MHz mode) of CntMod4 
          end else if (CntMod4==2'd1)
            if (CollDelayDone==1'b1) begin
              RamAddr   <= RamWrAddr+8'd1;
              RamWrAddr <= RamWrAddr+8'd1;
            end
          end
        
          // Shift / load with RdData internal data register
          if (CollLoadRdData == 1'b1)
            // Load 4 samples aligned at 16bits boundaries
            ShiftData1 <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
          else
            ShiftData1 <= {ShiftData1[11:0],ShiftData1[47:12]};

          // Set valid immediately, even if ShiftData1 is not yet loaded. Delay trigger is used
          // to remove transient from collected samples.
          CollEstValidInt <= 1'b1;
          // Send RAM data to both I/Q paths. Loopback path selection is done inside the RF
          CollEstDataI <= ShiftData1[11:0];
          CollEstDataQ <= ShiftData1[11:0];

          // Always shift / load WrData from selected I/Q path,
          // so that shift register is fully loaded when write trigger is received 
          if (CfgRegfIQCollPath==COLLPATH_I)
            ShiftData2 <= {rxInDataI,ShiftData2[47:12]};
          else
            ShiftData2 <= {rxInDataQ,ShiftData2[47:12]};

          // Write cycle
          if (CollDelayDone==1'b1) begin
            RamWrEn <= CollWrEnNext;
            if (CntMod4==2'd1)
              // Align 4 samples to 16bits boundaries
              RamWrData <= {4'd0,ShiftData2[47:36],4'd0,ShiftData2[35:24],4'd0,ShiftData2[23:12],4'd0,ShiftData2[11:0]};
          end else begin // waiting for trigger
            RamWrEn   <= 1'b0;
            RamWrData <= 64'd0;
          end
        
      // Estimation - Idle
      ///////////////////////////////////////////////////////
      end else begin
        // Reset Valid when sample has been sampled by next block.
        if (EstReady==1'b1)
          CollEstValidInt <= 1'b0;
        
        // From any state, stop when all iterations done
        if (CollEstDone==1'b1)
          EstState        <= EST_STATE_IDLE;

        case (EstState)
          // Idle. Reset signals / Prepare first RAM read access for I sample
          EST_STATE_IDLE : begin
            RamRdEn         <= 1'b0;
            CntMod4         <= 2'd3; // reset value needed for Sample collection
            Toggle          <= 1'b0;
            if (CollEstBusyRise == 1'b1) begin
              EstState      <= EST_STATE_ADDRI;
              RamAddr       <= CfgRegfIQEstIOffset;
              RamRdEn       <= 1'b1;
              RamRdAddr     <= 8'd0;
            end
          end

          // Address for I sample is on RAM address bus.
          // In 20 MHz, RAM clock is twice FE clock so data for I sample is already on RAM read data bus.
          // Prepare RAM read access for Q sample.
          EST_STATE_ADDRI : begin
            // 80 MHz: FE clock is twice RAM clock, so wait for Toggle so that address is kept for two cycles
            if ((Toggle == 1'b1) || (FeRamFreqRatio!=RATIO_TWICE)) begin
              EstState      <= EST_STATE_ADDRQ;
              RamAddr       <= CfgRegfIQEstQOffset+RamRdAddr;
              RamRdEn       <= 1'b1;
              // In 20 MHz, RAM clock is twice FE clock so I sample on RdData is ready to be sampled
              if (FeRamFreqRatio==RATIO_HALF)
                ShiftData1  <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
              // Prepare next offset
              if (RamRdAddr<EstSizeWrap)
                RamRdAddr <= RamRdAddr+8'd1;
              else
                RamRdAddr <= 8'd0;
            end
          end

          // Address for Q sample is on RAM address bus.
          EST_STATE_ADDRQ : begin
            case(FeRamFreqRatio)
              // 20 MHz - RAM clock is twice FE clock so Q sample on RdData is ready to be sampled
              RATIO_HALF  : begin
                RamRdEn       <= 1'b0;
                ShiftData2    <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
                CntMod4       <= 2'd0;
                // Skip EST_STATE_DATA1 as Q sample already sampled
                EstState      <= EST_STATE_DATA2;
              end
              // 40 MHz - FE clock is equal to RAM clock. I sample ready to be sampled on RdData
              RATIO_ONE  : begin
                RamRdEn       <= 1'b0;
                ShiftData1    <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
                EstState      <= EST_STATE_DATA1;
              end
              // 80 MHz - FE clock is twice RAM clock
              default : begin
                // Wait for Toggle so that address is kept for two cycles
                if (Toggle == 1'b1) begin
                  RamRdEn     <= 1'b0;
                  EstState    <= EST_STATE_DATA1;
                // Sample I data without waiting for Toggle, else RdData is gone
                end else begin
                  ShiftData1  <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
                end
              end
            endcase
          end

          // I samples available in ShiftData1
          // Store Q sample in ShiftData2 (for 20 and 40 MHz, this state is skipped in 80 MHz)
          EST_STATE_DATA1 : begin
            RamRdEn       <= 1'b0; // if 20 MHz, RdEn was still high
            EstState      <= EST_STATE_DATA2;
            // Sample Q data
            ShiftData2    <= {RamRdData[59:48],RamRdData[43:32],RamRdData[27:16],RamRdData[11:0]};
            CntMod4       <= 2'd0;
          end

          // I and Q samples available in ShiftData1/2, shift data out based on Valid/Ready 
          EST_STATE_DATA2 : begin
            if ((EstReady==1'b1) || (CollEstValidInt==1'b0)) begin
              CntMod4         <= CntMod4+2'd1; // Sample counter
              ShiftData2      <= {ShiftData2[11:0],ShiftData2[47:12]};
              ShiftData1      <= {ShiftData1[11:0],ShiftData1[47:12]};
              CollEstDataI    <= ShiftData1[11:0];
              CollEstDataQ    <= ShiftData2[11:0];
              CollEstValidInt <= 1'b1;
              // Last sample is shifted out, start over
              if (CntMod4==2'd3) begin 
                EstState      <= EST_STATE_ADDRI;
                RamAddr       <= CfgRegfIQEstIOffset+RamRdAddr;
                RamRdEn       <= 1'b1;
                Toggle        <= 1'b0;
              end
            end
          end

          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default  : begin
            EstState        <= EST_STATE_IDLE;
          end
          // pragma coverage block = on 
        endcase
      end

    end // Enable == 1

  end

  // Estimation state machine
  assign CollEstValid = CollEstValidInt && EstReady; // freeze output flow if estimation block not ready
  assign EstSizeWrap  = CfgRegfIQEstSize-8'd1;


// debug print in waves
`ifdef RW_SIMU_ON
  // pragma coverage block = off, expr = off, toggle = off
  reg [20*8-1:0] strEstState;
  always @(*)
    case(EstState)
      EST_STATE_IDLE    : strEstState = "Idle";
      EST_STATE_ADDRI   : strEstState = "AddrI";
      EST_STATE_ADDRQ   : strEstState = "AddrQ";
      EST_STATE_DATA1   : strEstState = "Data1";
      EST_STATE_DATA2   : strEstState = "Data2";
      default     : strEstState = "Error";
    endcase
  // pragma coverage block = on, expr = on, toggle = on
`endif

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
