//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 29038 $
// $Date: 2016-11-27 10:22:41 +0100 (Sun, 27 Nov 2016) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of iq_comp module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module iq_comp( 
  // Clock and reset
  input  wire        clk,     // FE Clock (160/80/40 MHz)
  input  wire        reset_n, // Global asynchronous reset (active LOW)

  // Control
  input  wire        iq_gain_comp_en,
  input  wire        iq_phase_comp_en,

  // Estimates
  input  wire [10:0] iq_gain,
  input  wire [8:0]  iq_phase,
    
  // Input data
  input  wire [11:0] in_i,
  input  wire [11:0] in_q,
  input  wire        in_en,

  // Outputs data
  output wire [11:0] out_i,
  output wire [11:0] out_q,
  output wire        out_en
  );

//////////////////////////////////////////////////////////////////////////////
// Internal Regs declarations
//////////////////////////////////////////////////////////////////////////////
  reg [11:0] in_i_1t;
  reg [10:0] iq_gain_1t;
  reg [9:0]  iq_phase_1t;
  reg        in_en_1t;
  reg        in_en_2t;
  reg        in_en_3t;
  reg        in_en_4t;
  reg        in_en_5t;

  reg [23:0] q_gain_mult;
  reg [11:0] gain_i_comp;
  reg [11:0] gain_q_comp;
  reg [11:0] gain_i_comp_1t;
  reg [11:0] gain_q_comp_1t;
  reg [11:0] gain_i_comp_2t;
  reg [11:0] gain_q_comp_2t;

  reg [21:0] i_phase_mult;
  reg [21:0] q_phase_mult;

  reg [11:0] i_phase_mult_rnd_1t;
  reg [11:0] q_phase_mult_rnd_1t;

  reg [11:0] phase_i_comp;
  reg [11:0] phase_q_comp;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  wire [13:0] q_gain_mult_rnd;
  wire [11:0] q_gain_mult_rnd_sat;

  wire [11:0] gain_i_mux;
  wire [11:0] gain_q_mux;
  wire        gain_en_mux;

  wire [11:0] i_phase_mult_rnd;
  wire [11:0] q_phase_mult_rnd;

  wire [12:0] i_phase_accu;
  wire [12:0] q_phase_accu;
  wire [11:0] i_phase_accu_sat;
  wire [11:0] q_phase_accu_sat;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Register inputs
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    iq_gain_1t  <= 11'b0;
    iq_phase_1t <= 10'b0;
    in_i_1t     <= 12'b0;
  end
  else 
  begin
    iq_gain_1t  <= iq_gain;
    iq_phase_1t <= (~{iq_phase[8],iq_phase}) + 10'd1;
    in_i_1t     <= in_i;
  end
end

// Register data enable
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    in_en_1t <= 1'b0;
    in_en_2t <= 1'b0;
    in_en_3t <= 1'b0;
    in_en_4t <= 1'b0;
    in_en_5t <= 1'b0;
  end
  else 
  begin
    in_en_1t <= in_en;
    in_en_2t <= in_en_1t;
    in_en_3t <= in_en_2t;
    in_en_4t <= in_en_3t;
    in_en_5t <= in_en_4t;
  end
end

///////////////////////////////////////////////
// Compute gain compensation
///////////////////////////////////////////////
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    q_gain_mult <= 24'b0;
  else 
    q_gain_mult <= $signed(in_q) * $signed({1'b0,iq_gain_1t});
end

// Rounding by 10-bit
Round #(
  .INPUT_WIDTH(24),
  .OUTPUT_WIDTH(14)
  )
  u_rnd_gain_q(
  .InputData(q_gain_mult),
  .RoundData(q_gain_mult_rnd)
  );

// Saturation 12-bit
SatSigned # (
  .INPUT_WIDTH(14),
  .OUTPUT_WIDTH(12)
  )
  u_sat_gain_q (
  .InputData(q_gain_mult_rnd),
  .SatData(q_gain_mult_rnd_sat)
  );

// Register gain compensation output
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    gain_i_comp <= 12'b0;
    gain_q_comp <= 12'b0;
  end
  else 
  begin
    gain_i_comp <= in_i_1t;
    gain_q_comp <= q_gain_mult_rnd_sat;
  end
end

// Mux to by-pass IQ gain compensation
assign gain_i_mux  = (iq_gain_comp_en) ? gain_i_comp : in_i;
assign gain_q_mux  = (iq_gain_comp_en) ? gain_q_comp : in_q;
assign gain_en_mux = (iq_gain_comp_en) ? in_en_2t    : in_en;

///////////////////////////////////////////////
// Compute phase compensation
///////////////////////////////////////////////
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    i_phase_mult <= 22'b0;
    q_phase_mult <= 22'b0;
  end
  else 
  begin
    i_phase_mult <= $signed(gain_i_mux) * $signed({{2{iq_phase_1t[9]}},iq_phase_1t});
    q_phase_mult <= $signed(gain_q_mux) * $signed({{2{iq_phase_1t[9]}},iq_phase_1t});
  end
end

// Rounding by 10-bit
Round #(
  .INPUT_WIDTH(22),
  .OUTPUT_WIDTH(12)
  )
  u_rnd_phase_i(
  .InputData(i_phase_mult),
  .RoundData(i_phase_mult_rnd)
  );

Round #(
  .INPUT_WIDTH(22),
  .OUTPUT_WIDTH(12)
  )
  u_rnd_phase_q(
  .InputData(q_phase_mult),
  .RoundData(q_phase_mult_rnd)
  );


// Register phase compensation after rounding
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    i_phase_mult_rnd_1t <= 12'b0;
    q_phase_mult_rnd_1t <= 12'b0;
  end
  else 
  begin
    i_phase_mult_rnd_1t <= i_phase_mult_rnd;
    q_phase_mult_rnd_1t <= q_phase_mult_rnd;
  end
end

// Delayed gain compensation output by 2cc
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    gain_i_comp_1t <= 12'b0;
    gain_q_comp_1t <= 12'b0;
    gain_i_comp_2t <= 12'b0;
    gain_q_comp_2t <= 12'b0;
  end
  else 
  begin
    gain_i_comp_1t <= gain_i_mux;
    gain_q_comp_1t <= gain_q_mux;
    gain_i_comp_2t <= gain_i_comp_1t;
    gain_q_comp_2t <= gain_q_comp_1t;
  end
end

// Compute adder
assign i_phase_accu = $signed({q_phase_mult_rnd_1t[11],q_phase_mult_rnd_1t}) + $signed({gain_i_comp_2t[11],gain_i_comp_2t});
assign q_phase_accu = $signed({i_phase_mult_rnd_1t[11],i_phase_mult_rnd_1t}) + $signed({gain_q_comp_2t[11],gain_q_comp_2t});

// Saturation 12-bit
SatSigned # (
  .INPUT_WIDTH(13),
  .OUTPUT_WIDTH(12)
  )
  u_sat_phase_i (
  .InputData(i_phase_accu),
  .SatData(i_phase_accu_sat)
  );

SatSigned # (
  .INPUT_WIDTH(13),
  .OUTPUT_WIDTH(12)
  )
  u_sat_phase_q (
  .InputData(q_phase_accu),
  .SatData(q_phase_accu_sat)
  );

// Register phase compensation output
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
  begin
    phase_i_comp <= 12'b0;
    phase_q_comp <= 12'b0;
  end
  else 
  begin
    phase_i_comp <= i_phase_accu_sat;
    phase_q_comp <= q_phase_accu_sat;
  end
end

// Mux to by-pass IQ phase compensation
assign out_i  = (iq_phase_comp_en) ? phase_i_comp : gain_i_mux;
assign out_q  = (iq_phase_comp_en) ? phase_q_comp : gain_q_mux;
assign out_en = (iq_phase_comp_en) ? in_en_5t     : gain_en_mux;

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
