//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39825 $
// $Date: 2019-09-19 11:14:27 +0200 (Thu, 19 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Estimation controller module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module iq_est_ctrl(
  // Clock and reset
  input  wire        clk,     // FE Clock (160/80/40 MHz)
  input  wire        reset_n, // Global asynchronous reset (active LOW)

  // registers (asynchronous)
  input  wire [1:0]  reg_cbw,
  //
  input  wire        reg_iq_est_iter_clr,
  //
  input  wire [7:0]  reg_iq_est_snr_min,
  input  wire [3:0]  reg_iq_est_mcs_min,
  input  wire [2:0]  reg_iq_est_bw_list,
  input  wire [7:0]  reg_iq_est_fo_min,
  //
  input  wire        reg_iq_est_phase_en,
  input  wire        reg_iq_est_gain_en,
  //
  input  wire [2:0]  reg_iq_est_del,
  //
  input  wire [2:0]  reg_iq_est_gain_step_init,
  input  wire [2:0]  reg_iq_est_gain_step_stop,
  input  wire [2:0]  reg_iq_est_phase_step_init,
  input  wire [2:0]  reg_iq_est_phase_step_stop,
  //
  input  wire [2:0]  reg_iq_est_step_del0,
  input  wire [2:0]  reg_iq_est_step_del1,
  input  wire [2:0]  reg_iq_est_step_del2,
  input  wire [2:0]  reg_iq_est_step_del3,
  input  wire [2:0]  reg_iq_est_step_del4,
  //
  output reg         stat_reg_valid,

  // Control from AGC
  input  wire        iq_est_gain_acc_en,
  input  wire        iq_est_phase_acc_en,
  input  wire [7:0]  snr,
  input  wire        agc_lock,

  // Control from modem
  input  wire        frame_leg,
  input  wire        frame_vht,
  input  wire [3:0]  legrate,
  input  wire [6:0]  mcs,
  input  wire        bweq20,
  input  wire        bweq40,
  input  wire        bweq80,
  input  wire [7:0]  fo_est,

  // Output control to error estimations
  output wire        err_clr,
  output wire        err_en,

  // Output control to loop filters
  output reg [2:0]  gain_step,
  output reg [2:0]  phase_step,
  output wire       iter_gain_inc,
  output wire       iter_phase_inc
  );

//////////////////////////////////////////////////////////////////////////////
// Local parameters declarations
//////////////////////////////////////////////////////////////////////////////
  localparam  LEG_6=4'hb,LEG_9=4'hf,LEG_12=4'ha,LEG_18=4'he,LEG_24=4'h9,
              LEG_36=4'hd,LEG_48=4'h8,LEG_54=4'hc;


//////////////////////////////////////////////////////////////////////////////
// Internal Regs declarations
//////////////////////////////////////////////////////////////////////////////
  reg [8:0]  iter_gain_count;
  reg [8:0]  iter_phase_count;
  reg        iter_inc;
  reg        agc_lock_1t;
  reg [9:0]  err_en_count;
  reg [7:0]  err_en_count_max;
  reg        iq_est_gain_acc_en_1t;
  reg        iq_est_gain_acc_en_2t;
  reg        iq_est_phase_acc_en_1t;
  reg        iq_est_phase_acc_en_2t;
  reg        acc_en_1t;
  reg [3:0]  mcs_decode;
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  wire        iter_inc_snr;
  wire        iter_inc_mcs;
  wire        iter_inc_bw;
  wire        iter_inc_fo;

  wire [3:0]  step_sum0;
  wire [4:0]  step_sum1;
  wire [4:0]  step_sum2;
  wire [5:0]  step_sum3;

  wire [9:0]  err_en_count_max_bw;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Decode MCS from frame format and legacy rate / (V)HT-MCS
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    mcs_decode <= 4'b0;
  else if (frame_leg) begin
    case(legrate)
      LEG_6:    mcs_decode <= 4'd0;
      LEG_9:    mcs_decode <= 4'd1;
      LEG_12:   mcs_decode <= 4'd2;
      LEG_18:   mcs_decode <= 4'd3;
      LEG_24:   mcs_decode <= 4'd4;
      LEG_36:   mcs_decode <= 4'd5;
      LEG_48:   mcs_decode <= 4'd6;
      default:  mcs_decode <= 4'd7;
    endcase   
  end
  else begin
    case (mcs)
      7'd0      ,7'd16,7'd24 : mcs_decode <= 4'd0;
      7'd1      ,7'd17,7'd25 : mcs_decode <= 4'd1;
      7'd2      ,7'd18,7'd26 : mcs_decode <= 4'd2;
      7'd3      ,7'd19,7'd27 : mcs_decode <= 4'd3;
      7'd4,7'd12,7'd20,7'd28 : mcs_decode <= 4'd4;
      7'd5,7'd13,7'd21,7'd29 : mcs_decode <= 4'd5;
      7'd6,7'd14,7'd22,7'd30 : mcs_decode <= 4'd6;
      7'd7,7'd15,7'd23,7'd31 : mcs_decode <= 4'd7;
      7'd8                   : begin
                               if (frame_vht)
                                 mcs_decode <= 4'd8;
                               else
                                 mcs_decode <= 4'd0;
                               end
      7'd9                   : begin
                               if (frame_vht)
                                 mcs_decode <= 4'd9;
                               else
                                 mcs_decode <= 4'd1;
                              end
      7'd10                  : begin
                               if (frame_vht)
                                 mcs_decode <= 4'd10;
                               else
                                 mcs_decode <= 4'd2;
                               end
      7'd11                  : begin
                               if (frame_vht)
                                 mcs_decode <= 4'd11;
                               else
                                 mcs_decode <= 4'd3;
                              end
      default                : mcs_decode <= 4'b0; // For unequal modulation, set MCS 0
    endcase
  end
end

// Iteration counters
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0) begin
    iter_gain_count  <= 9'b0;
    iter_phase_count <= 9'b0;
  end
  else if (reg_iq_est_iter_clr) begin // Clear on SW request
    iter_gain_count  <= 9'b0;
    iter_phase_count <= 9'b0;
  end
  else begin                          // Increment iteration
    if (iter_gain_inc && (iter_gain_count[8:5] != 4'd9)) // Limit iteration to 9'd288
      iter_gain_count  <= iter_gain_count + 9'd1;
    if (iter_phase_inc && (iter_phase_count[8:5] != 4'd9)) // Limit iteration to 9'd288
      iter_phase_count <= iter_phase_count + 9'd1;
  end
end

// Iteration counter increment
assign iter_inc_snr = (snr >= reg_iq_est_snr_min);
assign iter_inc_mcs = (mcs_decode >= reg_iq_est_mcs_min);
assign iter_inc_bw  = (bweq20 & reg_iq_est_bw_list[0]) |
                      (bweq40 & reg_iq_est_bw_list[1]) |
                      (bweq80 & reg_iq_est_bw_list[2]);
assign iter_inc_fo  = (fo_est >= reg_iq_est_fo_min);


always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    iter_inc <= 1'b0;
  else
    iter_inc <= iter_inc_snr & iter_inc_mcs & iter_inc_bw & iter_inc_fo;
end

assign iter_gain_inc  = iter_inc & iq_est_gain_acc_en_1t  & ~iq_est_gain_acc_en_2t  & reg_iq_est_phase_en;
assign iter_phase_inc = iter_inc & iq_est_phase_acc_en_1t & ~iq_est_phase_acc_en_2t & reg_iq_est_gain_en;

always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0) begin
    iq_est_gain_acc_en_1t  <= 1'b0;
    iq_est_gain_acc_en_2t  <= 1'b0;
    iq_est_phase_acc_en_1t <= 1'b0;
    iq_est_phase_acc_en_2t <= 1'b0;
  end
  else begin
    iq_est_gain_acc_en_1t  <= iq_est_gain_acc_en;
    iq_est_gain_acc_en_2t  <= iq_est_gain_acc_en_1t;
    iq_est_phase_acc_en_1t <= iq_est_phase_acc_en;
    iq_est_phase_acc_en_2t <= iq_est_phase_acc_en_1t;
  end
end

// IQ error control
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    err_en_count <= 10'b0;
  else if (!agc_lock_1t)
    err_en_count <= 10'b0;
  else if (err_en_count<=err_en_count_max_bw)
    err_en_count <= err_en_count + 10'd1;
end

// Compute err_en_count_max according to reg_iq_est_del
always @*
begin
  case(reg_iq_est_del)
    3'd0:    err_en_count_max=8'd32;  // 1x0.8us=0.8us
    3'd1:    err_en_count_max=8'd64;  // 2x0.8us=1.6us
    3'd2:    err_en_count_max=8'd96;  // 3x0.8us=2.4us
    3'd3:    err_en_count_max=8'd128; // 4x0.8us=3.2us
    3'd4:    err_en_count_max=8'd160; // 5x0.8us=4.0us
    3'd5:    err_en_count_max=8'd192; // 6x0.8us=4.8us
    default: err_en_count_max=8'd224; // 7x0.8us=5.6us
  endcase
end

// Compute err_en_count_max_bw according to reg_iq_est_del
assign err_en_count_max_bw =
`ifdef RW_NX_DERIV_FE_PATH80M
           (reg_cbw==2'd2) ? {err_en_count_max,2'b0} :
`endif
`ifdef RW_NX_DERIV_FE_PATH40M
           (reg_cbw==2'd1) ? {1'b0,err_en_count_max,1'b0} :
`endif
                             {2'b0,err_en_count_max};

always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    agc_lock_1t <= 1'b0;
  else
    agc_lock_1t <= agc_lock;
end

assign err_clr = agc_lock & ~agc_lock_1t;
assign err_en  = agc_lock_1t & (err_en_count<=err_en_count_max_bw);

// Decompose each increments for comparison
assign step_sum0 = {1'b0,reg_iq_est_step_del0} + {1'b0,reg_iq_est_step_del1};
assign step_sum1 = {1'b0,step_sum0} + {2'b0,reg_iq_est_step_del2};
assign step_sum2 = step_sum1        + {2'b0,reg_iq_est_step_del3};
assign step_sum3 = {1'b0,step_sum2} + {3'b0,reg_iq_est_step_del4};

// Compute gain and phase step before error shift
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    gain_step <= 3'b0;
  else if(iter_gain_count < {3'b0,reg_iq_est_step_del0,3'b0})
    gain_step <= reg_iq_est_gain_step_init;
  else if(iter_gain_count < {2'b0,step_sum0,3'b0}) begin
    if ((reg_iq_est_gain_step_init-3'd1) > reg_iq_est_gain_step_stop)
      gain_step <= reg_iq_est_gain_step_init-3'd1;
    else
      gain_step <= reg_iq_est_gain_step_stop;
  end
  else if(iter_gain_count < {1'b0,step_sum1,3'b0}) begin
    if ((reg_iq_est_gain_step_init-3'd2) > reg_iq_est_gain_step_stop)
      gain_step <= reg_iq_est_gain_step_init-3'd2;
    else
      gain_step <= reg_iq_est_gain_step_stop;
  end
  else if(iter_gain_count < {1'b0,step_sum2,3'b0}) begin
    if ((reg_iq_est_gain_step_init-3'd3) > reg_iq_est_gain_step_stop)
      gain_step <= reg_iq_est_gain_step_init-3'd3;
    else
      gain_step <= reg_iq_est_gain_step_stop;
  end
  else if(iter_gain_count < {step_sum3,3'b0}) begin
    if ((reg_iq_est_gain_step_init-3'd4) > reg_iq_est_gain_step_stop)
      gain_step <= reg_iq_est_gain_step_init-3'd4;
    else
      gain_step <= reg_iq_est_gain_step_stop;
  end
  else begin
    if ((reg_iq_est_gain_step_init-3'd5) > reg_iq_est_gain_step_stop)
      gain_step <= reg_iq_est_gain_step_init-3'd5;
    else
      gain_step <= reg_iq_est_gain_step_stop;
  end
end

always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0)
    phase_step <= 3'b0;
  else if(iter_phase_count < {3'b0,reg_iq_est_step_del0,3'b0})
    phase_step <= reg_iq_est_phase_step_init;
  else if(iter_phase_count < {2'b0,step_sum0,3'b0}) begin
    if ((reg_iq_est_phase_step_init-3'd1) > reg_iq_est_phase_step_stop)
      phase_step <= reg_iq_est_phase_step_init-3'd1;
    else
      phase_step <= reg_iq_est_phase_step_stop;
  end
  else if(iter_phase_count < {1'b0,step_sum1,3'b0}) begin
    if ((reg_iq_est_phase_step_init-3'd2) > reg_iq_est_phase_step_stop)
      phase_step <= reg_iq_est_phase_step_init-3'd2;
    else
      phase_step <= reg_iq_est_phase_step_stop;
  end
  else if(iter_phase_count < {1'b0,step_sum2,3'b0}) begin
    if ((reg_iq_est_phase_step_init-3'd3) > reg_iq_est_phase_step_stop)
      phase_step <= reg_iq_est_phase_step_init-3'd3;
    else
      phase_step <= reg_iq_est_phase_step_stop;
  end
  else if(iter_phase_count < {step_sum3,3'b0}) begin
    if ((reg_iq_est_phase_step_init-3'd4) > reg_iq_est_phase_step_stop)
      phase_step <= reg_iq_est_phase_step_init-3'd4;
    else
      phase_step <= reg_iq_est_phase_step_stop;
  end
  else begin
    if ((reg_iq_est_phase_step_init-3'd5) > reg_iq_est_phase_step_stop)
      phase_step <= reg_iq_est_phase_step_init-3'd5;
    else
      phase_step <= reg_iq_est_phase_step_stop;
  end
end

// Status register valid generation
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0) begin
    stat_reg_valid <= 1'b0;
    acc_en_1t      <= 1'b0;
  end
  else begin
    acc_en_1t      <= iter_gain_inc | iter_phase_inc;
    if (!agc_lock)
      stat_reg_valid <= 1'b0;
    else if (acc_en_1t)
      stat_reg_valid <= 1'b1;
  end
end

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
