//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 29349 $
// $Date: 2017-01-04 18:38:45 +0100 (Wed, 04 Jan 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : IQ estimation loop filter module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module iq_est_loop(
  // Clock and reset
  input  wire        clk,     // FE Clock (160/80/40 MHz)
  input  wire        reset_n, // Global asynchronous reset (active LOW)

  // registers (asynchronous)
  input  wire        reg_iq_est_iter_clr,
  //
  input  wire [10:0] reg_iq_gain,
  input  wire [8:0]  reg_iq_phase,

  // Input data from error estimation
  input  wire [21:0] var_i,
  input  wire [21:0] var_q,
  input  wire [22:0] m_iq,

  // Input from estimation controller
  input  wire [2:0] gain_step,
  input  wire [2:0] phase_step,
  input  wire       iter_gain_inc,
  input  wire       iter_phase_inc,

  // Output estimates
  output reg  [10:0] iq_gain_est,
  output reg  [8:0]  iq_phase_est
  );

//////////////////////////////////////////////////////////////////////////////
// Internal Regs declarations
//////////////////////////////////////////////////////////////////////////////
  reg [8:0]  gain_est;
  reg [8:0]  phase_est;
  reg [11:0] iq_gain_accu;
  reg [9:0]  iq_phase_accu;
  
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  wire [22:0] var_iq_sum;
  wire        gain_err;
  wire        phase_err;
  
  wire [10:0] iq_gain_accu_sat;
  wire [8:0]  iq_phase_accu_sat;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Compute gain error
assign var_iq_sum = $signed({1'b0,var_i}) - $signed({1'b0,var_q});
assign gain_err   = var_iq_sum[22];

// Compute phase error
assign phase_err = m_iq[22];

// Compute gain and phase error shift
always @*
begin
  case(gain_step)
    3'd0:    gain_est=(gain_err) ? 9'b111111111 : 9'b000000001;
    3'd1:    gain_est=(gain_err) ? 9'b111111110 : 9'b000000010;
    3'd2:    gain_est=(gain_err) ? 9'b111111100 : 9'b000000100;
    3'd3:    gain_est=(gain_err) ? 9'b111111000 : 9'b000001000;
    3'd4:    gain_est=(gain_err) ? 9'b111110000 : 9'b000010000;
    3'd5:    gain_est=(gain_err) ? 9'b111100000 : 9'b000100000;
    3'd6:    gain_est=(gain_err) ? 9'b111000000 : 9'b001000000;
    default: gain_est=(gain_err) ? 9'b110000000 : 9'b010000000;
  endcase
end

always @*
begin
  case(phase_step)
    3'd0:    phase_est=(phase_err) ? 9'b111111111 : 9'b000000001;
    3'd1:    phase_est=(phase_err) ? 9'b111111110 : 9'b000000010;
    3'd2:    phase_est=(phase_err) ? 9'b111111100 : 9'b000000100;
    3'd3:    phase_est=(phase_err) ? 9'b111111000 : 9'b000001000;
    3'd4:    phase_est=(phase_err) ? 9'b111110000 : 9'b000010000;
    3'd5:    phase_est=(phase_err) ? 9'b111100000 : 9'b000100000;
    3'd6:    phase_est=(phase_err) ? 9'b111000000 : 9'b001000000;
    default: phase_est=(phase_err) ? 9'b110000000 : 9'b010000000;
  endcase
end

// Compute gain and phase accumulators
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0) begin
    iq_gain_accu  <= 12'd1024;
    iq_phase_accu <= 10'b0;
  end
  else if (reg_iq_est_iter_clr) begin
    iq_gain_accu  <= {1'b0,reg_iq_gain};
    iq_phase_accu <= {reg_iq_phase[8],reg_iq_phase};
  end
  else begin
    if (iter_gain_inc)
      iq_gain_accu  <= $signed({1'b0,iq_gain_est}) + $signed({gain_est[8],gain_est});
    if (iter_phase_inc)   
      iq_phase_accu <= $signed({iq_phase_est[8],iq_phase_est}) + $signed({phase_est[8],phase_est});
  end
end

// Saturate gain and phase accumulators

// Gain must be always > 0
assign iq_gain_accu_sat = (iq_gain_accu[11:10]==2'b10) ? 11'd2047 :
                          (iq_gain_accu[11:10]==2'b11) ? 11'd0    :
                                                         iq_gain_accu[10:0];

SatSigned # (
  .INPUT_WIDTH(10),
  .OUTPUT_WIDTH(9)
  )
  u_sat_phase (
  .InputData(iq_phase_accu),
  .SatData(iq_phase_accu_sat)
  );

// Register gain and phase estimation
always @(posedge clk or negedge reset_n)
begin
  if (reset_n == 1'b0) begin
    iq_gain_est  <= 11'd1024;
    iq_phase_est <= 9'b0;
  end
  else begin
    iq_gain_est  <= iq_gain_accu_sat;
    iq_phase_est <= iq_phase_accu_sat;
  end
end

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
