//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15153 $
// $Date: 2014-06-02 17:43:26 +0200 (Mon, 02 Jun 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Karst controller FSM module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module KarstCtrlFsm
(
  /* SYSTEM */         
  input  wire          clk,
  input  wire          rst_n,
  
  /* PHY */
  input  wire          RCRxOn,
  output reg           RCRxAck,
  //
  input  wire          RCTxOn,
  output reg           RCTxAck,
  //
  input  wire  [1:0]   RCAntennaSet,
  //
  input  wire          RCProgRF,
  input  wire          RCProgRFCont,
  output reg           RCProgRFDone,
  //
  
  input  wire  [5:0]   RCLNAGain0,
  input  wire  [5:0]   RCLNAGain1,

  input  wire  [5:0]   TX0GAIN,
  input  wire  [5:0]   TX1GAIN,

  /* Registers */
  input wire           RegAUTO_TX_GAIN_EN,
  input wire           RegSAMEANTCONTROL,
  input wire           RegRF_SEL_2G4,

  input  wire	 [3:0]   RegTX0PAMODE,
  input  wire	 [3:0]   RegTX0MODE,
  input  wire	 [3:0]   RegTX0LOMODE,
  input  wire	 [3:0]   RegRX0MODE,
  input  wire	 [3:0]   RegRX0LOMODE,
  
  input  wire	 [3:0]   RegTX1PAMODE,
  input  wire	 [3:0]   RegTX1MODE,
  input  wire	 [3:0]   RegTX1LOMODE,
  input  wire	 [3:0]   RegRX1MODE,
  input  wire	 [3:0]   RegRX1LOMODE,

  input  wire  [9:0]   RegRX0GAINADDR,
  input  wire  [9:0]   RegRX1GAINADDR,

  input  wire  [9:0]   RegTX0GAINADDR,
  input  wire  [9:0]   RegTX1GAINADDR,
  input  wire  [3:0]   RegTX1GAINOFFSET,
  input  wire  [3:0]   RegTX0GAINOFFSET,

  input  wire  [9:0]   RegLO2RXON_DELAY,
  input  wire  [9:0]   RegRXON2LO_DELAY,

  input  wire  [9:0]   RegLO2TXON_DELAY,
  input  wire  [9:0]   RegTXON2LO_DELAY,
  
  input  wire  [9:0]   RegTXPAON_DELAY,
  input  wire  [9:0]   RegTXPAOFF_DELAY,

  input  wire  [9:0]   RegRX2TXON_DELAY,

  input  wire  [7:0]   RegEXTPA1_SEQ_ON_DELAY,
  input  wire  [7:0]   RegEXTPA0_SEQ_ON_DELAY,
  input  wire  [7:0]   RegEXTPA1_SEQ_OFF_DELAY,
  input  wire  [7:0]   RegEXTPA0_SEQ_OFF_DELAY,

  input  wire  [9:0]   RegTXSWITCH_ON_DELAY,
  input  wire  [9:0]   RegTXSWITCH_OFF_DELAY,

  input  wire          RCAGCFreeze,
  input  wire  [1:0]   GainRdReq,
  output reg   [1:0]   GainRdDone,
  output reg   [5:0]   RFLNAGain0,
  output reg   [5:0]   RFLNAGain1,

  /* SPI */
  input  wire          spi_done,
  //
  output reg           spi_start,
  output reg           spi_wrnrd,
  output reg   [9:0]   spi_addr,
  input  wire  [5:0]   spi_rddata,
  output reg   [5:0]   spi_wrdata,

  /* GPIO interface */
  output reg   [7:0]   RFGPIO,

  /* Interrupts */
  output reg           RCRxEndP,
  output reg           RCTxEndP,
  
  
  /* RF */
  output reg           RFADCOn,
  output reg           RFAGCFreeze,
  //
  output reg           RFTRXSwitch,
  output reg           RFDACOn,
  output reg           RFExtPA0On_5G9,
  output reg           RFExtPA0On_2G4,
  output reg           RFExtPA1On_5G9,
  output reg           RFExtPA1On_2G4,

  /* Status control */
  output wire [2:0]    RC_STATE
  );

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  //$fsm_sd KarstCtrlFSMState
  localparam   S_IDLE         = 3'd0,
               S_RX2ON        = 3'd1,
               S_TX2ON        = 3'd2,
               S_RXON         = 3'd3,
               S_TX2PAON      = 3'd4,
               S_TXPAON       = 3'd7;
  
  reg [2:0]    KarstCtrlState;
  
  reg [9:0]    fsm_counter;
  reg [7:0]    extpa_counter;
  reg          RCTxOnD;

  reg          Rx0OnDone;
  reg          Rx1OnDone;
  reg          Rx0GainDone;
  reg          Rx1GainDone;
  reg          Tx0PAOnDone;
  reg          Tx1PAOnDone;
  reg          Tx0PAOffDone;
  reg          Tx1PAOffDone;
  reg          Tx0OnDone;
  reg          Tx1OnDone;
  reg [1:0]    TxGainWrDone;
  
  reg          RFExtPA0Done;
  reg          RFExtPA1Done;
  reg          RFExtPA0On;
  reg          RFExtPA1On;
  
  reg [5:0]    gpio_data;
  reg          gpio_trigger, gpio_trigger_sync;
  reg          gpio_gaincmd;

  wire         RxGainUpdate;

  wire         Ant1ProgDisabled;

  /*****************************************************************************
  * assignment
  *****************************************************************************/
  assign RC_STATE = KarstCtrlState;
 
  
  // Rx gain request
  assign RxGainUpdate = RCProgRF | RCProgRFCont;
  
  // Set to 1 when TXR1 and TXR2 are coupled or when TXRX1 is not active
  assign Ant1ProgDisabled = (RegSAMEANTCONTROL & RCAntennaSet[0]) | !RCAntennaSet[1];

  // Rx gain acknowledgment
 // assign RCProgRFDone = (KarstCtrlState == S_RXON) ? RCProgRF & Rx1GainDone : 1'b0;
  always @(posedge clk or negedge rst_n)
  begin
    if(rst_n == 1'b0)
      RCProgRFDone <= 1'b0;
    else if (gpio_trigger_sync && RCProgRF &&  & Rx1GainDone && KarstCtrlState == S_RXON)
      RCProgRFDone <= 1'b1;
    else if (!RCProgRF)
      RCProgRFDone <= 1'b0;
  end
 
  /*****************************************************************************
  * fsm
  *****************************************************************************/
  always @(posedge clk or negedge rst_n)
  begin
    if(rst_n == 1'b0)
    begin
      KarstCtrlState <= S_IDLE;
    
      fsm_counter         <= 10'b0;

      gpio_trigger        <= 1'b0;
      gpio_gaincmd        <= 1'b0;
      gpio_data           <= 6'b0;

      RCRxAck             <= 1'b0;
      RCTxAck             <= 1'b0;
      
      RCRxEndP            <= 1'b0;
      RCTxEndP            <= 1'b0;

      RFADCOn             <= 1'b0;
      RFDACOn             <= 1'b0;

      GainRdDone          <= 2'b0;
      RFLNAGain0          <= 6'b0;
      RFLNAGain1          <= 6'b0;

      Rx0OnDone           <= 1'b0;
      Rx1OnDone           <= 1'b0;
      Rx0GainDone         <= 1'b0;
      Rx1GainDone         <= 1'b0;
      Tx0OnDone           <= 1'b0;
      Tx1OnDone           <= 1'b0;
      Tx0PAOnDone         <= 1'b0;
      Tx1PAOnDone         <= 1'b0;
      Tx0PAOffDone        <= 1'b0;
      Tx1PAOffDone        <= 1'b0;
      TxGainWrDone        <= 2'b0;
      spi_start           <= 1'b0;
      spi_wrnrd           <= 1'b1;
      spi_addr            <= 10'b0;
      spi_wrdata          <= 6'b0;
      
      RFAGCFreeze         <= 1'b1;

    end
    else
    begin
      // Pulse generation
      RCRxEndP   <= 1'b0;
      RCTxEndP   <= 1'b0;
      gpio_trigger <= 1'b0;
      
      // used to gate ADC clock, '|ack' ensures that last samples are well flushed
      // before gating the ADC.
      RFADCOn   <= RCRxOn | RCRxAck;
      
      // Counter decrement
      if(fsm_counter != 10'b0)
        fsm_counter <= fsm_counter + 10'h3ff;

      /*************************************************************************
      * FSM
      *************************************************************************/
      case(KarstCtrlState)
      
        /*********************************************************************
        * S_IDLE
        *********************************************************************/
        S_IDLE:
        begin
          // Reinit flags
          Rx0OnDone           <= 1'b0;
          Rx1OnDone           <= 1'b0;
          Rx0GainDone         <= 1'b0;
          Rx1GainDone         <= 1'b0;
          Tx0OnDone           <= 1'b0;
          Tx1OnDone           <= 1'b0;
          Tx0PAOnDone         <= 1'b0;
          Tx1PAOnDone         <= 1'b0;
          Tx0PAOffDone        <= 1'b0;
          Tx1PAOffDone        <= 1'b0;
          TxGainWrDone        <= 2'b0;
          GainRdDone          <= 2'b0;
          RFLNAGain0          <= 6'b0;
          RFLNAGain1          <= 6'b0;
          RFAGCFreeze         <= 1'b1;
          RCRxAck             <= 1'b0;
          RCTxAck             <= 1'b0;
          spi_addr            <= 10'b0;
          spi_start           <= 1'b0;
          spi_wrnrd           <= 1'b1;
          spi_wrdata          <= 6'b0;
          RFDACOn             <= 1'b0; 


          // Transitions for Rx
          if(RCRxOn)
          begin
            fsm_counter     <= RegLO2RXON_DELAY;
            KarstCtrlState  <= S_RX2ON;
            gpio_trigger    <= RCAntennaSet[0]; 
            gpio_gaincmd    <= 1'b0;
            gpio_data       <= {2'b0, RegRX0MODE};
            Rx0OnDone       <= 1'b1;
            Rx1OnDone       <= Ant1ProgDisabled;
          end                                                                           
          // Transitions for Tx
          else if(RCTxOn)
          begin
            RFDACOn         <= 1'b1; 
            fsm_counter     <= RegLO2TXON_DELAY;
            KarstCtrlState  <= S_TX2ON;
            gpio_trigger    <= RCAntennaSet[0] | RCAntennaSet[1];
            gpio_gaincmd    <= 1'b0;
            if (RCAntennaSet[0])
            begin
              Tx0OnDone       <= 1'b1;
              Tx1OnDone       <= Ant1ProgDisabled;
              gpio_trigger    <= RCAntennaSet[0];
              gpio_data       <= {2'b0, RegTX0MODE};
            end
            else 
            begin
              Tx0OnDone       <= 1'b0;
              Tx1OnDone       <= 1'b1;
              gpio_trigger    <= RCAntennaSet[1];
              gpio_data       <= {2'b0, RegTX1MODE};
            end
          end
        end

        /*********************************************************************
        * S_RX2ON
        *********************************************************************/
        S_RX2ON:
        begin
          if(Rx1OnDone && fsm_counter == 10'b0)
          begin
            // Rx On acknowledgment
            RCRxAck <= 1'b1;
            KarstCtrlState <= S_RXON;
          end
          else if(!Rx1OnDone) 
          begin
            Rx1OnDone       <= 1'b1;
            fsm_counter     <= RegLO2RXON_DELAY;
            gpio_trigger    <= 1'b1;
            gpio_gaincmd    <= 1'b0;
            gpio_data       <= {2'b0, RegRX1MODE};
          end
        end
                                                                                            
        /*********************************************************************
        * S_RXON
        *********************************************************************/
        S_RXON:
        begin
          RFAGCFreeze <= RCAGCFreeze;
          // RX GAIN UPDATE THROUGH GPIO interface
          if (RxGainUpdate && !Rx1GainDone)
          begin
            if(!Rx0GainDone) 
            begin
              Rx0GainDone     <= 1'b1;
              Rx1GainDone     <= Ant1ProgDisabled;
              gpio_trigger    <= RCAntennaSet[0];
              gpio_gaincmd    <= 1'b1;                                               
              gpio_data       <= RCLNAGain0;                                         
            end
            else
            begin
              Rx1GainDone     <= 1'b1;
              gpio_trigger    <= 1'b1;
              gpio_gaincmd    <= 1'b1;
              gpio_data       <= RCLNAGain1;
            end
          end
          
          // RX GAIN Read through SPI interface
          else if (spi_done && spi_start)
          begin
            spi_start   <= 1'b0;
            if (GainRdDone[0]) 
            begin
              RFLNAGain1 <= spi_rddata;
              GainRdDone <= 2'b11;
            end
            else 
            begin
              RFLNAGain0 <= spi_rddata;
              GainRdDone <= 2'b01;
            end
          end
          else if ((GainRdReq!=2'b00) && (GainRdReq != GainRdDone))
          begin
            spi_start     <= 1'b1;
            spi_wrnrd     <= 1'b0;
            if (GainRdDone[0]) 
              spi_addr      <= RegRX1GAINADDR;
            else 
              spi_addr      <= RegRX0GAINADDR;
          end

          else
          begin
            Rx0GainDone <= RxGainUpdate;
            Rx1GainDone <= RxGainUpdate;
            GainRdDone  <= GainRdReq;
            
            if(!RCRxOn && RCTxOn)
            begin
              // Tx request at the end of RX, transition from RX mode to Tx Mode
              RCRxAck         <= 1'b0;
              RCRxEndP        <= 1'b1;
              KarstCtrlState  <= S_TX2ON;
              gpio_trigger    <= RCAntennaSet[0];
              gpio_gaincmd    <= 1'b0;
              gpio_data       <= {2'b0, RegTX0MODE};
              Tx0OnDone       <= RCAntennaSet[0];
              Tx1OnDone       <= Ant1ProgDisabled;
              fsm_counter     <= RegRX2TXON_DELAY;
            end
            else if(!RCRxOn && Rx0OnDone)
            begin
              // Transition from RX0 mode to RegLO0MODE Mode
              gpio_trigger    <= RCAntennaSet[0];
              gpio_gaincmd    <= 1'b0;
              gpio_data       <= {2'b0, RegRX0LOMODE};
              Rx0OnDone       <= 1'b0;
              Rx1OnDone       <= !(RegSAMEANTCONTROL | !RCAntennaSet[1]);
              fsm_counter     <= RegRXON2LO_DELAY;
             end
            else if(!RCRxOn && Rx1OnDone)
            begin
              // Transition from RX1 mode to RegLO1MODE Mode
              gpio_trigger    <= RCAntennaSet[0]; 
              gpio_gaincmd    <= 1'b0;
              gpio_data       <= {2'b0, RegRX1LOMODE};
              Rx1OnDone       <= 1'b0;
              fsm_counter     <= RegRXON2LO_DELAY;
             end
            else if(!RCRxOn && fsm_counter == 10'b0)
            begin
              // End of RX, goes in Idle State
              RCRxEndP        <= 1'b1;
              KarstCtrlState <= S_IDLE;
            end
          end
        end
         
        /*********************************************************************
        * S_TX2ON
        *********************************************************************/
        S_TX2ON:
        begin
          if(!RCTxOn && fsm_counter == 10'b0)
          begin
            if(Tx0OnDone)
            begin
              gpio_trigger    <= RCAntennaSet[0];
              gpio_gaincmd    <= 1'b0;
              gpio_data       <= {2'b0, RegTX0LOMODE};
              Tx0OnDone       <= 1'b0;
              Tx1OnDone       <= !Ant1ProgDisabled;
              fsm_counter     <= RegTXON2LO_DELAY;
            end
            else if (Tx1OnDone)
            begin
              gpio_trigger    <= RCAntennaSet[1];
              gpio_gaincmd    <= 1'b0;
              gpio_data       <= {2'b0, RegTX0LOMODE};
              Tx1OnDone       <= 1'b0;
              fsm_counter     <= RegTXON2LO_DELAY;
            end
            else
              KarstCtrlState <= S_IDLE;
          end
          
          // TX GAIN write through SPI interface
          else if (spi_done && spi_start && RegAUTO_TX_GAIN_EN)
          begin
            spi_start   <= 1'b0;
            if (RCAntennaSet[1:0] != 2'b11 | TxGainWrDone[0]) 
              TxGainWrDone <= 2'b11;
            else
              TxGainWrDone <= 2'b01;
          end
          else if (TxGainWrDone!=2'b11 && RegAUTO_TX_GAIN_EN)
          begin
            spi_start     <= 1'b1;
            spi_wrnrd     <= 1'b1;
            if (!RCAntennaSet[0] && !TxGainWrDone[0])
            begin
              // If TxGainWrDone[0] is not set and RCAntennaSet[0] is not set, 
              // it indicates that the RF is configured in 1x1 (RCAntennaSet[1:0] == 2 
              // and only the TX PATH 1 Gain shall be set.
              // In this case, the TX PATH 1 of the RF is configured with TX0GAIN as only TX0GAIN is valid
              // Note that the offset applied is still TX1GAINOFFSET as is depends on the 
              // offset to be applied on the TX Path 1 whatever the RF configuration is (1x1 or 2x2)
              spi_addr    <= RegTX1GAINADDR;
              spi_wrdata  <= TX0GAIN+{{2{RegTX1GAINOFFSET[3]}},RegTX1GAINOFFSET};
            end
            else if (TxGainWrDone[0])
            begin
              // If TxGainWrDone[0] is set, it indicates that the TXGAIN 0 has already been set
              // and the RF is configured in 2x2 (RCAntennaSet[1:0] == 3
              // In this case, the TX PATH 1 of the RF is configured with TX1GAIN
              spi_addr    <= RegTX1GAINADDR;
              spi_wrdata  <= TX1GAIN+{{2{RegTX1GAINOFFSET[3]}},RegTX1GAINOFFSET};
            end
            else 
            begin
              // If TxGainWrDone[0] is not set and RCAntennaSet[0] is set, 
              // it indicates that the RF is configured in 1x1 (RCAntennaSet[1:0] == 1 
              // and only the TX PATH 0 Gain shall be set.
              // In this case, the TX PATH 0 of the RF is configured with TX0GAIN
              spi_addr    <= RegTX0GAINADDR;
              spi_wrdata  <= TX0GAIN+{{2{RegTX0GAINOFFSET[3]}},RegTX0GAINOFFSET};
            end
          end

          else if(Tx1OnDone && fsm_counter == 10'b0)
          begin
            fsm_counter     <= RegTXPAON_DELAY;
            KarstCtrlState  <= S_TX2PAON;
            gpio_gaincmd    <= 1'b0;
            if (RCAntennaSet[0])
            begin
              Tx0PAOnDone     <= 1'b1;
              Tx1PAOnDone     <= Ant1ProgDisabled;
              gpio_trigger    <= RCAntennaSet[0];
              gpio_data       <= {2'b0, RegTX0PAMODE};
            end
            else 
            begin
              Tx0PAOnDone     <= 1'b0;
              Tx1PAOnDone     <= 1'b1;
              gpio_trigger    <= RCAntennaSet[1];
              gpio_data       <= {2'b0, RegTX1PAMODE};
            end
          end

          else if(!Tx1OnDone && fsm_counter == 10'b0) 
          begin
            Tx1OnDone       <= 1'b1;
            gpio_trigger    <= 1'b1;
            gpio_gaincmd    <= 1'b0;
            gpio_data       <= {2'b0, RegTX1MODE};
          end
        end
                                                                                            


        /*********************************************************************
        * S_TX2PAON
        *********************************************************************/
        S_TX2PAON:
        begin
          if(RFExtPA1Done && Tx1PAOnDone && fsm_counter == 10'b0)
          begin
            KarstCtrlState <= S_TXPAON;
          end
          else if(!Tx1PAOnDone && Tx0PAOnDone) 
          begin
            fsm_counter     <= RegTXPAON_DELAY;
            gpio_trigger    <= 1'b1;
            gpio_gaincmd    <= 1'b1;
            gpio_data       <= {2'b0, RegTX1PAMODE};
            Tx1PAOnDone     <= 1'b1;
          end
         
        end

        /*********************************************************************
        * S_TXPA_ON
        *********************************************************************/
        S_TXPAON:
        begin
          // Tx On acknowledgment is set only when all enabled external PA are ON
          RCTxAck <= (RFExtPA0On | ~RCAntennaSet[0])  && (RFExtPA1On| ~RCAntennaSet[1]);
          
          if(!RCTxOn && Tx1PAOffDone && !RFExtPA0On && !RFExtPA1On && (fsm_counter == 10'b0) && !RFExtPA0Done && !RFExtPA1Done)
          begin
            // Tx Off acknowledgment
            RCTxAck <= 1'b0;                                                     
            // Tx end interrupt generation
            RCTxEndP <= 1'b1;
            // transition to S_IDLE state
            KarstCtrlState <= S_IDLE;
          end
          else if(!Tx0PAOffDone && !RCTxOn)
          begin
            RFDACOn         <= 1'b0; 
            fsm_counter     <= RegTXPAOFF_DELAY;
            gpio_trigger    <= RCAntennaSet[0];
            gpio_gaincmd    <= 1'b0;
            gpio_data       <= {2'b0, RegTX0LOMODE};
            Tx0PAOffDone    <= 1'b1;
            Tx1PAOffDone    <= RegSAMEANTCONTROL | !RCAntennaSet[1];
          end
          else if(!Tx1PAOffDone && Tx0PAOffDone)
          begin
            RFDACOn         <= 1'b0; 
            fsm_counter     <= RegTXPAOFF_DELAY;
            gpio_trigger    <= 1'b1;
            gpio_gaincmd    <= 1'b0;
            gpio_data       <= {2'b0, RegTX1LOMODE};
            Tx0PAOffDone    <= 1'b1;
            Tx1PAOffDone    <= 1'b1;
          end
        end

       // Disable coverage on the default state because it cannot be reached.
       // pragma coverage block = off 
        default:   
          KarstCtrlState <= S_IDLE;
       // pragma coverage block = on 

      endcase                                                                               
    end
  end

  /*****************************************************************************
  * GPIO interface
  *****************************************************************************/
  always @(posedge clk or negedge rst_n)
  begin
    if(rst_n == 1'b0)
    begin
      gpio_trigger_sync <= 1'b0;
      RFGPIO            <= 8'b0;
    end
    else 
    begin
      gpio_trigger_sync <= gpio_trigger;
      RFGPIO[7]   <= gpio_gaincmd;
      RFGPIO[5:0] <= gpio_data;
      if (gpio_trigger_sync)
        RFGPIO[6] <= !RFGPIO[6];
    end
  end



  /*****************************************************************************
  * Tx/Rx switch control
  *****************************************************************************/
  always @(posedge clk or negedge rst_n)
  begin
    if(rst_n == 1'b0)
    begin
      RFTRXSwitch  <= 1'b1;
      RCTxOnD      <= 1'b0;
    end
    else
    begin
      // RCTxOn delayed by 1cc for edge detection
      RCTxOnD <= RCTxOn;
      
      // RF TX/RX switch, by default, switch in TX position to get get
      // the DB RX led off
      if(RCRxOn)
        RFTRXSwitch <= 1'b0;
      else
        RFTRXSwitch <= 1'b1;
    end
  end

  



 /*****************************************************************************
  * External PA control
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(rst_n == 1'b0)
    begin
      RFExtPA0Done  <= 1'b0;
      RFExtPA1Done  <= 1'b0;
      RFExtPA0On    <= 1'b0;
      RFExtPA1On    <= 1'b0;
      extpa_counter <= 8'b0;
    end
    else
    begin
      // Counter decrement
      if(extpa_counter != 8'b0)
        extpa_counter <= extpa_counter + 8'hff;
      // External PA sequential activation
      else if ((RFDACOn && (KarstCtrlState == S_TXPAON)) || (KarstCtrlState == S_TX2PAON))
      begin
        if (!RFExtPA1Done)
        begin
          if(!RFExtPA0Done)
          begin
            RFExtPA0Done <= 1'b1;
            if (RegEXTPA0_SEQ_ON_DELAY == 8'b0)
              RFExtPA0On    <= RCAntennaSet[0];
            else
              extpa_counter <= RegEXTPA0_SEQ_ON_DELAY;
          end
          else
          begin
            RFExtPA0On    <= RCAntennaSet[0];
            RFExtPA1Done <= 1'b1;
            if (RegEXTPA1_SEQ_ON_DELAY == 8'b0)
              RFExtPA1On    <= RCAntennaSet[1];
            else
            begin
              extpa_counter <= RegEXTPA1_SEQ_ON_DELAY;
              RFExtPA0On    <= RCAntennaSet[0];
            end
          end
        end
        else
        begin
          RFExtPA1On <= RCAntennaSet[1];
        end
      end
     // External PA sequential desactivation
      else
      begin
        if (RFExtPA1Done)
        begin
          if(RFExtPA0Done)
          begin
            RFExtPA0Done <= 1'b0;
            if (RegEXTPA0_SEQ_OFF_DELAY == 8'b0)
              RFExtPA0On   <= 1'b0;
            else
              extpa_counter <= RegEXTPA0_SEQ_OFF_DELAY;
          end
          else
          begin
            RFExtPA0On   <= 1'b0;
            RFExtPA1Done <= 1'b0;
            if (RegEXTPA1_SEQ_OFF_DELAY == 8'b0)
              RFExtPA1On   <= 1'b0;
            else
              extpa_counter <= RegEXTPA1_SEQ_OFF_DELAY;
          end
        end
        else
          RFExtPA1On    <= 1'b0;
      end
    end
  end

  /*****************************************************************************
  * External PA control Muxing
  *****************************************************************************/
  always @(posedge clk or negedge rst_n)
  begin
    if(rst_n == 1'b0)
    begin
      RFExtPA0On_5G9     <= 1'b0;
      RFExtPA0On_2G4     <= 1'b0;
      RFExtPA1On_5G9     <= 1'b0;
      RFExtPA1On_2G4     <= 1'b0;
    end
    else
    begin
      RFExtPA0On_2G4   <=  RegRF_SEL_2G4 & RFExtPA0On;
      RFExtPA0On_5G9   <= !RegRF_SEL_2G4 & RFExtPA0On;

      RFExtPA1On_2G4   <=  RegRF_SEL_2G4 & RFExtPA1On;
      RFExtPA1On_5G9   <= !RegRF_SEL_2G4 & RFExtPA1On;
    end
  end

 
// pragma coverage block = off 
  `ifdef RW_SIMU_ON
   
    reg [16*8-1:0] KarstCtrlStateStr;
  
    always @(*)
      case (KarstCtrlState)
        S_IDLE    : KarstCtrlStateStr = "S_IDLE";
        S_RX2ON   : KarstCtrlStateStr = "S_RX2ON";
        S_TX2ON   : KarstCtrlStateStr = "S_TX2ON";
        S_RXON    : KarstCtrlStateStr = "S_RXON";
        S_TX2PAON : KarstCtrlStateStr = "S_TX2PAON";
        S_TXPAON  : KarstCtrlStateStr = "S_TXPAON";
        default   : KarstCtrlStateStr = "UNKNOWN";
      endcase
  
  `endif
// pragma coverage block = on

endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
