//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15015 $
// $Date: 2014-05-22 12:09:55 +0200 (Thu, 22 May 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Karst SPI controller module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module KarstCtrlSpi
(
  /* system */
  input  wire         clk,
  input  wire         rst_n,
 
  /* Registers */
  input  wire  [3:0]  prescaler,
  input  wire  [5:0]  rd_delay,

  /* SPI request */
  input  wire         spi_start,
  output reg          spi_done,
  //
  input  wire         spi_wrnrd,
  input  wire  [9:0]  spi_addr,
  input  wire  [15:0] spi_data_in,
  //
  output reg   [15:0] spi_data_out,

  /* SPI IF */
  input wire         rf_spi_miso,
  output reg         rf_spi_csn,
  output reg         rf_spi_sclk,
  output wire        rf_spi_mosi,

  /* Diag */
  output wire  [1:0] spi_fsm_diag
  );

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam    S_IDLE        = 2'b00,
                S_SPI         = 2'b01,
                S_READ        = 2'b11,
                S_DONE        = 2'b10;
                
  reg [ 1:0] KarstCtrlSPIState;
  reg [ 5:0] count;
  reg [ 3:0] prescaler_trig;
  reg [ 3:0] prescaler_count;
  reg [26:0] spi_sr;
  reg        rd_ongoing;


  /*****************************************************************************
  * assignments
  *****************************************************************************/
  // SPI MOSI
  assign rf_spi_mosi = spi_sr[26];
  
  // diag port
  assign spi_fsm_diag = KarstCtrlSPIState;

  /*****************************************************************************
  * serializer
  *****************************************************************************/
  always @(posedge clk or negedge rst_n)
  begin
  
    if(rst_n==1'b0)
    begin
      rf_spi_csn       <= 1'b1;
      rf_spi_sclk      <= 1'b0;
    
      spi_data_out     <= 16'b0;
      spi_done         <= 1'b0;
      
      spi_sr           <= 27'b0;
      
      count            <= 6'd0;
      prescaler_count  <= 4'b0;
      prescaler_trig   <= 4'b0;
      rd_ongoing       <= 1'b0;
      
      KarstCtrlSPIState <= S_IDLE;
    end
    else
    begin
    
      case(KarstCtrlSPIState)
      
        /***********************************************************************
        * IDLE: wait an incoming request
        ***********************************************************************/
        S_IDLE:
        begin
          // SPI mode access
          if(spi_start == 1'b1)
          begin
            KarstCtrlSPIState <= S_SPI;
            rd_ongoing       <= 1'b0;                            
            rf_spi_sclk      <= 1'b0;
            rf_spi_csn       <= 1'b0;
            spi_sr           <= {spi_wrnrd,spi_addr,spi_data_in};
            if (!spi_wrnrd)
              count  <= 6'd23; //11*2+1
            else  
              count  <= 6'd55; //27*2+1
            prescaler_count  <= prescaler;
            prescaler_trig   <= prescaler;
          end
        end
        
        /***********************************************************************
        * SPI: transfer
        ***********************************************************************/
        S_SPI:
        begin
          if(prescaler_count == 4'b0)
          begin
            prescaler_count <= prescaler_trig;
            count           <= count + 6'b111111;                          
            rf_spi_sclk     <= count[0] && !rd_ongoing;                                 
        
            // Shift SPI data out
            if(count[0] == 1'b0)
              spi_sr       <= {spi_sr[25:0],1'b0};  
            
            // Shift SPI data in
            if(count[0] == 1'b1 && count != 6'd1)
              spi_data_out <= {spi_data_out[14:0],rf_spi_miso};  
            
            if(count == 6'd1 && spi_wrnrd)                                 
            begin                                                
              // End of Write SPI access
              rf_spi_csn         <= 1'b1;                            
              rf_spi_sclk        <= 1'b0;
              KarstCtrlSPIState   <= S_DONE;
              spi_done           <= 1'b1;
            end
            else if(count == 6'd1 && !spi_wrnrd && !rd_ongoing)                         
            begin                                                
              // Enable Rd_delay counter to wait for available read data before ging in READ State
              rd_ongoing       <= 1'b1;                            
              rf_spi_csn       <= 1'b1;                            
              rf_spi_sclk      <= 1'b0;
              count            <= rd_delay;
              prescaler_count  <= prescaler;
              prescaler_trig   <= prescaler;
            end
            else  if(count == 6'd0)                         
            begin                                                
              // When Read data is avalaible
              rf_spi_csn       <= 1'b1;                            
              rf_spi_sclk      <= 1'b0;
              count            <= 6'd34; //11*2+1
              prescaler_count  <= prescaler;
              prescaler_trig   <= prescaler;
              KarstCtrlSPIState <= S_READ;
            end
         end
          else
          begin
            prescaler_count <= prescaler_count + 4'b1111;
          end
        end
        
        /***********************************************************************
        * SPI READ: transfer
        ***********************************************************************/
        S_READ:
        begin
          rf_spi_csn         <= 1'b0;                            
          if(prescaler_count == 4'b0)
          begin
            prescaler_count <= prescaler_trig;
            count           <= count + 6'b111111;                          
            rf_spi_sclk     <= count[0];                                 
        
            // Shift SPI data out
            if(count[0] == 1'b0)
              spi_sr       <= {spi_sr[25:0],1'b0};  
            
            // Shift SPI data in
            if(count[0] == 1'b1 && count != 6'd1)
              spi_data_out <= {spi_data_out[14:0],rf_spi_miso};  
            
            if(count == 6'd1 )                                 
            begin                                                
              rf_spi_csn         <= 1'b1;                            
              rf_spi_sclk        <= 1'b0;
              KarstCtrlSPIState   <= S_DONE;
              spi_done           <= 1'b1;
            end
          end
          else
          begin
            prescaler_count <= prescaler_count + 4'b1111;
          end
        end
        
        /***********************************************************************
        * DONE: indicates that SPI transfer is done, wait start de-assertion
        ***********************************************************************/
        S_DONE:
        begin
          spi_sr   <= 27'b0;
          if(spi_start == 1'b0)
          begin
            spi_done         <= 1'b0;
            KarstCtrlSPIState <= S_IDLE;
          end
        end
        
       // Disable coverage on the default state because it cannot be reached.
       // pragma coverage block = off 
        default:   
          KarstCtrlSPIState <= S_IDLE;
       // pragma coverage block = on 
        
      endcase
    end
  end
  
// pragma coverage block = off 
  `ifdef RW_SIMU_ON
   
    reg [16*8-1:0] KarstCtrlSPIStateStr;               
    always @(*)                             
      case (KarstCtrlSPIState)                          
        S_IDLE        : KarstCtrlSPIStateStr = "S_IDLE";
        S_SPI         : KarstCtrlSPIStateStr = "S_SPI";
        S_READ        : KarstCtrlSPIStateStr = "S_READ";
        S_DONE        : KarstCtrlSPIStateStr = "S_DONE";
        default       : KarstCtrlSPIStateStr = "UNKNOWN";
      endcase                               
  `endif
// pragma coverage block = on

endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
