////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 15271 $
// $Date: 2014-06-11 10:42:43 +0200 (Wed, 11 Jun 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Effective RF Gain computation Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module KarstGainComp 
(
            //Clock and Reset
            input    wire                                  rst_n,        
            input    wire                                  clk,         
  
            // AGC Interface
            input    wire      [7:0]                       AGCGainTgt0,
            input    wire      [7:0]                       AGCGainTgt1,
            input    wire                                  AGCRFGainCompReq,
            output   reg                                   AGCRFGainCompDone,
            
            // Register Interface
            input    wire      [6:0]                       RegGainIndMin, 
            input    wire      [6:0]                       RegGainIndMax, 

            // Karst AGC Gain Table Interface
            input    wire      [7:0]                       Rx0GainOffset, 
            input    wire      [7:0]                       Rx1GainOffset, 
            output   wire      [5:0]                       Rx0GainAddr, 
            output   wire      [5:0]                       Rx1GainAddr,  

            // Effective gain applied to RF Gain and gain Comparator
            output  reg        [7:0]                       RFGain0dB, //Gain for Rx Chain 0
            output  reg        [7:0]                       RFGain1dB, //Gain for Rx Chain 1

            // Effective gain index applied to Radio Controller
            output   reg       [5:0]                       RFGain0Index, // Gain index for Rx Chain 0
            output   reg       [5:0]                       RFGain1Index  // Gain index for Rx Chain 1
            );

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire            [7:0]                  Rx0GainTargetdBInt;
wire            [7:0]                  Rx1GainTargetdBInt;
wire            [7:0]                  Rx0GainAddrInt;
wire            [7:0]                  Rx1GainAddrInt;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg             [7:0]                  Rx0GainTargetdB;
reg             [7:0]                  Rx1GainTargetdB;
reg                                    AGCRFGainCompReqDel;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign Rx0GainTargetdBInt =  ($signed(AGCGainTgt0) <= $signed({1'h0,RegGainIndMin})) ? {1'h0,RegGainIndMin} + 8'd1 : 
                             ($signed(AGCGainTgt0) >  $signed({1'h0,RegGainIndMax})) ? {1'h0,RegGainIndMax} : AGCGainTgt0;
assign Rx1GainTargetdBInt =  ($signed(AGCGainTgt1) <= $signed({1'h0,RegGainIndMin})) ? {1'h0,RegGainIndMin} + 8'd1 : 
                             ($signed(AGCGainTgt1) >  $signed({1'h0,RegGainIndMax})) ? {1'h0,RegGainIndMax} : AGCGainTgt1;

always @ (posedge clk or negedge rst_n)
   begin: RxGainTargetdB_Blk
      if (rst_n == 1'b0)
      begin
         Rx0GainTargetdB <= 8'd61;
         Rx1GainTargetdB <= 8'd61;
       end
      else if (AGCRFGainCompReq)
      begin
         Rx0GainTargetdB <= Rx0GainTargetdBInt;
         Rx1GainTargetdB <= Rx1GainTargetdBInt;
      end
   end //RxGainTargetdB_Blk
   

/////////////////////////////////
// Effective RF Gain computation
/////////////////////////////////

assign  Rx0GainAddrInt = Rx0GainTargetdB - {1'h0,RegGainIndMin};
assign  Rx1GainAddrInt = Rx1GainTargetdB - {1'h0,RegGainIndMin};

assign  Rx0GainAddr =  (Rx0GainAddrInt > 8'd63) ? 6'd63 : Rx0GainAddrInt[5:0];
assign  Rx1GainAddr =  (Rx1GainAddrInt > 8'd63) ? 6'd63 : Rx1GainAddrInt[5:0];

always @ (posedge clk or negedge rst_n)
   begin: RFGainComp_Blk
      if (rst_n == 1'b0)
      begin
         RFGain0dB    <= 8'd66;
         RFGain1dB    <= 8'd66;
         RFGain0Index <= 6'd61;
         RFGain1Index <= 6'd61;
       end
      else if (AGCRFGainCompReqDel)
      begin
         RFGain0dB    <= Rx0GainTargetdB - {{4{Rx0GainOffset[3]}},Rx0GainOffset[3:0]};
         RFGain1dB    <= Rx1GainTargetdB - {{4{Rx1GainOffset[3]}},Rx1GainOffset[3:0]};
         RFGain0Index <= Rx0GainAddr - {Rx0GainOffset[7],Rx0GainOffset[7],Rx0GainOffset[7:4]};
         RFGain1Index <= Rx1GainAddr - {Rx1GainOffset[7],Rx1GainOffset[7],Rx1GainOffset[7:4]};
      end
   end //RFGainComp_Blk

//Delayed Versions of AGCRFGainCompReq
always @ (posedge clk or negedge rst_n)
   begin: CmdValidDel_Blk
      if (rst_n == 1'b0)
         AGCRFGainCompReqDel <= 1'b0;
      else
         AGCRFGainCompReqDel <= AGCRFGainCompReq;
   end //AGCRFGainCompValid_Blk

//Generate AGCRFGainCompDone
always @ (posedge clk or negedge rst_n)
   begin: AGCRFGainCompDone_Blk
      if (rst_n == 1'b0)
         AGCRFGainCompDone <= 1'b0;
      else if (!AGCRFGainCompReq)
         AGCRFGainCompDone <= 1'b0;
      else if (AGCRFGainCompReqDel)
         AGCRFGainCompDone <= 1'b1;
   end //AGCRFGainCompDone_Blk

endmodule //KarstGainComp

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////

