//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 15388 $
// $Date: 2014-06-24 18:55:38 +0200 (Tue, 24 Jun 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of KarstCtrl module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rc_karst
( 
  /*******************************************************************
  * Asynchronous Resets
  *******************************************************************/
  input  wire        nRCRst,  // Active low
  input  wire        nRegBusRst, // Active low
  
  /*******************************************************************
  * Clocks
  *******************************************************************/
  input  wire        RCClk,
  input  wire        RegBusClk,

  /*******************************************************************
  * Interrupts
  *******************************************************************/
  output wire        RCRxEndP,
  output wire        RCTxEndP,
  
  /*****************************************************************************
  * RegBus interface
  *****************************************************************************/
  input  wire        RegBusSel,
  input  wire        RegBusRdEn,
  input  wire        RegBusWrEn,
  input  wire [3:0]  RegBusAddr,
  input  wire [31:0] RegBusWData,
  output wire [31:0] RegBusRData,
  
  /*******************************************************************
  * RIU interface
  *******************************************************************/
  input wire         RCAGCFreeze,
  input  wire        RCTxOn,
  output wire        RCTxAck,

  input  wire [1:0]  RCAntennaSet,
  
  input  wire        RCRxOn,
  output wire        RCRxAck,

  input  wire        RCProgRFCont,
  input  wire        RCProgRF,
  input  wire        RCProgForceRF,
  output wire        RCProgRFDone,
  
  input  wire [5:0]  RCRxGain0,
  input  wire [5:0]  RCRxGain1,

  input  wire [5:0]  RCTxGain0,
  input  wire [5:0]  RCTxGain1,

  input  wire [1:0]  GainRdReq,
  output wire [5:0]  RFLNAGain0,
  output wire [5:0]  RFLNAGain1,
  output wire [1:0]  GainRdDone,

  // Register Interface
  output wire [6:0]  RegGainIndMin, 
  output wire [6:0]  RegGainIndMax, 
  /*******************************************************************
  * RF interface
  *******************************************************************/
  /* SPI interface */
  input  wire        RFSpiMiso,
  output wire        RFSpiCsn,
  output wire        RFSpiSclk,
  output wire        RFSpiMosi,
  
  /* GPIO interface */
  output wire  [7:0] RFGPIO,

  /* RF AGC Control */
  output wire        RFAGCFreeze,

  /* Force RF TX path */
  output wire        RFForceTxOn,

  /* RF Test mode */
  output wire        RFTestMode,

 /* Reset */
  output wire        RFResetn,

  /*******************************************************************
  * External interface
  *******************************************************************/
  output wire        RFADCOn,
  
  //
  output wire        RFTRXSwitch,
  output wire        RFDACOn,
  output wire        RFExtPA0On_5G9,
  output wire        RFExtPA1On_5G9,
  output wire        RFExtPA0On_2G4,
  output wire        RFExtPA1On_2G4,

  /*******************************************************************
  * Debug port
  *******************************************************************/
  output wire [15:0] DbgKarstCtrl
  );

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  /* Registers */
  wire          RegSTART_DONEIn;
  wire          RegSTART_DONEInValid;
  wire          RegDATAInValid;
  wire [2 : 0]  RegRC_STATE;
  wire	        RegSTART_DONE;
  wire	        RegSAMEANTCONTROL;
  wire	[3 : 0] RegHW_PRESCALER;
  wire	        RegAUTO_TX_GAIN_EN;
  wire	        RegRF_FORCE_TXON;
  wire	        RegRF_TESTMODE;
  wire	        RegRF_SEL_2G4;
  wire	        RegRF_RESETN;
  wire	[3 : 0] RegSW_PRESCALER;
  wire	        RegREADNOTWRITE;
  wire	[9 : 0] RegADDRESS;
  wire	[15 : 0] RegDATA;
  wire  [5 : 0] RegSPI_RD_DELAY;
  wire  [3 : 0] RegTX1GAINOFFSET;
  wire  [3 : 0] RegTX0GAINOFFSET;
  wire  [9 : 0] RegTX0GAINADDR;
  wire  [9 : 0] RegTX1GAINADDR;
  wire  [9 : 0] RegRX0GAINADDR;
  wire  [9 : 0] RegRX1GAINADDR;
  wire  [9 : 0] RegTXPAON_DELAY;
  wire  [9 : 0] RegTXPAOFF_DELAY;
  wire  [7 : 0] RegEXTPA1_SEQ_ON_DELAY;
  wire  [7 : 0] RegEXTPA0_SEQ_ON_DELAY;
  wire  [7 : 0] RegEXTPA1_SEQ_OFF_DELAY;
  wire  [7 : 0] RegEXTPA0_SEQ_OFF_DELAY;
  wire  [9 : 0] RegTXSWITCH_OFF_DELAY;
  wire  [9 : 0] RegRX2TXON_DELAY;
  wire  [9 : 0] RegTXSWITCH_ON_DELAY;
  wire  [3:0]   RegTX0PAMODE;
  wire  [3:0]   RegTX0MODE;
  wire  [3:0]   RegTX0LOMODE;
  wire  [3:0]   RegRX0MODE;
  wire  [3:0]   RegRX0LOMODE;
  wire  [3:0]   RegTX1PAMODE;
  wire  [3:0]   RegTX1MODE;
  wire  [3:0]   RegTX1LOMODE;
  wire  [3:0]   RegRX1MODE;
  wire  [3:0]   RegRX1LOMODE;
  wire  [9:0]   RegLO2RXON_DELAY;
  wire  [9:0]   RegRXON2LO_DELAY;
  wire  [9:0]   RegLO2TXON_DELAY;
  wire  [9:0]   RegTXON2LO_DELAY;
  wire          RCProgRFInt;

  /* Resync */
  wire          RegRF_SEL_2G4Sync;
  wire          RegSTART_DONESync;
  wire          RegSTART_DONEInValidSync;
  wire          RegDATAInValidSync;
  wire          RFExtPA0On_2G4Sync;
  wire          RFExtPA0On_5G9Sync;

  /* fsm */
  wire          fsm_spi_start;
  wire          fsm_spi_done;
  wire          fsm_spi_wrnrd;
  wire   [9:0]  fsm_spi_addr;
  wire   [5:0]  fsm_spi_wrddata;
  wire   [2:0]  RC_STATE;
  
  /* SPI */
  wire   [3:0] spi_prescaler;
  wire         spi_start;
  wire         spi_done;
  wire         spi_wrnrd;
  wire   [9:0] spi_addr;
  wire  [15:0] spi_data_in;
  wire   [1:0] spi_fsm_diag;
  wire  [15:0] spi_data_out;

  /* Arbitration */
  wire         reg_spi_mux_en;
  wire         fsm_spi_mux_en;  

  /*****************************************************************************
  * Outputs assignment
  *****************************************************************************/
  assign RCProgRFInt  = RCProgRF | RCProgForceRF;

  // Simple resync to RCClk
  ClkSyncSimple U_RegForceTxOn_Sync( 
    .dstclk     (RCClk),
    .dstresetn  (nRCRst),
    .srcdata    (RegRF_FORCE_TXON),
    .dstdata    (RFForceTxOn)
    );
  
  ClkSyncSimple U_RegRF_SEL_2G4_Sync( 
    .dstclk     (RCClk),
    .dstresetn  (nRCRst),
    .srcdata    (RegRF_SEL_2G4),
    .dstdata    (RegRF_SEL_2G4Sync)
    );
  
  assign RFTestMode = RegRF_TESTMODE;
  assign RFResetn   = RegRF_RESETN;
    
  /*****************************************************************************
  * KarstCtrl register
  *****************************************************************************/
  KarstCtrlReg U_KarstCtrlReg (
    .rst_n                      ( nRegBusRst),
    .clk                        ( RegBusClk),
    //
    .START_DONEIn               ( RegSTART_DONEIn),
    .START_DONEInValid          ( RegSTART_DONEInValidSync),
    .DATAIn                     ( spi_data_out),
    .DATAInValid                ( RegDATAInValidSync),
    //
    .RC_STATE                   ( RegRC_STATE),
    //
    .HW_PRESCALER               ( RegHW_PRESCALER),
    .AUTO_TX_GAIN_EN            ( RegAUTO_TX_GAIN_EN),
    .RF_FORCE_TXON              ( RegRF_FORCE_TXON ),
    .RF_TESTMODE                ( RegRF_TESTMODE ),
    .RF_SEL_2G4                 ( RegRF_SEL_2G4),
    .SAMEANTCTRL                ( RegSAMEANTCONTROL),
    .RF_RESET_N                 ( RegRF_RESETN),
    //
    .START_DONE                 ( RegSTART_DONE),
    .SW_PRESCALER               ( RegSW_PRESCALER),
    .READNOTWRITE               ( RegREADNOTWRITE),
    .ADDRESS                    ( RegADDRESS),
    .DATA                       ( RegDATA),
    //
    .RX0LOMODE                  ( RegRX0LOMODE),
    .RX0MODE                    ( RegRX0MODE),
    .TX0LOMODE                  ( RegTX0LOMODE),
    .TX0MODE                    ( RegTX0MODE),
    .TX0PAMODE                  ( RegTX0PAMODE),
     
    .RX1LOMODE                  ( RegRX1LOMODE),
    .RX1MODE                    ( RegRX1MODE),
    .TX1LOMODE                  ( RegTX1LOMODE),
    .TX1MODE                    ( RegTX1MODE),
    .TX1PAMODE                  ( RegTX1PAMODE),
    //
    .TX0GAINADDR                ( RegTX0GAINADDR),
    .TX1GAINADDR                ( RegTX1GAINADDR),
    .TX1GAINOFFSET              ( RegTX1GAINOFFSET),
    .TX0GAINOFFSET              ( RegTX0GAINOFFSET),
    //
    .RX0GAINADDR                ( RegRX0GAINADDR),
    .RX1GAINADDR                ( RegRX1GAINADDR),
    //
    .EXTPA1_SEQ_ON_DELAY        ( RegEXTPA1_SEQ_ON_DELAY),
    .EXTPA0_SEQ_ON_DELAY        ( RegEXTPA0_SEQ_ON_DELAY),
    //
    .EXTPA1_SEQ_OFF_DELAY       ( RegEXTPA1_SEQ_OFF_DELAY),
    .EXTPA0_SEQ_OFF_DELAY       ( RegEXTPA0_SEQ_OFF_DELAY),
    //
    .SPI_RD_DELAY               ( RegSPI_RD_DELAY),
     //    
    .LO_2_RXON_DELAY            ( RegLO2RXON_DELAY),
    .RXON_2_LO_DELAY            ( RegRXON2LO_DELAY),
    //    
    .LO_2_TXON_DELAY            ( RegLO2TXON_DELAY),
    .TXON_2_LO_DELAY            ( RegTXON2LO_DELAY),
    .PAON_DELAY                 ( RegTXPAON_DELAY),
    .PAOFF_DELAY                ( RegTXPAOFF_DELAY),
    //    
    .RX2TXON_DELAY              ( RegRX2TXON_DELAY),
    //    
    .TXSWITCH_ON_DELAY          ( RegTXSWITCH_ON_DELAY),
    .TXSWITCH_OFF_DELAY         ( RegTXSWITCH_OFF_DELAY),
    //    
    .RXAGCGAININDMAX            ( RegGainIndMax),
    .RXAGCGAININDMIN            ( RegGainIndMin),
    //    
    .regbus_sel(     RegBusSel), 
    .regbus_rden(    RegBusRdEn),
    .regbus_wren(    RegBusWrEn), 
    .regbus_addr(    RegBusAddr), 
    .regbus_wdata(   RegBusWData),
    .regbus_rdata(   RegBusRData)  
    );
  

  assign RegSTART_DONEIn        = 1'b0;
  
  // Pulse resync to RegBus clock domain
  ClkSyncPulse2Pulse U_RegSTART_DONEInValidSync ( 
    .srcclk     (RCClk),
    .srcresetn  (nRCRst),
    .dstclk     (RegBusClk),
    .dstresetn  (nRegBusRst),
    .srcdata    (RegSTART_DONEInValid),
    .dstdata    (RegSTART_DONEInValidSync)
    );

  ClkSyncPulse2Pulse U_RegDATAInValidSync ( 
    .srcclk     (RCClk),
    .srcresetn  (nRCRst),
    .dstclk     (RegBusClk),
    .dstresetn  (nRegBusRst),
    .srcdata    (RegDATAInValid),
    .dstdata    (RegDATAInValidSync)
    );

  // Simple resync to RegBus clock domain
  ClkSyncSimpleTop  
  #(
    .SIZE(      4)
  ) 
  u_rcstate2ahbResync 
  (
    .dstclk(    RegBusClk),
    .dstresetn( nRegBusRst),
    .srcdata({  RCRxAck,
                RCTxAck,
                RFExtPA0On_2G4,
                RFExtPA0On_5G9}),
    .dstdata({  RegRC_STATE[0],
                RegRC_STATE[1],
                RFExtPA0On_2G4Sync,
                RFExtPA0On_5G9Sync})
  );
  
  assign RegRC_STATE[2] = RFExtPA0On_2G4Sync | RFExtPA0On_5G9Sync;

  /*****************************************************************************
  * FSM
  *****************************************************************************/
  KarstCtrlFsm u_KarstCtrlFsm (
    .clk                          (RCClk),
    .rst_n                        (nRCRst),
    //
    .RCRxOn                       (RCRxOn),
    .RCRxAck                      (RCRxAck),
    //
    .RCAGCFreeze                  (RCAGCFreeze),
    .RCTxOn                       (RCTxOn),
    .RCTxAck                      (RCTxAck),
    .RCAntennaSet                 (RCAntennaSet),
    //
    .RCProgRF                     (RCProgRFInt),
    .RCProgRFCont                 (RCProgRFCont),
    .RCProgRFDone                 (RCProgRFDone),
    //
    .RCLNAGain0                   (RCRxGain0),
    .RCLNAGain1                   (RCRxGain1),
    //
    .TX0GAIN                      (RCTxGain0),
    .TX1GAIN                      (RCTxGain1),
    //
    .RegAUTO_TX_GAIN_EN           (RegAUTO_TX_GAIN_EN),
    .RegSAMEANTCONTROL            (RegSAMEANTCONTROL),
    .RegRF_SEL_2G4                (RegRF_SEL_2G4Sync),
    //    
    .RegRX0LOMODE                 (RegRX0LOMODE),
    .RegRX0MODE                   (RegRX0MODE),
    .RegTX0LOMODE                 (RegTX0LOMODE),
    .RegTX0MODE                   (RegTX0MODE),
    .RegTX0PAMODE                 (RegTX0PAMODE),
    
    .RegRX1LOMODE                 (RegRX1LOMODE),
    .RegRX1MODE                   (RegRX1MODE),
    .RegTX1LOMODE                 (RegTX1LOMODE),
    .RegTX1MODE                   (RegTX1MODE),
    .RegTX1PAMODE                 (RegTX1PAMODE),
    //    
    .RegLO2RXON_DELAY             (RegLO2RXON_DELAY),
    .RegRXON2LO_DELAY             (RegRXON2LO_DELAY),
    //    
    .RegLO2TXON_DELAY             (RegLO2TXON_DELAY),
    .RegTXON2LO_DELAY             (RegTXON2LO_DELAY),
    .RegTXPAON_DELAY              (RegTXPAON_DELAY),
    .RegTXPAOFF_DELAY             (RegTXPAOFF_DELAY),
    //    
    .RegRX2TXON_DELAY             (RegRX2TXON_DELAY),
    //    
    .RegEXTPA1_SEQ_ON_DELAY       (RegEXTPA1_SEQ_ON_DELAY),
    .RegEXTPA0_SEQ_ON_DELAY       (RegEXTPA0_SEQ_ON_DELAY),
    .RegEXTPA1_SEQ_OFF_DELAY      (RegEXTPA1_SEQ_OFF_DELAY),
    .RegEXTPA0_SEQ_OFF_DELAY      (RegEXTPA0_SEQ_OFF_DELAY),
    //    
    .RegTXSWITCH_ON_DELAY         (RegTXSWITCH_ON_DELAY),
    .RegTXSWITCH_OFF_DELAY        (RegTXSWITCH_OFF_DELAY),
    //    
    .RegRX0GAINADDR               (RegRX0GAINADDR),
    .RegRX1GAINADDR               (RegRX1GAINADDR),
    //    
    .RegTX0GAINADDR               (RegTX0GAINADDR),
    .RegTX1GAINADDR               (RegTX1GAINADDR),
    .RegTX0GAINOFFSET             (RegTX0GAINOFFSET),
    .RegTX1GAINOFFSET             (RegTX1GAINOFFSET),
    //
    .GainRdReq                    (GainRdReq),
    .GainRdDone                   (GainRdDone),
    .RFLNAGain0                   (RFLNAGain0),
    .RFLNAGain1                   (RFLNAGain1),
    //   
    .RFGPIO                       (RFGPIO),
    .RFAGCFreeze                  (RFAGCFreeze),
    //
    .RCRxEndP                     (RCRxEndP),
    .RCTxEndP                     (RCTxEndP),
    //
    .spi_done                     (fsm_spi_done),
    .spi_start                    (fsm_spi_start),
    .spi_wrnrd                    (fsm_spi_wrnrd),
    .spi_addr                     (fsm_spi_addr),
    .spi_rddata                   (spi_data_out[5:0]),
    .spi_wrdata                   (fsm_spi_wrddata),
    //
    .RFADCOn                      (RFADCOn),
    //
    .RFTRXSwitch                  (RFTRXSwitch),
    .RFDACOn                      (RFDACOn),
    .RFExtPA0On_5G9               (RFExtPA0On_5G9),
    .RFExtPA0On_2G4               (RFExtPA0On_2G4),
    .RFExtPA1On_5G9               (RFExtPA1On_5G9),
    .RFExtPA1On_2G4               (RFExtPA1On_2G4),
    //
    .RC_STATE                     (RC_STATE)
    );


   
  /*****************************************************************************
  * FSM/REGBANK SPI/FAST WRITE bus arbitration
  *****************************************************************************/
  KarstCtrlArbiter U_KarstCtrlArbiter
  (
    /* system */
    .clk(                      RCClk),
    .rst_n(                    nRCRst),
    
    /* requests */
    .a_start(                  RegSTART_DONESync),
    .b_start(                  fsm_spi_start),
    .c_start(                  1'b0),
    .d_start(                  1'b0),
    
    /* done */
    .done(                     spi_done),
    
    /* mux command */
    .a_mux_en(                 reg_spi_mux_en),
    .b_mux_en(                 fsm_spi_mux_en),
    .c_mux_en(                               ),
    .d_mux_en(                               )
    );   

  // Simple resync to RCClk
  ClkSyncSimple U_RegSTART_DONESync( 
    .dstclk     (RCClk),
    .dstresetn  (nRCRst),
    .srcdata    (RegSTART_DONE),
    .dstdata    (RegSTART_DONESync)
    );

  /* SPI Multiplexer */
  assign RegSTART_DONEInValid = reg_spi_mux_en & spi_done;
  assign RegDATAInValid       = reg_spi_mux_en & spi_done & !spi_wrnrd;
  assign fsm_spi_done         = fsm_spi_mux_en & spi_done;
  
  assign {spi_start,spi_wrnrd,spi_addr,spi_data_in} =
         {28{reg_spi_mux_en}} & {RegSTART_DONESync, !RegREADNOTWRITE  ,RegADDRESS  , RegDATA        } |
         {28{fsm_spi_mux_en}} & {fsm_spi_start    , fsm_spi_wrnrd    , fsm_spi_addr, {10'b0,fsm_spi_wrddata}};

  assign spi_prescaler = (reg_spi_mux_en) ? RegSW_PRESCALER : RegHW_PRESCALER;
  
  /*****************************************************************************
  * SPI serializer
  *****************************************************************************/
  KarstCtrlSpi U_KarstCtrlSpi
  (
    /* */
    .clk(                      RCClk),
    .rst_n(                    nRCRst),

    /* */
    .prescaler(                spi_prescaler),
    .rd_delay(                 RegSPI_RD_DELAY),
    
    /* */
    .spi_start(                spi_start),
    .spi_done(                 spi_done),
    .spi_wrnrd(                spi_wrnrd),
    .spi_addr(                 spi_addr),
    .spi_data_in(              spi_data_in),
    .spi_data_out(             spi_data_out),
    
    /* */
    .rf_spi_miso(              RFSpiMiso),
    .rf_spi_csn(               RFSpiCsn),
    .rf_spi_sclk(              RFSpiSclk),
    .rf_spi_mosi(              RFSpiMosi),
    
    /* */
    .spi_fsm_diag(             spi_fsm_diag)
    );

  
  /*****************************************************************************
  * Debug port
  *****************************************************************************/
  assign DbgKarstCtrl  = {RCTxOn,
                         RCTxAck,
                         RCRxOn,
                         RCRxAck,
                         RCProgRF,
                         RFADCOn,
                         RCProgRFCont,
                         RFTRXSwitch,
                         RFDACOn,
                         RFExtPA1On_2G4 | RFExtPA0On_2G4,
                         RFExtPA1On_5G9 | RFExtPA0On_5G9,
                         spi_fsm_diag,
                         RC_STATE[2:0]};
  
endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
