--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandebu $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 50 $
--/ $Date: 2011-09-21 18:10:05 +0200 (Wed, 21 Sep 2011) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      : Error Vector Magnitude calculator.
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/evm_calculator/vhdl/rtl/evm_calculator.vhd $
--/
--////////////////////////////////////////////////////////////////////////////

--               It gives information of reception quality into register.


--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
 
--library commonlib;
library work;
--use commonlib.mdm_math_func_pkg.all;
use work.mdm_math_func_pkg.all;


--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity evm_calculator is
  port (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
     reset_n          : in std_logic; -- global reset  
     clk              : in std_logic; -- main clock
    --------------------------------------
    -- Control
    --------------------------------------
    symbol_sync       : in std_logic;
    sfd_found         : in std_logic;    
    --------------------------------------
    -- Error from phase estimator
    --------------------------------------
    phase_error       : in std_logic_vector(12 downto 0);
    --------------------------------------
    -- Register information
    --------------------------------------
    evm               : out std_logic_vector(12 downto 0)
    );

end evm_calculator;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture RTL of evm_calculator is

  ------------------------------------------------------------------------------
  -- Constants
  ------------------------------------------------------------------------------
  constant T16US_CT  : std_logic_vector(8 downto 0) := "000010000";
  constant T32US_CT  : std_logic_vector(8 downto 0) := "000100000";
  constant T64US_CT  : std_logic_vector(8 downto 0) := "001000000";
  constant T128US_CT : std_logic_vector(8 downto 0) := "010000000";
  constant T256US_CT : std_logic_vector(8 downto 0) := "100000000";

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  signal us_counter          : std_logic_vector(8 downto 0);
  signal phase_error_sat     : std_logic_vector(7 downto 0);
  signal error_mult          : std_logic_vector(15 downto 0);
  signal error_accu          : std_logic_vector(22 downto 0);
  signal error_accu_mul4     : std_logic_vector(24 downto 0);
  signal error_accu_mul4_sel : std_logic_vector(24 downto 0);
  signal error_accu_div_m    : std_logic_vector(24 downto 0);
  signal m_round             : std_logic_vector(2 downto 0);
  signal error_accu_div_rnd  : std_logic_vector(23 downto 0);

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin

  -- Phase input rounding of 4lsb and saturation of 8msb
  phase_error_sat <= sat_round_signed_slv(phase_error,1,4);
  
  -- Square module
  error_mult <= abs (signed(phase_error_sat) * signed(phase_error_sat));
  
  -- Accumulator
  error_mod_accu_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      error_accu <= (others => '0');
    elsif clk'event and clk = '1' then
      if sfd_found = '1' and symbol_sync = '1' then
        error_accu <= unsigned(error_accu) + unsigned(error_mult);
      elsif sfd_found = '0' then
        error_accu <= (others => '0');
      end if;
    end if;
  end process error_mod_accu_p;
  
  -- Counter
  us_counter_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      us_counter <= (others => '0');
    elsif clk'event and clk = '1' then
      if sfd_found = '1' and symbol_sync = '1' and us_counter <= T256US_CT then
        us_counter <= us_counter + '1';
      elsif sfd_found = '0' then
        us_counter <= (others => '0');
      end if;
    end if;
  end process us_counter_p;
  
  -- Multiply accumulator by 4
  error_accu_mul4 <= error_accu & "00";
  
  -- Select bit rounding according to counter
  with us_counter select
    m_round <= "011" when T16US_CT,  -- 2^4 -> round bit 3
               "100" when T32US_CT,  -- 2^5 -> round bit 4
               "101" when T64US_CT,  -- 2^6 -> round bit 5
               "110" when T128US_CT, -- 2^7 -> round bit 6
               "111" when T256US_CT, -- 2^8 -> round bit 7
               "000" when others;

  -- Select correct vector according to m_round
  with m_round select
    error_accu_mul4_sel <=
      ext(error_accu_mul4(24 downto 3),error_accu_mul4_sel'length) when "011",
      ext(error_accu_mul4(24 downto 4),error_accu_mul4_sel'length) when "100",
      ext(error_accu_mul4(24 downto 5),error_accu_mul4_sel'length) when "101",
      ext(error_accu_mul4(24 downto 6),error_accu_mul4_sel'length) when "110",
      ext(error_accu_mul4(24 downto 7),error_accu_mul4_sel'length) when "111",
      error_accu_mul4 when others;
  
  -- Accumulator rounding of m lsb
  error_accu_div_m <= ext((error_accu_mul4_sel + '1'), error_accu_div_m'length);
  error_accu_div_rnd <= error_accu_div_m(24 downto 1);
  
  -- EVM output
  evm_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      evm <= (others => '0');
    elsif clk'event and clk = '1' then
      case us_counter is
        when T16US_CT | T32US_CT | T64US_CT | T128US_CT | T256US_CT =>
          evm <= sat_unsigned_slv(error_accu_div_rnd, 11);
        when others =>
           null;
      end case;
    end if;
  end process evm_p;

  
end RTL;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------
