--               This block compensates the mismatch between the expected
--               and the actual power level.
--               The power estimation is compared to a grid of threshold to
--               choose the scaling constant to be used to multiply the 
--               sample stream. A data_in_enable signals allows to stop the 
--               computation.
--               Grid of threshold : 
--               --------------------------------------------
--               | Power Estimation Result  |  Gain |  Value |
--               --------------------------------------------
--               |             P > 10111010 | 00110 | 0.750 |
--               | 10111010 >= P > 10001011 | 00111 | 0.875 |
--               | 10001011 >= P > 01101100 | 01000 | 1.000 |
--               | 01101100 >= P > 01010111 | 01001 | 1.125 |
--               | 01010111 >= P > 01000111 | 01010 | 1.250 |
--               | 01000111 >= P > 00111011 | 01011 | 1.375 |
--               | 00111011 >= P > 00110010 | 01100 | 1.500 |
--               | 00110010 >= P > 00101011 | 01101 | 1.625 |
--               | 00101011 >= P > 00100010 | 01110 | 1.750 |
--               | 00100010 >= P > 00011011 | 10000 | 2.000 |
--               | 00011011 >= P > 00010101 | 10010 | 2.250 |
--               | 00010101 >= P > 00010001 | 10100 | 2.500 |
--               | 00010001 >= P            | 10110 | 2.750 |
--               --------------------------------------------
--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all; 

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_gain_compensation is
  generic 
  (
    dsize_g : integer := 8
  );
  port 
  (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
    clk     : in  std_logic; -- clock
    reset_n : in  std_logic; -- reset
    --------------------------------------
    -- Control signal
    --------------------------------------
    gain_enable       : in  std_logic;
    power_estimation  : in  std_logic_vector(20 downto 0);
    data_in_enable    : in  std_logic;
    data_out_enable   : out std_logic;
    --------------------------------------
    -- Data
    --------------------------------------
	data_i_in  : in  std_logic_vector(dsize_g - 1 downto 0);
	data_q_in  : in  std_logic_vector(dsize_g - 1 downto 0);
    data_i_out : out std_logic_vector(dsize_g - 1 downto 0);
    data_q_out : out std_logic_vector(dsize_g - 1 downto 0)
  );
end fe11b_gain_compensation;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture rtl of fe11b_gain_compensation is


  ------------------------------------------------------------------------------
  -- Types
  ------------------------------------------------------------------------------
  
  ------------------------------------------------------------------------------
  -- Constants
  ------------------------------------------------------------------------------
  constant DSIZE_P3_CT : integer := dsize_g + 3;
  constant DSIZE_P2_CT : integer := dsize_g + 2;
  constant DSIZE_P1_CT : integer := dsize_g + 1;
  constant DSIZE_M1_CT : integer := dsize_g - 1;
  constant DSIZE_M2_CT : integer := dsize_g - 2;
  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------

  signal data_i_out_ext : std_logic_vector(DSIZE_P3_CT downto 0);
  signal data_q_out_ext : std_logic_vector(DSIZE_P3_CT downto 0);
  
  signal data_i_sign : std_logic;
  signal data_i_sign_ff1 : std_logic;
  
  signal data_q_sign : std_logic;
  signal data_q_sign_ff1 : std_logic;

  signal gain : std_logic_vector(4 downto 0);

  signal cnt     : std_logic;
  signal cnt_ff1 : std_logic;

  signal data_in_enable_ff1 : std_logic;
  signal data_in_enable_int : std_logic;
  
  signal data_out_enable_o : std_logic;

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin
  
  
  -- Outut assignment.  
  data_i_out <= data_i_out_ext(DSIZE_P2_CT downto 3);
  data_q_out <= data_q_out_ext(DSIZE_P2_CT downto 3);

  data_out_enable <= data_out_enable_o;
  
  ------------------------------------------------------------------------
  ------------------------------------------------------------------------
  -- Capture and extend the inputs and multiply the inputs by the gain. --
  ------------------------------------------------------------------------
  ------------------------------------------------------------------------
  
  Multi_p : process(reset_n, clk)
--  variable cnt : std_logic;
  variable op1_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable op2_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable op3_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable op1_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable op2_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable op3_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable operand1_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable operand2_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable operand1_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable operand2_q : std_logic_vector(DSIZE_P3_CT downto 0);

  variable result_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable result_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable result_sat_i : std_logic_vector(DSIZE_P3_CT downto 0);
  variable result_sat_q : std_logic_vector(DSIZE_P3_CT downto 0);
  variable data_i_sync : std_logic_vector(dsize_g downto 0);
  variable data_q_sync : std_logic_vector(dsize_g downto 0);
  variable gain_sync : std_logic_vector(4 downto 0);
  begin
    if reset_n='0' then
      data_i_sync := (others => '0');
      data_q_sync := (others => '0');
     
      data_i_out_ext <= (others => '0');
      data_q_out_ext <= (others => '0');
      result_i := (others => '0');
      result_q := (others => '0');
      result_sat_i := (others => '0');
      result_sat_q := (others => '0');

      op1_i := (others => '0');
      op2_i := (others => '0');
      op3_i := (others => '0');
      operand1_i := (others => '0');
      operand2_i := (others => '0');
      op1_q := (others => '0');
      op2_q := (others => '0');
      op3_q := (others => '0');
      operand1_q := (others => '0');
      operand2_q := (others => '0');

      gain_sync := (others => '0');
      data_i_sign_ff1 <= '0';
      data_q_sign_ff1 <= '0';
      data_i_sign <= '0';
      data_q_sign <= '0';
      data_out_enable_o <= '0';
--      cnt := '1';
    elsif clk'event and clk='1' then
      -- Data Output Formating.
      -- Saturation at 1.25 and
      -- sign addition
      ------------------------
      if cnt='1' then
        -- Saturation at 1.25 
         if result_i(DSIZE_P2_CT downto DSIZE_M1_CT) >= "0101" then
           result_sat_i(DSIZE_P3_CT) := '0';
           result_sat_i(DSIZE_P2_CT downto DSIZE_M1_CT) := "0101";
           result_sat_i(DSIZE_M2_CT downto 0) := (others => '0');
         else
           result_sat_i := result_i;
         end if;
        
         if result_q(DSIZE_P2_CT downto DSIZE_M1_CT) >= "0101" then
           result_sat_q(DSIZE_P3_CT) := '0';
           result_sat_q(DSIZE_P2_CT downto DSIZE_M1_CT) := "0101";
           result_sat_q(DSIZE_M2_CT downto 0) := (others => '0');
         else
           result_sat_q := result_q;
         end if;

        -- Add sign
        if data_i_sign_ff1 = '1' then
          data_i_out_ext <= not (result_sat_i) + '1';
        else
          data_i_out_ext <= result_sat_i;
        end if;  
        if data_q_sign_ff1 = '1' then
          data_q_out_ext <= not (result_sat_q) + '1';
        else
          data_q_out_ext <= result_sat_q;
        end if; 
        data_out_enable_o <= not data_out_enable_o;
      end if;
      
      -- Calculate.
      --------------
      if (cnt or cnt_ff1) = '1' then
        result_i := operand1_i + operand2_i;
        result_q := operand1_q + operand2_q;
      end if;
     -- cnt := not cnt;

      -- First Pipe of the multiplier.
      --------------------------------
      if cnt = '0' then
        op1_i := (others => '0');
        op2_i := (others => '0');
        op3_i := (others => '0');

        op1_q := (others => '0');
        op2_q := (others => '0');
        op3_q := (others => '0');


        if gain_sync(4) = '1' then
        -- Gain Values :
        -- 10000
        -- 10010
        -- 10100
        -- 10110
          op1_i(DSIZE_P3_CT downto 4) := data_i_sync(DSIZE_M1_CT downto 0);
          op1_q(DSIZE_P3_CT downto 4) := data_q_sync(DSIZE_M1_CT downto 0);
          if gain_sync(2) = '1' then
            op2_i(DSIZE_P1_CT downto 2) := data_i_sync(DSIZE_M1_CT downto 0);
            op2_q(DSIZE_P1_CT downto 2) := data_q_sync(DSIZE_M1_CT downto 0);
          end if;  
          if gain_sync(1) = '1' then
            op3_i(dsize_g downto 1) := data_i_sync(DSIZE_M1_CT downto 0);
            op3_q(dsize_g downto 1) := data_q_sync(DSIZE_M1_CT downto 0);
          end if;  
        elsif gain_sync(3) = '1' then
          -- Gain Value
          -- 01XXX
          op1_i(DSIZE_P2_CT downto 3) := data_i_sync(DSIZE_M1_CT downto 0);
          op1_q(DSIZE_P2_CT downto 3) := data_q_sync(DSIZE_M1_CT downto 0);
          if gain_sync(2) = '1' then
            -- 011XX
            op2_i(DSIZE_P1_CT downto 2) := data_i_sync(DSIZE_M1_CT downto 0);
            op2_q(DSIZE_P1_CT downto 2) := data_q_sync(DSIZE_M1_CT downto 0);
            if gain_sync(1) = '1' then
              -- 01110
              op3_i(dsize_g downto 1) := data_i_sync(DSIZE_M1_CT downto 0);
              op3_q(dsize_g downto 1) := data_q_sync(DSIZE_M1_CT downto 0);
            elsif gain_sync(0) = '1' then
              -- 01101
              op3_i(DSIZE_M1_CT downto 0) := data_i_sync(DSIZE_M1_CT downto 0);
              op3_q(DSIZE_M1_CT downto 0) := data_q_sync(DSIZE_M1_CT downto 0);
            end if;  
          elsif gain_sync(1) = '1' then
            -- 0101X
            op2_i(dsize_g downto 1) := data_i_sync(DSIZE_M1_CT downto 0);
            op2_q(dsize_g downto 1) := data_q_sync(DSIZE_M1_CT downto 0);
            if gain_sync(0) = '1' then
              -- 01011
              op3_i(DSIZE_M1_CT downto 0) := data_i_sync(DSIZE_M1_CT downto 0);
              op3_q(DSIZE_M1_CT downto 0) := data_q_sync(DSIZE_M1_CT downto 0);
            end if;  
          else
            if gain_sync(0) = '1' then
              -- 01001
              op2_i(DSIZE_M1_CT downto 0) := data_i_sync(DSIZE_M1_CT downto 0);
              op2_q(DSIZE_M1_CT downto 0) := data_q_sync(DSIZE_M1_CT downto 0);
            end if;  

          end if;  
        else
          -- Gain Value
          -- 0011X
          op1_i(DSIZE_P1_CT downto 2) := data_i_sync(DSIZE_M1_CT downto 0);
          op1_q(DSIZE_P1_CT downto 2) := data_q_sync(DSIZE_M1_CT downto 0);
          op2_i(dsize_g downto 1) := data_i_sync(DSIZE_M1_CT downto 0);
          op2_q(dsize_g downto 1) := data_q_sync(DSIZE_M1_CT downto 0);
          if gain_sync(0) = '1' then
            -- 00111
            op3_i(DSIZE_M1_CT downto 0) := data_i_sync(DSIZE_M1_CT downto 0);
            op3_q(DSIZE_M1_CT downto 0) := data_q_sync(DSIZE_M1_CT downto 0);
          else
            
          end if;              
        end if;  
        operand1_i := op1_i;
        operand1_q := op1_q;
        operand2_i := op2_i;
        operand2_q := op2_q;
      end if;

      -- Second Pipe of the multiplier.
      --------------------------------
      if cnt = '1' then
        
        data_i_sign <= data_i_in(DSIZE_M1_CT);
        data_i_sign_ff1 <= data_i_sign;
          
        data_q_sign <= data_q_in(DSIZE_M1_CT);
        data_q_sign_ff1 <= data_q_sign;

        operand1_i := (others => '0');
        operand2_i := (others => '0');
        operand1_q := (others => '0');
        operand2_q := (others => '0');


        operand1_i := op3_i;
        operand1_q := op3_q;
        operand2_i := result_i;
        operand2_q := result_q;
        
        -- Take the absolute value of the input.
        ----------------------------------------
        if data_i_in(DSIZE_M1_CT) = '1' then
          data_i_sync(DSIZE_M1_CT downto 0) := not (data_i_in) + '1';
        else
          data_i_sync(DSIZE_M1_CT downto 0) := data_i_in;
        end if;  
          
        if data_q_in(DSIZE_M1_CT) = '1' then
          data_q_sync(DSIZE_M1_CT downto 0) := not (data_q_in) + '1';
        else
          data_q_sync(DSIZE_M1_CT downto 0) := data_q_in;
        end if;  
        gain_sync := gain;
      end if;  

    end if;
  end process Multi_p;
    
    
  cnt_p : process (reset_n, clk)
  begin
    if reset_n='0' then
      cnt_ff1 <= '1';
--       data_i_in_ff1 <= (others => '0');
--       data_q_in_ff1 <= (others => '0');
    elsif clk'event and clk='1' then
      cnt_ff1 <= cnt;
--       data_i_in_ff1 <= data_i_in;
    end if;
  end process cnt_p;
   

--  data_in_enable_int <= data_in_enable xor data_in_enable_ff1;
  cnt <= data_in_enable xor data_in_enable_ff1;

  data_in_enable_ff1_p : process (reset_n,clk)
  begin
    if reset_n='0' then
      data_in_enable_ff1 <= '0';
    elsif clk'event and clk='1' then
      data_in_enable_ff1 <= data_in_enable; 
    end if;
  end process data_in_enable_ff1_p;
   

  ---------------------
  ---------------------
  -- Gain selection. --
  ---------------------
  ---------------------

  gain_selection_p : process (reset_n, clk)
  begin
    if reset_n='0' then
      gain <= "00000";
    elsif clk'event and clk='1' then
      if gain_enable = '1' and power_estimation /= "000000000000000000000" then  
        case power_estimation(20 downto 17) is
          when "1111" => 
              gain <= "00110";

          when "1110" => 
              gain <= "00110";

          when "1101" => 
              gain <= "00110";

          when "1100" => 
              gain <= "00110";

          when "1011" => 
            if power_estimation(16 downto 13) > "1010" then
              gain <= "00110";
            else
              gain <= "00111";
            end if;    

          when "1010" => 
              gain <= "00111";

          when "1001" => 
              gain <= "00111";

          when "1000" => 
            if power_estimation(16 downto 13) > "1011" then
              gain <= "00111";
            else
              gain <= "01000";
            end if;    

          when "0111" => 
              gain <= "01000";

          when "0110" => 
            if power_estimation(16 downto 13) > "1100" then
              gain <= "01000";
            else
              gain <= "01001";
            end if;    
          when "0101" => 
            if power_estimation(16 downto 13) > "0111" then
              gain <= "01001";
            else
              gain <= "01010";
            end if;    
          when "0100" => 
            if power_estimation(16 downto 13) > "0111" then
              gain <= "01010";
            else
              gain <= "01011";
            end if;    
          
          when others => 
            case power_estimation(18 downto 15) is
              when "1111" => 
                  gain <= "01011";
                  
              when "1110" => 
                  gain <= "01100";
      
              when "1101" => 
                  gain <= "01100";

              when "1100" => 
                if power_estimation(14 downto 13) > "10" then
                  gain <= "01100";
                else
                  gain <= "01101";
                end if;    
      
              when "1011" => 
                  gain <= "01101";

              when "1010" => 
                  gain <= "01110";
      
              when "1001" => 
                  gain <= "01110";
      

              when "1000" => 
                if power_estimation(14 downto 13) > "10" then
                  gain <= "01110";
                else
                  gain <= "10000";
                end if;    
      
              when "0111" => 
                  gain <= "10000";
      

              when "0110" => 
                  gain <= "10010";
      
              when "0101" => 
                if power_estimation(14 downto 13) > "01" then
                  gain <= "10010";
                else
                  gain <= "10100";
                end if;    
      
              when "0100" => 
                if power_estimation(14 downto 13) > "01" then
                  gain <= "10100";
                else
                  gain <= "10110";
                end if;    
      
              when others =>
                 gain <= "10110";
            end case;  
        end case;  
      else    
        gain <= "01000";                
      end if;
    end if;
  end process gain_selection_p;
      
end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

