--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.std_logic_signed.all;
use ieee.std_logic_arith.all; 

use work.mdm_math_func_pkg.all;

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_interpolator is
  generic 
  (
    dsize_g   : integer := 7; -- data size
    tausize_g : integer := 13 -- tau size
  );
  port 
  (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
    clk       : in std_logic;  -- 60 MHz clock
    reset_n   : in std_logic;
    --------------------------------------
    -- Test control signal
    --------------------------------------
    interpolator_enable : in std_logic;  -- enable the Interpolation.
    clock_lock          : in std_logic;
    -- Value of timing offset
    tau_clock_unlocked   : in  std_logic_vector(tausize_g - 1 downto 0);
    tau_clock_locked     : in  std_logic_vector(tausize_g - 1 downto 0);
    -- Input datas
    shift_sample         : in  std_logic; -- high when new data
    data_in_i            : in  std_logic_vector(dsize_g - 1 downto 0);
    data_in_q            : in  std_logic_vector(dsize_g - 1 downto 0);
    -- Interpolated datas
    shift_sample_interp  : out std_logic; -- high when new data - 0 when sample to skip
    interpolated_data_i  : out std_logic_vector(dsize_g - 1 downto 0);
    interpolated_data_q  : out std_logic_vector(dsize_g - 1 downto 0);
    -- Max value of stage which is filled -> from register (max value 40)
    interp_max_stage_i   : in  std_logic_vector(5 downto 0);
    -- Indicate when the datas are right.
    clk_skip             : out std_logic  -- when '1' -> clk_skip  (stop clk during 1 period)
  );
end fe11b_interpolator;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture rtl of fe11b_interpolator is


  ------------------------------------------------------------------------------
  -- Constants
  ------------------------------------------------------------------------------
  constant DSIZE_MINUS_1_CT    : integer := dsize_g-1;
  constant DSIZE_MINUS_2_CT    : integer := dsize_g-2;
  constant STAGE_NUMBER_CT     : integer := 40;
  constant STAGE_NUMBER_BIT_CT : integer := 64;
  constant TAU_LIMIT_POS : std_logic_vector(12 downto 0) := "0000001000000";
  constant TAU_LIMIT_NEG : std_logic_vector(12 downto 0) := "1111111000000";
  ------------------------------------------------------------------------------
  -- Types
  ------------------------------------------------------------------------------
  type ArrayOfSLVdsize is array (0 to STAGE_NUMBER_CT) of 
                                     std_logic_vector(dsize_g-1 downto 0);

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------

  signal tau        : std_logic_vector(tausize_g - 1 downto 0);
  signal tau_r_add  : std_logic_vector(tausize_g downto 0);
  signal tau_r      : std_logic_vector(tausize_g - 4 downto 0);
  signal interp_max_stage_10_bits_pos: std_logic_vector(tausize_g - 4 downto 0);
  signal interp_max_stage_10_bits_neg: std_logic_vector(tausize_g - 4 downto 0);
  signal tau_r_zero : std_logic_vector(tausize_g - 4 downto 0);
  signal tau_r_limit : std_logic_vector(tausize_g - 5 downto 0);
  signal tau_0_4lsb : std_logic_vector(tausize_g -5 downto 0);
  signal tau_s      : std_logic_vector(tausize_g -5 downto 0);
  signal n_int      : std_logic_vector(tausize_g -7 downto 0);
  signal n_int_ff   : std_logic_vector(tausize_g -7 downto 0);
  signal n_int_invert : std_logic_vector(tausize_g-7 downto 0);
  signal n_invert   : integer range STAGE_NUMBER_CT-1 downto 0;
--  signal n_invert   : std_logic_vector(tausize_g - 7 downto 0);

  
  -- array of input data delayed
  signal input_data_delay_i  : ArrayOfSLVdsize;    
  signal input_data_delay_q  : ArrayOfSLVdsize;    
  -- input data delay n and n+1
  signal data_delay_n0_i     : std_logic_vector(dsize_g-1 downto 0); 
  signal data_delay_n0_q     : std_logic_vector(dsize_g-1 downto 0); 
  signal data_delay_n1_i     : std_logic_vector(dsize_g-1 downto 0); 
  signal data_delay_n1_q     : std_logic_vector(dsize_g-1 downto 0); 

  -- input data delay n and n+1 extented. 
  signal data_delay_n0_i_ext : std_logic_vector(dsize_g downto 0); 
  signal data_delay_n0_q_ext : std_logic_vector(dsize_g downto 0); 
  signal data_delay_n1_i_ext : std_logic_vector(dsize_g downto 0); 
  signal data_delay_n1_q_ext : std_logic_vector(dsize_g downto 0); 
  
  -- result of input_data_delay(n+1) - input_data_delay(n)
  signal diff_data_i         : std_logic_vector(dsize_g downto 0);    
  signal diff_data_q         : std_logic_vector(dsize_g downto 0);    

  -- result of diff_data/2
  signal diff_data_d2_i      : std_logic_vector(dsize_g downto 0);    
  signal diff_data_d2_q      : std_logic_vector(dsize_g downto 0);    
  -- result of diff_data/4
  signal diff_data_d4_i      : std_logic_vector(dsize_g downto 0);    
  signal diff_data_d4_q      : std_logic_vector(dsize_g downto 0); 
  -- result of diff_data*2
  signal diff_data_m2_i      : std_logic_vector(dsize_g+1 downto 0);    
  signal diff_data_m2_q      : std_logic_vector(dsize_g+1 downto 0);
  -- Result of addition of diff_data*2 + diff_data
  signal diff_data_m2_plus_diff_data_i     : std_logic_vector(dsize_g+2 downto 0);
  signal diff_data_m2_plus_diff_data_q     : std_logic_vector(dsize_g+2 downto 0);
  -- Result of (diff_data*2 + diff_data)/4
  signal diff_data_m2_plus_diff_data_d4_i  : std_logic_vector(dsize_g downto 0);
  signal diff_data_m2_plus_diff_data_d4_q  : std_logic_vector(dsize_g downto 0);

  -- Data interpolated before resynchronization
  signal interpolated_data_c_i : std_logic_vector(dsize_g downto 0);
  signal interpolated_data_c_q : std_logic_vector(dsize_g downto 0);

  -- n : value of delay 
  signal n             : integer range STAGE_NUMBER_CT-1 downto 0;
  signal tau_ent       : integer range STAGE_NUMBER_BIT_CT downto 0- STAGE_NUMBER_BIT_CT;
  signal tau_ent_limit : integer range STAGE_NUMBER_BIT_CT downto 0- STAGE_NUMBER_BIT_CT;
  signal tau_ent_old   : integer range STAGE_NUMBER_CT-1 downto 0;

  -- m : interpolation ratio
  signal m   : std_logic_vector(1 downto 0);
  signal m_int   : std_logic_vector(1 downto 0);

  -- Clock Skipping
  signal clk_skip_reg     : std_logic;
  signal clk_skip_reg_ff0  : std_logic;
  signal clk_skip_reg_ff1  : std_logic;
  signal clk_skip_reg_ff2  : std_logic;
--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------

begin

  tau_r_zero <= (others => '0');

  tau <= tau_clock_locked when clock_lock = '1' else tau_clock_unlocked;

  -----------------------------------------------------------------------------
  -- Input Data Delay Generation
  -----------------------------------------------------------------------------
  input_data_delay_p : process (clk,reset_n)
  begin
    if reset_n = '0' then
      input_data_delay_i(0) <= (others => '0');
      input_data_delay_q(0) <= (others => '0');
      for i in 1 to STAGE_NUMBER_CT loop
        input_data_delay_i(i) <= (others => '0');
        input_data_delay_q(i) <= (others => '0');
      end loop;
    elsif clk'event and clk = '1' then
      if shift_sample = '1' then -- shift only when a new data has arrived
        input_data_delay_i(0) <= data_in_i;
        input_data_delay_q(0) <= data_in_q;
        for i in 1 to STAGE_NUMBER_CT loop
          input_data_delay_i(i) <= input_data_delay_i(i-1);
          input_data_delay_q(i) <= input_data_delay_q(i-1);
        end loop;        
      end if;
    end if;
  end process input_data_delay_p;

  -----------------------------------------------------------------------------
  -- Treatment of tau
  -----------------------------------------------------------------------------
  -- tau/16 with rounding
  tau_r_add <= signed(sxt(tau,tau_r_add'high + 1)) + 8;
  tau_r <= tau_r_add(tau_r_add'high downto 4);

  interp_max_stage_10_bits_pos <= sxt('0' & interp_max_stage_i & "00", interp_max_stage_10_bits_pos'high + 1);
  interp_max_stage_10_bits_neg <= sxt('1' & (not(interp_max_stage_i & "00") + 1), interp_max_stage_10_bits_neg'high + 1);
  
  limit_tau_r_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      tau_r_limit <= (others => '0');
    elsif clk'event and clk = '1' then
      if shift_sample = '1' and interpolator_enable = '1' then
      if (tau_r > tau_r_zero) then
        if (tau_r > interp_max_stage_10_bits_pos) then
          tau_r_limit <= interp_max_stage_10_bits_pos(interp_max_stage_10_bits_pos'high-1 downto 0);
        else
          tau_r_limit <= tau_r(tau_r'high -1 downto 0);
        end if;
      else                              -- Tau negative
        if (tau_r < interp_max_stage_10_bits_neg) then
          tau_r_limit <= interp_max_stage_10_bits_neg(interp_max_stage_10_bits_neg'high-1 downto 0);
        else
          tau_r_limit <= tau_r(tau_r'high-1 downto 0);
        end if;
      end if;
      end if;
    end if;
  end process limit_tau_r_p;
  
  -- Tau_s = max( tau(i), tau(i-1) ) with tau(i-1) with 4lsb=0
  tau_s_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      tau_s <= (others => '0');
      tau_0_4lsb <= (others => '0');
    elsif clk'event and clk = '1' then
      if shift_sample = '1' and interpolator_enable = '1' then
      if (signed(tau_s)>0) then
        tau_0_4lsb <= tau_s(tau_s'high downto 2) & "00";
        if signed(tau_0_4lsb) > signed(tau_r_limit) then
          tau_s <= tau_0_4lsb;
        else
          tau_s <= tau_r_limit;
        end if;
      else
        tau_s <= tau_r_limit;
      end if;
      end if;
    end if;
  end process tau_s_p;

  -- Generation of n = tau_s/4
  n_int <= tau_s(tau_s'high downto 2);

  -----------------------------------------------------------------------------
  -- Clock skip and shift_sample
  -----------------------------------------------------------------------------
  -- Clock_skip.
  clock_skip_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      --clk_skip_reg <= '0';
      n_int_invert <= (others => '0');
      m_int <= "00";
      m <= "00";
      n <= 0;
--      n_int_ff <= (others => '0');      
    elsif clk'event and clk = '1' then
--      if shift_sample = '1' and interpolator_enable = '1' then
        -- Flip-flop on n_int
--        n_int_ff <= n_int;
        if shift_sample = '1' and interpolator_enable = '1' then
        -- Generation of n (inverted sign):
        n_int_invert <= sxt(not n_int,n_int_invert'high+1) + '1';
        -- Generation of n (integer):
        if (n_int_invert>0) then
          n <= conv_integer(n_int_invert);
        else
          n <= 0;
        end if;
        -- Generation of m = tau_s - n*4
        m_int <= tau_s(1 downto 0);
        m <= m_int;
      end if;
    end if;
  end process clock_skip_p;

  -- Generation of clock_skip :
  clk_skip_reg <= '1' when (n_int_ff /= n_int) and tau(tausize_g-1)='0' and conv_integer(n_int) /= 0 else '0';

  -- Generation of output shift_sample_interp :
  shift_sample_interp_p: process (clk, reset_n)
    variable shift_sample_low_2cycles : std_logic;
  begin
    if reset_n = '0' then
      n_int_ff <= (others => '0');   
      shift_sample_interp <= '0';
      clk_skip <= '0';
      clk_skip_reg_ff0 <= '0';
      clk_skip_reg_ff1 <= '0';
      clk_skip_reg_ff2 <= '0';
      shift_sample_low_2cycles := '0';
    elsif clk'event and clk = '1' then
        n_int_ff <= n_int;
      clk_skip_reg_ff0 <= clk_skip_reg;
      clk_skip_reg_ff1 <= clk_skip_reg_ff0;
      clk_skip_reg_ff2 <= clk_skip_reg_ff1;
      if shift_sample_low_2cycles='1' then
        shift_sample_interp <= '0';
        shift_sample_low_2cycles := '0';
      elsif shift_sample = '1' and interpolator_enable = '1' and clk_skip_reg_ff2 = '1' then
        shift_sample_interp <= '0';
      elsif shift_sample = '0' and interpolator_enable = '1' and clk_skip_reg_ff2 = '1' then
        shift_sample_interp <= '0';
        shift_sample_low_2cycles := '1';
      else
        shift_sample_interp <= shift_sample;
      end if;
--      shift_sample_interp <= shift_sample and not clk_skip_reg_ff2;
      clk_skip <= clk_skip_reg_ff2;
    end if;
  end process shift_sample_interp_p;
  

  -----------------------------------------------------------------------------
  -- Interpolation Computing
  -----------------------------------------------------------------------------
--
--  With data_delay_n0 : Input data delayed of n clock period
--       data_delay_n1 : Input data delayed of n+1 clock period
--
--  The extension _ext indicate that the sign of the signal had been extented
--  for computation.
--
--  diff_data = data_delay_n0 - data_delay_n1
--  
--       data_delay_n0                                                   
--       --------.              m/Q                                        
--               |               |                                       
--              _|_  diff_data  _|_ diff_data_c ___                        
--             |_-_|---------->|_x_|---------->|_-_|----> interpolated_data_c  
--               |                               |                         
-- data_delay_n1 |      data_delay_n1            |                         
--       --------'-------------------------------'
--

    -- a0= r(n)
    -- b0= r(n+1)
  data_delay_n1_i <= input_data_delay_i(n+1);
  data_delay_n1_q <= input_data_delay_q(n+1);
  data_delay_n0_i <= input_data_delay_i(n);
  data_delay_n0_q <= input_data_delay_q(n);
 
  -- Signe extansion
  data_delay_n0_i_ext <= sxt(data_delay_n0_i,data_delay_n0_i_ext'high+1);
  data_delay_n0_q_ext <= sxt(data_delay_n0_q,data_delay_n0_q_ext'high+1);
  data_delay_n1_i_ext <= sxt(data_delay_n1_i,data_delay_n1_i_ext'high+1);
  data_delay_n1_q_ext <= sxt(data_delay_n1_q,data_delay_n1_q_ext'high+1);

  -- d= b0 - a0
  diff_data_i <= data_delay_n0_i_ext - data_delay_n1_i_ext;
  diff_data_q <= data_delay_n0_q_ext - data_delay_n1_q_ext;

  -- d/2
  diff_data_d2_i <= diff_data_i(dsize_g) & 
                    diff_data_i(dsize_g downto 1);
  diff_data_d2_q <= diff_data_q(dsize_g) & 
                    diff_data_q(dsize_g downto 1);

  -- d/4
  diff_data_d4_i <= diff_data_i(dsize_g) & diff_data_i(dsize_g) & 
                    diff_data_i(dsize_g downto 2);
  diff_data_d4_q <= diff_data_q(dsize_g) & diff_data_q(dsize_g) & 
                    diff_data_q(dsize_g downto 2);

  -- d*2
  diff_data_m2_i <= diff_data_i & '0';
  diff_data_m2_q <= diff_data_q & '0';

  -- d*2 + d
  diff_data_m2_plus_diff_data_i <= diff_data_m2_i + sxt(diff_data_i, diff_data_m2_plus_diff_data_i'high + 1);
  diff_data_m2_plus_diff_data_q <= diff_data_m2_q + sxt(diff_data_q, diff_data_m2_plus_diff_data_q'high + 1);

  -- (d*2 + d)/4
  diff_data_m2_plus_diff_data_d4_i <= diff_data_m2_plus_diff_data_i(diff_data_m2_plus_diff_data_i'high downto 2);
  diff_data_m2_plus_diff_data_d4_q <= diff_data_m2_plus_diff_data_q(diff_data_m2_plus_diff_data_q'high downto 2);
  
--
-- When m=0 : interpolated_data = data_delay_n1
--      m=1 : interpolated_data = data_delay_n1 + diff_data/4
--      m=2 : interpolated_data = data_delay_n1 + diff_data/2
--      m=3 : interpolated_data = data_delay_n1 + diff_data*3/4
--interpolator_rw_wlanrf.vhd
  
  interpolated_data_c_i <= sxt(data_delay_n1_i_ext, interpolated_data_c_i'high + 1) when m="00" else
                           sxt(data_delay_n1_i_ext, interpolated_data_c_i'high + 1) + sxt(diff_data_d4_i, interpolated_data_c_i'high + 1) when m="01" else
                           sxt(data_delay_n1_i_ext, interpolated_data_c_i'high + 1) + sxt(diff_data_d2_i, interpolated_data_c_i'high + 1) when m="10" else
                           sxt(data_delay_n1_i_ext, interpolated_data_c_i'high + 1) + sxt(diff_data_m2_plus_diff_data_d4_i, interpolated_data_c_i'high + 1);


  interpolated_data_c_q <= sxt(data_delay_n1_q_ext, interpolated_data_c_q'high + 1) when m="00" else
                           sxt(data_delay_n1_q_ext, interpolated_data_c_q'high + 1) + sxt(diff_data_d4_q, interpolated_data_c_q'high + 1) when m="01" else
                           sxt(data_delay_n1_q_ext, interpolated_data_c_q'high + 1) + sxt(diff_data_d2_q, interpolated_data_c_q'high + 1) when m="10" else
                           sxt(data_delay_n1_q_ext, interpolated_data_c_q'high + 1) + sxt(diff_data_m2_plus_diff_data_d4_q, interpolated_data_c_q'high + 1);

  -----------------------------------------------------------------------------
  -- Output data.
  -----------------------------------------------------------------------------
  interpolated_data_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      interpolated_data_i <= (others => '0');
      interpolated_data_q <= (others => '0');
    elsif clk'event and clk = '1' then
      if shift_sample = '1' then
        -- disable the interpolator
        if interpolator_enable = '0' then
          interpolated_data_i <= data_in_i;
          interpolated_data_q <= data_in_q;
        else
          -- Saturation of interpolated_data_i and interpolated_data_q of 1 msb
          interpolated_data_i <= sat_signed_slv(interpolated_data_c_i,1);
          interpolated_data_q <= sat_signed_slv(interpolated_data_c_q,1);
        end if;        
      end if;
    end if;
  end process interpolated_data_p;

end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

