--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.std_logic_unsigned.all; 
use ieee.std_logic_arith.all;

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_power_estim is
  generic
  (
    m_size_g  : integer := 7 -- nb of input bits from radio 
  );
  port 
  (
    -- clock and reset
    clk               : in std_logic; -- 60 MHz sampling clock
    reset_n           : in std_logic;

    -- inputs
    blocken           : in std_logic;   -- Activate the block
    update            : in std_logic;   -- Indicates end of integration
    -- data I/Q
    data_in_i         : in std_logic_vector(m_size_g-1 downto 0); -- Data I
    data_in_q         : in std_logic_vector(m_size_g-1 downto 0); -- Data Q
    -- outputs
    power_estimation  : out std_logic_vector (20 downto 0)
    --                  power estimation value. The size max is
    --                  2*m_size + length of "2C" (44 accumulations)
  );

end fe11b_power_estim;

--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture rtl of fe11b_power_estim is

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  signal last_pw_e_activate : std_logic;
  signal sample_delay       : std_logic_vector (21 downto 0);
  signal sample_delay_ff1   : std_logic_vector (21 downto 0);
  signal mult_pow_i         : std_logic_vector(2*m_size_g-1 downto 0); 
  signal mult_pow_q         : std_logic_vector(2*m_size_g-1 downto 0); 
  signal add_pow            : std_logic_vector(2*m_size_g downto 0);   
  signal abs_2_rf_rx        : std_logic_vector (2*m_size_g-1  downto 0);
  signal data_in_i_1t       : std_logic_vector(m_size_g-1 downto 0);
  signal data_in_q_1t       : std_logic_vector(m_size_g-1 downto 0);

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin

  resync_input_p : process (clk, reset_n)                                             
  begin                                                                             
   
    if reset_n = '0' then                                                           
   
      data_in_i_1t <= (others=>'0');
      data_in_q_1t <= (others=>'0');
    
    elsif clk = '1'and clk'event then                                               
    
      if blocken = '1' then
      
        data_in_i_1t <= data_in_i;
        data_in_q_1t <= data_in_q;
      
      else
    
        data_in_i_1t <= (others=>'0');
        data_in_q_1t <= (others=>'0');
      
      end if;
    
    end if;                                                                         
  
  end process resync_input_p;                                                         

  -- energy calculation
  mult_pow_i  <= signed(data_in_i_1t) * signed(data_in_i_1t);
  mult_pow_q  <= signed(data_in_q_1t) * signed(data_in_q_1t);
  add_pow     <= sxt(mult_pow_i, 2*m_size_g+1) + sxt(mult_pow_q, 2*m_size_g+1);
  abs_2_rf_rx <= add_pow(2*m_size_g-1 downto 0);

  -- Accumulation process
  sample_delay_proc:process(clk, reset_n)
  begin
  
    if reset_n = '0' then
    
      last_pw_e_activate <= '0';
      sample_delay_ff1   <= (others => '0');
      power_estimation   <= (others => '0');
    
    elsif clk'event and clk = '1' then
    
      last_pw_e_activate <= blocken;

      if blocken = '1' then   -- block is activated        
      
        if last_pw_e_activate = '0' then
        
          -- first time it has been actived => 
          power_estimation   <= (others => '0'); -- reset register
          sample_delay_ff1  <= (others => '0');
       
        else
        
          -- As long as the power estimation block is enabled the
          -- inputs are accumulated
          if sample_delay_ff1(21) = '0' then      
          
            sample_delay_ff1 <= sample_delay;         
          
          end if;
          
          if update = '1' then
          
            -- End of integration, the output is updated
            if sample_delay(21) = '1' then
           
              power_estimation  <=  (others=>'1');
           
            else
            
              power_estimation  <= sample_delay(20 downto 0);              
           
            end if;
          
          end if;
       
        end if;    
     
      end if;      
   
    end if;
 
  end process sample_delay_proc;

  sample_delay <= sample_delay_ff1 + abs_2_rf_rx;

end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

