--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

use work.mdm_math_func_pkg.all;
 
--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_rx_filter is
  generic 
  (
    dsize_g          : integer := 7
  );                             
  port 
  (
    ----------------------------
    -- Clock and reset          
    ----------------------------
    clk            : in std_logic;
    reset_n        : in std_logic;
    blocken        : in std_logic;
    
    ----------------------------
    -- I & Q      
    ----------------------------
    filter_in_en   : in  std_logic;
    filter_in_i    : in  std_logic_vector(dsize_g-1 downto 0);
    filter_in_q    : in  std_logic_vector(dsize_g-1 downto 0);
    
    filter_out_tog : out std_logic;
    filter_out_i   : out std_logic_vector(dsize_g downto 0);
    filter_out_q   : out std_logic_vector(dsize_g downto 0);

    ----------------------------
    -- I & Q      
    ----------------------------
    firdisb       : in std_logic        -- Disable filter
  );

end fe11b_rx_filter;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture rtl of fe11b_rx_filter is

  -----------------------------------------------------------------------------
  -- Constant
  -----------------------------------------------------------------------------
  constant F0_CT : std_logic_vector(5 downto 0):= "000010";  -- coefficient f0
  constant F1_CT : std_logic_vector(5 downto 0):= "111101";  -- coefficient f1
  constant F2_CT : std_logic_vector(5 downto 0):= "001010";  -- coefficient f2
  constant F3_CT : std_logic_vector(5 downto 0):= "010000";  -- coefficient f3
  
  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  -- filter_in_i input delayed
  signal filter_in_i_ff1  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff2  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff3  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff4  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff5  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff6  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff7  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff8  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff9  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff10 : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_i_ff11 : std_logic_vector(dsize_g-1 downto 0);

  -- filter_in_q input delayed
  signal filter_in_q_ff1  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff2  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff3  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff4  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff5  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff6  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff7  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff8  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff9  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff10 : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff11 : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_q_ff12 : std_logic_vector(dsize_g-1 downto 0);

  -- Multiplexed data to compute multiplication by the coefficient
  signal filter_in_ff1  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff3  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff5  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff6  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff7  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff9  : std_logic_vector(dsize_g-1 downto 0);
  signal filter_in_ff11 : std_logic_vector(dsize_g-1 downto 0);

  -- Sum data before multiplication
  signal sum_f0 : std_logic_vector(dsize_g downto 0);
  signal sum_f1 : std_logic_vector(dsize_g downto 0);
  signal sum_f2 : std_logic_vector(dsize_g downto 0);

  -- Multiplication results
  signal filter_in_mul0 : std_logic_vector(dsize_g+6 downto 0);
  signal filter_in_mul1 : std_logic_vector(dsize_g+6 downto 0);
  signal filter_in_mul2 : std_logic_vector(dsize_g+6 downto 0);
  signal filter_in_mul3 : std_logic_vector(dsize_g+5 downto 0);
  signal sum_iq         : std_logic_vector(dsize_g+6 downto 0);  -- sum of q values
  signal sum_iq_ff1     : std_logic_vector(dsize_g+6 downto 0);  -- sum of q values
  signal sum_int_i_for_gbl : std_logic_vector(dsize_g+6 downto 0);  -- sum of i values
  signal sum_int_q_for_gbl : std_logic_vector(dsize_g+6 downto 0);  -- sum of q values
  
  signal filter_out_tog_o : std_logic;
  signal clk22_en         : std_logic;  -- 22 MHz clock enable
 
 

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin

  ---------------------------------------------------------------------------       
  -- Shift registers                                                                
  ---------------------------------------------------------------------------       
  shift_i_p: process (clk, reset_n)                                                 
  begin                                                                             
    if reset_n = '0' then                                                           
   
      filter_in_i_ff1  <= (others => '0');                                          
      filter_in_i_ff2  <= (others => '0');                                          
      filter_in_i_ff3  <= (others => '0');                                          
      filter_in_i_ff4  <= (others => '0');                                          
      filter_in_i_ff5  <= (others => '0');                                          
      filter_in_i_ff6  <= (others => '0');                                          
      filter_in_i_ff7  <= (others => '0');                                          
      filter_in_i_ff8  <= (others => '0');                                          
      filter_in_i_ff9  <= (others => '0');                                          
      filter_in_i_ff10 <= (others => '0');                                          
      filter_in_i_ff11 <= (others => '0');                                          
                                                                                    
    elsif clk'event and clk = '1' then                                              
     
      if blocken = '0' then
                                                               
        filter_in_i_ff1  <= (others => '0');                                        
        filter_in_i_ff2  <= (others => '0');                                        
        filter_in_i_ff3  <= (others => '0');                                        
        filter_in_i_ff4  <= (others => '0');                                        
        filter_in_i_ff5  <= (others => '0');                                        
        filter_in_i_ff6  <= (others => '0');                                        
        filter_in_i_ff7  <= (others => '0');                                        
        filter_in_i_ff8  <= (others => '0');                                        
        filter_in_i_ff9  <= (others => '0');                                        
        filter_in_i_ff10 <= (others => '0');                                        
        filter_in_i_ff11 <= (others => '0');                                        
                                                                                    
      elsif filter_in_en = '1' then       
                                                 
        filter_in_i_ff1  <= filter_in_i;                                            
        filter_in_i_ff2  <= filter_in_i_ff1;                                        
        filter_in_i_ff3  <= filter_in_i_ff2;                                        
        filter_in_i_ff4  <= filter_in_i_ff3;                                        
        filter_in_i_ff5  <= filter_in_i_ff4;                                        
        filter_in_i_ff6  <= filter_in_i_ff5;                                        
        filter_in_i_ff7  <= filter_in_i_ff6;                                        
        filter_in_i_ff8  <= filter_in_i_ff7;                                        
        filter_in_i_ff9  <= filter_in_i_ff8;                                        
        filter_in_i_ff10 <= filter_in_i_ff9;                                        
        filter_in_i_ff11 <= filter_in_i_ff10;                                       
                                                                                    
      end if;                                                                       

    end if;                                                                         
  
  end process shift_i_p;                                                            

  shift_q_p : process (clk, reset_n)                                                
  begin                                                                             
    
    if reset_n = '0' then                                                           
      filter_in_q_ff1  <= (others => '0');                                          
      filter_in_q_ff2  <= (others => '0');                                          
      filter_in_q_ff3  <= (others => '0');                                          
      filter_in_q_ff4  <= (others => '0');                                          
      filter_in_q_ff5  <= (others => '0');                                          
      filter_in_q_ff6  <= (others => '0');                                          
      filter_in_q_ff7  <= (others => '0');                                          
      filter_in_q_ff8  <= (others => '0');                                          
      filter_in_q_ff9  <= (others => '0');                                          
      filter_in_q_ff10 <= (others => '0');                                          
      filter_in_q_ff11 <= (others => '0');                                          
      filter_in_q_ff12 <= (others => '0');                                          
                                                                                    
    elsif clk'event and clk = '1' then                                              
  
      if blocken = '0' then                                                         
       
        filter_in_q_ff1  <= (others => '0');                                        
        filter_in_q_ff2  <= (others => '0');                                        
        filter_in_q_ff3  <= (others => '0');                                        
        filter_in_q_ff4  <= (others => '0');                                        
        filter_in_q_ff5  <= (others => '0');                                        
        filter_in_q_ff6  <= (others => '0');                                        
        filter_in_q_ff7  <= (others => '0');                                        
        filter_in_q_ff8  <= (others => '0');                                        
        filter_in_q_ff9  <= (others => '0');                                        
        filter_in_q_ff10 <= (others => '0');                                        
        filter_in_q_ff11 <= (others => '0');                                        
                                                                                    
      elsif filter_in_en = '1' then   
                                             
        filter_in_q_ff1  <= filter_in_q;                                            
        filter_in_q_ff2  <= filter_in_q_ff1;                                        
        filter_in_q_ff3  <= filter_in_q_ff2;                                        
        filter_in_q_ff4  <= filter_in_q_ff3;                                        
        filter_in_q_ff5  <= filter_in_q_ff4;                                        
        filter_in_q_ff6  <= filter_in_q_ff5;                                        
        filter_in_q_ff7  <= filter_in_q_ff6;                                        
        filter_in_q_ff8  <= filter_in_q_ff7;                                        
        filter_in_q_ff9  <= filter_in_q_ff8;                                        
        filter_in_q_ff10 <= filter_in_q_ff9;                                        
        filter_in_q_ff11 <= filter_in_q_ff10;                                       
        filter_in_q_ff12 <= filter_in_q_ff11;                                       
    
      end if;                                                                       
   
    end if;                                                                         
 
  end process shift_q_p;                                                            

  ---------------------------------------------------------------------------       
  -- Inputs for sum are multiplexed as there is a downsample on the block           
  -- output                                                                         
  ---------------------------------------------------------------------------       
  filter_in_ff1 <= filter_in_i_ff1 when clk22_en = '1' else filter_in_q_ff2;        
  filter_in_ff3 <= filter_in_i_ff3 when clk22_en = '1' else filter_in_q_ff4;        
  filter_in_ff5 <= filter_in_i_ff5 when clk22_en = '1' else filter_in_q_ff6;        
  filter_in_ff6 <= filter_in_i_ff6 when clk22_en = '1' else filter_in_q_ff7;        
  filter_in_ff7 <= filter_in_i_ff7 when clk22_en = '1' else filter_in_q_ff8;        
  filter_in_ff9 <= filter_in_i_ff9 when clk22_en = '1' else filter_in_q_ff10;       
  filter_in_ff11 <= filter_in_i_ff11 when clk22_en = '1' else filter_in_q_ff12;     

  ---------------------------------------------------------------------------       
  -- Coefficients                                                                   
  -- As the filter coefficient are symetric, the FFs are first added and then       
  -- multiplied. All bits are kept for sum accuracy.                                
  ---------------------------------------------------------------------------       
  sum_f0 <= sxt(filter_in_ff1,dsize_g+1) + sxt(filter_in_ff11,dsize_g+1);           
  sum_f1 <= sxt(filter_in_ff3,dsize_g+1) + sxt(filter_in_ff9,dsize_g+1);            
  sum_f2 <= sxt(filter_in_ff5,dsize_g+1) + sxt(filter_in_ff7,dsize_g+1);            

  filter_in_mul0 <= signed(sum_f0) * signed(F0_CT);                                 
  filter_in_mul1 <= signed(sum_f1) * signed(F1_CT);                                 
  filter_in_mul2 <= signed(sum_f2) * signed(F2_CT);                                 
  filter_in_mul3 <= signed(filter_in_ff6) * signed(F3_CT);                          

  
  sum_iq <= filter_in_mul0 + filter_in_mul1 + filter_in_mul2 +                      
            sxt(filter_in_mul3,dsize_g+7);                                          

  ---------------------------------------------------------------------------       
  -- Outputs                                                                        
  ---------------------------------------------------------------------------       

  output_p : process (clk, reset_n)                                                 
    variable sum_int_i : std_logic_vector(dsize_g+6 downto 0);  -- sum of i values  
    variable sum_int_q : std_logic_vector(dsize_g+6 downto 0);  -- sum of q values  
  begin                                                                             
    if reset_n = '0' then                                                           
    
      clk22_en         <= '0';                                                      
      filter_out_tog_o <= '0';                                                      
      sum_int_i        := (others => '0');                                          
      sum_int_q        := (others => '0');                                          
      sum_int_i_for_gbl<= (others => '0');                                          
      sum_int_q_for_gbl<= (others => '0');                                          
      sum_iq_ff1       <= (others => '0');                                          
      --                                                                            
      filter_out_i      <= (others => '0');                                         
      filter_out_q      <= (others => '0');                                         
                                                                                    
    elsif clk'event and clk = '1' then                                              
      
      if blocken = '0' then                                                         
      
        clk22_en          <= '0';                                                   
        sum_int_i         := (others => '0');                                       
        sum_int_q         := (others => '0');                                       
        sum_int_i_for_gbl <= (others => '0');                                       
        sum_int_q_for_gbl <= (others => '0');                                       
        sum_iq_ff1        <= (others => '0');                                       
        --                                                                          
        filter_out_i      <= (others => '0');                                       
        filter_out_q      <= (others => '0');                                       
    
      else                                                                          

        -- Generate enable to downsample by 2                                       
        if filter_in_en = '1' then                                                 
        
          clk22_en   <= not clk22_en;                                               
          sum_iq_ff1 <= sum_iq;                                                     
       
        end if;                                                                     

        -- Outputs are downsampled to 22 MHz                                        
        if clk22_en = '0' and filter_in_en = '1' then                              
         
          filter_out_tog_o <= not filter_out_tog_o;                                 
          
          if firdisb = '1' then                                                     
            -- Filter is bypassed                                                   
            sum_int_i         := sxt(filter_in_i, dsize_g+2) & "00000";             
            sum_int_q         := sxt(filter_in_q, dsize_g+2) & "00000";             
            sum_int_i_for_gbl <= sxt(filter_in_i, dsize_g+2) & "00000";             
            sum_int_q_for_gbl <= sxt(filter_in_q, dsize_g+2) & "00000";             
            --                                                                      
            filter_out_i      <= sum_int_i(dsize_g+5 downto 5) + sum_int_i(4);      
            filter_out_q      <= sum_int_q(dsize_g+5 downto 5) + sum_int_q(4);      
          else                                                                      
            sum_int_i         := sum_iq_ff1;                                        
            sum_int_q         := sum_iq;                                            
            sum_int_i_for_gbl <= sum_iq_ff1;                                        
            sum_int_q_for_gbl <= sum_iq;                                            
            --                                                                      
            filter_out_i      <= sum_int_i(dsize_g+5 downto 5) + sum_int_i(4);      
            filter_out_q      <= sum_int_q(dsize_g+5 downto 5) + sum_int_q(4);      

          end if;                                                                   
        
        end if;                                                                     
     
      end if;                                                                       
   
    end if;                                                                         
  end process output_p;                                                             

  filter_out_tog <= filter_out_tog_o;                                               
  
end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

