--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

use work.mdm_math_func_pkg.all;
use work.cordic_vect_pkg.all;

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_timingoff_estim is
  generic 
  (
    m_size_g   : integer := 7;          -- I&Q size
    tau_size_g : integer := 13
    );        -- Tau size
  port 
  (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
    clk     : in std_logic;             -- Clock
    reset_n : in std_logic;             -- Reset

    --------------------------------------
    -- I & Q
    --------------------------------------
    data_in_i     : in  std_logic_vector(m_size_g-1 downto 0); -- I data
    data_in_q     : in  std_logic_vector(m_size_g-1 downto 0); -- Q data
    datain_enable : in std_logic;       -- Data enable
    tau           : out std_logic_vector(tau_size_g-1 downto 0); -- Tau

    --------------------------------------
    -- Control
    --------------------------------------
    timingoff_en  : in std_logic         -- Enables the timing offset estimation 
  );

end fe11b_timingoff_estim ;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture rtl of fe11b_timingoff_estim  is

  ------------------------------------------------------------------------------
  -- Types
  ------------------------------------------------------------------------------

  ------------------------------------------------------------------------------
  -- Constant
  ------------------------------------------------------------------------------
  constant FIVEHUNDRED_US_CT  : std_logic_vector(8 downto 0)  := "111110011";
  -- Five hundred micro second
  constant THREEHUNDRED_US_CT : std_logic_vector(8 downto 0)  := "100101011";
  -- Five hundred micro second
  constant GAIN_CT            : std_logic_vector(3 downto 0)  := "1011"; 
  -- gain coefficient: -8/2Pi
  constant NULL_CT        : std_logic_vector(16 downto 0) := (others => '0');
  constant ONE_US_CT      : std_logic_vector(5 downto 0)  := "101011";
  constant ACCU_MAX_CT    : std_logic_vector(15 downto 0) := "0111111111111111";
  constant ACCU_MIN_CT    : std_logic_vector(15 downto 0) := "1000000000000001";
  

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  -- I & Q sums
  signal abs_i  : std_logic_vector(m_size_g-1 downto 0);  -- Data in i abs value
  signal abs_q  : std_logic_vector(m_size_g-1 downto 0);  -- Data in q abs value 
  signal iq_sum : std_logic_vector(m_size_g-1 downto 0);  -- I and Q sum
  signal iq_sum_int : std_logic_vector(2 downto 0);
  signal datain_enable_ff1 : std_logic;  -- Datain_enable resync
  
  -- Block enable
  signal timingoff_en_ff1 : std_logic;  -- Timing offset estimation enable

  -- Lbox 1
  signal index_lbox1 : std_logic_vector(1 downto 0);
  signal l_box1_x    : std_logic_vector(16 downto 0); -- Input of accu in L-box1
  signal l_box1_buf_x: std_logic_vector(15 downto 0); -- L-box1 accu value 
  signal l_box1_y    : std_logic_vector(16 downto 0); -- Input of accu in L-box1
  signal l_box1_buf_y: std_logic_vector(15 downto 0); -- L-box1 accu value 

  -- Lbox 2
  signal index_lbox2 : std_logic_vector(1 downto 0);
  signal l_box2_x    : std_logic_vector(16 downto 0);  -- Input of accu in L-box1
  signal l_box2_buf_x: std_logic_vector(15 downto 0);  -- L-box1 accu value 
  signal l_box2_y    : std_logic_vector(16 downto 0);  -- Input of accu in L-box1
  signal l_box2_buf_y: std_logic_vector(15 downto 0);  -- L-box1 accu value 

  -- Cordic
  signal cordic_in_x      : std_logic_vector(15 downto 0); -- Cordic input x
  signal cordic_in_y      : std_logic_vector(15 downto 0); -- Cordic input y
  signal cordic_in_x_int  : std_logic_vector(15 downto 0);
  signal cordic_in_y_int  : std_logic_vector(15 downto 0);
  signal cordic_angle     : std_logic_vector(11 downto 0); -- Recovered angle
                                                           -- in range [-Pi;Pi]
  signal cordic_angle_ff1 : std_logic_vector(11 downto 0);
  signal angle_out        : std_logic_vector(tau_size_g-2 downto 0); 
                                                            -- Cordic output
  signal cordic_ready     : std_logic; -- Value computed by cordic ready
  signal cordic_ready_ff1 : std_logic;
  signal load             : std_logic; -- load input in cordic

  -- Unwrap and realign
  signal k_al     : std_logic_vector(15 downto 0);  -- Realignement variable
  signal k_al_buf : std_logic_vector(15 downto 0);  -- Realignement variable
  signal k_uw     : std_logic_vector(12 downto 0);  -- Wrapping variable
  signal k_uw_ff1 : std_logic_vector(12 downto 0);  -- Wrapping variable

  -- Reconstructed angle
  signal tau_o   : std_logic_vector(15 downto 0);
  signal tau_mul : std_logic_vector(19 downto 0);

  -- Counters and switch control
  signal switch_counter  : std_logic_vector(8 downto 0);  -- Counts up for
                                        -- switch between the 2 L-boxes
  signal l_box1_active   : std_logic;   -- L_box 1 is active
  signal switch_lbox     : std_logic;   -- Switch between the 2 L-boxes
  signal firstinit       : std_logic;   -- First initialization phase
  signal one_us_en_ff1   : std_logic;   -- One microsecond enable
  signal one_us_en       : std_logic;
  signal one_us_counter  : std_logic_vector(5 downto 0);
  signal l_box_changed   : std_logic;   -- switch between l-boxes occured
  signal firstinit_cordic : std_logic;  -- Indicates that the first 300 us are
                                        -- over ut waiting for cordic ready

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
  
begin


  -----------------------------------------------------------------------------
  -- Absolute value
  -----------------------------------------------------------------------------
  abs_val_p : process (reset_n, clk)
  begin
    if reset_n = '0' then
      abs_i <= (others => '0');
      abs_q <= (others => '0');
      datain_enable_ff1 <= '0';
    elsif clk'event and clk = '1' then
      datain_enable_ff1 <= datain_enable;
      
      if data_in_i(m_size_g-1) = '0' then
        abs_i <= data_in_i;
      else
        abs_i <= (not data_in_i +'1');
      end if;
      if data_in_q(m_size_g-1) = '0' then
        abs_q <= data_in_q;
      else
        abs_q <= (not data_in_q +'1');
      end if;
    end if;
  end process abs_val_p;

  
  -- absolute value sum
  iq_sum <= abs_i + abs_q;
  -- Sum is done to the nearest value i.e. bit after coma is checked
  iq_sum_int <= iq_sum(m_size_g-1 downto m_size_g-3)+ '1' when
                      iq_sum(m_size_g-4) = '1'
                      and iq_sum(m_size_g-1 downto m_size_g-3) /= "111"
                else iq_sum(m_size_g-1 downto m_size_g-3);

  ------------------------------------------------------------------------------
  -- L_box
  ------------------------------------------------------------------------------
  -- Control the 2 L-boxes: activate them one after the other
  l_box_ctrl_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      l_box1_active    <= '0';
      timingoff_en_ff1 <= '0';
      firstinit        <= '0';
    elsif clk'event and clk = '1' then
      timingoff_en_ff1 <= timingoff_en;
      if timingoff_en = '1' and timingoff_en_ff1 = '0' then
        l_box1_active <= '1';
        firstinit     <= '1';
      elsif switch_counter = THREEHUNDRED_US_CT and firstinit = '1' and
            one_us_en = '1' then
        firstinit <= '0';
      elsif switch_lbox = '1' then
        l_box1_active <= not l_box1_active;
      end if;
    end if;
  end process l_box_ctrl_p;

  -- Controls switch fron one L-box to the other
  switch_lbox <= '1' when
                     one_us_en = '1' and (switch_counter = FIVEHUNDRED_US_CT)
                 else '0';

  ------------
  -- L-box 1
  ------------
  l_box1_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      index_lbox1 <= (others => '0');
      l_box1_buf_x    <= (others => '0');
      l_box1_buf_y    <= (others => '0');
    elsif clk'event and clk = '1' then
      if (l_box1_active = '1' and timingoff_en = '1' and switch_lbox = '1')
        or (index_lbox1 = "11" and datain_enable_ff1 = '1')
        or (timingoff_en = '1' and timingoff_en_ff1 = '0') then
        -- The l-box is reinitialized at the beginning of each run cycle
        index_lbox1 <= (others => '0');
      elsif timingoff_en = '1' and datain_enable_ff1 = '1' then
        index_lbox1 <= index_lbox1 + '1';
      end if;

      if (l_box1_active = '1' and timingoff_en = '1' and switch_lbox = '1') 
        or (timingoff_en = '1' and timingoff_en_ff1 = '0') then
        -- Accu are reset when the block is enabled and a switch occur
        -- and the L-box will be activated
        l_box1_buf_x <= (others => '0');
        l_box1_buf_y <= (others => '0');
      elsif datain_enable_ff1  = '1' then
           
        if l_box1_x(16) = l_box1_x(15) then
          l_box1_buf_x <= l_box1_x(15 downto 0);
        else 
          if l_box1_buf_x(15) = '1' then
            l_box1_buf_x <= ACCU_MIN_CT;
          else 
            l_box1_buf_x <= ACCU_MAX_CT;           
          end if;
        end if;
        if l_box1_y(16) = l_box1_y(15) then
          l_box1_buf_y <= l_box1_y(15 downto 0);
        else 
          if l_box1_buf_y(15) = '1' then
            l_box1_buf_y <= ACCU_MIN_CT;
          else
            l_box1_buf_y <= ACCU_MAX_CT;   
          end if;             
        end if;
      end if;
    end if;
  end process l_box1_p;

 
  with index_lbox1 select
    l_box1_x <=
       sxt(l_box1_buf_x,17) + (NULL_CT(16 downto 3)&iq_sum_int)
        when "00",                          -- Coefficient is 1
       sxt(l_box1_buf_x,17) - (NULL_CT(16 downto 3)&iq_sum_int)
        when "10" ,                         -- Coefficient is -1
       sxt(l_box1_buf_x,17) when others;

  with index_lbox1 select
    l_box1_y <=
       sxt(l_box1_buf_y,17) - (NULL_CT(16 downto 3)&iq_sum_int)
        when "01",                          -- Coefficient is -1
       sxt(l_box1_buf_y,17) + (NULL_CT(16 downto 3)&iq_sum_int)
        when "11",                           -- Coefficient is 1
       sxt(l_box1_buf_y,17) when others;

  ------------
  -- L-box 2
  ------------
  l_box2_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      index_lbox2  <= (others => '0');
      l_box2_buf_x <= (others => '0');
      l_box2_buf_y <= (others => '0');
    elsif clk'event and clk = '1' then

      if (l_box1_active = '0' and timingoff_en = '1' and switch_lbox = '1')
        or (index_lbox2 = "11" and datain_enable_ff1 = '1')  then
        -- The l-box is reinitialized at the beginning of each run cycle
        index_lbox2 <= (others => '0');
      elsif timingoff_en_ff1 = '1' and datain_enable_ff1 = '1' and
           (firstinit = '0' or (one_us_en = '1' and
                                       switch_counter = THREEHUNDRED_US_CT ))then
        index_lbox2 <= index_lbox2 + '1';
      end if;

      if (l_box1_active = '0' and timingoff_en = '1' and switch_lbox = '1')
          or (one_us_en = '1' and
                 switch_counter = THREEHUNDRED_US_CT and firstinit = '1') then
        l_box2_buf_x <= (others => '0');
        l_box2_buf_y <= (others => '0');
      elsif datain_enable_ff1 = '1' then
        
        if l_box2_x(16) = l_box2_x(15) then
          l_box2_buf_x <= l_box2_x(15 downto 0);
        else
          if l_box2_buf_x(15) = '1' then
            l_box2_buf_x <= ACCU_MIN_CT;
          else 
            l_box2_buf_x <= ACCU_MAX_CT;                
          end if;  
        end if;
        
        if l_box2_y(16) = l_box2_y(15) then
          l_box2_buf_y <= l_box2_y(15 downto 0);
        else
          if l_box2_buf_y(15) = '1' then
            l_box2_buf_y <= ACCU_MIN_CT;
          else
            l_box2_buf_y <= ACCU_MAX_CT;                
          end if;            
        end if;
      end if;
    end if;
  end process l_box2_p;
    
  with index_lbox2 select
    l_box2_x <=
       sxt(l_box2_buf_x,17) + (NULL_CT(16 downto 3)&iq_sum_int)
        when "00",                          -- Coefficient is 1
       sxt(l_box2_buf_x,17) - (NULL_CT(16 downto 3)&iq_sum_int)
        when "10" ,                         -- Coefficient is -1
       sxt(l_box2_buf_x,17) when others;

  with index_lbox2 select
    l_box2_y <=
       sxt(l_box2_buf_y,17) - (NULL_CT(16 downto 3)&iq_sum_int)
        when "01",                          -- Coefficient is 1
       sxt(l_box2_buf_y,17) + (NULL_CT(16 downto 3)&iq_sum_int)
        when "11" ,                         -- Coefficient is -1
       sxt(l_box2_buf_y,17) when others;

  -----------------------------------------------------------------------------
  -- Downsampling at 1 MHz
  -----------------------------------------------------------------------------
  downsampling_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      cordic_in_x_int <= (others => '0');
      cordic_in_y_int <= (others => '0');
    elsif clk'event and clk = '1' then
      if one_us_en = '1' and datain_enable_ff1 = '1' then
        if (l_box1_active xor switch_lbox) = '1' then
          -- Take L-box1 value
          if l_box1_x(15) = l_box1_x(16) then
            cordic_in_x_int <= l_box1_x(15 downto 0);
          else
            if l_box1_buf_x(15) = '1' then
              cordic_in_x_int <= ACCU_MIN_CT;
            else 
              cordic_in_x_int <= ACCU_MAX_CT;    
            end if;            
          end if;  
  
          if l_box1_y(15) = l_box1_y(16) then
            cordic_in_y_int <= l_box1_y(15 downto 0);
          else
            if l_box1_buf_y(15) = '1' then
              cordic_in_y_int <= ACCU_MIN_CT;
            else 
              cordic_in_y_int <= ACCU_MAX_CT;    
            end if;            
          end if;  

        else
          -- Take L-box2 value
          if l_box2_x(15) = l_box2_x(16) then
            cordic_in_x_int <= l_box2_x(15 downto 0);
          else
            if l_box2_buf_x(15) = '1' then
              cordic_in_x_int <= ACCU_MIN_CT;
            else 
              cordic_in_x_int <= ACCU_MAX_CT;    
            end if;            
          end if;  
  
          if l_box2_y(15) = l_box2_y(16) then
            cordic_in_y_int <= l_box2_y(15 downto 0);
          else
            if l_box2_buf_y(15) = '1' then
              cordic_in_y_int <= ACCU_MIN_CT;
            else 
              cordic_in_y_int <= ACCU_MAX_CT;    
            end if;            
          end if;            
        end if;
      end if;
    end if;
  end process downsampling_p;

  -- Angle must be between -Pi/2 and Pi/2
  cordic_in_x <=  ACCU_MAX_CT when 
                      cordic_in_x_int(15) = '1' and 
                      signed(cordic_in_x_int) = signed(ACCU_MIN_CT) else
                      (not cordic_in_x_int + '1')  when cordic_in_x_int(15) = '1'  
                 else cordic_in_x_int;
  cordic_in_y <= ACCU_MAX_CT when 
                      cordic_in_x_int(15) = '1' and 
                      signed(cordic_in_y_int) = signed(ACCU_MIN_CT) else
                 (not cordic_in_y_int + '1') when cordic_in_x_int(15) = '1'else 
                 cordic_in_y_int;
  
  -- Recover real angle if it has been modified before the CORDIC
  -- Cordic value is divided by 2 because of cordic algorithm see cordic_vect
  update_angle_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      cordic_angle <= (others => '0');
    elsif clk'event and clk = '1' then
      if cordic_ready = '1'  and timingoff_en = '1' then
        if cordic_in_x_int(15) = '1' and cordic_in_y_int(15) = '1' then
          cordic_angle <= angle_out(11) &angle_out(11 downto 1) - "011001001000";
        elsif cordic_in_x_int(15) = '1' and cordic_in_y_int(15) = '0' then
          cordic_angle <= angle_out(11)&angle_out(11 downto 1) + "011001001000";
        else
          cordic_angle <= angle_out(11) & angle_out(11 downto 1);
        end if;
      end if;
    end if;
  end process update_angle_p;


  -----------------------------------------------------------------------------
  -- Unwrap and realign
  -----------------------------------------------------------------------------
  -- At each switch between the 2 L-boxes the phase has to be realigned to the
  -- current value of tau.
    align_p : process (clk, reset_n)
    begin
      if reset_n = '0' then
        cordic_ready_ff1 <= '0';
        k_uw_ff1         <= (others => '0');
        cordic_angle_ff1 <= (others => '0');
        l_box_changed    <= '0';
        k_al_buf         <= (others => '0');
        firstinit_cordic <= '0';
      elsif clk'event and clk = '1' then
        k_uw_ff1         <= k_uw;
        cordic_angle_ff1 <= cordic_angle;
        cordic_ready_ff1 <= cordic_ready;
        
        if timingoff_en = '1' and timingoff_en_ff1 = '0' then
          k_al_buf <= (others => '0');
          k_uw_ff1 <= (others => '0');
          l_box_changed <= '0';
          firstinit_cordic <= '0';
        elsif firstinit = '1' and switch_counter = THREEHUNDRED_US_CT
               and one_us_en = '1' then
          -- Keep information that 300 us are just over and wait until
          -- CORDIC result is ready
            firstinit_cordic <= '1';               
        elsif switch_lbox = '1' then
          l_box_changed <= '1';
        elsif cordic_ready_ff1 = '1' and
          (l_box_changed = '1' or firstinit_cordic = '1' ) then        
          l_box_changed <= '0';
          firstinit_cordic <= '0';
          k_al_buf      <= k_al;
          k_uw_ff1      <= (others => '0');
        end if;
        
      end if;
    end process align_p;

  -- Alignment constant: at l-box switch tau has to be realigned with the
  -- last value
  k_al <= tau_o - sxt(cordic_angle,16) when
                     (cordic_ready_ff1 = '1' and
                      (l_box_changed = '1' or firstinit_cordic = '1'))
          else k_al_buf;

  -- Unwrap constant: updated when a shift of more than Pi is detected
  -- 2*Pi is then added or substracted
  k_uw <= k_uw_ff1 + "0110010010000" when    
                 (signed(cordic_angle_ff1) > signed(cordic_angle) and
                  cordic_angle_ff1 - cordic_angle >= "011001001000" and 
                  firstinit = '0' and l_box_changed = '0')  else 
          k_uw_ff1 + "1001101110000" when 
                 (signed(cordic_angle_ff1) < signed(cordic_angle) and
                  cordic_angle - cordic_angle_ff1 >= "011001001000" and 
                   firstinit = '0' and l_box_changed = '0') else
         (others => '0') when
                 (l_box_changed = '1' and cordic_ready_ff1 = '1') else 
          k_uw_ff1;

  -- Reconstructed angle
  tau_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      tau_o <= (others => '0');
    elsif clk'event and clk = '1' then
      if firstinit = '0' and firstinit_cordic = '0'then
        tau_o <= sxt(cordic_angle, 16) + sxt(k_uw, 16) + k_al;
      else
        tau_o <= (others => '0');
      end if;
    end if;
  end process tau_p;
  
  -- Constant multiplication
  tau_mul <= signed(tau_o) * signed(GAIN_CT);

  -- Tau saturation
  tau_out_sat_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      tau <= (others => '0');
    elsif clk'event and clk = '1' then
      tau <= sat_signed_slv(tau_mul(19 downto 5),2);
    end if;
  end process tau_out_sat_p;

  ------------------------------------------------------------------------------
  -- Counter
  ------------------------------------------------------------------------------
  -- Counts up for switch between L-boxes
  switch_counter_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      switch_counter <= (others => '0');
      one_us_en_ff1  <= '0';
    elsif clk'event and clk = '1' then
      one_us_en_ff1 <= one_us_en;
      if (one_us_en = '1' and
          (switch_counter = FIVEHUNDRED_US_CT or
           (switch_counter = THREEHUNDRED_US_CT and firstinit = '1')))
        or (timingoff_en = '1' and timingoff_en_ff1 = '0') then
        switch_counter <= (others => '0');
      elsif one_us_en = '1' and timingoff_en = '1' then
        switch_counter <= switch_counter + '1';
      end if;
    end if;
  end process switch_counter_p;


  -- Generate one microsecond enable
  one_us_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      one_us_counter <= (others => '0');
    elsif clk'event and clk = '1' then
      if (one_us_counter = ONE_US_CT  and datain_enable_ff1 = '1') or
        (timingoff_en = '1' and timingoff_en_ff1 = '0') then
        one_us_counter <= (others => '0');
      elsif timingoff_en = '1' and datain_enable_ff1 = '1' then
        one_us_counter <= one_us_counter + '1';
      end if;
    end if;
  end process one_us_p;

  one_us_en <= '1' when one_us_counter = ONE_US_CT and datain_enable_ff1 = '1' else '0';
  

  -----------------------------------------------------------------------------
  -- Port map 
  -----------------------------------------------------------------------------
  cordic_vect_1 : cordic_vect
    generic map (
      datasize_g  => m_size_g+9,
      errorsize_g => tau_size_g-1,
      scaling_g   => 0
      )
    port map (
      -- clock and reset.
      clk          => clk,
      reset_n      => reset_n,
      --
      load         => load,             -- Load input values.
      x_in         => cordic_in_x,      -- Real part in.
      y_in         => cordic_in_y,      -- Imaginary part in.
      --
      angle_out    => angle_out,        -- Angle out.
      cordic_ready => cordic_ready      -- Angle ready.
      );

  load <= one_us_en_ff1 and timingoff_en;
  


end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

