--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.std_logic_unsigned.all; 
use ieee.std_logic_arith.all;

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity fe11b_tx_filter is
  generic(
    out_length_g : integer := 7; -- number of significant output bits
    phi_degree_g : integer := 5  -- deg of filter (3 -> 4 phis on equation)
    );
  port (
    -- clocks and reset
    clk          : in  std_logic; -- 60 MHz clock (and not 44 MHz as tx_filter)
    reset_n      : in  std_logic;   
    -- input signals
    fir_activate : in  std_logic; -- activate the block (when disact, it finishes the transfer) 
    fir_disb     : in  std_logic; -- when disb, i and q are transfered without modif
    init_fir     : in  std_logic; -- init the registers
    fir_coef_sel : in  std_logic;
    phi_angle    : in  std_logic_vector (1 downto 0); -- phi input 
    -- output signals
    data_valid   : out std_logic;
    data_i       : out std_logic_vector (out_length_g -1 downto 0);
    data_q       : out std_logic_vector (out_length_g -1 downto 0)
  );

end fe11b_tx_filter;

--------------------------------------------------------------------------------
-- Architecture
--
--                   ___    ___          ___
--  -- phi_angle -->|  |-->|  |-......->|  |-->
--                  |__|   |__|         |__|
--                  _|_    _|_          _|_
--                  \/ C0  \/ C1        \/ C11
--                  |      |            |___________________(   ) 
--                  |      |________________________________( + |---> I/Q
--                  |_______________________________________(   )
--
-- As the phi_angle arrives at 11 MHz speed, 3x0 are inserted between. 
-- Therefore, there is only 2^6 different values (3 following phi_angles) 
-- that will be shifted.
--  p1 0  0  0  p2 0  0  0  p3 0  0  0  -> s1
--  0  p1 0  0  0  p2 0  0  0  p3 0  0  -> s2
--  0  0  p1 0  0  0  p2 0  0  0  p3 0  -> s3
--  0  0  0  p1 0  0  0  p2 0  0  0  p3 -> s4
--  
-- A table (ROM) is implemented that gives the 4 following outputs for 5 
-- phi_angle given.
--
-- As a pi/4 rotation is performed Q = phi_angle(1) and I = phi_angle(0)
--------------------------------------------------------------------------------
architecture rtl of fe11b_tx_filter is


  ------------------------------------------------------------------------------
  -- Constants
  ------------------------------------------------------------------------------
  -- For disable mode:
  constant ONES_CT       : std_logic_vector (out_length_g-2 downto 0) := (others => '1');
  constant ZEROS_CT      : std_logic_vector (out_length_g-2 downto 0) := (others => '0');
  constant ZEROS2_CT     : std_logic_vector (out_length_g-1 downto 0) := (others => '0');
  constant SIGNED_MAX_CT : std_logic_vector (out_length_g-1 downto 0) := '0' & ONES_CT;
  constant SIGNED_MIN_CT : std_logic_vector (out_length_g-1 downto 0) := '1' & ZEROS_CT;

  -- Dummy i/q to insert at the end of a packet
  constant DUMMY_I_CT : std_logic := '1';  -- -1
  constant DUMMY_Q_CT : std_logic := '1';  -- -1

  -- Output values for initialization (for out_length_g = 7)
  constant I_OUTPUT_INIT_CT : std_logic_vector (6 downto 0) := "0111111";
  constant Q_OUTPUT_INIT_CT : std_logic_vector (6 downto 0) := "0111111";

  -- Val of shift_counter
  constant SC_INIT_VAL_CT       : std_logic_vector(3 downto 0) := "1100";  -- init is 12d

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  -- Modified Input for dummy bits insertion
  signal reg_in_i       : std_logic;
  signal reg_in_q       : std_logic;
  -- Registered Modified Input
  signal reg_in_i_reg   : std_logic;
  signal reg_in_q_reg   : std_logic;  
  -- Registered phis
  signal phi_reg_i      : std_logic_vector (phi_degree_g-2 downto 0); 
  signal phi_reg_q      : std_logic_vector (phi_degree_g-2 downto 0); 

  -- luts addresses
  signal lut_add_i      : std_logic_vector (phi_degree_g-1 downto 0);
  signal lut_add_q      : std_logic_vector (phi_degree_g-1 downto 0);
  -- luts outputs  (4 next values to send)
  signal lut_out_i      : std_logic_vector (4*out_length_g-1 downto 0);
  signal lut_out_q      : std_logic_vector (4*out_length_g-1 downto 0);

  -- counter that determines 1 of the 4 outputs
  signal four_count     : std_logic_vector (1 downto 0);

  -- Output Registers
  signal i_out_reg      : std_logic_vector (out_length_g-1 downto 0);
  signal q_out_reg      : std_logic_vector (out_length_g-1 downto 0);

  -- Shift_sample Generation
  signal shift_samp     : std_logic; -- high when 44MB/s data needed
  signal shift_samp_ff1  : std_logic; -- delayed for upsample block
  
  -- For finishing a packet
  signal fir_activate_dly : std_logic;
  signal fir_activate_tot : std_logic;
  signal end_count        : std_logic_vector (3 downto 0);

  -- When FIR is disabled
  signal i_out_disb     : std_logic_vector (out_length_g-1 downto 0);
  signal q_out_disb     : std_logic_vector (out_length_g-1 downto 0);  
  
  ------------------------------------------------------------------------------
  -- components
  ------------------------------------------------------------------------------
  component fe11b_tx_lut_iq
  port 
  (
    lut_add      : in std_logic_vector (4 downto 0);  
    fir_coef_sel : in std_logic;
    lut_out      : out std_logic_vector(27 downto 0)  
  );
  end component;

--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin

  
  -----------------------------------------------------------------------------
  -- phi registers processsor
  -----------------------------------------------------------------------------
  -- phi_angle ->---|----------|   
  --             __\/__        |    
  --            |  |  |        |     
  --            |__|__|--------|-----> lut_add
  --               |           |      
  --             __\/__        |
  --            |  |  |        | 
  --            |__|__|--------|  
  --               |
  --             __\/__        |
  --            |  |  |        | 
  --            |__|__|--------|  
  -----------------------------------------------------------------------------

  -- reg_in gets some DUMMY_CT at the end of a packet to finish it.
  reg_in_i <= phi_angle(0) when fir_activate = '1' else DUMMY_I_CT;
  reg_in_q <= phi_angle(1) when fir_activate = '1' else DUMMY_Q_CT;

  -- Register reg_in_i/q when needed (as they come from a resynchro)
  reg_in_p : process (clk, reset_n)
  begin  -- process reg_in_p
    if reset_n = '0' then
      reg_in_i_reg <= DUMMY_I_CT;
      reg_in_q_reg <= DUMMY_Q_CT;
    elsif clk'event and clk = '1' then
      if fir_activate_tot = '1' then
        if four_count = "00" and shift_samp = '1'  and init_fir = '0' then
          -- register the incoming data
          reg_in_i_reg <= reg_in_i;
          reg_in_q_reg <= reg_in_q;
        end if;
      else
        reg_in_i_reg <= DUMMY_I_CT;
        reg_in_q_reg <= DUMMY_Q_CT;        
      end if;
    end if;
  end process reg_in_p;


  -- Phi Generation
  phi_reg_p : process (clk,reset_n)
  begin
    if reset_n = '0' then
      phi_reg_i <= "0101";
      phi_reg_q <= "0101";
    elsif clk'event and clk ='1' then
      if fir_activate_tot = '1' then
        if init_fir = '1' then
          phi_reg_i <= "0101";
          phi_reg_q <= "0101";
        elsif four_count = "00"  and shift_samp = '1' then
          phi_reg_i <= phi_reg_i (phi_degree_g-3 downto 0) & reg_in_i_reg;
          phi_reg_q <= phi_reg_q (phi_degree_g-3 downto 0) & reg_in_q_reg;
        end if;
      end if;
    end if;
  end process;
  
  lut_add_i <= phi_reg_i & reg_in_i_reg;
  lut_add_q <= phi_reg_q & reg_in_q_reg;
  
  -----------------------------------------------------------------------------
  -- Look_up_table implemetation
  -----------------------------------------------------------------------------
  --                _____      
  --               |     | 24  
  -- lut_add   --->| LUT |--/--> lut_out
  --               |_____|       
  --
  -----------------------------------------------------------------------------

  fe11b_tx_lut_iq_1 : fe11b_tx_lut_iq
  port map 
  ( 
    lut_add      => lut_add_i,
    fir_coef_sel => fir_coef_sel,
    lut_out      => lut_out_i
  );
   
  fe11b_tx_lut_iq_2 : fe11b_tx_lut_iq
  port map 
  ( 
    lut_add      => lut_add_q,
    fir_coef_sel => fir_coef_sel,
    lut_out      => lut_out_q
  );
   
  -----------------------------------------------------------------------------
  -- reg output I/Q + counter
  -----------------------------------------------------------------------------
  --          counter 1,2,3,4
  --              | 
  --             \/      __________ 
  --             |\      |i_out_reg|   i_output
  --         /-->| |-/-- |q_out_reg|--> q_output
  --             |/      |_______/\|
  --
  -----------------------------------------------------------------------------
  output_i_q_p : process (clk,reset_n)
  begin
    if reset_n = '0' then
      i_out_reg          <= I_OUTPUT_INIT_CT;
      q_out_reg          <= Q_OUTPUT_INIT_CT; 
      four_count         <= "01"; 
    elsif clk'event and clk ='1' then

     if fir_activate_tot = '1' then
       if init_fir = '1' then
          i_out_reg      <= I_OUTPUT_INIT_CT;
          q_out_reg      <= Q_OUTPUT_INIT_CT;
          four_count     <= "01";
       else
         if shift_samp = '1' then 
          four_count <= four_count + '1'; -- new data only when needed           
         end if;
          case four_count is
            when "01" =>
              i_out_reg  <= lut_out_i (out_length_g-1 downto 0);
              q_out_reg  <= lut_out_q (out_length_g-1 downto 0);
       
            when "10" =>
              i_out_reg  <= lut_out_i ((2*out_length_g-1) downto out_length_g);
              q_out_reg  <= lut_out_q ((2*out_length_g-1) downto out_length_g);
              
            when "11" =>
              i_out_reg  <= lut_out_i ((3*out_length_g-1) downto 2*out_length_g);
              q_out_reg  <= lut_out_q ((3*out_length_g-1) downto 2*out_length_g);
           
            when others =>
              i_out_reg  <= lut_out_i ((4*out_length_g-1) downto 3*out_length_g);
              q_out_reg  <= lut_out_q ((4*out_length_g-1) downto 3*out_length_g);
          end case;
       end if;
       else
          four_count     <= "01";
          i_out_reg      <= I_OUTPUT_INIT_CT;
          q_out_reg      <= Q_OUTPUT_INIT_CT;         
       
      end if;
    end if;
  end process;

  -----------------------------------------------------------------------------
  -- Shift_Sample Generation
  -----------------------------------------------------------------------------
  -- The upsample_44to60 block needs very regular data :
  -- period :   0  1  2  3  4  5  6  7  8  9  10 11 12 13 14 0  1  ....
  -- data   :   d0 d1    d2 d3 d4    d5 d6 d7    d8 d9 dA    dB ....
  --
  -- In order to get this type of output data, the shift_counter generate a
  -- shift_sample signal which will increase the four_counter. When the
  -- four_counter reachs "00", then a new data is needed.
  shift_count_p: process (clk, reset_n)
  begin  -- process shift_count_p
    if reset_n = '0' then              
      shift_samp     <= '0';
      shift_samp_ff1 <= '0';
    elsif clk'event and clk = '1' then
      shift_samp_ff1 <= shift_samp;
      if init_fir = '1' or fir_activate_tot = '0' then
        shift_samp    <= '0';
        -- 13d has been chosen as it will let enough time to delay reg_in_i and
        -- to not miss an other data, no matter of the duration of the first
        -- incomings data (5 periods or 6 periods)
      elsif fir_activate_tot = '1' then
        -- Generate shift_sample signal
        shift_samp <= '1'; -- one 44MB/s data needed 

      end if;

    end if;
  end process shift_count_p;

--  shift_sample <= shift_samp_ff1;
--  addr_table   <= shift_counter;
    
  -----------------------------------------------------------------------------
  -- end_counter
  -----------------------------------------------------------------------------
  -- When the fir_activate => 0, the last phi has been sent.
  -- The fir actually stops when the last phi has finished to be sent throw the
  -- tx_filter and the upsample_44to60.
  end_count_p : process (clk,reset_n)
  begin
    if reset_n ='0' then 
      end_count      <= (others => '0');
      fir_activate_dly <= '0';
    elsif clk'event and clk ='1' then
      if init_fir = '1' then
        end_count <= (others => '0');
        fir_activate_dly <= '1';
      elsif fir_activate = '0' then
        if four_count = "01" then
          end_count <= end_count + '1';
          if end_count = phi_degree_g + 3 then
            fir_activate_dly <= '0';
          end if;
        end if;
      end if;
    end if;
  end process;  
          
  fir_activate_tot <= fir_activate_dly or fir_activate;
  
  -----------------------------------------------------------------------------
  -- outputs
  -----------------------------------------------------------------------------
  
  -- when test mode, I and Q are sent without being filtered
  with phi_angle select
          i_out_disb <=
          SIGNED_MAX_CT when "00",
          ZEROS2_CT     when "01",
          ZEROS2_CT     when "10",
          SIGNED_MIN_CT when others; -- 11
    
  with phi_angle select
          q_out_disb <=
          ZEROS2_CT     when "00",
          SIGNED_MAX_CT when "01",
          SIGNED_MIN_CT when "10",
          ZEROS2_CT     when others; -- 11
  
  data_i     <= not(i_out_reg(out_length_g-1))&i_out_reg(out_length_g-2 downto 0) when fir_disb ='0'  else i_out_disb;
  data_q     <= not(q_out_reg(out_length_g-1))&q_out_reg(out_length_g-2 downto 0) when fir_disb ='0'  else q_out_disb;
  data_valid <= shift_samp;

end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------
