--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandeburie $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 36271 $
--/ $Date: 2018-11-22 12:16:03 +0100 (Thu, 22 Nov 2018) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      : Modem 802_11b core
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/modem802_11b_core/vhdl/rtl/modem802_11b_core.vhd $
--/
--////////////////////////////////////////////////////////////////////////////


--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

--library modem802_11b_rtl;
library work;
--use modem802_11b_rtl.modem802_11b_pkg.all;
use work.modem802_11b_core_pkg.all;

--library modem_sm_b_rtl;
library work;

--library crc16_8_rtl;
library work;

--library tx_path_rtl;
library work;
--library rx_path_rtl;
library work;
--library rx_ctrl_rtl;
library work;

--library evm_calculator_rtl;
library work;

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity modem802_11b_core is
  port (
   -- clocks and reset
   clk                 : in  std_logic; -- main clock (not gated)
   rx_path_b_gclk      : in  std_logic; -- gated clock for RX path
   tx_path_b_gclk      : in  std_logic; -- gated clock for TX path
   reset_n             : in  std_logic; -- global reset async
   --
   rx_gating           : out std_logic; -- Gating condition for Rx path
   tx_gating           : out std_logic; -- Gating condition for Tx path
  
   --------------------------------------------
   -- Interface with RW_WLAN Bup
   --------------------------------------------
   -- inputs signals                                                           
   bup_txdata          : in  std_logic_vector(7 downto 0); -- data to send         
   phy_txstartend_req  : in  std_logic; -- request to start a packet transmission    
   phy_data_req        : in  std_logic; -- request to send a byte                  
   txv_length          : in  std_logic_vector(11 downto 0);  -- RX PSDU length     
   txv_service         : in  std_logic_vector(7 downto 0);  -- tx service field   
   txv_datarate        : in  std_logic_vector( 3 downto 0); -- PSDU transm. rate
   txv_immstop         : in std_logic;  -- request from Bup to stop tx.
    
   -- outputs signals                                                          
   phy_txstartend_conf : out std_logic; -- transmission started, ready for data  
   phy_rxstartend_ind  : out std_logic; -- indication of RX packet                     
   phy_data_conf       : out std_logic; -- last byte was read, ready for new one 
   phy_data_ind        : out std_logic; -- received byte ready                  
   rxv_length          : out std_logic_vector(11 downto 0);  -- RX PSDU length  
   rxv_service         : out std_logic_vector(7 downto 0);  -- rx service field
   rxv_datarate        : out std_logic_vector( 3 downto 0); -- PSDU rec. rate
   rxe_errorstat       : out std_logic_vector(1 downto 0);-- packet recep. stat
   phy_cca_ind         : out std_logic; -- CCA status                           
   bup_rxdata          : out std_logic_vector(7 downto 0); -- data received      
   
   --------------------------------------------
   -- Radio controller interface
   --------------------------------------------
   rf_txonoff_conf     : in  std_logic;  -- Radio controller in TX mode conf
   rf_rxonoff_conf     : in  std_logic;  -- Radio controller in RX mode conf
   --
   rf_txonoff_req      : out std_logic;  -- Radio controller in TX mode req
   rf_rxonoff_req      : out std_logic;  -- Radio controller in RX mode req
   
   --------------------------------------------
   -- AGC
   --------------------------------------------
   agcproc_end         : in std_logic;
   cca_busy            : in std_logic;
   correl_rst_n        : in std_logic;
   --
   psdu_duration       : out std_logic_vector(15 downto 0);
   correct_header      : out std_logic;
   plcp_state          : out std_logic;
   plcp_error          : out std_logic;
   listen_start_o      : out std_logic; -- high when start to listen
   rx_end_for_timing   : out std_logic; -- indication of end of frame
   --
   sfd_found           : out std_logic;
   symbol_sync2        : out std_logic;
   --------------------------------------------
   -- Data Inputs
   --------------------------------------------
   -- data from gain compensation (inside rx_b_frontend)
   rf_rxi              : in  std_logic_vector(7 downto 0);
   rf_rxq              : in  std_logic_vector(7 downto 0);
   
   --------------------------------------------
   -- Modemb Registers Outputs
   --------------------------------------------
   -- MDMbSTAT0 register
   reg_eqsumq          : out std_logic_vector(7 downto 0);
   reg_eqsumi          : out std_logic_vector(7 downto 0);
   reg_dcoffsetq       : out std_logic_vector(5 downto 0);
   reg_dcoffseti       : out std_logic_vector(5 downto 0);

   -- MDMbSTAT1 register
   reg_iqgainestim     : out std_logic_vector(6 downto 0);
   reg_freqoffestim    : out std_logic_vector(7 downto 0);
   reg_evm             : out std_logic_vector(12 downto 0);

   --------------------------------------------
   -- Modem Register Inputs
   --------------------------------------------
   -- MDMbCNTL register
   reg_gaindisb        : in std_logic; -- '0' to enable Gain compensation.
   reg_interpdisb      : in std_logic; -- '0' to enable Interpolation.
   reg_iqmmdisb        : in std_logic; -- '0' to enable I/Q mismatch compensation.
   reg_precompdisb     : in std_logic; -- '0' to enable timing offset compensation
   reg_dcoffdisb       : in std_logic; -- '0' to enable the DC offset compensation
   reg_compdisb        : in std_logic; -- '0' to enable the compensation.
   reg_eqdisb          : in std_logic; -- '0' to enable the Equalizer.
   reg_spreaddisb      : in std_logic; -- '0' to enable spreading.                       
   reg_scrambdisb      : in std_logic; -- '0' to enable scrambling.
   reg_sfderr          : in std_logic_vector( 2 downto 0); -- Error number for SFD

   -- Number of preamble bits to be considered in short SFD comparison
   reg_sfdlen          : in std_logic_vector( 2 downto 0);
   reg_prepre          : in std_logic_vector( 5 downto 0); -- pre-preamble count.

   -- MDMbPRMINIT register
   -- Values for phase correction parameters.
   reg_rho             : in std_logic_vector( 1 downto 0);
   reg_mu              : in std_logic_vector( 1 downto 0);
   -- Values for phase feedforward equalizer parameters.
   reg_beta            : in std_logic_vector( 1 downto 0);
   reg_alpha           : in std_logic_vector( 1 downto 0);

   -- MDMbTALPHA register
   -- TALPHA time interval value for equalizer alpha parameter.
   reg_talpha3         : in std_logic_vector( 3 downto 0);
   reg_talpha2         : in std_logic_vector( 3 downto 0);
   reg_talpha1         : in std_logic_vector( 3 downto 0);
   reg_talpha0         : in std_logic_vector( 3 downto 0);

   -- MDMbTBETA register
   -- TBETA time interval value for equalizer beta parameter.
   reg_tbeta3          : in std_logic_vector( 3 downto 0);
   reg_tbeta2          : in std_logic_vector( 3 downto 0);
   reg_tbeta1          : in std_logic_vector( 3 downto 0);
   reg_tbeta0          : in std_logic_vector( 3 downto 0);

   -- MDMbTMU register
   -- TMU time interval value for phase correction and offset comp. mu param
   reg_tmu3            : in std_logic_vector( 3 downto 0);
   reg_tmu2            : in std_logic_vector( 3 downto 0);
   reg_tmu1            : in std_logic_vector( 3 downto 0);
   reg_tmu0            : in std_logic_vector( 3 downto 0);

   -- MDMbCNTL1 register
   reg_rxlenchken      : in std_logic;
   reg_rxmaxlength     : in std_logic_vector(11 downto 0);

   -- MDMbRFCNTL register
   -- Delay of the Tx front-end inside the RW_WLAN RF, in number of 44 MHz cycles.
   reg_txenddel        : in std_logic_vector(7 downto 0);

   -- MDMbEQCNTL register
   -- Delay to stop the equalizer adaptation after the last param update, in s
   reg_eqhold          : in std_logic_vector(11 downto 0);
   -- Delay to start the compensation after the start of the estimation, in s.
   reg_comptime        : in std_logic_vector( 4 downto 0);
   -- Delay to start the estimation after the enabling of the equalizer, in s.
   reg_esttime         : in std_logic_vector( 4 downto 0);
   -- Delay to switch on the equalizer after the fine gain setting, in s.
   reg_eqtime          : in std_logic_vector( 3 downto 0);

   -- MDMbCNTL2 register
   reg_precomp         : in std_logic_vector( 5 downto 0); -- in us.
   reg_synctime        : in std_logic_vector( 5 downto 0);
   reg_looptime        : in std_logic_vector( 3 downto 0);

   --------------------------------------------
   -- Tx FIR controls
   --------------------------------------------
   init_fir            : out std_logic;
   fir_activate        : out std_logic;
   fir_phi_out_tog_o   : out std_logic;
   fir_phi_out         : out std_logic_vector (1 downto 0);
   
   --------------------------------------------
   -- Interface with RX Frontend
   --------------------------------------------
   -- Controls
   clock_lock          : out std_logic;
   gain_enable         : out std_logic;  -- gain compensation control.
   tau_est             : out std_logic_vector(17 downto 0);
   enable_error        : out std_logic;

   --------------------------------------------
   -- Error
   --------------------------------------------
   symbol_sync_error_o : out std_logic;
   error_cart_i_o      : out std_logic_vector(13 downto 0);  -- real part.
   error_cart_q_o      : out std_logic_vector(13 downto 0);  -- Imaginary part    

   --------------------------------------------
   -- Diagnostic port
   --------------------------------------------
   modem_diag          : out std_logic_vector(15 downto 0);
   modem_diag0         : out std_logic_vector(15 downto 0);
   modem_diag1         : out std_logic_vector(15 downto 0);
   modem_diag2         : out std_logic_vector(15 downto 0)
  );

end modem802_11b_core;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture RTL of modem802_11b_core is

  ------------------------------------------------------------------------------
  -- Types
  ------------------------------------------------------------------------------

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  -- Registers signals.
  signal scrambling_disb     : std_logic;  -- disable the scrambling
  signal spread_disb         : std_logic;  -- disable the spreading
  signal eq_disb             : std_logic;  -- disable the equalizer
  signal interpdisb          : std_logic;  -- disable interpolator
  signal comp_disb           : std_logic;  -- disable the compensation (error calc)
  signal iqmmdisb            : std_logic;  -- disable I/Q mismatch compensation.
  signal gaindisb            : std_logic;  -- disable the gain compensation.
  signal sfd_detect_enable   : std_logic;  -- enable SFD detection when high
  signal iq_gain_sat_stat    : std_logic_vector(6 downto 0);
  signal dc_offset_i_stat    : std_logic_vector(5 downto 0);
  signal dc_offset_q_stat    : std_logic_vector(5 downto 0);
  signal coeff_sum_i_stat    : std_logic_vector(7 downto 0);
  signal coeff_sum_q_stat    : std_logic_vector(7 downto 0);
  signal freqoffestim_stat   : std_logic_vector(7 downto 0);
  signal evm_stat            : std_logic_vector(12 downto 0);
  signal precompdisb         : std_logic;  -- disable timing offset compensation.
  signal dcoffdisb           : std_logic;  -- disable the DC offset compensation.
  signal preamble_type_tx    : std_logic;
  signal preamble_type_rx    : std_logic;
  signal seria_data_conf     : std_logic;  -- Serializer is ready for new data
  signal scr_data_in         : std_logic_vector(7 downto 0);  -- data sent to scr
  signal rx_data             : std_logic_vector(7 downto 0);  -- data sent to scr
  signal sm_data_req         : std_logic;  -- State machines data request
  signal tx_psk_mode         : std_logic;  -- 0 = BPSK; 1 = QPSK
  signal rx_psk_mode         : std_logic;  -- 0 = BPSK; 1 = QPSK
  signal tx_activated        : std_logic;  -- indicate when tx_path has finished
  signal sfd_found_int       : std_logic;  -- high when SFD is found on RX
  signal byte_ind            : std_logic;  -- pulse when an RX byte is available
                                           -- from RX path
  signal activate_seria      : std_logic;  -- activate Serializer
  signal shift_period        : std_logic_vector(3 downto 0);  -- Serializer speed
  signal activate_cck        : std_logic;  -- activate CCK modulator
  signal rx_cck_rate         : std_logic;  -- CCK speed (0 = 5.5 Mb/s; 1 = 11 Mb/s)
  signal tx_cck_rate         : std_logic;  -- CCK speed (0 = 5.5 Mb/s; 1 = 11 Mb/s)
  signal crc_out_1st         : std_logic_vector( 7 downto 0);  -- CRC 1st data 
  signal crc_out_2nd         : std_logic_vector( 7 downto 0);  -- CRC 2nd data
  signal crc_init            : std_logic;  -- init CRC computation
  signal crc_data_valid      : std_logic;  -- compute CRC on packet header
  signal data_to_crc         : std_logic_vector(7 downto 0);  -- byte data to CRC
  -- Values applied for phase correction parameters.
  signal applied_mu          : std_logic_vector( 2 downto 0);
  -- Values applied for phase feedforward equalizer parameters.
  signal applied_beta        : std_logic_vector( 2 downto 0);
  signal applied_alpha       : std_logic_vector( 2 downto 0);
  signal alpha_accu_disb     : std_logic;
  signal beta_accu_disb      : std_logic;

  signal mod_type : std_logic;

  signal rx_idle_state   : std_logic;   -- high when rx sm is idle

  signal remod_data     : std_logic_vector(1 downto 0);  -- Data from the TX path
  signal remod_enable   : std_logic;  -- High when the remodulation is enabled
  signal remod_data_req : std_logic;    -- request to send a byte 
  signal remod_type     : std_logic;    -- CCK : 0 ; PBCC : 1
  signal remod_bq       : std_logic;    -- BPSK = 0 - QPSK = 1 
  signal demod_data     : std_logic_vector(7 downto 0);  -- Data to the TX path

  -- signals for iq_mismatch
  signal iq_estim_enable : std_logic;   -- enable the I/Q Mismatch estim when 1
  signal iq_comp_enable  : std_logic;   -- enable the I/Q Mismatch comp when 1

  signal equalizer_activate   : std_logic;
  signal equalizer_init_n     : std_logic;
  signal equalizer_disb       : std_logic;
  signal synctime_enable      : std_logic;
  signal decode_path_activate : std_logic;
  signal diff_decod_first_val : std_logic;
  signal precomp_enable       : std_logic;
  signal enable_error_int     : std_logic;
  signal rec_mode             : std_logic_vector(1 downto 0);

  signal symbol_sync       : std_logic;
  signal symbol_sync_ff1   : std_logic;
  signal symbol_sync_ff2   : std_logic;
  signal symbol_sync_ff3   : std_logic;
  signal symbol_sync_ff4   : std_logic;
  signal symbol_sync_ff5   : std_logic;
  signal symbol_sync_ff6   : std_logic;
  signal symbol_sync_ff7   : std_logic;
  signal symbol_sync_ff8   : std_logic;
  signal symbol_sync_ff9   : std_logic;
  signal symbol_sync_ff10  : std_logic;
  signal symbol_sync_ff11  : std_logic;
  signal symbol_sync_ff12  : std_logic;
  signal symbol_sync_ff13  : std_logic;
  signal symbol_sync_ff14  : std_logic;

  signal rxv_service_o : std_logic_vector(7 downto 0);

  signal rf_txonoff_req_o           : std_logic;
  signal phy_txstartend_conf_o      : std_logic;  -- transmission started, ready for data
  signal phy_rxstartend_ind_o       : std_logic;  -- end of the reception
  -- EVM calculator
  signal phase_error                : std_logic_vector(12 downto 0);
  signal phase_error_ready          : std_logic; -- Symbol sync pulse
  -- DIAG PORTS.
  signal rx_state_diag              : std_logic_vector(2 downto 0);  -- diag from sm
  signal diag_error_i               : std_logic_vector(7 downto 0);  -- diag from equalizer
  signal diag_error_q               : std_logic_vector(7 downto 0);  -- diag from equalizer
  signal rxe_errorstat_int          : std_logic_vector(1 downto 0);
  signal phy_cca_ind_int            : std_logic;
  signal phy_data_ind_int           : std_logic;
  -- Resynchronization signals.
  signal cca_busy_ff1_resync        : std_logic;
  signal cca_busy_ff2_resync        : std_logic;
  signal agcproc_end_ff1_resync     : std_logic;
  signal agcproc_end_ff2_resync     : std_logic;
  signal correl_rst_n_ff1_resync    : std_logic;
  signal correl_rst_n_ff2_resync    : std_logic;
  signal rf_txonoff_conf_ff1_resync : std_logic;  -- Radio controller in TX mode conf
  signal rf_txonoff_conf_ff2_resync : std_logic;  -- Radio controller in TX mode conf
  signal rf_rxonoff_conf_ff1_resync : std_logic;  -- Radio controller in RX mode conf
  signal rf_rxonoff_conf_ff2_resync : std_logic;  -- Radio controller in RX mode conf
  
--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin
  
  -----------------------------------------------------------------------------
  -- Output linking
  -----------------------------------------------------------------------------
  clock_lock  <= rxv_service_o(2);
  rxv_service <= rxv_service_o;
    
  rf_txonoff_req      <= rf_txonoff_req_o;
  phy_txstartend_conf <= phy_txstartend_conf_o;
  phy_rxstartend_ind  <= phy_rxstartend_ind_o;
  
  phy_data_conf       <= seria_data_conf;
  
  sfd_found           <= sfd_found_int;

  rxe_errorstat       <= rxe_errorstat_int;

  phy_cca_ind         <= phy_cca_ind_int;
  
  phy_data_ind        <= phy_data_ind_int;

  --Status registers
  reg_eqsumq          <= coeff_sum_q_stat;
  reg_eqsumi          <= coeff_sum_i_stat;
  reg_dcoffsetq       <= dc_offset_q_stat;
  reg_dcoffseti       <= dc_offset_i_stat;
                
  reg_iqgainestim     <= iq_gain_sat_stat;
  reg_freqoffestim    <= freqoffestim_stat;
  reg_evm             <= evm_stat; 



  -----------------------------------------------------------------------------
  -- Resynchronisation.
  -----------------------------------------------------------------------------
  mdm_resync_p : process (clk, reset_n)
  begin
    if reset_n = '0' then
      cca_busy_ff1_resync     <= '0';
      cca_busy_ff2_resync     <= '0';
      agcproc_end_ff1_resync  <= '0';
      agcproc_end_ff2_resync  <= '0';
      correl_rst_n_ff1_resync <= '1';
      correl_rst_n_ff2_resync <= '1';
    elsif clk'event and clk = '1' then
      cca_busy_ff1_resync     <= cca_busy;
      cca_busy_ff2_resync     <= cca_busy_ff1_resync;
      agcproc_end_ff1_resync  <= agcproc_end;
      agcproc_end_ff2_resync  <= agcproc_end_ff1_resync;
      correl_rst_n_ff1_resync <= correl_rst_n;
      correl_rst_n_ff2_resync <= correl_rst_n_ff1_resync;
    end if;
  end process mdm_resync_p;
  
  -----------------------------------------------------------------------------
  -- Gating condition
  -----------------------------------------------------------------------------
  modemb_gating_p : process(clk, reset_n)
    variable nb_clock_cnt : std_logic_vector(1 downto 0);
  begin
    if reset_n = '0' then
      rx_gating <= '1';
      tx_gating <= '1';
      nb_clock_cnt := "00";
    elsif clk'event and clk = '1' then
      -- Gating condition for Rx path
      if (cca_busy_ff2_resync = '1' or phy_rxstartend_ind_o = '1') then
          rx_gating <= '0';
          nb_clock_cnt := "01";
      else
        if nb_clock_cnt = "00" then
          rx_gating <= '1';
        elsif nb_clock_cnt = "11" then
          rx_gating <= '1';
        else
          nb_clock_cnt := nb_clock_cnt + '1';          
        end if;
        
      end if;

      -- Gating condition for Tx path
      if (txv_datarate(3) = '0' and
         (phy_txstartend_conf_o = '1' or phy_txstartend_req  = '1')) or
         (cca_busy_ff2_resync = '1' or phy_rxstartend_ind_o = '1') then
        tx_gating <= '0';
      else
        tx_gating <= '1';
      end if;

    end if;
  end process modemb_gating_p;


  -- One micro second counter                  
  one_us_p : process (rx_path_b_gclk, reset_n)
  begin
    if reset_n = '0' then
      symbol_sync_ff1       <= '0';
      symbol_sync_ff2       <= '0';
      symbol_sync_ff3       <= '0';
      symbol_sync_ff4       <= '0';
      symbol_sync_ff5       <= '0';
      symbol_sync_ff6       <= '0';
      symbol_sync_ff7       <= '0';
      symbol_sync_ff8       <= '0';
      symbol_sync_ff9       <= '0';
      symbol_sync_ff10      <= '0';
      symbol_sync_ff11      <= '0';
      symbol_sync_ff12      <= '0';
      symbol_sync_ff13      <= '0';
      symbol_sync_ff14      <= '0';
    elsif rx_path_b_gclk'event and rx_path_b_gclk = '1' then
      symbol_sync_ff1  <= symbol_sync;
      symbol_sync_ff2  <= symbol_sync_ff1;
      symbol_sync_ff3  <= symbol_sync_ff2;
      symbol_sync_ff4  <= symbol_sync_ff3;
      symbol_sync_ff5  <= symbol_sync_ff4;
      symbol_sync_ff6  <= symbol_sync_ff5;
      symbol_sync_ff7  <= symbol_sync_ff6;
      symbol_sync_ff8  <= symbol_sync_ff7;
      symbol_sync_ff9  <= symbol_sync_ff8;
      symbol_sync_ff10 <= symbol_sync_ff9;
      symbol_sync_ff11 <= symbol_sync_ff10;
      symbol_sync_ff12 <= symbol_sync_ff11;
      symbol_sync_ff13 <= symbol_sync_ff12;
      symbol_sync_ff14 <= symbol_sync_ff13;
    end if;
  end process one_us_p;
  
  symbol_sync2 <= symbol_sync_ff14; --transport symbol_sync after 340 ns;
  
  ------------------------------------------------------------------------------
  -- Tx Path Core Port Map
  ------------------------------------------------------------------------------
  tx_path_core_1 : tx_path_core
  generic map(
    dec_freq_g               => 4
          )
  port map (
    --------------------------------------
    -- Clocks & Reset
    -------------------------------------- 
    clk                   => tx_path_b_gclk,
    reset_n               => reset_n,
    --------------------------------------------
    -- Interface with Modem State Machines
    --------------------------------------------
    low_r_flow_activate   => activate_seria,
    psk_mode              => tx_psk_mode,
    shift_period          => shift_period,
    cck_flow_activate     => activate_cck,
    cck_speed             => tx_cck_rate,
    tx_activated          => tx_activated,
    --------------------------------------------
    -- Interface with RW_WLAN Bup - via or not Modem State Machines
    --------------------------------------------
    scrambling_disb       => scrambling_disb,
    spread_disb           => spread_disb,
    bup_txdata            => scr_data_in,
    phy_data_req          => sm_data_req,
    txv_prtype            => preamble_type_tx,
    txv_immstop           => txv_immstop,
    phy_data_conf         => seria_data_conf,
    --------------------------------------------
    -- Interface for remodulation
    --------------------------------------------
    remod_data            => remod_data, 
    --                          
    remod_enable          => remod_enable,
    remod_data_req        => remod_data_req,
    remod_type            => remod_type,    
    remod_bq              => remod_bq,    
    demod_data            => demod_data,
    --------------------------------------------
    -- FIR controls
    --------------------------------------------
    init_fir              => init_fir,
    fir_activate          => fir_activate,
    fir_phi_out_tog_o     => fir_phi_out_tog_o,
    fir_phi_out           => fir_phi_out
    );

  
  ------------------------------------------------------------------------------
  -- RX Path Core Port map
  ------------------------------------------------------------------------------
  rx_path_core_1 : rx_path_core
    generic map (
      data_length_g       => 9, -- data size
      angle_length_g      => 15 -- agle size for phase compensation cordic
    )
    port map (
      --------------------------------------------
      -- clock and reset
      --------------------------------------------
      reset_n             => reset_n,            -- Global reset.
      rx_path_b_gclk      => rx_path_b_gclk,    -- Gated Clock for RX Path.
      
      --------------------------------------------
      -- Data In
      --------------------------------------------
      data_in_i           => rf_rxi,
      data_in_q           => rf_rxq,
      
      --------------------------------------------
      -- Control inputs
      --------------------------------------------
      -- Control for gain compensation.
      dcoffdisb           => dcoffdisb,          -- disable dc_offset compensation when high.
      
      -- Control for IQ Mismatch Compensation
      iq_estimation_enable   => iq_estim_enable, -- enable the I/Q estimation when high.
      iq_compensation_enable => iq_comp_enable,  -- enable the I/Q compensation when high
      -- Control for equalization
      equ_activate        => equalizer_activate, -- enable the equalizer when high.
      equalizer_disb      => equalizer_disb,     -- disable the equalizer filter when high.
      equalizer_init_n    => equalizer_init_n,   -- equalizer filter coeffs set to 0 when low.
      alpha_accu_disb     => alpha_accu_disb,    -- stop coeff accu when high.
      beta_accu_disb      => beta_accu_disb,     -- stop dc accu when high.
      alpha               => applied_alpha,      -- alpha parameter value.
      beta                => applied_beta,       -- beta parameter value.
      -- Control for DSSS / CCK demodulation
      interp_disb         => interpdisb,         -- disable the interpolation when high 
      rx_enable           => cca_busy_ff2_resync,-- enable rx path when high 
      mod_type            => mod_type,           -- '0' for DSSS, '1' for CCK.
      enable_error        => enable_error_int,       -- Enable Error Calculation
      precomp_enable      => precomp_enable,     -- Reload the omega accumulator
      demod_rate          => rx_psk_mode,        -- '0' for BPSK, '1' for QPSK
      cck_rate            => rx_cck_rate,
      rho                 => reg_rho,            -- rho parameter value.
      mu                  => applied_mu,         -- mu parameter value.
      --
      tau_est             => tau_est,           
      -- Control for Decode Path
      scrambling_disb     => scrambling_disb,     -- scrambling disable
      decode_path_activate=> decode_path_activate,-- enable the diff. decoder
      diff_decod_first_val=> diff_decod_first_val,-- initialize the diff. decoder
      sfd_detect_enable   => sfd_detect_enable,
      sfderr              => reg_sfderr,
      sfdlen              => reg_sfdlen,
      rec_mode            => rec_mode,
           
      --------------------------------------------
      -- Remodulation interface
      --------------------------------------------
      remod_data          => remod_data,      -- Data from the TX path
      --
      remod_enable        => remod_enable,    -- High when the remodulation is enabled
      remod_data_req      => remod_data_req,  -- request to send a byte 
      remod_type          => remod_type,      -- CCK : 0 ; PBCC : 1
      remod_bq            => remod_bq,        -- BPSK = 0 - QPSK = 1 
      demod_data          => demod_data,      -- Data to the TX path

      --------------------------------------------
      -- AGC-CCA interface
      --------------------------------------------
      correl_rst_n        => correl_rst_n_ff2_resync, -- reset the Barker correlator when low
      synchro_en          => synctime_enable, -- enable the synchronisation when high
      --
      symbol_synchro      => symbol_sync,     -- pulse at the beginning of a symbol.
      
      --------------------------------------------
      -- Modem B state machines interface
      --------------------------------------------
      sfd_found           => sfd_found_int,   -- sfd found when high
      preamble_type       => preamble_type_rx,-- Type of preamble 
      phy_data_ind        => byte_ind,        -- pulse when an RX byte is available.
      data_to_bup         => rx_data,         -- Received data.
      --------------------------------------------
      -- Status registers.
      --------------------------------------------
      iq_gain_sat_stat    => iq_gain_sat_stat,
      dc_offset_i_stat    => dc_offset_i_stat,
      dc_offset_q_stat    => dc_offset_q_stat,  
      coeff_sum_i_stat    => coeff_sum_i_stat,
      coeff_sum_q_stat    => coeff_sum_q_stat,
      freqoffestim_stat   => freqoffestim_stat,
      --------------------------------------------
      -- Error
      --------------------------------------------
      phase_error_o       => phase_error,
      phase_error_ready_o => phase_error_ready,
      --
      symbol_sync_error_o => symbol_sync_error_o,
      error_cart_i_o      => error_cart_i_o,
      error_cart_q_o      => error_cart_q_o,
      -------------------------------
      -- Diag ports
      -------------------------------
      diag_error_i        => diag_error_i,
      diag_error_q        => diag_error_q
      );

  --------------------------------------------
  -- RX path control
  --------------------------------------------
  rx_ctrl_1 : rx_ctrl
  port map (
    --------------------------------------
    -- Clocks & Reset
    -------------------------------------- 
    hresetn             => reset_n,
    hclk                => clk,

    --------------------------------------------
    -- Registers interface
    --------------------------------------------
    eq_disb             => eq_disb,
    precomp             => reg_precomp,
    eqtime              => reg_eqtime,
    eqhold              => reg_eqhold,
    looptime            => reg_looptime,
    synctime            => reg_synctime,
    alpha               => reg_alpha,
    beta                => reg_beta,
    mu                  => reg_mu,
    talpha3             => reg_talpha3,
    talpha2             => reg_talpha2,
    talpha1             => reg_talpha1,
    talpha0             => reg_talpha0,
    tbeta3              => reg_tbeta3,
    tbeta2              => reg_tbeta2,
    tbeta1              => reg_tbeta1,
    tbeta0              => reg_tbeta0,
    tmu3                => reg_tmu3,
    tmu2                => reg_tmu2,
    tmu1                => reg_tmu1,
    tmu0                => reg_tmu0,

    --------------------------------------------
    -- Input control
    --------------------------------------------
    energy_detect       => cca_busy_ff2_resync,
    agcproc_end         => agcproc_end_ff2_resync,
    rx_psk_mode         => rx_psk_mode,
    rx_idle_state       => rx_idle_state,
    precomp_disb        => precompdisb,
    comp_disb           => comp_disb,
    iqmm_disb           => iqmmdisb,
    gain_disb           => gaindisb,
    --------------------------------------------
    -- RX path control signals
    --------------------------------------------
    equalizer_activate  => equalizer_activate,
    equalizer_init_n    => equalizer_init_n,
    equalizer_disb      => equalizer_disb,
    precomp_enable      => precomp_enable,
    synctime_enable     => synctime_enable,
    phase_estim_enable  => enable_error_int,
    iq_comp_enable      => iq_comp_enable,
    iq_estim_enable     => iq_estim_enable,
    gain_enable         => gain_enable,
    sfd_detect_enable   => sfd_detect_enable,
    applied_alpha       => applied_alpha,
    applied_beta        => applied_beta,
    alpha_accu_disb     => alpha_accu_disb,
    beta_accu_disb      => beta_accu_disb,
    applied_mu          => applied_mu
    );

  enable_error <= enable_error_int;  

  ------------------------------------------------------------------------------
  -- Modem State Machines 802.11b Port Map
  ------------------------------------------------------------------------------
  -- Resync of radio-controller signals.
  txonoff_conf_resync_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      rf_txonoff_conf_ff1_resync <= '0';
      rf_txonoff_conf_ff2_resync <= '0';
      rf_rxonoff_conf_ff1_resync <= '0';
      rf_rxonoff_conf_ff2_resync <= '0';
    elsif clk'event and clk = '1' then
      rf_txonoff_conf_ff1_resync <= rf_txonoff_conf;
      rf_txonoff_conf_ff2_resync <= rf_txonoff_conf_ff1_resync;
      rf_rxonoff_conf_ff1_resync <= rf_rxonoff_conf; 
      rf_rxonoff_conf_ff2_resync <= rf_rxonoff_conf_ff1_resync;      
    end if;
  end process txonoff_conf_resync_p;   

  
  modem_sm_b_1 : modem_sm_b
    port map (
    --------------------------------------
    -- Clocks & Reset
    -------------------------------------- 
    hresetn              => reset_n,
    hclk                 => clk,
    --------------------------------------
    -- TX path block
    -------------------------------------- 
    seria_data_conf      => seria_data_conf,
    tx_activated         => tx_activated,
    --                      
    scr_data_in          => scr_data_in,
    sm_data_req          => sm_data_req,
    tx_psk_mode          => tx_psk_mode,
    activate_seria       => activate_seria,
    shift_period         => shift_period,
    activate_cck         => activate_cck,
    tx_cck_rate          => tx_cck_rate,
    preamble_type_tx     => preamble_type_tx,
    --------------------------------------
    -- RX path block
    -------------------------------------- 
    cca_busy             => cca_busy_ff2_resync,
    preamble_type_rx     => preamble_type_rx,
    sfd_found            => sfd_found_int,
    byte_ind             => byte_ind,
    symbol_sync          => symbol_sync_ff14,
    rx_data              => rx_data,
    --  
    decode_path_activate => decode_path_activate,
    diff_decod_first_val => diff_decod_first_val,
    rec_mode             => rec_mode,
    mod_type             => mod_type,                   
    rx_psk_mode          => rx_psk_mode,
    rx_cck_rate          => rx_cck_rate,
    rx_idle_state        => rx_idle_state,
    rx_plcp_state        => plcp_state,
    --------------------------------------------
    -- Registers
    --------------------------------------------
    reg_prepre           => reg_prepre,
    txenddel_reg         => reg_txenddel,
    rxlenchken           => reg_rxlenchken,
    rxmaxlength          => reg_rxmaxlength,
    --------------------------------------------
    -- CCA
    --------------------------------------------
    psdu_duration        => psdu_duration,
    correct_header       => correct_header,
    plcp_error           => plcp_error,
    listen_start_o       => listen_start_o,
    rx_end_for_timing    => rx_end_for_timing,
    --------------------------------------
    -- CRC
    -------------------------------------- 
    crc_data_1st         => crc_out_1st,
    crc_data_2nd         => crc_out_2nd,
    --                      
    crc_init             => crc_init,
    crc_data_valid       => crc_data_valid,
    data_to_crc          => data_to_crc,
    --------------------------------------------
    -- Radio controller interface
    --------------------------------------------
    rf_txonoff_req       => rf_txonoff_req_o,
    rf_txonoff_conf      => rf_txonoff_conf_ff2_resync,
    rf_rxonoff_req       => rf_rxonoff_req,
    rf_rxonoff_conf      => rf_rxonoff_conf_ff2_resync,
    --------------------------------------
    -- BuP
    -------------------------------------- 
    -- TX
    phy_txstartend_req   => phy_txstartend_req,
    txv_service          => txv_service,
    phy_data_req         => phy_data_req,
    txv_datarate         => txv_datarate,
    txv_length           => txv_length,
    bup_txdata           => bup_txdata,
    phy_txstartend_conf  => phy_txstartend_conf_o,
    txv_immstop          => txv_immstop,
    -- RX
    phy_cca_ind          => phy_cca_ind_int,
    phy_rxstartend_ind   => phy_rxstartend_ind_o,
    rxv_service          => rxv_service_o,
    phy_data_ind         => phy_data_ind_int,
    rxv_datarate         => rxv_datarate,
    rxv_length           => rxv_length,
    rxe_errorstat        => rxe_errorstat_int,
    bup_rxdata           => bup_rxdata, 
    --------------------------------------
    -- Tx sfd active
    --------------------------------------
    tx_sfd              => open,
    --------------------------------------
    -- Diag
    --------------------------------------
    rx_state_diag        => rx_state_diag
    );

  
  ------------------------------------------------------------------------------
  -- Registers inteface Port Map
  ------------------------------------------------------------------------------
  modemb_registers_if_1 : modemb_registers_if
  port map (
    -- Clocks & Reset
    reset_n               => reset_n,
    hclk                  => clk,
    -- Controls
    -- Registers inputs
    reg_gaindisb          => reg_gaindisb,
    reg_interpdisb        => reg_interpdisb,
    reg_iqmmdisb          => reg_iqmmdisb,
    reg_precompdisb       => reg_precompdisb,
    reg_dcoffdisb         => reg_dcoffdisb,
    reg_compdisb          => reg_compdisb,
    reg_eqdisb            => reg_eqdisb,
    reg_spreaddisb        => reg_spreaddisb,
    reg_scrambdisb        => reg_scrambdisb,
    -- Registers outputs =>
    reg_gaindisb_sync     => gaindisb,
    reg_interpdisb_sync   => interpdisb,
    reg_iqmmdisb_sync     => iqmmdisb,
    reg_precompdisb_sync  => precompdisb,
    reg_dcoffdisb_sync    => dcoffdisb,
    reg_compdisb_sync     => comp_disb,
    reg_eqdisb_sync       => eq_disb,
    reg_spreaddisb_sync   => spread_disb,
    reg_scrambdisb_sync   => scrambling_disb
  );
  
  
  ------------------------------------------------------------------------------
  -- CRC16_8 Port Map
  ------------------------------------------------------------------------------
  crc16_8_1 : crc16_8
  port map (
    clk                 => tx_path_b_gclk,
    resetn              => reset_n,
    data_in             => data_to_crc,
    ld_init             => crc_init,
    calc                => crc_data_valid,
    crc_out_1st         => crc_out_1st,
    crc_out_2nd         => crc_out_2nd
    );

  ------------------------------------------------------------------------------
  -- EVM calculator Port Map
  ------------------------------------------------------------------------------
  evm_calculator_1 : evm_calculator
  port map (
    reset_n      => reset_n,
    clk          => rx_path_b_gclk,
    symbol_sync  => phase_error_ready,
    sfd_found    => sfd_found_int,
    phase_error  => phase_error,
    evm          => evm_stat
    );

  --------------------------------------------
  -- Diagnostic port assignment
  --------------------------------------------
  modem_diag(0)           <= sfd_found_int;
  modem_diag(1)           <= symbol_sync;
  modem_diag(15 downto 2) <= "00000000000000";
  -- Input of modem b.
  modem_diag0(7 downto 0)  <= rf_rxi;
  modem_diag0(15 downto 8) <= rf_rxq;
  -- Control/error signals of modem b.
  modem_diag1(0)  <= '0';
  modem_diag1(1)  <= sfd_found_int;
  modem_diag1(2)  <= symbol_sync;
  modem_diag1(3)  <= cca_busy_ff2_resync;
  modem_diag1(4)  <= agcproc_end_ff2_resync;
  modem_diag1(5)  <= preamble_type_rx;
  modem_diag1(6)  <= rxe_errorstat_int(0);
  modem_diag1(7)  <= rxe_errorstat_int(1);
  modem_diag1(8)  <= rx_state_diag(0);
  modem_diag1(9)  <= rx_state_diag(1);
  modem_diag1(10) <= rx_state_diag(2);
  modem_diag1(11) <= phy_rxstartend_ind_o;
  modem_diag1(12) <= phy_cca_ind_int;
  modem_diag1(13) <= phy_data_ind_int;
  modem_diag1(14) <= phy_txstartend_req;
  modem_diag1(15) <= '0';
  -- Equalizer error signals.
  modem_diag2(7 downto 0)  <= diag_error_i;
  modem_diag2(15 downto 8) <= diag_error_q;

end rtl;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

