--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandebu $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 4387 $
--/ $Date: 2012-09-21 15:41:09 +0200 (Fri, 21 Sep 2012) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      : Package for modem802_11b.
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/modem802_11b_core/vhdl/rtl/modem802_11b_core_pkg.vhd $
--/
--////////////////////////////////////////////////////////////////////////////



--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library IEEE; 
    use IEEE.STD_LOGIC_1164.ALL; 


--------------------------------------------------------------------------------
-- Package
--------------------------------------------------------------------------------
package modem802_11b_core_pkg is


--------------------------------------------------------------------------------
-- Components list declaration done by <fb> script.
--------------------------------------------------------------------------------

----------------------
-- File: /home/cvandebu/PROJECTS/RW_WLAN_IP_LIB/IPs/RW_WLAN/MODEM802_11b/rx_path/vhdl/rtl/rx_path_core.vhd
----------------------
  component rx_path_core

  generic (
    -- number of bits for the complex data :
    data_length_g  : integer := 9;
    -- number of bits for the input angle z_in :
    angle_length_g : integer := 15
    );
  port (
    --------------------------------------------
    -- clocks and reset.
    --------------------------------------------
    reset_n                : in  std_logic;  -- Global reset.
    rx_path_b_gclk         : in  std_logic;  -- Gated Clock for RX Path (44 Mhz).

    --------------------------------------------
    -- Data In. (from gain compensation)
    --------------------------------------------
    data_in_i              : in  std_logic_vector(7 downto 0);
    data_in_q              : in  std_logic_vector(7 downto 0);

    --------------------------------------------
    -- Control for dc_offset compensation.
    --------------------------------------------
    dcoffdisb              : in  std_logic;  -- disable dc_offset compensation when high

    --------------------------------------------
    -- Control for IQ Mismatch Compensation
    --------------------------------------------
    iq_estimation_enable   : in  std_logic;  -- enable the I/Q estimation when high
    iq_compensation_enable : in  std_logic;  -- enable the I/Q compensation when high

    --------------------------------------------
    -- Control for equalization
    --------------------------------------------
    equ_activate           : in  std_logic;  -- enable the equalizer when high.
    equalizer_disb         : in  std_logic;  -- disable the equalizer filter when high.
    equalizer_init_n       : in  std_logic;  -- equalizer filter coeffs set to 0 when low.
    alpha_accu_disb        : in  std_logic;  -- stop coeff accu when high.
    beta_accu_disb         : in  std_logic;  -- stop dc accu when high.
    alpha                  : in  std_logic_vector(2 downto 0);  -- alpha parameter value.
    beta                   : in  std_logic_vector(2 downto 0);  -- beta parameter value.

    --------------------------------------------
    -- Control for DSSS / CCK demodulation
    --------------------------------------------
    interp_disb            : in  std_logic;  -- disable the interpolation when high
    rx_enable              : in  std_logic;  -- enable rx path when high 
    mod_type               : in  std_logic;  -- '0' for DSSS, '1' for CCK.
    enable_error           : in  std_logic;  -- Enable error calculation when high.
    precomp_enable         : in  std_logic;  -- Reload the omega accumulator.
    demod_rate             : in  std_logic;  -- '0' for BPSK, '1' for QPSK.
    cck_rate               : in  std_logic;  -- '0' for 5.5 Mhz, '1' for 11 Mhz.
    rho                    : in  std_logic_vector(1 downto 0);  -- rho parameter value
    mu                     : in  std_logic_vector(2 downto 0);  -- mu parameter value.
    --
    tau_est                : out std_logic_vector(17 downto 0);

    --------------------------------------------
    -- Control for Decode Path
    --------------------------------------------
    scrambling_disb        : in  std_logic;  -- scrambling disable (test mode) 
    decode_path_activate   : in  std_logic;  -- enable the differential decoder
    diff_decod_first_val   : in  std_logic;  -- initialize the diff_decoder block
    sfd_detect_enable      : in  std_logic;  -- enable the sfd detection 
    -- Number of errors allowed.
    sfderr                 : in  std_logic_vector (2 downto 0);
    -- Number of pramble bits used for Start Frame Delimiter search.
    sfdlen                 : in  std_logic_vector (2 downto 0);
    -- Receive mode        : 0=BPSK, 1=QPSK, 2=CCK5.5, 3=CCK11.
    rec_mode               : in  std_logic_vector (1 downto 0);

    --------------------------------------------
    -- Remodulation interface
    --------------------------------------------
    remod_data             : in  std_logic_vector(1 downto 0);  -- Data from the TX path
    --
    remod_enable           : out std_logic;  -- High when the remodulation is enabled
    remod_data_req         : out std_logic;  -- request to send a byte 
    remod_type             : out std_logic;  -- CCK : 0 ; PBCC : 1
    remod_bq               : out std_logic;  -- BPSK = 0 - QPSK = 1 
    demod_data             : out std_logic_vector(7 downto 0);  -- Data to the TX path

    --------------------------------------------
    -- AGC-CCA interface
    --------------------------------------------
    correl_rst_n           : in  std_logic;  -- reset the Barker correlator when low
    synchro_en             : in  std_logic;  -- enable the synchronisation when high 
    --
    symbol_synchro         : out std_logic;  -- pulse at the beginning of a symbol.

    --------------------------------------------
    -- Modem B state machines interface
    --------------------------------------------
    sfd_found              : out std_logic;  -- sfd found when high
    preamble_type          : out std_logic;  -- Type of preamble 
    phy_data_ind           : out std_logic;  -- pulse when an RX byte is available.
    data_to_bup            : out std_logic_vector(7 downto 0); -- RX data.
    --------------------------------------------
    -- Status registers.
    --------------------------------------------
    iq_gain_sat_stat       : out std_logic_vector(6 downto 0);
    dc_offset_i_stat       : out std_logic_vector(5 downto 0);
    dc_offset_q_stat       : out std_logic_vector(5 downto 0);
    coeff_sum_i_stat       : out std_logic_vector(7 downto 0);
    coeff_sum_q_stat       : out std_logic_vector(7 downto 0);
    freqoffestim_stat      : out std_logic_vector(7 downto 0);
    --------------------------------------------
    -- Error
    --------------------------------------------
    phase_error_o       : out std_logic_vector(data_length_g + 3 downto 0);
    phase_error_ready_o : out std_logic; -- Symbol sync pulse
    --
    symbol_sync_error_o : out std_logic;
    error_cart_i_o      : out std_logic_vector(data_length_g+4 downto 0);-- real part
    error_cart_q_o      : out std_logic_vector(data_length_g+4 downto 0);-- Imaginary part
    -------------------------------
    -- Diag ports
    -------------------------------
    diag_error_i     : out std_logic_vector(data_length_g-2 downto 0); 
    diag_error_q     : out std_logic_vector(data_length_g-2 downto 0)
    
    );

  end component;


----------------------
-- File: /share/hw_projects/PROJECTS/PRJ_RW_WLAN_IP_LIB_V1_7/IPs/RW_WLAN/MODEM802_11b/tx_path/vhdl/rtl/tx_path_core.vhd
----------------------
  component tx_path_core
  generic(
   dec_freq_g : integer := 1 -- shift the register every dec_freq_g period.
                               -- (1 to 8) - should correspond to 11 MHz.
          );
  port (
   -- clocks and reset
   clk                : in  std_logic;
   reset_n            : in  std_logic;
   
   --------------------------------------------
   -- Interface with Modem State Machines
   --------------------------------------------
   low_r_flow_activate: in  std_logic;   
   --                   made high by the state machines for 1 or 2 Mb/s trans
   psk_mode           : in  std_logic;    
   --                   BPSK = 0 - QPSK = 1
   shift_period       : in  std_logic_vector (3 downto 0); 
   --                   period to shift of the serializer (1010 for low rate)  
   cck_flow_activate  : in  std_logic;            
   --                   made high by the state machines for CCK 5.5 or 11 Mb/s
   cck_speed          : in  std_logic;                     
   --                   5.5 Mbits/s = 0 - 11 Mbits/s = 1
   tx_activated       : out std_logic;
   --                   indicate to the sm when the tx_path is activated
   
   --------------------------------------------
   -- Interface with RW_WLAN Bup - via or not Modem State Machines
   --------------------------------------------
   -- inputs signals                                                           
   scrambling_disb    : in std_logic;
   --                   disable the scrambler when high (for modem tests) 
   spread_disb        : in std_logic;
   --                   disable the spreading when high (for modem tests) 
   bup_txdata         : in  std_logic_vector(7 downto 0); 
   --                   data to send
   phy_data_req       : in  std_logic; 
   --                   request to send a byte                  
   txv_prtype         : in  std_logic; 
   --                   def the type of preamble (short or long)
   txv_immstop        : in std_logic;
   --                   for BT co-existence, stop tx immediately if high.
   -- outputs signals                                                          
   phy_data_conf      : out std_logic; 
   --                   last byte was read, ready for new one 

   --------------------------------------------
   -- Interface with the RX Path for the remodulation
   --------------------------------------------
   remod_enable     : in  std_logic; -- High when the remodulation is enabled
   remod_data_req   : in  std_logic; -- request to send a byte 
   remod_type       : in  std_logic; -- CCK : 0 ; PBCC : 1
   remod_bq         : in  std_logic; -- BPSK = 0 - QPSK = 1 
   demod_data       : in  std_logic_vector(7 downto 0); -- Data to the TX path
   --
   remod_data       : out std_logic_vector(1 downto 0); -- Data from the TX path

   --------------------------------------------
   -- FIR controls
   --------------------------------------------
   init_fir         : out std_logic;
   fir_activate     : out std_logic;
   fir_phi_out_tog_o: out std_logic; -- when toggle a new data has arrived
   fir_phi_out      : out std_logic_vector (1 downto 0)
   );

  end component;


----------------------
-- File: /share/hw_projects/PROJECTS/PRJ_RW_WLAN_IP_LIB_V1_7/IPs/RW_WLAN/MODEM802_11b/modem_sm_b/vhdl/rtl/modem_sm_b.vhd
----------------------
  component modem_sm_b
  port (
    --------------------------------------
    -- Clocks & Reset
    -------------------------------------- 
    hresetn             : in  std_logic; -- AHB reset line.
    hclk                : in  std_logic; -- AHB clock line.
    --------------------------------------
    -- TX path block
    -------------------------------------- 
    seria_data_conf     : in  std_logic; -- Serializer is ready for new data
    tx_activated        : in  std_logic; -- the tx_path is transmitting    
    -- 
    scr_data_in         : out std_logic_vector(7 downto 0); -- data sent to scrambler
    sm_data_req         : out std_logic; -- State machines data request
    tx_psk_mode         : out std_logic; -- 0 = BPSK; 1 = QPSK
    activate_seria      : out std_logic; -- activate Serializer
    shift_period        : out std_logic_vector(3 downto 0); -- Serializer speed
    activate_cck        : out std_logic; -- activate CCK modulator
    tx_cck_rate         : out std_logic; -- CCK speed (0 = 5.5 Mbit/s; 1 = 11 Mbit/s)
    preamble_type_tx    : out std_logic; -- preamble type (0 = short; 1 = long)
    --------------------------------------
    -- RX path block
    -------------------------------------- 
    cca_busy            : in  std_logic; -- CCA busy
    preamble_type_rx    : in  std_logic; -- 1: long preamble ; 0: short preamble
    sfd_found           : in  std_logic; -- pulse when SFD is detected
    byte_ind            : in  std_logic; -- byte indication  
    symbol_sync         : in  std_logic; -- symbol indication
    rx_data             : in  std_logic_vector(7 downto 0); -- rx descrambled data
    --
    decode_path_activate: out std_logic; -- decode path activate
    diff_decod_first_val: out std_logic; -- pulse on first byte to decode
    rec_mode            : out std_logic_vector(1 downto 0); -- BPSK, QPSK, CCK5.5, CCK 11
    mod_type            : out std_logic; -- 0 : DSSS ; 1 : CCK
    rx_psk_mode         : out std_logic; -- 0 = BPSK; 1 = QPSK
    rx_cck_rate         : out std_logic; -- CCK rate (0 = 5.5 Mb/s; 1 = 11 Mb/s)
    rx_idle_state       : out std_logic; -- high when sm is idle
    rx_plcp_state       : out std_logic; -- high when sm is in plcp state
    --------------------------------------------
    -- Registers
    --------------------------------------------
    reg_prepre          : in  std_logic_vector(5 downto 0); -- pre-preamble count.
    txenddel_reg        : in  std_logic_vector(7 downto 0);
    rxlenchken          : in  std_logic; -- select ckeck on rx data lenght.
    rxmaxlength         : in  std_logic_vector(11 downto 0); -- Max accepted received length.    
    --------------------------------------------
    -- CCA
    --------------------------------------------
    psdu_duration       : out std_logic_vector(15 downto 0); --length in us
    correct_header      : out std_logic; -- high when header is correct.
    plcp_error          : out std_logic; -- high when plcp error occures
    listen_start_o      : out std_logic; -- high when start to listen
    rx_end_for_timing   : out std_logic; -- indication of end of frame
    --------------------------------------
    -- CRC
    -------------------------------------- 
    crc_data_1st        : in  std_logic_vector(7 downto 0); -- CRC data
    crc_data_2nd        : in  std_logic_vector(7 downto 0); -- CRC data
    --
    crc_init            : out std_logic; -- init CRC computation
    crc_data_valid      : out std_logic; -- compute CRC on packet header
    data_to_crc         : out std_logic_vector(7 downto 0); -- byte data to CRC
    --------------------------------------------
    -- Radio controller interface
    --------------------------------------------
    rf_txonoff_req     : out std_logic;  -- tx on off request
    rf_txonoff_conf    : in  std_logic;  -- tx on off confirmation
    rf_rxonoff_req     : out std_logic;  -- rx on off request
    rf_rxonoff_conf    : in  std_logic;  -- rx on off confirmation
    --------------------------------------
    -- BuP
    -------------------------------------- 
    -- TX
    phy_txstartend_req  : in  std_logic; -- request to start a packet transmission
                                         -- or request for end of transmission
    txv_service         : in  std_logic_vector(7 downto 0); -- service field
    phy_data_req        : in  std_logic; -- request from BuP to send a byte
    txv_datarate        : in  std_logic_vector( 3 downto 0); -- PSDU transmission rate
    txv_length          : in  std_logic_vector(11 downto 0); -- packet length in bytes
    bup_txdata          : in  std_logic_vector( 7 downto 0); -- data from BuP
    phy_txstartend_conf : out std_logic; -- transmission started, ready for data
                                         -- or transmission ended
    txv_immstop         : in std_logic;  -- request from Bup to stop tx.
    -- RX
    phy_cca_ind         : out  std_logic; -- indication of a carrier
    phy_rxstartend_ind  : out  std_logic; -- indication of a received PSDU
    rxv_service         : out  std_logic_vector(7 downto 0); -- service field
    phy_data_ind        : out  std_logic; -- indication of a received byte
    rxv_datarate        : out  std_logic_vector( 3 downto 0); -- PSDU RX rate
    rxv_length          : out  std_logic_vector(11 downto 0); -- packet length in bytes
    rxe_errorstat       : out  std_logic_vector(1 downto 0); -- error
    bup_rxdata          : out  std_logic_vector( 7 downto 0);  -- data to BuP
    --------------------------------------------
    -- Misc
    --------------------------------------------
    tx_sfd              : out std_logic;  -- Tx sfd avtive
    --------------------------------------
    -- Diag
    --------------------------------------
    rx_state_diag       : out std_logic_vector(2 downto 0)  -- Diag port
    );
  end component;


----------------------
-- File: /share/hw_projects/PROJECTS/PRJ_RW_WLAN_IP_LIB_V1_7/IPs/RW_WLAN/MODEM802_11b/rx_ctrl/vhdl/rtl/rx_ctrl.vhd
----------------------
  component rx_ctrl
  port (
    --------------------------------------
    -- Clocks & Reset
    -------------------------------------- 
    hresetn             : in  std_logic; -- AHB reset line.
    hclk                : in  std_logic; -- AHB clock line.

    --------------------------------------------
    -- Registers interface
    --------------------------------------------
    eq_disb             : in std_logic; --equalizer disable
    -- delay before enabling the precompensation :
    precomp             : in  std_logic_vector(5 downto 0); 
    -- delay before enabling the equalizer after energy detect
    eqtime              : in  std_logic_vector(3 downto 0); 
    -- delay before disabling the equalizer after last parameter update
    eqhold              : in  std_logic_vector(11 downto 0); 
    -- delay before enabling the phase correction after energy detect
    looptime            : in  std_logic_vector(3 downto 0); 
    -- delay before switching off the timing synchro after energy detect
    synctime            : in  std_logic_vector(5 downto 0); 
    -- initial value of equalizer parameters
    alpha               : in  std_logic_vector(1 downto 0); 
    beta                : in  std_logic_vector(1 downto 0); 
    -- initial value of phase estimation parameters
    mu                  : in  std_logic_vector(1 downto 0); 
    -- Talpha time intervals values for alpha equalizer parameter.
    talpha3             : in  std_logic_vector( 3 downto 0);
    talpha2             : in  std_logic_vector( 3 downto 0);
    talpha1             : in  std_logic_vector( 3 downto 0);
    talpha0             : in  std_logic_vector( 3 downto 0);
    -- Tbeta time intervals values for beta equalizer parameter.
    tbeta3              : in  std_logic_vector( 3 downto 0);
    tbeta2              : in  std_logic_vector( 3 downto 0);
    tbeta1              : in  std_logic_vector( 3 downto 0);
    tbeta0              : in  std_logic_vector( 3 downto 0);
    -- Tmu time interval value for phase correction mu parameter.
    tmu3                : in  std_logic_vector( 3 downto 0);
    tmu2                : in  std_logic_vector( 3 downto 0);
    tmu1                : in  std_logic_vector( 3 downto 0);
    tmu0                : in  std_logic_vector( 3 downto 0);

    --------------------------------------------
    -- Input control
    --------------------------------------------
    energy_detect       : in  std_logic;
    agcproc_end         : in  std_logic; -- pulse on AGC procedure end
    rx_psk_mode         : in  std_logic; -- 0 = BPSK; 1 = QPSK
    rx_idle_state       : in  std_logic;
    precomp_disb        : in  std_logic; -- disable the precompensation 
    comp_disb           : in  std_logic; -- disable the compensation 
                                         -- (error calculation)
    iqmm_disb           : in  std_logic; -- disable iq mismatch
    gain_disb           : in  std_logic; -- disable gain

    --------------------------------------------
    -- RX path control signals
    --------------------------------------------
    equalizer_activate  : out std_logic; -- equalizer enable
    equalizer_init_n    : out std_logic; -- equalizer initialization
    equalizer_disb      : out std_logic; -- equalizer disable
    precomp_enable      : out std_logic; -- frequency precompensation enable
    synctime_enable     : out std_logic; -- timing synchronization enable
    phase_estim_enable  : out std_logic; -- phase estimation enable
    iq_comp_enable      : out std_logic; -- iq mismatch compensation enable
    iq_estim_enable     : out std_logic; -- iq mismatch estimation enable
    gain_enable         : out std_logic; -- gain enable
    sfd_detect_enable   : out std_logic; -- enable SFD detection when high
    -- parameters value sent to the equalizer
    applied_alpha       : out std_logic_vector(2 downto 0);
    applied_beta        : out std_logic_vector(2 downto 0);
    alpha_accu_disb     : out std_logic;
    beta_accu_disb      : out std_logic;
    -- parameters value sent to the phase estimation
    applied_mu          : out std_logic_vector(2 downto 0)
    );

  end component;


----------------------
-- File: /home/cvandebu/PROJECTS/RW_WLAN_IP_LIB/IPs/RW_WLAN/MODEM802_11b/evm_calculator/vhdl/rtl/evm_calculator.vhd
----------------------
  component evm_calculator
  port (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
     reset_n          : in std_logic; -- global reset  
     clk              : in std_logic; -- main clock
    --------------------------------------
    -- Control
    --------------------------------------
    symbol_sync       : in std_logic;
    sfd_found         : in std_logic;    
    --------------------------------------
    -- Error from phase estimator
    --------------------------------------
    phase_error       : in std_logic_vector(12 downto 0);
    --------------------------------------
    -- Register information
    --------------------------------------
    evm               : out std_logic_vector(12 downto 0)
    );

  end component;


----------------------
-- File: /share/hw_projects/PROJECTS/PRJ_RW_WLAN_IP_LIB_V1_7/IPs/RW_WLAN/MODEM802_11b/crc16_8/vhdl/rtl/crc16_8.vhd
----------------------
  component crc16_8
  port (
    -- clock and reset
    clk       : in  std_logic;                    
    resetn    : in  std_logic;                   
     
    -- inputs
    data_in   : in  std_logic_vector ( 7 downto 0);
    --          8-bits inputs for parallel computing. 
    ld_init   : in  std_logic;
    --          initialize the CRC
    calc      : in  std_logic;
    --          ask of calculation of the available data.
 
    -- outputs
    crc_out_1st  : out std_logic_vector (7 downto 0); 
    crc_out_2nd  : out std_logic_vector (7 downto 0) 
    --          CRC result
   );

  end component;


----------------------
-- File: modemb_registers_if.vhd
----------------------
  component modemb_registers_if
  port (
    --------------------------------------
    -- Clocks & Reset
    --------------------------------------
    reset_n : in std_logic;
    hclk    : in std_logic;             -- 44 MHz clock.

    --------------------------------------
    -- Controls
    --------------------------------------
    -- Registers inputs :
    reg_gaindisb          : in  std_logic;  -- '0' to enable Gain compensation.
    reg_interpdisb        : in  std_logic;  -- '0' to enable Interpolation.
    reg_iqmmdisb          : in  std_logic;  -- '0' to enable I/Q mismatch compensation.
    reg_precompdisb       : in  std_logic;  -- '0' to enable timing offset compensation
    reg_dcoffdisb         : in  std_logic;  -- '0' to enable the DC offset compensation
    reg_compdisb          : in  std_logic;  -- '0' to enable the compensation.
    reg_eqdisb            : in  std_logic;  -- '0' to enable the Equalizer.
    reg_spreaddisb        : in  std_logic;  -- '0' to enable spreading.                        
    reg_scrambdisb        : in  std_logic;  -- '0' to enable scrambling.
    -- Registers outputs :
    reg_gaindisb_sync     : out std_logic;  -- '0' to enable Gain compensation.
    reg_interpdisb_sync   : out std_logic;  -- '0' to enable Interpolation.
    reg_iqmmdisb_sync     : out std_logic;  -- '0' to enable I/Q mismatch compensation.
    reg_precompdisb_sync  : out std_logic;  -- '0' to enable timing offset compensation
    reg_dcoffdisb_sync    : out std_logic;  -- '0' to enable the DC offset compensation
    reg_compdisb_sync     : out std_logic;  -- '0' to enable the compensation.
    reg_eqdisb_sync       : out std_logic;  -- '0' to enable the Equalizer.
    reg_spreaddisb_sync   : out std_logic;  -- '0' to enable spreading.                        
    reg_scrambdisb_sync   : out std_logic   -- '0' to enable scrambling.
    );

  end component;


----------------------
-- File: modem802_11b_core.vhd
----------------------
  component modem802_11b_core
  generic (
    radio_interface_g : integer := 3   -- 0 -> reserved
    );                                 -- 1 -> only Analog interface
                                       -- 2 -> only HISS interface
  port (                               -- 3 -> both interfaces (HISS and Analog)
   -- clocks and reset
   clk                 : in  std_logic; -- main clock (not gated)
   rx_path_b_gclk      : in  std_logic; -- gated clock for RX path
   tx_path_b_gclk      : in  std_logic; -- gated clock for TX path
   reset_n             : in  std_logic; -- global reset async
   --
   rx_gating           : out std_logic; -- Gating condition for Rx path
   tx_gating           : out std_logic; -- Gating condition for Tx path
  
   --------------------------------------------
   -- Interface with RW_WLAN Bup
   --------------------------------------------
   -- inputs signals                                                           
   bup_txdata          : in  std_logic_vector(7 downto 0); -- data to send         
   phy_txstartend_req  : in  std_logic; -- request to start a packet transmission    
   phy_data_req        : in  std_logic; -- request to send a byte                  
   phy_ccarst_req      : in  std_logic; -- request to reset CCA state machine                 
   txv_length          : in  std_logic_vector(11 downto 0);  -- RX PSDU length     
   txv_service         : in  std_logic_vector(7 downto 0);  -- tx service field   
   txv_datarate        : in  std_logic_vector( 3 downto 0); -- PSDU transm. rate
   txpwr_level         : in  std_logic_vector( 2 downto 0); -- TX power level.
   txv_immstop         : in  std_logic;  -- request from Bup to stop tx.
    
   -- outputs signals                                                          
   phy_txstartend_conf : out std_logic; -- transmission started, ready for data  
   phy_rxstartend_ind  : out std_logic; -- indication of RX packet                     
   phy_data_conf       : out std_logic; -- last byte was read, ready for new one 
   phy_data_ind        : out std_logic; -- received byte ready                  
   rxv_length          : out std_logic_vector(11 downto 0);  -- RX PSDU length  
   rxv_service         : out std_logic_vector(7 downto 0);  -- rx service field
   rxv_datarate        : out std_logic_vector( 3 downto 0); -- PSDU rec. rate
   rxe_errorstat       : out std_logic_vector(1 downto 0);-- packet recep. stat
   phy_cca_ind         : out std_logic; -- CCA status                           
   bup_rxdata          : out std_logic_vector(7 downto 0); -- data received      
   
   --------------------------------------------
   -- Radio controller interface
   --------------------------------------------
   rf_txonoff_conf     : in  std_logic;  -- Radio controller in TX mode conf
   rf_rxonoff_conf     : in  std_logic;  -- Radio controller in RX mode conf
   --
   rf_txonoff_req      : out std_logic;  -- Radio controller in TX mode req
   rf_rxonoff_req      : out std_logic;  -- Radio controller in RX mode req
   rf_dac_enable       : out std_logic;  -- DAC enable
   
   --------------------------------------------
   -- AGC
   --------------------------------------------
   agcproc_end         : in std_logic;
   cca_busy            : in std_logic;
   correl_rst_n        : in std_logic;
   agc_diag            : in std_logic_vector(15 downto 0);
   --
   psdu_duration       : out std_logic_vector(15 downto 0);
   correct_header      : out std_logic;
   plcp_state          : out std_logic;
   plcp_error          : out std_logic;
   listen_start_o      : out std_logic; -- high when start to listen
   rx_end_for_timing   : out std_logic; -- indication of end of frame
   --
   sfd_found           : out std_logic;
   symbol_sync2        : out std_logic;
   --------------------------------------------
   -- Data Inputs
   --------------------------------------------
   -- data from gain compensation (inside rx_b_frontend)
   rf_rxi              : in  std_logic_vector(7 downto 0);
   rf_rxq              : in  std_logic_vector(7 downto 0);
   
   --------------------------------------------
   -- Modemb Registers Input
   --------------------------------------------
   -- MDMbSTAT0 register
   reg_eqsumq          : out std_logic_vector(7 downto 0);
   reg_eqsumi          : out std_logic_vector(7 downto 0);
   reg_dcoffsetq       : out std_logic_vector(5 downto 0);
   reg_dcoffseti       : out std_logic_vector(5 downto 0);

   -- MDMbSTAT1 register
   reg_iqgainestim     : out std_logic_vector(6 downto 0);
   reg_freqoffestim    : out std_logic_vector(7 downto 0);
   reg_evm             : out std_logic_vector(12 downto 0);

   --------------------------------------------
   -- Modem Register Outputs
   --------------------------------------------
   -- MDMbCNTL register
   reg_gaindisb        : in std_logic; -- '0' to enable Gain compensation.
   reg_interpdisb      : in std_logic; -- '0' to enable Interpolation.
   reg_iqmmdisb        : in std_logic; -- '0' to enable I/Q mismatch compensation.
   reg_precompdisb     : in std_logic; -- '0' to enable timing offset compensation
   reg_dcoffdisb       : in std_logic; -- '0' to enable the DC offset compensation
   reg_compdisb        : in std_logic; -- '0' to enable the compensation.
   reg_eqdisb          : in std_logic; -- '0' to enable the Equalizer.
   reg_spreaddisb      : in std_logic; -- '0' to enable spreading.                       
   reg_scrambdisb      : in std_logic; -- '0' to enable scrambling.
   reg_sfderr          : in std_logic_vector( 2 downto 0); -- Error number for SFD
   reg_interfildisb    : in std_logic; -- '1' to bypass rx_11b_interf_filter 

   -- Number of preamble bits to be considered in short SFD comparison
   reg_sfdlen          : in std_logic_vector( 2 downto 0);
   reg_prepre          : in std_logic_vector( 5 downto 0); -- pre-preamble count.

   -- MDMbPRMINIT register
   -- Values for phase correction parameters.
   reg_rho             : in std_logic_vector( 1 downto 0);
   reg_mu              : in std_logic_vector( 1 downto 0);
   -- Values for phase feedforward equalizer parameters.
   reg_beta            : in std_logic_vector( 1 downto 0);
   reg_alpha           : in std_logic_vector( 1 downto 0);

   -- MDMbTALPHA register
   -- TALPHA time interval value for equalizer alpha parameter.
   reg_talpha3         : in std_logic_vector( 3 downto 0);
   reg_talpha2         : in std_logic_vector( 3 downto 0);
   reg_talpha1         : in std_logic_vector( 3 downto 0);
   reg_talpha0         : in std_logic_vector( 3 downto 0);

   -- MDMbTBETA register
   -- TBETA time interval value for equalizer beta parameter.
   reg_tbeta3          : in std_logic_vector( 3 downto 0);
   reg_tbeta2          : in std_logic_vector( 3 downto 0);
   reg_tbeta1          : in std_logic_vector( 3 downto 0);
   reg_tbeta0          : in std_logic_vector( 3 downto 0);

   -- MDMbTMU register
   -- TMU time interval value for phase correction and offset comp. mu param
   reg_tmu3            : in std_logic_vector( 3 downto 0);
   reg_tmu2            : in std_logic_vector( 3 downto 0);
   reg_tmu1            : in std_logic_vector( 3 downto 0);
   reg_tmu0            : in std_logic_vector( 3 downto 0);

   -- MDMbCNTL1 register
   reg_rxlenchken      : in std_logic;
   reg_rxmaxlength     : in std_logic_vector(11 downto 0);

   -- MDMbRFCNTL register
   -- Delay of the Tx front-end inside the RW_WLAN RF, in number of 44 MHz cycles.
   reg_txenddel        : in std_logic_vector(7 downto 0);

   -- MDMbEQCNTL register
   -- Delay to stop the equalizer adaptation after the last param update, in s
   reg_eqhold          : in std_logic_vector(11 downto 0);
   -- Delay to start the compensation after the start of the estimation, in s.
   reg_comptime        : in std_logic_vector( 4 downto 0);
   -- Delay to start the estimation after the enabling of the equalizer, in s.
   reg_esttime         : in std_logic_vector( 4 downto 0);
   -- Delay to switch on the equalizer after the fine gain setting, in s.
   reg_eqtime          : in std_logic_vector( 3 downto 0);

   -- MDMbCNTL2 register
   reg_precomp         : in std_logic_vector( 5 downto 0); -- in us.
   reg_synctime        : in std_logic_vector( 5 downto 0);
   reg_looptime        : in std_logic_vector( 3 downto 0);

   --------------------------------------------
   -- Tx FIR controls
   --------------------------------------------
   init_fir            : out std_logic;
   fir_activate        : out std_logic;
   fir_phi_out_tog_o   : out std_logic;
   fir_phi_out         : out std_logic_vector (1 downto 0);
   
   --------------------------------------------
   -- Interface with RX Frontend
   --------------------------------------------
   -- Controls
   clock_lock          : out std_logic;
   gain_enable         : out std_logic;  -- gain compensation control.
   tau_est             : out std_logic_vector(17 downto 0);
   enable_error        : out std_logic;

   --------------------------------------------
   -- Error
   --------------------------------------------
   symbol_sync_error_o : out std_logic;
   error_cart_i_o : out std_logic_vector(13 downto 0);  -- real part.
   error_cart_q_o : out std_logic_vector(13 downto 0);  -- Imaginary part    

   --------------------------------------------
   -- Diagnostic port
   --------------------------------------------
   modem_diag          : out std_logic_vector(31 downto 0);
   modem_diag0         : out std_logic_vector(15 downto 0);
   modem_diag1         : out std_logic_vector(15 downto 0);
   modem_diag2         : out std_logic_vector(15 downto 0)
  );

  end component;



 
end modem802_11b_core_pkg;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------
