--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandebu $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 50 $
--/ $Date: 2011-09-21 18:10:05 +0200 (Wed, 21 Sep 2011) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      : Error generator for phase and carrier offset estimation.
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/phase_estimation/vhdl/rtl/error_gen.vhd $
--/
--////////////////////////////////////////////////////////////////////////////



--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library IEEE; 
use IEEE.STD_LOGIC_1164.ALL; 
use IEEE.std_logic_unsigned.all;  
 
--library phase_estimation_rtl;
library work;
--use phase_estimation_rtl.phase_estimation_pkg.all;
use work.phase_estimation_pkg.all;

--library cordic_vect_rtl;
library work;
--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity error_gen is
  generic (
    datasize_g  : integer := 28;-- Max value is 28.
    errorsize_g : integer := 28 -- Max value is 28.
  );
  port (
    -- clock and reset.
    clk            : in  std_logic;
    reset_n        : in  std_logic;
    --
    symbol_sync    : in  std_logic;     -- Symbol synchronization pulse.
    -- Demodulated datain (real and im).
    data_i         : in  std_logic_vector(datasize_g-1 downto 0);
    data_q         : in  std_logic_vector(datasize_g-1 downto 0);
    -- Demapped data.
    demap_data     : in  std_logic_vector(1 downto 0);
    enable_error   : in  std_logic;
    --
    -- Error detected.
    phase_error    : out std_logic_vector(errorsize_g-1 downto 0);
    -- Error 
    error_cart_i_o : out std_logic_vector(datasize_g-1 downto 0);  -- real part.
    error_cart_q_o : out std_logic_vector(datasize_g-1 downto 0);  -- Imaginary part
    -- Error ready.
    error_ready    : out std_logic
    );

end error_gen;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture RTL of error_gen is

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  -- Signals for multiplication of demodulated data with demapped data conjugate
  signal neg_data_i    : std_logic_vector(datasize_g-1 downto 0); -- (- data_i).
  signal neg_data_q    : std_logic_vector(datasize_g-1 downto 0); -- (- data_q).
  -- Detected error, cartesian coordonates.
  signal error_cart_i  : std_logic_vector(datasize_g-1 downto 0); -- real part.
  signal error_cart_q  : std_logic_vector(datasize_g-1 downto 0); -- Imaginary part
 
  signal phase_error_o : std_logic_vector(errorsize_g-1 downto 0); -- Error detected.
--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin
  
  -- error_cart = input data * demap_data conjugate.
  neg_data_i <= not(data_i) + '1';
  neg_data_q <= not(data_q) + '1';
  
  with demap_data select
    error_cart_i <=
      data_i      when "00",
      data_q      when "01",
      neg_data_q  when "10",
      neg_data_i  when others;

  with demap_data select
    error_cart_q <=
      data_q      when "00",
      neg_data_i  when "01",
      data_i      when "10",
      neg_data_q  when others;
           
                  
  -- Block for cordic algorithm: the angle output belongs to [-pi/2, pi/2].
  cordic_vect_1 : cordic_vect
    generic map (
      datasize_g          =>  datasize_g,
      errorsize_g         =>  errorsize_g,
      scaling_g           =>  0 -- no scaling needed
      
      )
    port map (
      -- clock and reset.
      clk                 => clk,
      reset_n             => reset_n,
      --
      load                => symbol_sync,  -- Load input values.
      x_in                => error_cart_i, -- Real part in.
      y_in                => error_cart_q, -- Imaginary part in.
      --
      angle_out           => phase_error_o,  -- Angle out.
      cordic_ready        => error_ready   -- Angle ready.
      );
      
      
  phase_error <= phase_error_o when enable_error='1' else (others=> '0');

  error_cart_output_p: process (clk, reset_n)
  begin
    if reset_n = '0' then
      error_cart_i_o <= (others => '0');
      error_cart_q_o <= (others => '0');
    elsif clk'event and clk = '1' then
      error_cart_i_o <= error_cart_i;
      error_cart_q_o <= error_cart_q;      
    end if;
  end process error_cart_output_p;
end RTL;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------
