--////////////////////////////////////////////////////////////////////////////
--/  Copyright (C) by RivieraWaves.
--/  This module is a confidential and proprietary property of RivieraWaves
--/  and a possession or use of this module requires written permission
--/  from RivieraWaves.
--/---------------------------------------------------------------------------
--/ $Author: cvandebu $
--/ Company          : RivieraWaves
--/---------------------------------------------------------------------------
--/ $Revision: 50 $
--/ $Date: 2011-09-21 18:10:05 +0200 (Wed, 21 Sep 2011) $
--/ --------------------------------------------------------------------------
--/ Dependencies     : None
--/ Description      : Phase and carrier offset estimation.
--/ Application Note :
--/ Terms & concepts :
--/ Bugs             :
--/ Open issues and future enhancements :
--/ References       :
--/ Revision History :
--/ --------------------------------------------------------------------------
--/
--/ $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MODEM80211BCORE/phase_estimation/vhdl/rtl/phase_estimation.vhd $
--/
--////////////////////////////////////////////////////////////////////////////



--------------------------------------------------------------------------------
-- Library
--------------------------------------------------------------------------------
library IEEE; 
use IEEE.STD_LOGIC_1164.ALL; 
 
--library phase_estimation_rtl;
library work;
--use phase_estimation_rtl.phase_estimation_pkg.all;
use work.phase_estimation_pkg.all;
-- ambit synthesis off
-- synopsys translate_off
-- synthesis translate_off 
--use phase_estimation_rtl.phase_estimation_global_pkg.all;
use work.phase_estimation_global_pkg.all;
-- ambit synthesis on
-- synopsys translate_on
-- synthesis translate_on
 

--------------------------------------------------------------------------------
-- Entity
--------------------------------------------------------------------------------
entity phase_estimation is
  generic (
    dsize_g      : integer := 13; -- size of data in
    esize_g      : integer := 13; -- size of error (must >= dsize_g).
    phisize_g    : integer := 15; -- size of angle phi
    omegasize_g  : integer := 12; -- size of angle omega
    sigmasize_g  : integer := 10; -- size of angle sigma
    tausize_g    : integer := 18  -- size of tau
  );
  port (
    -- clock and reset.
    clk                  : in  std_logic;
    reset_n              : in  std_logic;
    --
    symbol_sync          : in  std_logic;  -- Symbol synchronization pulse.
    precomp_enable       : in  std_logic;  -- Enable the precompensation
    interpolation_enable : in  std_logic;  -- Load the tau accumulator
    data_i               : in  std_logic_vector(dsize_g-1 downto 0);  -- Real data in
    data_q               : in  std_logic_vector(dsize_g-1 downto 0);  -- Im data in.
    demap_data           : in  std_logic_vector(1 downto 0);  -- Data from demapping.
    enable_error         : in  std_logic;  -- Enable the error calculation.
    mod_type             : in  std_logic;  -- Modulation type: '0' for DSSS, '1' for CCK.
    rho                  : in  std_logic_vector(3 downto 0);  -- rho parameter value.
    mu                   : in  std_logic_vector(3 downto 0);  -- mu parameter value.
    -- Filtered outputs.
    freqoffestim_stat    : out std_logic_vector(7 downto 0);  -- Status register.
    phi                  : out std_logic_vector(phisize_g-1 downto 0);  -- phi angle.
    sigma                : out std_logic_vector(sigmasize_g-1 downto 0);  -- sigma angle.
    omega                : out std_logic_vector(omegasize_g-1 downto 0);  -- omega angle.
    tau                  : out std_logic_vector(tausize_g-1 downto 0);   -- tau.
    -- Phase Error 
    phase_error_o        : out std_logic_vector(esize_g-1 downto 0);
    phase_error_ready_o  : out std_logic;
    -- Error 
    error_cart_i_o : out std_logic_vector(dsize_g downto 0);  -- real part.
    error_cart_q_o : out std_logic_vector(dsize_g downto 0)  -- Imaginary part    
    );

end phase_estimation;


--------------------------------------------------------------------------------
-- Architecture
--------------------------------------------------------------------------------
architecture RTL of phase_estimation is

  ------------------------------------------------------------------------------
  -- Signals
  ------------------------------------------------------------------------------
  signal data_i_ext   : std_logic_vector(dsize_g downto 0);
  signal data_q_ext   : std_logic_vector(dsize_g downto 0);
  signal phase_error  : std_logic_vector(esize_g-1 downto 0);
  signal error_ready  : std_logic; -- phase_error valid.


--------------------------------------------------------------------------------
-- Architecture Body
--------------------------------------------------------------------------------
begin


  data_i_ext(dsize_g) <= data_i(dsize_g-1);
  data_q_ext(dsize_g) <= data_q(dsize_g-1);

  data_i_ext(dsize_g-1 downto 0) <= data_i;
  data_q_ext(dsize_g-1 downto 0) <= data_q;
  --------------------------------------
  -- Port map for phase error generator.
  --------------------------------------
  error_gen_1: error_gen
  generic map (
    datasize_g  => dsize_g+1,
    errorsize_g => esize_g
  )
  port map (
      -- clock and reset.
      clk                 => clk,
      reset_n             => reset_n,
      --
      symbol_sync         => symbol_sync, -- Symbol synchronization pulse.
      data_i              => data_i_ext,  -- Real data in.
      data_q              => data_q_ext,  -- Imaginary data in.
      demap_data          => demap_data,  -- Data from demapping.
      enable_error        => enable_error,
      --
      phase_error         => phase_error, -- Phase error.
      error_cart_i_o      => error_cart_i_o, 
      error_cart_q_o      => error_cart_q_o,    
      error_ready         => error_ready  -- Error ready.
      );
  
  -- Phase error output
  phase_error_o       <= phase_error;
  phase_error_ready_o <= error_ready;
  
  --------------------------------------
  -- Port map for filter.
  --------------------------------------
  filter_1: filter
  generic map (
    esize_g     => esize_g,     -- size of error (must >= dsize_g).
    phisize_g   => phisize_g,   -- size of angle phi
    omegasize_g => omegasize_g, -- size of angle omega
    sigmasize_g => sigmasize_g, -- size of angle sigma
    tausize_g   => tausize_g    -- size of tau
  )
  port map (
      -- clock and reset.
      clk                  => clk,
      reset_n              => reset_n,
      --
      load                 => error_ready, -- Filter synchronization.
      precomp_enable       => precomp_enable,  -- Precompensation enable
      interpolation_enable => interpolation_enable, -- Interpolation enable  
      enable_error         => enable_error,-- Enable the compensation. 
      symbol_sync          => symbol_sync, -- Symbol synchronization.
      mod_type             => mod_type,    -- Modulation type (DSSS or CCK).
      phase_error          => phase_error, -- Phase error.
      rho                  => rho,         -- rho parameter value.
      mu                   => mu,          -- mu parameter value.
      --                   
      freqoffestim_stat    => freqoffestim_stat,               
      phi                  => phi,         -- phi angle.
      sigma                => sigma,       -- theta angle.
      omega                => omega,       -- omega angle.
      tau                  => tau          -- tau.
      );

  ------------------------------------------------------------------------------
  -- Global Signals for test
  ------------------------------------------------------------------------------
-- ambit synthesis off
-- synopsys translate_off
-- synthesis translate_off 
  phase_error_tglobal(31 downto dsize_g-1) <= (others => phase_error(dsize_g-1));
  phase_error_tglobal(dsize_g-1 downto 0) <= phase_error;
  phase_error_gbl <= phase_error;
-- ambit synthesis on
-- synopsys translate_on
-- synthesis translate_on
end RTL;

--------------------------------------------------------------------------------
-- End of file
--------------------------------------------------------------------------------

